#
# dt - DITrack command-line client globals
#
# Copyright (c) 2006-2007 The DITrack Project, www.ditrack.org.
#
# $Id: globals.py 2329 2007-11-06 19:34:08Z vss $
# $HeadURL: https://127.0.0.1/ditrack/src/tags/0.7/DITrack/dt/globals.py $
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#  * Redistributions of source code must retain the above copyright notice,
# this list of conditions and the following disclaimer.
#  * Redistributions in binary form must reproduce the above copyright notice,
# this list of conditions and the following disclaimer in the documentation
# and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
#

import datetime
import os.path
import stat
import sys
import time

import DITrack.Util.common

VERSION = "0.7"

class Globals:
    def __init__(self):

        # Our version.
        self.version = VERSION
        self.dt_title = "DITrack command-line client, version %s" % \
            self.version

        # Figure out the binary name.
        self.binname = os.path.basename(sys.argv[0])

        # Username can be defined with get_username()
        self.username = None

        # Timezone string.
        if time.daylight:
            tz = time.altzone
        else:
            tz = time.timezone

        if tz > 0:
            offs = "-"
        elif tz == 0:
            offs = " "
        else:
            offs = "+"
            tz = -tz

        self.timezone_string = offs + \
            ("%02d%02d" % (tz / 3600, (tz / 60) % 60))

        # Path to svn executable.
        if os.name == "posix":
            self.svn_path = "svn"
        elif os.name == "nt":
            self.svn_path = "svn.exe"

        # Editor is not necessarily required for any command.
        self.editor = ""

        self.text_delimiter = "=" * 78

    def fmt_timestamp(self):
        """
        Returns the current time string formatted.
        """

        weekdays = ("Mon", "Tue", "Wed", "Thu", "Fri", "Sat", "Sun")

        now = datetime.datetime.now()

        return now.strftime("%s %Y-%m-%d %H:%M:%S ") + \
            weekdays[now.weekday()] + " " + self.timezone_string

    def get_editor(self):
        "Figure out user editor; bail out if something goes wrong."

        if len(self.editor):
            return

        notdefined = "Text editor is not defined (use EDITOR "  \
            "environment variable)"

        if not os.environ.has_key("EDITOR"):
            DITrack.Util.common.err(notdefined)

        self.editor = os.environ["EDITOR"]

        msg = "Text editor '" + self.editor + "' (set with "    \
            "EDITOR environment variable) "

        if len(self.editor) == 0:
            DITrack.Util.common.err(notdefined)

        if self.editor[0] == "/":

            if not os.path.exists(self.editor):
                DITrack.Util.common.err(msg + "doesn't exist")

            if not os.path.isfile(self.editor):
                DITrack.Util.common.err(msg + "is not a file")

            st = os.stat(self.editor)
            if not (st[stat.ST_MODE] & stat.S_IEXEC):
                DITrack.Util.common.err(msg + "is not executable")

    def get_username(self, opts):
        """
        Figure out the user name, using the environment and passed options
        OPTS.
        """

        name = None

        if (opts) and ("user" in opts.var):
            name = opts.var["user"]
        elif "DITRACK_USER" in os.environ:
            name = os.environ["DITRACK_USER"]
        elif "USER" in os.environ:
            name = os.environ["USER"]
        self.username = name
