#
# Configuration.py - database configuration
#
# Copyright (c) 2006-2007 The DITrack Project, www.ditrack.org.
#
# $Id: Configuration.py 1846 2007-08-04 11:25:09Z gli $
# $HeadURL: https://127.0.0.1/ditrack/src/tags/0.7/DITrack/DB/Configuration.py $
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#  * Redistributions of source code must retain the above copyright notice,
# this list of conditions and the following disclaimer.
#  * Redistributions in binary form must reproduce the above copyright notice,
# this list of conditions and the following disclaimer in the documentation
# and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
#

import email
import email.Message
import os.path

import ConfigParser

# DITrack modules
import DITrack.DB.Common
import DITrack.DB.Exceptions

#
# Classes
#
class VersionSet:
    """
    A representation of a single version set.
    """

    def __init__(self, past, current, future):
        self.past = past
        self.current = current
        self.future = future

class VersionCfg:
    """
    A representation of version sets configuration.
    """

    def __contains__(self, key):
        return self.items.has_key(key)

    def __getitem__(self, key):
        return self.items[key]

    def __init__(self, path):
        f = open(path)
        versions = email.message_from_file(f)
        f.close()

        if len(versions.get_payload()):
            raise DITrack.DB.Exceptions.CorruptedDB_UnparseableVersionsError(
                "Empty line in version set configuration file")

        self.items = {}
        for s in versions.keys():

            if s in self.items:
                raise DITrack.DB.Exceptions.CorruptedDB_UnparseableVersionsError(
                    "Duplicat version set '" + s + "' definition")

            v = versions[s].strip().split("/")
            if len(v) != 3:
                raise DITrack.DB.Exceptions.CorruptedDB_UnparseableVersionsError(
                    "Invalid version set '" + s + "' definition")

            self.items[s] = VersionSet(
                v[0].strip().split(),
                v[1].strip().split(),
                v[2].strip().split()
                )

class Category:
    def __init__(self, version_set, versions, default_owner, enabled):
        """
        VERSION_SET
            Version set name.

        XXX: do we really need to pass both version_set and versions?
        """

        self.version_set = version_set
        self.versions = versions
        self.default_owner = default_owner
        self.enabled = enabled

class CategoryCfg(object):
    """
    Representation of per-database category configuration.
    """

    def __getitem__(self, key):
        return self.items[key]

    def __init__(self, cfgfile, users, versions):
        """
        Parse category configuration file 'cfgfile'.

        Defined versions and users are passed in 'versions' and 'users'
        parameters respectively. They are used only for consistency checks and
        are not stored anywhere inside the object.
        """

        f = open(cfgfile)
        sections = f.read().split("\n\n")
        f.close()

        category = {}
        for s in sections:
            if not len(s):
                continue

            c = email.message_from_string(s)

            if "Category" not in c:
                raise DITrack.DB.Exceptions.CorruptedDB_UnparseableCategoriesError(
                    "Invalid category definition (no 'Category' field)"
                    )

            if not len(c["Category"]):
                raise DITrack.DB.Exceptions.CorruptedDB_UnparseableCategoriesError(
                    "Invalid category definition (blank 'Category' field)"
                    )

            if " " in c["Category"]:
                raise DITrack.DB.Exceptions.CorruptedDB_UnparseableCategoriesError(
                    "Invalid category '" + c["Category"] + "' definition: "
                    "blank characters in name"
                    )


            if c["Category"] in category:
                raise DITrack.DB.Exceptions.CorruptedDB_UnparseableCategoriesError(
                    "Duplicate category '" + c["Category"] + "' definition")

            if "Version-set" not in c:
                raise DITrack.DB.Exceptions.CorruptedDB_UnparseableCategoriesError(
                    "Invalid category '" + c["Category"] + "' definition: "
                    "no version set defined"
                    )

            if c["Version-set"] not in versions:
                raise DITrack.DB.Exceptions.CorruptedDB_UnparseableCategoriesError(
                    "Invalid category '" + c["Category"] + "' definition: "
                    "unknown version set"
                    )

            if "Default-owner" not in c:
                raise DITrack.DB.Exceptions.CorruptedDB_UnparseableCategoriesError(
                    "Invalid category '" + c["Category"] + "' definition: "
                    "no default owner defined"
                    )

            if c["Default-owner"] not in users:
                raise DITrack.DB.Exceptions.CorruptedDB_UnparseableCategoriesError(
                    "Invalid category '" + c["Category"] + "' definition: "
                    "unknown user assigned as default owner"
                    )

            if c.has_key("Status"):
                status = c["Status"].strip()
                if status == "enabled":
                    enabled = True
                elif status == "disabled":
                    enabled = False
                else:
                    raise DITrack.DB.Exceptions.CorruptedDB_UnparseableCategoriesError(
                        "Invalid category '" + c["Category"] + "' definition: "
                        "invalid status"
                        )
            else:
                enabled = True

            category[c["Category"]] = Category(
                c["Version-set"],
                versions[c["Version-set"]],
                c["Default-owner"],
                enabled
                )

        self.items = category

    def __iter__(self):
        return self.items.__iter__()

    def __len__(self):
        return len(self.items)

    def keys(self):
        return self.items.keys()

class ListingFormatCfg:
    """
    A representation of version sets configuration.
    """

    def __contains__(self, key):
        return self.items.has_key(key)

    def __getitem__(self, key):
        try:
            return self.items[key]
        except KeyError:
            raise DITrack.DB.Exceptions.InvalidListingFormatError(key)

    def __init__(self, path):
        cfg = ConfigParser.ConfigParser()
        try:
            cfg.read(path)
        except ConfigParser.MissingSectionHeaderError:
            # XXX: Remove mention about listing format when single
            # configuration file will be used
            raise DITrack.DB.Exceptions.CorruptedDB_UnparseableListingFormatError(
                "ERROR: Incorrect configuration file %s:\n"
                "No section [listing formats] found" % path)

        if not cfg.has_section("listing-formats"):
            raise DITrack.DB.Exceptions.CorruptedDB_UnparseableListingFormatError(
                "ERROR: Incorrect configuration file %s:\n"
                "ERROR: No section [listing formats] found" % path)

        self.items = {}
        for key, value in cfg.items("listing-formats", True):
            self.items[key] = value;

class UserCfg:
    """
    A representation of user accounts configuration.
    """

    def __contains__(self, key):
        return self.items.has_key(key)

    def __init__(self, fname):
        """
        Parses user accounts configuration file 'fname'.
        """

        users = {}
        f = open(fname)
        while 1:
            str = f.readline()
            if not str: break

            user = str.strip()

            if user in users:
                raise DITrack.DB.Exceptions.CorruptedDB_DuplicateUserError(
                    "Duplicate user entry '" + user + "' in" " '" + fname + "'")

            users[user] = user

        f.close()

        self.items = users

    def __len__(self):
        return len(self.items)

    def keys(self):
        return self.items.keys()

    has_key = __contains__


class FilterCfg:
    """
    A representation of filters configuration.
    """

    def __contains__(self, key):
        return self.items.has_key(key)

    def __getitem__(self, key):
        return self.items[key]

    def __init__(self, path, username):
        f = open(path)
        filters = email.message_from_file(f)
        f.close()

        if len(filters.get_payload()):
            raise DITrack.DB.Exceptions.CorruptedDB_UnparseableFiltersError(
                "Empty line in filter configuration file")

        self.items = {}
        for s in filters.keys():

            if s in self.items:
                raise DITrack.DB.Exceptions.CorruptedDB_UnparseableFiltersError(
                    "Duplicate filter " + s + "' definition")

            filter = filters[s].strip()

            if not filter:
                raise DITrack.DB.Exceptions.CorruptedDB_UnparseableFiltersError(
                    "Empty filter " + s + "' definition")

            try:
                self.items[s] = DITrack.DB.Common.Filter(filter, username)
            except (DITrack.DB.Exceptions.FilterIsPredefinedError,
                    DITrack.DB.Exceptions.FilterExpressionError):
                raise DITrack.DB.Exceptions.CorruptedDB_UnparseableFiltersError(
                    "Syntax error in filter " + s + "' definition")

    def __len__(self):
        return len(self.items)


    def keys(self):
        return self.items.keys()

    has_key = __contains__

class Configuration:
    """
    Database configuration object (everything under /etc in a database).

    Exported:

    category
        Categories configuration object, CategoryCfg.

    filters
        Filters configuration object, FilterCfg.

    path
        A mapping of strings identifying various objects to their respective
        locations. The keys currently supported are:

        /
            Root of the issue database.

        categories
            Location of categories configuration file.

        data
            Root of the issue data.

        filters
            Location of predefined filters configuration file.

        users
            Location of user accounts configuration file.

        version
            Location of versions configuration file.

    users
        User accounts configuration object, UserCfg.

    versions
        Version sets configuration object, VersionCfg.
    """

    def __init__(self, path, username):

        # Prepare pathnames mapping.
        self.path = {}
        self.path["/"] = path
        self.path["categories"] = os.path.join(path, "etc", "categories")
        self.path["data"] = os.path.join(path, "data")
        self.path["filters"] = os.path.join(path, "etc", "filters")
        self.path["users"] = os.path.join(path, "etc", "users")
        self.path["versions"] = os.path.join(path, "etc", "versions")
        self.path["listing_format"] = os.path.join(path, "etc", 
                "listing-format")

        self.users = UserCfg(self.path["users"])

        self.versions = VersionCfg(self.path["versions"])

        self.category = CategoryCfg(self.path["categories"], self.users,
            self.versions)

        self.listing_format = ListingFormatCfg(self.path["listing_format"])

        try:
            self.filters = FilterCfg(self.path["filters"], username)
        except DITrack.DB.Exceptions.CorruptedDB_UnparseableFiltersError:
            raise
