/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "protocolmanager.h"
#include "protocolmanageradaptor.h"

#include <QtCore/QSettings>

#include <QtTapioca/ConnectionManagerFactory>
#include <QtTapioca/ConnectionManager>

namespace
{
    static const QString protocolmanager_group("ProtocolManager");
} // namespace

/// @cond INCLUDE_PRIVATE

/**
 * @brief Private class implementing the ProtocolManager.
 *
 * A private class implementing the ProtocolManager.
 *
 * @author Tobias Hunger <info@basyskom.de>
 */
class ProtocolManagerPrivate
{
public:
    /** @brief Constructor. */
    ProtocolManagerPrivate()
    { m_settings.beginGroup(protocolmanager_group); }

    /** @brief Destructor. */
    ~ProtocolManagerPrivate()
    { m_settings.endGroup(); }

    /** @brief The configuration settings object. */
    QSettings m_settings;
    /** @brief A pointer to the ProtocolManager's D-Bus adaptor. */
    ProtocolManagerAdaptor * adaptor;
};

/// @endcond

// ****************************************************************************

ProtocolManager::ProtocolManager(QObject * parent) :
    QObject(parent),
    d(new ProtocolManagerPrivate())
{
    Q_ASSERT(0 != d);
    d->adaptor = new ProtocolManagerAdaptor(this);
    Q_ASSERT(0 != d->adaptor);
}

ProtocolManager::~ProtocolManager()
{
    delete d;
}

QStringList ProtocolManager::allConnectionManagers() const
{
    QStringList results;
    QList<QtTapioca::ConnectionManager*> cms =
        QtTapioca::ConnectionManagerFactory::self()->
            getAllConnectionManagers();
    QtTapioca::ConnectionManager* cm;
    foreach(cm, cms)
    { results.append(cm->name()); }
    return results;
}

QStringList ProtocolManager::supportedProtocols() const
{
    QStringList results;
    QList<QtTapioca::ConnectionManager*> cms =
        QtTapioca::ConnectionManagerFactory::self()->
            getAllConnectionManagers();
    QtTapioca::ConnectionManager* cm;
    foreach(cm, cms)
    {
        QStringList protocols = cm->supportedProtocols();
        QString protocol;
        foreach(protocol, protocols)
        {
            if (results.indexOf(protocol) < 0)
            { results.append(protocol); }
        }
    }
    return results;
}

QStringList ProtocolManager::connectionManagersForProtocol(const QString &protocol) const
{
    QStringList results;
    QList<QtTapioca::ConnectionManager*> cms =
        QtTapioca::ConnectionManagerFactory::self()->
            getAllConnectionManagers();
    QtTapioca::ConnectionManager* cm;
    foreach(cm, cms)
    {
        if (cm->supportedProtocols().indexOf(protocol) >= 0)
        { results.append(cm->name()); }
    }
    return results;
}

bool ProtocolManager::setDefaultConnectionManagerForProtocol(const QString &protocol,
                                                             const QString &cm_name)
{
    QtTapioca::ConnectionManager* cm =
        QtTapioca::ConnectionManagerFactory::self()->
            getConnectionManagerByName(cm_name);
    if (cm == 0)
    {
        return false; /* requested CM is not installed. */
    }

    Q_ASSERT(cm != 0);
    if (cm->supportedProtocols().indexOf(protocol) < 0)
    { return false; /* protocol is unknown */ }

     d->m_settings.setValue(protocol, cm_name);
    return true;
}

QString ProtocolManager::defaultConnectionManagerForProtocol(const QString &protocol) const
{
    QString result = d->m_settings.value(protocol, "").toString();
    QStringList cms = connectionManagersForProtocol(protocol);
    cms << "";

    Q_ASSERT(!cms.isEmpty()); // there is at least the "" entry we just added.
    Q_ASSERT(cms.last() == ""); // invalid entry is last

    // Do we have no default?
    if (result.isEmpty())
    {
        return cms[0]; /* [0] exists and is "" if list was empty. */
    }

    if (cms.indexOf(result) < 0)
    {
        return cms[0]; // [0] exists and is "" if list was empty.
    }

    Q_ASSERT( cms.indexOf(result ) >= 0 );
    return result;
}
