# -*- coding: utf-8 -*-
#
# Author: Ingelrest François (Athropos@gmail.com)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

import asf, flac, mp3, mp4, mpc, ogg, os, playlist, tools, traceback

from tools.log    import logger
from tools.consts import UNKNOWN_TRACKNUMBER, UNKNOWN_TITLE, UNKNOWN_ARTIST, UNKNOWN_ALBUM, UNKNOWN_LENGTH


# The format of the list returned by the readInfo() and getTracks() functions
(
    NFO_NUM,  # Track number
    NFO_TIT,  # Title
    NFO_ART,  # Artist
    NFO_ALB,  # Album
    NFO_LEN,  # Length in seconds
    NFO_FIL   # Full path to the file
) = range(6)


# Supported formats with associated modules
mFormats = { '.flac': flac, '.m4a': mp4, '.mp3': mp3, '.mp4': mp4, '.mpc': mpc,'.ogg': ogg, '.wma': asf}


def isSupported(file):
    """ Return True if the given file is a supported format """
    return os.path.splitext(file)[1].lower() in mFormats


def getSupportedFormats():
    """ Return a list of all formats from which tags can be extracted """
    return ['*' + ext for ext in mFormats.iterkeys()]


def readInfo(files):
    """
        Given a list of files, return a list such as:
        [[TRACKNUMBER1, TITLE1, ARTIST1, ALBUM1, LENGTH1, FILE1], [TRACKNUMBER2, TITLE2, ARTIST2, ALBUM2, LENGTH2, FILE2], ...]
    """
    info = []
    for file in files:
        try:
            module = mFormats[os.path.splitext(file)[1].lower()]
            info.append(module.readInfo(file))
        except:
            logger.error('Unable to extract information from %s\n\n%s' % (file, traceback.format_exc()))
            info.append([UNKNOWN_TRACKNUMBER, UNKNOWN_TITLE, UNKNOWN_ARTIST, UNKNOWN_ALBUM, UNKNOWN_LENGTH, file])

    return info


def getTracks(filenames):
    """
        Return all the tracks contained in the list of files:
            * For directory, load all media files (playable files and playlists)
            * For a playlist, load its tracks
        The content of playlists is never sorted
    """
    allTracks = []

    # Directories
    for directory in [filename for filename in filenames if os.path.isdir(filename)]:
        files, playlists = [], []
        for (filename, path) in tools.listDir(directory):
            if os.path.isfile(path):
                if isSupported(filename):            files.append(path)
                elif playlist.isSupported(filename): playlists.append(path)

        allTracks.extend(sorted(readInfo(files), key=lambda track: track[NFO_NUM]))

        for pl in playlists:
            allTracks.extend(readInfo(playlist.load(pl)))

    # Files
    tracks = readInfo([filename for filename in filenames if os.path.isfile(filename) and isSupported(filename)])
    allTracks.extend(sorted(tracks, key=lambda track: track[NFO_NUM]))

    # Playlists
    for pl in [filename for filename in filenames if os.path.isfile(filename) and playlist.isSupported(filename)]:
        allTracks.extend(readInfo(playlist.load(pl)))

    return allTracks
