# Written by Bram Cohen
# Modified by Cameron Dale
# see LICENSE.txt for license information
#
# $Id: StorageWrapper.py 266 2007-08-18 02:06:35Z camrdale-guest $

"""Manage the storage of data at the piece level.

@type logger: C{logging.Logger}
@var logger: the logger to send all log messages to for this module
@type STATS_INTERVAL: C{float}
@var STATS_INTERVAL: how often to print a status update during an old-style init

"""

from DebTorrent.bitfield import Bitfield
from sha import sha
from DebTorrent.clock import clock
from random import randrange
import logging
from bisect import insort

logger = logging.getLogger('DebTorrent.BT1.StorageWrapper')

STATS_INTERVAL = 0.2

def dummy_status(fractionDone = None, activity = None):
    """Dummy function to do nothing with the status info."""
    pass

class Olist:
    """A list that will be popped in order.
    
    @type d: C{dictionary}
    @ivar d: the ordered list, keys are the list entries
    
    """
    
    def __init__(self, l = []):
        """Initialize the instance.
        
        @type l: C{list}
        @param l: the list to initialize the instance with
            (optional, defaults to an empty initialization)
        
        """
        
        self.d = {}
        for i in l:
            self.d[i] = 1

    def __len__(self):
        """Get the length of the list.
        
        @rtype: C{int}
        @return: the length of the list
        
        """
        
        return len(self.d)

    def includes(self, i):
        """Check if the list includes a value.
        
        @type i: unknown
        @param i: the value to check the list for
        @rtype: C{boolean}
        @return: whether the list already includes the value
        
        """
        
        return self.d.has_key(i)

    def add(self, i):
        """Add a value to the list.
        
        @type i: unknown
        @param i: the value to add to the list
        
        """
        
        self.d[i] = 1

    def extend(self, l):
        """Extend the list with multiple new values.
        
        @type l: C{list}
        @param l: the list of values to add to the list
        
        """
        
        for i in l:
            self.d[i] = 1

    def pop(self, n=0):
        """Remove a value from the list and return it.
        
        @type n: C{int}
        @param n: the list item to pop::
            -1 -- the smallest item is popped
             0 -- the largest item is popped (default)
             n -- the nth item in the list is popped
        @rtype: unknown
        @return: the value popped from the list
        
        """
        
        # assert self.d
        k = self.d.keys()
        if n == 0:
            i = min(k)
        elif n == -1:
            i = max(k)
        else:
            k.sort()
            i = k[n]
        del self.d[i]
        return i

    def remove(self, i):
        """Remove a value from the list.
        
        @type i: unknown
        @param i: the value to remove from the list
        
        """
        
        if self.d.has_key(i):
            del self.d[i]

class fakeflag:
    """A fake flag to use if one is not supplied.
    
    @type state: C{boolean}
    @ivar state: the current state of the flag
    
    """
    
    def __init__(self, state=False):
        """Set the new flag.
        
        @type state: C{boolean}
        @param state: the initial state of the flag
        
        """
        
        self.state = state
        
    def wait(self):
        """Do nothing."""
        pass

    def isSet(self):
        """Check if the current state is set.
        
        @rtype: C{boolean}
        @return: whether the flag is set
        
        """
        
        return self.state


class StorageWrapper:
    """Manage the storage of data at the piece level.
    
    @type storage: L{BT1.Storage.Storage}
    @ivar storage: the file storage instance
    @type request_size: C{int}
    @ivar request_size: the number of bytes to query for per request
    @type hashes: C{list} of C{string}
    @ivar hashes: the hashes of the pieces
    @type piece_sizes: C{list} of C{int}
    @ivar piece_sizes: the lengths of the pieces
    @type piece_lengths: C{list} of C{int}
    @ivar piece_lengths: the lengths of the pieces
    @type piece_begins: C{list} of C{long}
    @ivar piece_begins: the beginning offset of the pieces in the total download
    @type datalength: C{long}
    @ivar datalength: the total length of the download
    @type finished: C{method}
    @ivar finished: method to call when the download is complete
    @type failed: C{method}
    @ivar failed: method to call when a critical failure occurs
    @type statusfunc: C{method}
    @ivar statusfunc: the method to use to diplay status updates
    @type flag: C{threading.Event}
    @ivar flag: the flag that indicates when the program is to be shutdown
    @type check_hashes: C{boolean}
    @ivar check_hashes: whether to re-check hashes of data already
        downloaded, during startup
    @type data_flunked: C{method}
    @ivar data_flunked: method to call when a hash check fails for a piece
    @type backfunc: C{method}
    @ivar backfunc: method to call to schedule future invocation of requester functions
    @type config: C{dictionary}
    @ivar config: the configuration parameters
    @type unpauseflag: C{threading.Event}
    @ivar unpauseflag: the flag to unset to pause the download
    @type alloc_type: C{string}
    @ivar alloc_type: the type of allocation to do, always 'pre-allocate'
    @type double_check: C{boolean}
    @ivar double_check: whether to double-check data being written to the disk for errors
    @type triple_check: C{boolean}
    @ivar triple_check: whether to thoroughly check data being written to the disk
    @type bgalloc_enabled: C{boolean}
    @ivar bgalloc_enabled: whether the allocation of all pieces has been enabled
    @type bgalloc_active: C{boolean}
    @ivar bgalloc_active: whether background allocation is in process
    @type total_length: C{long}
    @ivar total_length: the total length of the download
    @type amount_left: C{long}
    @ivar amount_left: the amount of the total download that is still left to download
    @type numactive: C{list} of C{int}
    @ivar numactive: the number of active requests pending for a piece
    @type inactive_requests: C{list} of C{int} or C{list} of (C{int}, C{int})
    @ivar inactive_requests: for each piece, the inactive requests remaining
        for the piece. Will be 1 if the requests have not yet been generated,
        None if thee piece has already been downloaded, or a list of offsets
        within the piece and request lengths.
    @type amount_inactive: C{length}
    @ivar amount_inactive: the amount of data that is still in inactive requests
    @type amount_obtained: C{long}
    @ivar amount_obtained: the amount of the desired download that has been received
    @type amount_desired: C{long}
    @ivar amount_desired: the amount of the total download that is desired
    @type have: L{DebTorrent.bitfield.Bitfield}
    @ivar have: the bitfield that this peer has
    @type have_cloaked_data: (C{string}, C{list} of C{int})
    @ivar have_cloaked_data: the cached incomplete bitfiled as a binary string,
        and the haves needed to complete it, or None if it has not yet been generated
    @type blocked: C{list} of C{boolean}
    @ivar blocked: whether the piece location is blocked
    @type blocked_holes: C{list} of C{int}
    @ivar blocked_holes: the holes that have been blocked
    @type blocked_movein: L{Olist}
    @ivar blocked_movein: the piece locations that are currently blocking a
        piece from moving into its proper place
    @type blocked_moveout: L{Olist}
    @ivar blocked_moveout: the pieces that are currently blocked from moving
        into their proper place
    @type waschecked: C{list} of C{boolean}
    @ivar waschecked: whether the piece has been hash checked
    @type places: C{dictionary}
    @ivar places: keys are the pieces, values are their current piece locations
    @type holes: C{list} of C{int}
    @ivar holes: the piece locations that are currently empty and unallocated
    @type stat_active: C{dictionary}
    @ivar stat_active: keys are the pieces that currently have active requests
    @type stat_new: C{dictionary}
    @ivar stat_new: keys are the pieces that have a newly sent request but
        have not downloaded anything
    @type dirty: C{dictionary}
    @ivar dirty: keys are the pieces that have received partial downloads,
        values are a list of the offset within the piece and length of the
        partial downloads received
    @type stat_numflunked: C{int}
    @ivar stat_numflunked: the number of downloaded pieces that have failed a hash check
    @type stat_numdownloaded: C{int}
    @ivar stat_numdownloaded: total number of pieces that have successfully downloaded
    @type stat_numfound: C{int}
    @ivar stat_numfound: number of pieces that were found from a previous download
    @type download_history: C{dictionary}
    @ivar download_history: keys are downloaded pieces, values are dictionaries
        with keys of the request offset within the piece and values of the
        source of the download
    @type failed_pieces: C{dictionary}
    @ivar failed_pieces: keys are downloaded pieces, values are dictionaries
        with keys of the source of the download
    @type out_of_place: C{int}
    @ivar out_of_place: for sparse allocation, the number of pieces that are
        out of place during initialization
    @type write_buf_max: C{long}
    @ivar write_buf_max: the maximum number of bytes to store in the piece buffer
    @type write_buf_size: C{long}
    @ivar write_buf_size: the current number of bytes stored in the piece buffer
    @type write_buf: C{dictionary}
    @ivar write_buf: keys are pieces in the piece buffer, values are lists of
        the offset within the piece and the piece data: [(start, data), ...]
    @type write_buf_list: C{list} of C{int}
    @ivar write_buf_list: the pieces currently in the piece buffer, ordered
        so that the most recently received piece is at the end of the list
    @type initialize_tasks: C{list} of [C{string}, C{int}, C{method}, C{method}]
    @ivar initialize_tasks: the initialization tasks to perform, the status message,
        initial fraction done, method to call to initialize the task, and the task
    @type initialize_done: C{method}
    @ivar initialize_done: method to call when the background initialization is complete
    @type initialize_next: C{method}
    @ivar initialize_next: current method to call in the initialization tasks
    @type check_list: C{list} of C{int}
    @ivar check_list: the pieces needing hash checking during initialization
    @type check_total: C{int}
    @ivar check_total: the number of pieces needing hash checking during initialization
    @type check_targets: C{dictionary}
    @ivar check_targets: keys are piece hashes, values are a list of potential pieces
    @type check_numchecked: C{float}
    @ivar check_numchecked: not used
    @type numchecked: C{float}
    @ivar numchecked: the number of pieces that have been hash checked during initialization
    @type movelist: C{list} of C{int}
    @ivar movelist: for sparse allocation, the pieces that need to be moved
        during initialization
    @type tomove: C{float}
    @ivar tomove: for sparse allocation, the numebr of pieces that still need
        to be moved during initialization
    @type numholes: C{int}
    @ivar numholes: the number of holes that need to be filled by allocation
        during initialization
    @type alloc_buf: C{string}
    @ivar alloc_buf: dummy data to write to allocated but not downloaded
        piece locations

    @group Initialization: __init__, _bgsync, old_style_init, initialize,
        _initialize, init_hashcheck, _markgot, hashcheckfunc, init_movedata,
        movedatafunc, init_alloc, _allocfunc, allocfunc, bgalloc, _bgalloc
    @group Information: _waspre, _piecelen, get_amount_left, do_I_have_anything,
        am_I_complete, get_have_list, get_have_list_cloaked, do_I_have,
        get_hash, get_stats
    @group Requests: _make_inactive, new_request, request_lost,
        do_I_have_requests, is_unstarted, is_endgame, reset_endgame
    @group Read/Write: write_raw, _write_to_buffer, _flush_buffer, sync,
        _move_piece, _clear_space, piece_came_in, get_piece, read_raw,
        set_file_readonly
    @group Resuming: has_data, doublecheck_data, reblock, pickle, unpickle
    
    """
    
    def __init__(self, storage, request_size, hashes, 
            piece_sizes, datalength, finished, failed, 
            statusfunc = dummy_status, flag = fakeflag(), check_hashes = True,
            data_flunked = lambda x: None, backfunc = None,
            config = {}, unpauseflag = fakeflag(True) ):
        """Initialize the instance and begin the background initialization tasks.
        
        @type storage: L{BT1.Storage.Storage}
        @param storage: the file storage instance
        @type request_size: C{int}
        @param request_size: the number of bytes to query for per request
        @type hashes: C{list} of C{string}
        @param hashes: the hashes of the pieces
        @type piece_sizes: C{list} of C{int}
        @param piece_sizes: the lengths of the pieces
        @type datalength: C{long}
        @param datalength: the total length of the download
        @type finished: C{method}
        @param finished: method to call when the download is complete
        @type failed: C{method}
        @param failed: method to call when a critical failure occurs
        @type statusfunc: C{method}
        @param statusfunc: the method to use to diplay status updates
            (optional, defaults to not displaying status updates)
        @type flag: C{threading.Event}
        @param flag: the flag that indicates when the program is to be shutdown
            (optional, defaults to using a dummy flag)
        @type check_hashes: C{boolean}
        @param check_hashes: whether to re-check hashes of data already
            downloaded, during startup (optional, defaults to True)
        @type data_flunked: C{method}
        @param data_flunked: method to call when a hash check fails for a piece
            (optional, defaults to doing nothing)
        @type backfunc: C{method}
        @param backfunc: method to call to schedule future invocation of requester functions
        @type config: C{dictionary}
        @param config: the configuration parameters
        @type unpauseflag: C{threading.Event}
        @param unpauseflag: the flag to unset to pause the download
            (optional, defaults to using a dummy flag)
        
        """
        
        self.storage = storage
        self.request_size = long(request_size)
        self.hashes = hashes
        self.piece_sizes = piece_sizes
        self.piece_lengths = piece_sizes
        self.piece_begins = [0l]*len(piece_sizes)
        for i in xrange(1,len(piece_sizes)):
            self.piece_begins[i] = self.piece_begins[i-1] + self.piece_sizes[i-1]
        self.datalength = datalength
        self.finished = finished
        self.failed = failed
        self.statusfunc = statusfunc
        self.flag = flag
        self.check_hashes = check_hashes
        self.data_flunked = data_flunked
        self.backfunc = backfunc
        self.config = config
        self.unpauseflag = unpauseflag
        
        self.alloc_type = config.get('alloc_type','pre-allocate')
        self.double_check = config.get('double_check', 0)
        self.triple_check = config.get('triple_check', 0)
        if self.triple_check:
            self.double_check = True
        self.bgalloc_enabled = False
        self.bgalloc_active = False
        self.total_length = storage.get_total_length()
        self.amount_left = self.total_length
        if self.total_length <= self.datalength - self.piece_sizes[-1]:
            raise ValueError, 'bad data in responsefile - total too small'
        if self.total_length > self.datalength:
            raise ValueError, 'bad data in responsefile - total too big'
        self.numactive = [0] * len(hashes)
        self.inactive_requests = [1] * len(hashes)
        self.amount_inactive = self.total_length
        self.amount_obtained = 0
        self.amount_desired = self.total_length
        self.have = Bitfield(len(hashes))
        self.have_cloaked_data = None
        self.blocked = [False] * len(hashes)
        self.blocked_holes = []
        self.blocked_movein = Olist()
        self.blocked_moveout = Olist()
        self.waschecked = [False] * len(hashes)
        self.places = {}
        self.holes = []
        self.stat_active = {}
        self.stat_new = {}
        self.dirty = {}
        self.stat_numflunked = 0
        self.stat_numdownloaded = 0
        self.stat_numfound = 0
        self.download_history = {}
        self.failed_pieces = {}
        self.out_of_place = 0
        self.write_buf_max = config['write_buffer_size']*1048576L
        self.write_buf_size = 0L
        self.write_buf = {}
        self.write_buf_list = []

        self.initialize_tasks = [
            ['checking existing data', 0, self.init_hashcheck, self.hashcheckfunc],
            ['moving data', 1, self.init_movedata, self.movedatafunc],
            ['allocating disk space', 1, self.init_alloc, self.allocfunc] ]

        self.backfunc(self._bgalloc,0.1)
        self.backfunc(self._bgsync,max(self.config['auto_flush']*60,60))

    def _bgsync(self):
        """Periodically flush the files to disk."""
        if self.config['auto_flush']:
            self.sync()
        self.backfunc(self._bgsync,max(self.config['auto_flush']*60,60))


    def old_style_init(self):
        """Initialize the storage in the foreground.
        
        @rtype: C{boolean}
        @return: whether the init completed
        
        """
        
        while self.initialize_tasks:
            msg, done, init, next = self.initialize_tasks.pop(0)
            if init():
                self.statusfunc(activity = msg, fractionDone = done)
                t = clock() + STATS_INTERVAL
                x = 0
                while x is not None:
                    if t < clock():
                        t = clock() + STATS_INTERVAL
                        self.statusfunc(fractionDone = x)
                    self.unpauseflag.wait()
                    if self.flag.isSet():
                        return False
                    x = next()

        self.statusfunc(fractionDone = 0)
        logger.info('StorageWrapper old_style_init complete')
        return True


    def initialize(self, donefunc):
        """Schedule the background initialization of the storage.
        
        @type donefunc: C{method}
        @param donefunc: method to call when the initialization is complete
        
        """
        
        self.initialize_done = donefunc
        self.initialize_next = None
            
        self.backfunc(self._initialize)

    def _initialize(self):
        """Perform the background initialization of the storage."""
        if not self.unpauseflag.isSet():
            self.backfunc(self._initialize, 1)
            return
        
        if self.initialize_next:
            x = self.initialize_next()
            if x is None:
                self.initialize_next = None
            else:
                self.statusfunc(fractionDone = x)
        else:
            if not self.initialize_tasks:
                self.initialize_done()
                logger.info('StorageWrapper initialize complete')
                return
            msg, done, init, next = self.initialize_tasks.pop(0)
            logger.debug('Starting initialization task: %s', msg)
            if init():
                self.statusfunc(activity = msg, fractionDone = done)
                self.initialize_next = next

        self.backfunc(self._initialize)


    def init_hashcheck(self):
        """Initialize the hash check initialization task.
        
        @rtype: C{boolean}
        @return: whether the task should proceed
        
        """
        
        if self.flag.isSet():
            return False
        self.check_list = []
        if len(self.hashes) == 0 or self.amount_left == 0:
            self.check_total = 0
            self.finished()
            return False

        self.check_targets = {}
        got = {}
        for p,v in self.places.items():
            assert not got.has_key(v)
            got[v] = 1
        for i in xrange(len(self.hashes)):
            if self.places.has_key(i):  # restored from pickled
                self.check_targets[self.hashes[i]] = []
                if self.places[i] == i:
                    continue
                else:
                    assert not got.has_key(i)
                    self.out_of_place += 1
            if got.has_key(i):
                continue
            if self._waspre(i):
                if self.blocked[i]:
                    self.places[i] = i
                else:
                    self.check_list.append(i)
                continue
            if not self.check_hashes:
                self.failed('told file complete on start-up, but data is missing')
                return False
            self.holes.append(i)
            if self.blocked[i] or self.check_targets.has_key(self.hashes[i]):
                self.check_targets[self.hashes[i]] = [] # in case of a hash collision, discard
            else:
                self.check_targets[self.hashes[i]] = [i]
        self.check_total = len(self.check_list)
        self.check_numchecked = 0.0
#        self.lastlen = self._piecelen(len(self.hashes) - 1)
        self.numchecked = 0.0
        return self.check_total > 0

    def _markgot(self, piece, pos):
        """Mark a piece as complete during the initial hash check.
        
        @type piece: C{int}
        @param piece: the piece to mark as complete
        @type pos: C{int}
        @param pos: the place the piece currently occupies
        
        """
        
        logger.debug('Marking '+str(piece)+' at '+str(pos)+' as complete')
        self.places[piece] = pos
        self.have[piece] = True
        len = self._piecelen(piece)
        self.amount_obtained += len
        self.amount_left -= len
        self.amount_inactive -= len
        self.inactive_requests[piece] = None
        self.waschecked[piece] = self.check_hashes
        self.stat_numfound += 1

    def hashcheckfunc(self):
        """Hash check the current data as part of the initialization tasks.
        
        @rtype: C{float}
        @return: the fraction of the task that is complete, or None if the
            task is complete
        
        """
        
        if self.flag.isSet():
            return None
        if not self.check_list:
            return None
        
        i = self.check_list.pop(0)
        if not self.check_hashes:
            self._markgot(i, i)
        else:
            d1 = self.read_raw(i,0,self._piecelen(i))
            if d1 is None:
                return None
            sh = sha(d1[:])
            d1.release()
#            sp = sh.digest()
#            d2 = self.read_raw(i,self.lastlen,self._piecelen(i)-self.lastlen)
#            if d2 is None:
#                return None
#            sh.update(d2[:])
#            d2.release()
            s = sh.digest()
            if s == self.hashes[i]:
                self._markgot(i, i)
#            elif ( self.check_targets.get(s)
#                   and self._piecelen(i) == self._piecelen(self.check_targets[s][-1]) ):
#                self._markgot(self.check_targets[s].pop(), i)
#                self.out_of_place += 1
#            elif ( not self.have[-1] and sp == self.hashes[-1]
#                   and (i == len(self.hashes) - 1
#                        or not self._waspre(len(self.hashes) - 1)) ):
#                self._markgot(len(self.hashes) - 1, i)
#                self.out_of_place += 1
            else:
                self.places[i] = i
        self.numchecked += 1
        if self.amount_left == 0:
            self.finished()
        return (self.numchecked / self.check_total)


    def init_movedata(self):
        """Initialize the move data initialization task (only for sparse allocation).
        
        @rtype: C{boolean}
        @return: whether the task should proceed
        
        """
        
        if self.flag.isSet():
            return False
        if self.alloc_type != 'sparse':
            return False
        self.storage.top_off()  # sets file lengths to their final size
        self.movelist = []
        if self.out_of_place == 0:
            for i in self.holes:
                self.places[i] = i
            self.holes = []
            return False
        self.tomove = float(self.out_of_place)
        for i in xrange(len(self.hashes)):
            if not self.places.has_key(i):
                self.places[i] = i
            elif self.places[i] != i:
                self.movelist.append(i)
        self.holes = []
        return True

    def movedatafunc(self):
        """Move pieces around as part of the initialization tasks.
        
        @rtype: C{float}
        @return: the fraction of the task that is complete, or None if the
            task is complete
        
        """
        
        if self.flag.isSet():
            return None
        if not self.movelist:
            return None
        i = self.movelist.pop(0)
        old = self.read_raw(self.places[i], 0, self._piecelen(i))
        if old is None:
            return None
        if not self.write_raw(i, 0, old):
            return None
        if self.double_check and self.have[i]:
            if self.triple_check:
                old.release()
                old = self.read_raw( i, 0, self._piecelen(i),
                                            flush_first = True )
                if old is None:
                    return None
            if sha(old[:]).digest() != self.hashes[i]:
                self.failed('download corrupted; please restart and resume')
                return None
        old.release()

        self.places[i] = i
        self.tomove -= 1
        return (self.tomove / self.out_of_place)

        
    def init_alloc(self):
        """Initialize the allocation initialization task.
        
        Only runs if the allocation type is pre-allocate, or background and
        there are pieces waiting to move into their proper place.
        
        @rtype: C{boolean}
        @return: whether the task should proceed
        
        """
        
        if self.flag.isSet():
            return False
        if not self.holes:
            return False
        self.numholes = float(len(self.holes))
        self.alloc_buf = chr(0xFF) * max(self.piece_sizes)
        if self.alloc_type == 'pre-allocate':
            self.bgalloc_enabled = True
            return True
        if self.alloc_type == 'background':
            self.bgalloc_enabled = True
        if self.blocked_moveout:
            return True
        return False


    def _allocfunc(self):
        """Find the next location that needs allocating
        
        @rtype: C{int}
        @return: the next piece hole to allocate, or None if there are none
        
        """
        
        while self.holes:
            n = self.holes.pop(0)
            if self.blocked[n]:
                if not self.blocked_movein:
                    self.blocked_holes.append(n)
                    continue
                if not self.places.has_key(n):
                    b = self.blocked_movein.pop(0)
                    oldpos = self._move_piece(b, n)
                    self.places[oldpos] = oldpos
                    return None
            if self.places.has_key(n):
                oldpos = self._move_piece(n, n)
                self.places[oldpos] = oldpos
                return None
            return n
        return None

    def allocfunc(self):
        """Allocate space for pieces.
        
        Allocation is done both as part of the initial startup (see L{init_alloc}),
        and also periodically (see L{_bgalloc}).
        
        @rtype: C{float}
        @return: the fraction of the task that is complete, or None if the
            task is complete
        
        """
        
        if self.flag.isSet():
            return None
        
        if self.blocked_moveout:
            self.bgalloc_active = True
            n = self._allocfunc()
            if n is not None:
                if self.blocked_moveout.includes(n):
                    self.blocked_moveout.remove(n)
                    b = n
                else:
                    b = self.blocked_moveout.pop(0)
                oldpos = self._move_piece(b,n)
                self.places[oldpos] = oldpos
            return len(self.holes) / self.numholes

        if self.holes and self.bgalloc_enabled:
            self.bgalloc_active = True
            n = self._allocfunc()
            if n is not None:
                self.write_raw(n, 0, self.alloc_buf[:self._piecelen(n)])
                self.places[n] = n
            return len(self.holes) / self.numholes

        self.bgalloc_active = False
        return None

    def bgalloc(self):
        """Enable the backround allocation of all pieces.
        
        @rtype: C{boolean}
        @return: False
        
        """
        
        if self.bgalloc_enabled:
            if not self.holes and not self.blocked_moveout and self.backfunc:
                self.backfunc(self.storage.flush)
                # force a flush whenever the "finish allocation" button is hit
        self.bgalloc_enabled = True
        return False

    def _bgalloc(self):
        """Allocate the file storage in the background."""
        self.allocfunc()
        if self.config.get('alloc_rate',0) < 0.1:
            self.config['alloc_rate'] = 2.0
        self.backfunc( self._bgalloc,
              (float(self.datalength)/float(len(self.hashes)))/(self.config['alloc_rate']*1048576) )


    def _waspre(self, piece):
        """Determine if the piece has been pre-allocated.
        
        @type piece: C{int}
        @param piece: the piece to check
        @rtype: C{boolean}
        @return: whether a storage location for the piece has been allocated
        
        """
        
        return self.storage.was_preallocated(self.piece_begins[piece], self._piecelen(piece))

    def _piecelen(self, piece):
        """Get the size of a piece.
        
        @type piece: C{int}
        @param piece: the piece to get the size of
        @rtype: C{int}
        @return: the size of the piece
        
        """
        
        return self.piece_sizes[piece]

    def get_amount_left(self):
        """Determine the amount of the download still remaining.
        
        @rtype: C{long}
        @return: the amount still remaining to download
        
        """
        
        return self.amount_left

    def do_I_have_anything(self):
        """Determine if anything has been downloaded.
        
        @rtype: C{boolean}
        @return: whether anything has been downloaded
        
        """
        
        return self.amount_left < self.total_length

    def _make_inactive(self, index):
        """Create the list of inactive requests for a piece.
        
        @type index: C{int}
        @param index: the piece to make the inactive list for
        
        """
        
        length = self._piecelen(index)
        l = []
        x = 0
        while x + self.request_size < length:
            l.append((x, self.request_size))
            x += self.request_size
        l.append((x, length - x))
        self.inactive_requests[index] = l

    def is_endgame(self):
        """Determine if the download is in end-game mode.
        
        End-game mode occurs when there are pending requests for all the
        chunks of pieces in the download.
        
        @rtype: C{boolean}
        @return: whether the download is in end-game mode
        
        """
        
        return not self.amount_inactive

    def am_I_complete(self):
        """Determine if the download is currently complete.
        
        Complete here means that all the pieces that are desired have been
        downloaded, not that all possible pieces ahve been downloaded.
        
        @rtype: C{boolean}
        @return: whether the download is complete
        
        """
        
        return self.amount_obtained == self.amount_desired

    def reset_endgame(self, requestlist):
        """Reset all lost requests to inactive.
        
        Only occurs as the result of a loss of all peers, or a pause during
        end-game mode.
        
        @type requestlist: C{list} of (C{int}, C{int}, C{int})
        @param requestlist: the requests that were lost, the piece index,
            offset within the piece, and length of the request
        
        """
        
        for index, begin, length in requestlist:
            self.request_lost(index, begin, length)

    def get_have_list(self):
        """Get this peer's bitfield as a string.
        
        @rtype: C{string}
        @return: the have list as a binary string
        
        """
        
        return self.have.tostring()

    def get_have_list_cloaked(self):
        """Get this peer's incomplete bitfield as a string, and a have list.
        
        Random bits are removed from the bitfield and added to the list of haves.
        
        @rtype: (C{string}, C{list} of C{int})
        @return: the incomplete bitfiled as a binary string, and the haves to
            complete it
        
        """
        
        if self.have_cloaked_data is None:
            newhave = Bitfield(copyfrom = self.have)
            unhaves = []
            n = min(randrange(2,5),len(self.hashes))    # between 2-4 unless torrent is small
            while len(unhaves) < n:
                unhave = randrange(min(32,len(self.hashes)))    # all in first 4 bytes
                if not unhave in unhaves:
                    unhaves.append(unhave)
                    newhave[unhave] = False
            self.have_cloaked_data = (newhave.tostring(), unhaves)
        return self.have_cloaked_data

    def do_I_have(self, index):
        """Determine if the piece has been downloaded.
        
        @type index: C{int}
        @param index: the piece to check
        @rtype: C{boolean}
        @return: whether the piece has been downloaded
        
        """
        
        return self.have[index]

    def do_I_have_requests(self, index):
        """Determine if the piece has requests that could be sent out.
        
        @type index: C{int}
        @param index: the piece to check
        @rtype: C{boolean}
        @return: whether the piece has unsent requests to send
        
        """
        
        return not not self.inactive_requests[index]

    def is_unstarted(self, index):
        """Determine if the piece has been started downloading.
        
        @type index: C{int}
        @param index: the piece to check
        @rtype: C{boolean}
        @return: whether the piece has been started
        
        """
        
        return ( not self.have[index] and not self.numactive[index]
                 and not self.dirty.has_key(index) )

    def get_hash(self, index):
        """Get the piece's hash.
        
        @type index: C{int}
        @param index: the piece to get the hash of
        @rtype: C{string}
        @return: the SHA1 hash of the piece
        
        """
        
        return self.hashes[index]

    def get_stats(self):
        """Get some statistics about the download.
        
        @rtype: C{long}, C{long}
        @return: the amount of the download that has been completed, and the
            amount that is still desired to be downloaded
        
        """
        
        return self.amount_obtained, self.amount_desired

    def new_request(self, index):
        """Get a new request for the piece.
        
        @type index: C{int}
        @param index: the piece to get a request for
        @rtype: C{int}, C{int}
        @return: the offset within the piece and length of the request
        
        """
        
        if self.inactive_requests[index] == 1:
            self._make_inactive(index)
        self.numactive[index] += 1
        self.stat_active[index] = 1
        if not self.dirty.has_key(index):
            self.stat_new[index] = 1
        rs = self.inactive_requests[index]
#        r = min(rs)
#        rs.remove(r)
        r = rs.pop(0)
        self.amount_inactive -= r[1]
        return r


    def write_raw(self, index, begin, data):
        """Write some data for the piece to disk.
        
        @type index: C{int}
        @param index: the piece to write to
        @type begin: C{int}
        @param begin: the offset within the piece to start writing at
        @type data: C{string}
        @param data: the data to write
        @rtype: C{boolean}
        @return: whether the write was successful
        
        """
        
        try:
            self.storage.write(self.piece_begins[index] + begin, data)
            return True
        except IOError, e:
            self.failed('IO Error: ' + str(e))
            return False


    def _write_to_buffer(self, piece, start, data):
        """Write some data to the piece buffer.
        
        If the piece buffer has exceeded the maximum, the oldest pieces in it
        are first flushed to disk to make space. If the maximum write buffer
        size is set to 0, the data is immediately written to disk.
        
        @type piece: C{int}
        @param piece: the piece buffer to write to
        @type start: C{int}
        @param start: the offset within the piece to start writing at
        @type data: C{string}
        @param data: the data to write
        @rtype: C{boolean}
        @return: whether the write was successful
        
        """
        
        if not self.write_buf_max:
            return self.write_raw(self.places[piece], start, data)
        self.write_buf_size += len(data)
        while self.write_buf_size > self.write_buf_max:
            old = self.write_buf_list.pop(0)
            if not self._flush_buffer(old, True):
                return False
        if self.write_buf.has_key(piece):
            self.write_buf_list.remove(piece)
        else:
            self.write_buf[piece] = []
        self.write_buf_list.append(piece)
        self.write_buf[piece].append((start,data))
        return True

    def _flush_buffer(self, piece, popped = False):
        """Flush the piece in the buffer to disk.
        
        @type piece: C{int}
        @param piece: the piece buffer to write to disk
        @type popped: C{boolean}
        @param popped: whether the piece was already removed from
            the write_buf_list (optional, defaults to False)
        @rtype: C{boolean}
        @return: whether the flush was successful
        
        """
        
        if not self.write_buf.has_key(piece):
            return True
        if not popped:
            self.write_buf_list.remove(piece)
        l = self.write_buf[piece]
        del self.write_buf[piece]
        l.sort()
        for start, data in l:
            self.write_buf_size -= len(data)
            if not self.write_raw(self.places[piece], start, data):
                return False
        return True

    def sync(self):
        """Flush all the buffered pieces to disk in order by their current place in the download."""
        spots = {}
        for p in self.write_buf_list:
            spots[self.places[p]] = p
        l = spots.keys()
        l.sort()
        for i in l:
            try:
                self._flush_buffer(spots[i])
            except:
                pass
        try:
            self.storage.sync()
        except IOError, e:
            self.failed('IO Error: ' + str(e))
        except OSError, e:
            self.failed('OS Error: ' + str(e))


    def _move_piece(self, index, newpos):
        """Move a piece from it's current location to a new one.
        
        The piece hash may be double checked if this is it's final position,
        or triple checked regardless.
        
        @type index: C{int}
        @param index: the piece to move
        @type newpos: C{int}
        @param newpos: new position to move the piece to
        @rtype: C{int}
        @return: the old position od the piece, or -1 if an error occurred
        
        """
        
        oldpos = self.places[index]
        logger.debug('moving '+str(index)+' from '+str(oldpos)+' to '+str(newpos))
        assert oldpos != index
        assert oldpos != newpos
        assert index == newpos or not self.places.has_key(newpos)
        old = self.read_raw(oldpos, 0, self._piecelen(index))
        if old is None:
            return -1
        if not self.write_raw(newpos, 0, old):
            return -1
        self.places[index] = newpos
        if self.have[index] and (
                self.triple_check or (self.double_check and index == newpos) ):
            if self.triple_check:
                old.release()
                old = self.read_raw(newpos, 0, self._piecelen(index),
                                    flush_first = True)
                if old is None:
                    return -1
            if sha(old[:]).digest() != self.hashes[index]:
                self.failed('download corrupted; please restart and resume')
                return -1
        old.release()

        if self.blocked[index]:
            self.blocked_moveout.remove(index)
            if self.blocked[newpos]:
                self.blocked_movein.remove(index)
            else:
                self.blocked_movein.add(index)
        else:
            self.blocked_movein.remove(index)
            if self.blocked[newpos]:
                self.blocked_moveout.add(index)
            else:
                self.blocked_moveout.remove(index)
                    
        return oldpos
            
    def _clear_space(self, index):
        """Clear a space for a new piece to be written to the storage.
        
        If the proper place is available it will use it, otherwise a new
        place is created. The new place is alos first checked to make
        sure a piece is not waiting to be moved in there.
        
        @type index: C{int}
        @param index: the piece that needs a space
        @rtype: C{boolean}
        @return: whether the function needs to be called again to complete the
            operation
        
        """
        
        h = self.holes.pop(0)
        n = h
        if self.blocked[n]: # assume not self.blocked[index]
            if not self.blocked_movein:
                self.blocked_holes.append(n)
                return True    # repeat
            if not self.places.has_key(n):
                b = self.blocked_movein.pop(0)
                oldpos = self._move_piece(b, n)
                if oldpos < 0:
                    return False
                n = oldpos
        if self.places.has_key(n):
            oldpos = self._move_piece(n, n)
            if oldpos < 0:
                return False
            n = oldpos
        if index == n or index in self.holes:
            if n == h:
                self.write_raw(n, 0, self.alloc_buf[:self._piecelen(n)])
            self.places[index] = n
            if self.blocked[n]:
                # because n may be a spot cleared 10 lines above, it's possible
                # for it to be blocked.  While that spot could be left cleared
                # and a new spot allocated, this condition might occur several
                # times in a row, resulting in a significant amount of disk I/O,
                # delaying the operation of the engine.  Rather than do this,
                # queue the piece to be moved out again, which will be performed
                # by the background allocator, with which data movement is
                # automatically limited.
                self.blocked_moveout.add(index)
            return False
        for p, v in self.places.items():
            if v == index:
                break
        else:
            self.failed('download corrupted; please restart and resume')
            return False
        self._move_piece(p, n)
        self.places[index] = index
        return False


    def piece_came_in(self, index, begin, piece, source = None):
        """Process some received data for a piece.
        
        If this is the first data for a piece, it will first clear a space for
        the piece. If the piece is complete then the data structures are
        cleaned up, and the data is flushed to disk and hash checked.
        
        @type index: C{int}
        @param index: the piece that data was received for
        @type begin: C{int}
        @param begin: the offset within the piece that the data starts at
        @type piece: C{string}
        @param piece: the data that was received
        @rtype: C{boolean}
        @return: False if the piece is complete but failed the hash check,
            otherwise returns True
        
        """
        
        assert not self.have[index]
        
        if not self.places.has_key(index):
            while self._clear_space(index):
                pass
            logger.debug('new place for '+str(index)+' at '+str(self.places[index]))
        if self.flag.isSet():
            return

        if self.failed_pieces.has_key(index):
            old = self.read_raw(self.places[index], begin, len(piece))
            if old is None:
                return True
            if old[:].tostring() != piece:
                try:
                    self.failed_pieces[index][self.download_history[index][begin]] = 1
                except:
                    self.failed_pieces[index][None] = 1
            old.release()
        self.download_history.setdefault(index,{})[begin] = source
        
        if not self._write_to_buffer(index, begin, piece):
            return True
        
        self.amount_obtained += len(piece)
        self.dirty.setdefault(index,[]).append((begin, len(piece)))
        self.numactive[index] -= 1
        assert self.numactive[index] >= 0
        if not self.numactive[index]:
            del self.stat_active[index]
        if self.stat_new.has_key(index):
            del self.stat_new[index]

        if self.inactive_requests[index] or self.numactive[index]:
            return True
        
        del self.dirty[index]
        if not self._flush_buffer(index):
            return True
        length = self._piecelen(index)
        data = self.read_raw(self.places[index], 0, length,
                                 flush_first = self.triple_check)
        if data is None:
            return True
        hash = sha(data[:]).digest()
        data.release()
        if hash != self.hashes[index]:

            self.amount_obtained -= length
            self.data_flunked(length, index)
            self.inactive_requests[index] = 1
            self.amount_inactive += length
            self.stat_numflunked += 1

            self.failed_pieces[index] = {}
            allsenders = {}
            for d in self.download_history[index].values():
                allsenders[d] = 1
            if len(allsenders) == 1:
                culprit = allsenders.keys()[0]
                if culprit is not None:
                    culprit.failed(index, bump = True)
                del self.failed_pieces[index] # found the culprit already
            
            return False

        self.have[index] = True
        self.inactive_requests[index] = None
        self.waschecked[index] = True
        self.amount_left -= length
        self.stat_numdownloaded += 1

        for d in self.download_history[index].values():
            if d is not None:
                d.good(index)
        del self.download_history[index]
        if self.failed_pieces.has_key(index):
            for d in self.failed_pieces[index].keys():
                if d is not None:
                    d.failed(index)
            del self.failed_pieces[index]

        if self.amount_left == 0:
            self.finished()
        return True


    def request_lost(self, index, begin, length):
        """Add a lost request back to the list of inactive requests.
        
        @type index: C{int}
        @param index: the piece requested
        @type begin: C{int}
        @param begin: the offset within the piece of the request
        @type length: C{int}
        @param length: the length of the request
        
        """
        
        assert not (begin, length) in self.inactive_requests[index]
        insort(self.inactive_requests[index], (begin, length))
        self.amount_inactive += length
        self.numactive[index] -= 1
        if not self.numactive[index]:
            del self.stat_active[index]
            if self.stat_new.has_key(index):
                del self.stat_new[index]


    def get_piece(self, index, begin, length):
        """Get a chunk of a piece.
        
        If the piece has never been hash checked, the whole piece will be read
        and checked first. Otherwise only the requested data will be read.
        
        @type index: C{int}
        @param index: the piece requested
        @type begin: C{int}
        @param begin: the offset within the piece of the request
        @type length: C{int}
        @param length: the length of the request
        @rtype: L{DebTorrent.piecebuffer.SingleBuffer} or C{string}
        @return: the requested data (in a piecebuffer if the request was for
            the entire piece), or None if there was a problem
        
        """
        
        if not self.have[index]:
            return None
        data = None
        if not self.waschecked[index]:
            data = self.read_raw(self.places[index], 0, self._piecelen(index))
            if data is None:
                return None
            if sha(data[:]).digest() != self.hashes[index]:
                self.failed('told file complete on start-up, but piece failed hash check')
                return None
            self.waschecked[index] = True
            if length == -1 and begin == 0:
                return data     # optimization
        if length == -1:
            if begin > self._piecelen(index):
                return None
            length = self._piecelen(index)-begin
            if begin == 0:
                return self.read_raw(self.places[index], 0, length)
        elif begin + length > self._piecelen(index):
            return None
        if data is not None:
            s = data[begin:begin+length]
            data.release()
            return s
        data = self.read_raw(self.places[index], begin, length)
        if data is None:
            return None
        s = data.getarray()
        data.release()
        return s

    def read_raw(self, piece, begin, length, flush_first = False):
        """Read some piece data from the disk.
        
        @type piece: C{int}
        @param piece: the piece requested
        @type begin: C{int}
        @param begin: the offset within the piece of the request
        @type length: C{int}
        @param length: the length of the request
        @type flush_first: C{boolean}
        @param flush_first: whether to flush the files before reading the data
            (optional, default is not to flush)
        @rtype: L{DebTorrent.piecebuffer.SingleBuffer}
        @return: the requested data, or None if there was a problem
        
        """
        
        try:
            return self.storage.read(self.piece_begins[piece] + begin,
                                                     length, flush_first)
        except IOError, e:
            self.failed('IO Error: ' + str(e))
            return None


    def set_file_readonly(self, n):
        """Sync a file to disk and set it read only.
        
        @type n: C{int}
        @param n: the file index to set read only
        
        """
        
        try:
            self.storage.set_readonly(n)
        except IOError, e:
            self.failed('IO Error: ' + str(e))
        except OSError, e:
            self.failed('OS Error: ' + str(e))


    def has_data(self, index):
        """Determine whether a piece location has data in it.
        
        @type index: C{int}
        @param index: the piece location to check
        @rtype: C{boolean}
        @return: whether the location has data in it
        
        """
        
        return index not in self.holes and index not in self.blocked_holes

    def doublecheck_data(self, pieces_to_check):
        """Double check all the listed pieces hashes.
        
        @type pieces_to_check: C{dictionary}
        @param pieces_to_check: keys are the piece indexes to hash check
        @rtype: C{boolean}
        @return: whether all the pieces passed the hash check
        
        """
        
        if not self.double_check:
            return
        sources = []
        for p,v in self.places.items():
            if pieces_to_check.has_key(v):
                sources.append(p)
        assert len(sources) == len(pieces_to_check)
        sources.sort()
        for index in sources:
            if self.have[index]:
                piece = self.read_raw(self.places[index],0,self._piecelen(index),
                                       flush_first = True )
                if piece is None:
                    return False
                if sha(piece[:]).digest() != self.hashes[index]:
                    self.failed('download corrupted; please restart and resume')
                    return False
                piece.release()
        return True


    def reblock(self, new_blocked):
        """Recreate the various blocked lists based on new blocked pieces.
        
        Assume downloads have already been canceled and chunks made inactive.
        
        @type new_blocked: C{list} of C{boolean}
        @param new_blocked: whether the pieces are now blocked
        
        """
        
        for i in xrange(len(new_blocked)):
            if new_blocked[i] and not self.blocked[i]:
                length = self._piecelen(i)
                self.amount_desired -= length
                if self.have[i]:
                    self.amount_obtained -= length
                    continue
                if self.inactive_requests[i] == 1:
                    self.amount_inactive -= length
                    continue
                inactive = 0
                for nb, nl in self.inactive_requests[i]:
                    inactive += nl
                self.amount_inactive -= inactive
                self.amount_obtained -= length - inactive
                
            if self.blocked[i] and not new_blocked[i]:
                length = self._piecelen(i)
                self.amount_desired += length
                if self.have[i]:
                    self.amount_obtained += length
                    continue
                if self.inactive_requests[i] == 1:
                    self.amount_inactive += length
                    continue
                inactive = 0
                for nb, nl in self.inactive_requests[i]:
                    inactive += nl
                self.amount_inactive += inactive
                self.amount_obtained += length - inactive

        self.blocked = new_blocked

        self.blocked_movein = Olist()
        self.blocked_moveout = Olist()
        for p,v in self.places.items():
            if p != v:
                if self.blocked[p] and not self.blocked[v]:
                    self.blocked_movein.add(p)
                elif self.blocked[v] and not self.blocked[p]:
                    self.blocked_moveout.add(p)

        self.holes.extend(self.blocked_holes)    # reset holes list
        self.holes.sort()
        self.blocked_holes = []


    def pickle(self):
        """Pickle the current state for later resuming.
        
        Pickled data format::
    
            d['pieces'] = either a string containing a bitfield of complete pieces,
                        or the numeric value "1" signifying a seed.  If it is
                        a seed, d['places'] and d['partials'] should be empty
                        and needn't even exist.
            d['partials'] = [ piece, [ offset, length... ]... ]
                        a list of partial data that had been previously
                        downloaded, plus the given offsets.  Adjacent partials
                        are merged so as to save space, and so that if the
                        request size changes then new requests can be
                        calculated more efficiently.
            d['places'] = [ piece, place, {,piece, place ...} ]
                        the piece index, and the place it's stored.
                        If d['pieces'] specifies a complete piece or d['partials']
                        specifies a set of partials for a piece which has no
                        entry in d['places'], it can be assumed that
                        place[index] = index.  A place specified with no
                        corresponding data in d['pieces'] or d['partials']
                        indicates allocated space with no valid data, and is
                        reserved so it doesn't need to be hash-checked.
        
        @rtype: C{dictionary}
        @return: the pickled state
        
        """
        
        if self.have.complete():
            return {'pieces': 1}
        pieces = Bitfield(len(self.hashes))
        places = []
        partials = []
        for p in xrange(len(self.hashes)):
            if self.blocked[p] or not self.places.has_key(p):
                continue
            h = self.have[p]
            pieces[p] = h
            pp = self.dirty.get(p)
            if not h and not pp:  # no data
                places.extend([self.places[p],self.places[p]])
            elif self.places[p] != p:
                places.extend([p, self.places[p]])
            if h or not pp:
                continue
            pp.sort()
            r = []
            while len(pp) > 1:
                if pp[0][0]+pp[0][1] == pp[1][0]:
                    pp[0] = list(pp[0])
                    pp[0][1] += pp[1][1]
                    del pp[1]
                else:
                    r.extend(pp[0])
                    del pp[0]
            r.extend(pp[0])
            partials.extend([p,r])
        return {'pieces': pieces.tostring(), 'places': places, 'partials': partials}


    def unpickle(self, data, valid_places):
        """Unpickle the state for resuming.
        
        @type data: C{dictionary}
        @param data: the pickled state (see L{pickle})
        @type valid_places: C{list} of C{int}
        @param valid_places: the pieces that are still considered valid based
            on their modification time and size
        @rtype: C{list} of C{int}
        @return: the pieces that are partially downloaded
        
        """
        
        got = {}
        places = {}
        dirty = {}
        download_history = {}
        stat_active = {}
        stat_numfound = self.stat_numfound
        amount_obtained = self.amount_obtained
        amount_inactive = self.amount_inactive
        amount_left = self.amount_left
        inactive_requests = [x for x in self.inactive_requests]
        restored_partials = []

        try:
            if data['pieces'] == 1:     # a seed
                assert not data.get('places',None)
                assert not data.get('partials',None)
                have = Bitfield(len(self.hashes))
                for i in xrange(len(self.hashes)):
                    have[i] = True
                assert have.complete()
                _places = []
                _partials = []
            else:
                have = Bitfield(len(self.hashes), data['pieces'])
                _places = data['places']
                assert len(_places) % 2 == 0
                _places = [_places[x:x+2] for x in xrange(0,len(_places),2)]
                _partials = data['partials']
                assert len(_partials) % 2 == 0
                _partials = [_partials[x:x+2] for x in xrange(0,len(_partials),2)]
                
            for index, place in _places:
                if place not in valid_places:
                    continue
                assert not got.has_key(index)
                assert not got.has_key(place)
                places[index] = place
                got[index] = 1
                got[place] = 1

            for index in xrange(len(self.hashes)):
                if have[index]:
                    if not places.has_key(index):
                        if index not in valid_places:
                            have[index] = False
                            continue
                        assert not got.has_key(index)
                        places[index] = index
                        got[index] = 1
                    length = self._piecelen(index)
                    amount_obtained += length
                    stat_numfound += 1
                    amount_inactive -= length
                    amount_left -= length
                    inactive_requests[index] = None

            for index, plist in _partials:
                assert not dirty.has_key(index)
                assert not have[index]
                if not places.has_key(index):
                    if index not in valid_places:
                        continue
                    assert not got.has_key(index)
                    places[index] = index
                    got[index] = 1
                assert len(plist) % 2 == 0
                plist = [plist[x:x+2] for x in xrange(0,len(plist),2)]
                dirty[index] = plist
                stat_active[index] = 1
                download_history[index] = {}
                # invert given partials
                length = self._piecelen(index)
                l = []
                if plist[0][0] > 0:
                    l.append((0,plist[0][0]))
                for i in xrange(len(plist)-1):
                    end = plist[i][0]+plist[i][1]
                    assert not end > plist[i+1][0]
                    l.append((end,plist[i+1][0]-end))
                end = plist[-1][0]+plist[-1][1]
                assert not end > length
                if end < length:
                    l.append((end,length-end))
                # split them to request_size
                ll = []
                amount_obtained += length
                amount_inactive -= length
                for nb, nl in l:
                    while nl > 0:
                        r = min(nl,self.request_size)
                        ll.append((nb,r))
                        amount_inactive += r
                        amount_obtained -= r
                        nb += self.request_size
                        nl -= self.request_size
                inactive_requests[index] = ll
                restored_partials.append(index)

            assert amount_obtained + amount_inactive == self.amount_desired
        except:
            if 'pieces' in data:
                logger.exception('Error unpickling data cache')
            return []   # invalid data, discard everything

        self.have = have
        self.places = places
        self.dirty = dirty
        self.download_history = download_history
        self.stat_active = stat_active
        self.stat_numfound = stat_numfound
        self.amount_obtained = amount_obtained
        self.amount_inactive = amount_inactive
        self.amount_left = amount_left
        self.inactive_requests = inactive_requests
                
        return restored_partials
    
