# Written by Bram Cohen
# modified for multitracker operation by John Hoffman
# Modified by Cameron Dale
# see LICENSE.txt for license information
#
# $Id: Rerequester.py 266 2007-08-18 02:06:35Z camrdale-guest $

"""Communicate with a tracker.

@type logger: C{logging.Logger}
@var logger: the logger to send all log messages to for this module
@type mapbase64: C{string}
@var mapbase64: the 64 characters to use for a base64 representation
@type keys: C{dictionary}
@var keys: the key parameters to send to tracker, keys are the announce addresses
@type basekeydata: C{string}
@var basekeydata: semi-random data to use to create the key

"""

from DebTorrent.zurllib import urlopen, quote
from urlparse import urlparse, urlunparse
from socket import gethostbyname
from btformats import check_peers
from DebTorrent.bencode import bdecode
from threading import Thread, Lock
from cStringIO import StringIO
from traceback import print_exc
from socket import error, gethostbyname
from random import shuffle
from sha import sha
from time import time
import logging
try:
    from os import getpid
except ImportError:
    def getpid():
        return 1
    
logger = logging.getLogger('DebTorrent.BT1.Rerequester')

mapbase64 = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz.-'
keys = {}
basekeydata = str(getpid()) + repr(time()) + 'tracker'

def add_key(tracker):
    """Create a random base64 key to send to the tracker.
    
    @type tracker: C{string}
    @param tracker: the announce address for the tracker
    
    """
    
    key = ''
    for i in sha(basekeydata+tracker).digest()[-6:]:
        key += mapbase64[ord(i) & 0x3F]
    keys[tracker] = key

def get_key(tracker):
    """Get the query parameter for the key to send to the tracker.
    
    @type tracker: C{string}
    @param tracker: the announce address for the tracker
    @rtype: C{string}
    @return: the query parameter to send to the tracker
    
    """
    
    try:
        return "&key="+keys[tracker]
    except:
        add_key(tracker)
        return "&key="+keys[tracker]

class fakeflag:
    """A fake flag to use if one is not supplied.
    
    @type state: C{boolean}
    @ivar state: the current state of the flag
    
    """
    
    def __init__(self, state=False):
        """Set the new flag.
        
        @type state: C{boolean}
        @param state: the initial state of the flag
        
        """
        
        self.state = state
        
    def wait(self):
        """Do nothing."""
        pass

    def isSet(self):
        """Check if the current state is set.
        
        @rtype: C{boolean}
        @return: whether the flag is set
        
        """
        
        return self.state

class Rerequester:
    """Communicate with a tracker.
    
    @type sched: C{method}
    @ivar sched: method to call to schedule future invocation of requester functions
    @type externalsched: C{method}
    @ivar externalsched: method to call to schedule future invocation of other functions
    @type errorfunc: C{method}
    @ivar errorfunc: method to call when an error occurs
    @type connect: C{method}
    @ivar connect: method to call to start connections to new peers
    @type howmany: C{method}
    @ivar howmany: method to call to determine how many connections are open
    @type amount_left: C{method}
    @ivar amount_left: method to call to determine how much is left to download
    @type up: C{method}
    @ivar up: method to call to determine how much has been uploaded
    @type down: C{method}
    @ivar down: method to call to determine how much has been downloaded
    @type upratefunc: C{method}
    @ivar upratefunc: method to call to determine the current upload rate
    @type downratefunc: C{method}
    @ivar downratefunc: method to call to determine the current download rate
    @type doneflag: C{threading.Event}
    @ivar doneflag: the flag that indicates when the program is to be shutdown
    @type unpauseflag: C{threading.Event}
    @ivar unpauseflag: the flag to unset to pause the download
    @type seededfunc: C{method}
    @ivar seededfunc: method to call if the tracker reports the torrent is seeded
    @type force_rapid_update: C{boolean}
    @ivar force_rapid_update: whether to do quick tracker updates when requested
    @type ip: C{string}
    @ivar ip: IP address to report to the tracker
    @type minpeers: C{int}
    @ivar minpeers: minimum number of peers to not do rerequesting
    @type maxpeers: C{int}
    @ivar maxpeers: number of peers at which to stop initiating new connections
    @type interval: C{int}
    @ivar interval: minimum time to wait between requesting more peers
    @type timeout: C{int}
    @ivar timeout: time to wait before assuming that a connection has timed out
    @type trackerlist: C{list} of C{list} of C{string}
    @ivar trackerlist: the trackers to connect to
    @type lastsuccessful: C{string}
    @ivar lastsuccessful: the last tracker address that was successfully contacted
    @type rejectedmessage: C{string}
    @ivar rejectedmessage: the start of the error messages to use when a failure occurs
    @type url: C{string}
    @ivar url: the query parameters to send to all trackers
    @type last: unknown
    @ivar last: a tracker parameter
    @type trackerid: unknown
    @ivar trackerid: a tracker parameter
    @type announce_interval: C{int}
    @ivar announce_interval: the tracker-specified announce interval to use
    @type last_failed: C{boolean}
    @ivar last_failed: whether the last request was successful
    @type never_succeeded: C{boolean}
    @ivar never_succeeded: whether there has ever been a successful request
    @type errorcodes: C{dictionary}
    @ivar errorcodes: error codes and messages that have occurred
    @type lock: L{SuccessLock}
    @ivar lock: the locks to use to synchronize threaded tracker requests
    @type special: C{string}
    @ivar special: a special tracker announce address to send a single request to
    @type stopped: C{boolean}
    @ivar stopped: whether the download is stopped
    
    """
    
    def __init__( self, port, myid, infohash, trackerlist, config,
                  sched, externalsched, errorfunc, connect,
                  howmany, amount_left, up, down, upratefunc, downratefunc,
                  doneflag, unpauseflag = fakeflag(True),
                  seededfunc = None, force_rapid_update = False ):
        """Initialize the instance.
        
        @type port: C{int}
        @param port: port to connect to this peer on
        @type myid: C{string}
        @param myid: the peer ID to use
        @type infohash: C{string}
        @param infohash: the info hash of the torrent being downloaded
        @type trackerlist: C{list} of C{list} of C{string}
        @param trackerlist: the trackers to connect to
        @type config: C{dictionary}
        @param config: the configuration parameters
        @type sched: C{method}
        @param sched: method to call to schedule future invocation of requester functions
        @type externalsched: C{method}
        @param externalsched: method to call to schedule future invocation of other functions
        @type errorfunc: C{method}
        @param errorfunc: method to call when an error occurs
        @type connect: C{method}
        @param connect: method to call to start connections to new peers
        @type howmany: C{method}
        @param howmany: method to call to determine how many connections are open
        @type amount_left: C{method}
        @param amount_left: method to call to determine how much is left to download
        @type up: C{method}
        @param up: method to call to determine how much has been uploaded
        @type down: C{method}
        @param down: method to call to determine how much has been downloaded
        @type upratefunc: C{method}
        @param upratefunc: method to call to determine the current upload rate
        @type downratefunc: C{method}
        @param downratefunc: method to call to determine the current download rate
        @type doneflag: C{threading.Event}
        @param doneflag: the flag that indicates when the program is to be shutdown
        @type unpauseflag: C{threading.Event}
        @param unpauseflag: the flag to unset to pause the download
            (optional, defaults to an always True dummy flag)
        @type seededfunc: C{method}
        @param seededfunc: method to call if the tracker reports the torrent
            is seeded (optional, defaults to not checking)
        @type force_rapid_update: C{boolean}
        @param force_rapid_update: whether to do quick tracker updates when 
            requested (optional, defaults to False)
        
        """

        self.sched = sched
        self.externalsched = externalsched
        self.errorfunc = errorfunc
        self.connect = connect
        self.howmany = howmany
        self.amount_left = amount_left
        self.up = up
        self.down = down
        self.upratefunc = upratefunc
        self.downratefunc = downratefunc
        self.doneflag = doneflag
        self.unpauseflag = unpauseflag
        self.seededfunc = seededfunc
        self.force_rapid_update = force_rapid_update

        self.ip = config.get('ip','')
        self.minpeers = config['min_peers']
        self.maxpeers = config['max_initiate']
        self.interval = config['rerequest_interval']
        self.timeout = config['http_timeout']

        newtrackerlist = []        
        for tier in trackerlist:
            if len(tier)>1:
                shuffle(tier)
            newtrackerlist += [tier]
        self.trackerlist = newtrackerlist

        self.lastsuccessful = ''
        self.rejectedmessage = 'rejected by tracker - '

        self.url = ('info_hash=%s&peer_id=%s' %
            (quote(infohash), quote(myid)))
        if not config.get('crypto_allowed'):
            self.url += "&port="
        else:
            self.url += "&supportcrypto=1"
            if not config.get('crypto_only'):
                    self.url += "&port="
            else:
                self.url += "&requirecrypto=1"            
                if not config.get('crypto_stealth'):
                    self.url += "&port="
                else:
                    self.url += "&port=0&cryptoport="
        self.url += str(port)

        seed_id = config.get('dedicated_seed_id')
        if seed_id:
            self.url += '&seed_id='+quote(seed_id)
        if self.seededfunc:
            self.url += '&check_seeded=1'

        self.last = None
        self.trackerid = None
        self.announce_interval = 30 * 60
        self.last_failed = True
        self.never_succeeded = True
        self.errorcodes = {}
        self.lock = SuccessLock()
        self.special = None
        self.stopped = False

    def start(self):
        """Start the tracker requester."""
        self.sched(self.c, self.interval/2)
        self.d(0)

    def c(self):
        """Start a periodic general announce request for more peers."""
        if self.stopped:
            return
        if not self.unpauseflag.isSet() and (
            self.howmany() < self.minpeers or self.force_rapid_update ):
            self.announce(3, self._c)
        else:
            self._c()

    def _c(self):
        """Schedule another general announce request for more peers."""
        self.sched(self.c, self.interval)

    def d(self, event = 3):
        """Start a periodic announce request.
        
        @type event: C{int}
        @param event: the type of announce request to do (optional, defaults to 3)::
            0 -- started
            1 -- completed
            2 -- stopped
            3 -- general announce
        
        """
        
        if self.stopped:
            return
        if not self.unpauseflag.isSet():
            self._d()
            return
        self.announce(event, self._d)

    def _d(self):
        """Schedule another announce request"""
        if self.never_succeeded:
            self.sched(self.d, 60)  # retry in 60 seconds
        elif self.force_rapid_update:
            return
        else:
            self.sched(self.d, self.announce_interval)


    def hit(self, event = 3):
        """Start a specific type of announce request for more peers.
        
        @type event: C{int}
        @param event: the type of announce request to do (optional, defaults to 3)::
            0 -- started
            1 -- completed
            2 -- stopped
            3 -- general announce
        
        """
        
        if not self.unpauseflag.isSet() and (
            self.howmany() < self.minpeers or self.force_rapid_update ):
            self.announce(event)

    def announce(self, event = 3, callback = lambda: None, specialurl = None):
        """Create an announce request.
        
        @type event: C{int}
        @param event: the type of announce request to do (optional, defaults to 3)::
            0 -- started
            1 -- completed
            2 -- stopped
            3 -- general announce
        @type callback: C{method}
        @param callback: the method to call when the announce is complete
            (optional, defaults to doing nothing)
        @type specialurl: C{string}
        @param specialurl: a special tracker announce address to send this
            request to (optional, defaults to the regular tracker list)
        
        """
        
        if specialurl is not None:
            s = self.url+'&uploaded=0&downloaded=0&left=1'   # don't add to statistics
            if self.howmany() >= self.maxpeers:
                s += '&numwant=0'
            else:
                s += '&no_peer_id=1&compact=1'
            self.last_failed = True         # force true, so will display an error
            self.special = specialurl
            self.rerequest(s, callback)
            return
        
        else:
            s = ('%s&uploaded=%s&downloaded=%s&left=%s' %
                (self.url, str(self.up()), str(self.down()), 
                str(self.amount_left())))
        if self.last is not None:
            s += '&last=' + quote(str(self.last))
        if self.trackerid is not None:
            s += '&trackerid=' + quote(str(self.trackerid))
        if self.howmany() >= self.maxpeers:
            s += '&numwant=0'
        else:
            s += '&no_peer_id=1&compact=1'
        if event != 3:
            s += '&event=' + ['started', 'completed', 'stopped'][event]
        if event == 2:
            self.stopped = True
        self.rerequest(s, callback)


    def snoop(self, peers, callback = lambda: None):
        """Send an immediate request to the tracker.
        
        Tracker call support for the tracker-to-tracker communication.
        
        @type peers: C{int}
        @param peers: the number of peers to request
        @type callback: C{method}
        @param callback: the method to call when the announce is complete
            (optional, defaults to doing nothing)
        
        """
        
        self.rerequest(self.url
            +'&event=stopped&port=0&uploaded=0&downloaded=0&left=1&tracker=1&numwant='
            +str(peers), callback)


    def rerequest(self, s, callback):
        """Start a threaded request to a tracker.
        
        Uses the L{SuccessLock} to make sure concurrent requests are not allowed.
        
        @type s: C{string}
        @param s: the query to add to the URL for the request
        @type callback: C{method}
        @param callback: the method to call when the announce is complete
        
        """
        
        if not self.lock.isfinished():  # still waiting for prior cycle to complete??
            def retry(self = self, s = s, callback = callback):
                self.rerequest(s, callback)
            self.sched(retry,5)         # retry in 5 seconds
            return
        logger.info('Sending request: '+s)
        self.lock.reset()
        rq = Thread(target = self._rerequest, args = [s, callback],
                    name = 'Rerequester._rerequest')
        rq.setDaemon(False)
        rq.start()

    def _rerequest(self, s, callback):
        """Try all the trackers in the list for an announce request.
        
        @type s: C{string}
        @param s: the query to add to the URL for the request
        @type callback: C{method}
        @param callback: the method to call when the announce is complete
        
        """
        
        try:
            def fail (self = self, callback = callback):
                self._fail(callback)
            if self.ip:
                try:
                    s += '&ip=' + gethostbyname(self.ip)
                except:
                    self.errorcodes['troublecode'] = 'unable to resolve: '+self.ip
                    self.externalsched(fail)
            self.errorcodes = {}
            if self.special is None:
                for t in range(len(self.trackerlist)):
                    for tr in range(len(self.trackerlist[t])):
                        tracker  = self.trackerlist[t][tr]
                        logger.debug('Trying tracker: '+tracker)
                        if self.rerequest_single(tracker, s, callback):
                            if not self.last_failed and tr != 0:
                                del self.trackerlist[t][tr]
                                self.trackerlist[t] = [tracker] + self.trackerlist[t]
                            logger.info('Succesful with tracker: '+tracker)
                            return
            else:
                tracker = self.special
                self.special = None
                logger.debug('Trying special tracker: '+tracker)
                if self.rerequest_single(tracker, s, callback):
                    logger.info('Succesful with special tracker: '+tracker)
                    return
            # no success from any tracker
            self.externalsched(fail)
        except:
            logger.exception('Error occurred while trying list of trackers')
            self.externalsched(callback)


    def _fail(self, callback):
        """Process the failed request.
        
        @type callback: C{method}
        @param callback: the method to call when the announce is complete
        
        """
        
        if ( (self.upratefunc() < 100 and self.downratefunc() < 100)
             or not self.amount_left() ):
            for f in ['rejected', 'bad_data', 'troublecode']:
                if self.errorcodes.has_key(f):
                    r = self.errorcodes[f]
                    break
            else:
                r = 'Problem connecting to tracker - unspecified error'
            logger.error(r)
            self.errorfunc(r)

        self.last_failed = True
        self.lock.give_up()
        self.externalsched(callback)


    def rerequest_single(self, t, s, callback):
        """Start a threaded request to a single tracker and wait for it to complete.
        
        @type t: C{string}
        @param t: the announce address of the tracker to contact
        @type s: C{string}
        @param s: the query to add to the URL for the request
        @type callback: C{method}
        @param callback: the method to call when the announce is complete
        
        """
        
        l = self.lock.set()
        rq = Thread(target = self._rerequest_single, args = [t, s+get_key(t), l, callback],
                    name = 'Rerequester._rerequest_single')
        rq.setDaemon(False)
        rq.start()
        self.lock.wait()
        if self.lock.success:
            self.lastsuccessful = t
            self.last_failed = False
            self.never_succeeded = False
            return True
        if not self.last_failed and self.lastsuccessful == t:
            # if the last tracker hit was successful, and you've just tried the tracker
            # you'd contacted before, don't go any further, just fail silently.
            self.last_failed = True
            self.externalsched(callback)
            self.lock.give_up()
            return True
        return False    # returns true if it wants rerequest() to exit


    def _rerequest_single(self, t, s, l, callback):
        """Perform a request to a single tracker.
        
        @type t: C{string}
        @param t: the announce address of the tracker to contact
        @type s: C{string}
        @param s: the query to add to the URL for the request
        @type l: C{long}
        @param l: the current hold on the lock
        @type callback: C{method}
        @param callback: the method to call when the announce is complete
        
        """
        
        try:        
            closer = [None]
            def timedout(self = self, l = l, closer = closer):
                if self.lock.trip(l):
                    self.errorcodes['troublecode'] = 'Problem connecting to tracker - timeout exceeded'
                    self.lock.unwait(l)
                try:
                    closer[0]()
                except:
                    pass
                    
            self.externalsched(timedout, self.timeout)

            err = None
            try:
                url,q = t.split('?',1)
                q += '&'+s
            except:
                url = t
                q = s
            try:
                h = urlopen(url+'?'+q)
                closer[0] = h.close
                data = h.read()
            except (IOError, error), e:
                err = 'Problem connecting to tracker - ' + str(e)
            except:
                err = 'Problem connecting to tracker'
            try:
                h.close()
            except:
                pass
            if err:        
                if self.lock.trip(l):
                    self.errorcodes['troublecode'] = err
                    self.lock.unwait(l)
                return

            if data == '':
                if self.lock.trip(l):
                    self.errorcodes['troublecode'] = 'no data from tracker'
                    self.lock.unwait(l)
                return
            
            try:
                r = bdecode(data, sloppy=1)
                check_peers(r)
            except ValueError, e:
                if self.lock.trip(l):
                    self.errorcodes['bad_data'] = 'bad data from tracker - ' + str(e)
                    self.lock.unwait(l)
                return
            
            if r.has_key('failure reason'):
                if self.lock.trip(l):
                    self.errorcodes['rejected'] = self.rejectedmessage + r['failure reason']
                    self.lock.unwait(l)
                return
                
            if self.lock.trip(l, True):     # success!
                self.lock.unwait(l)
            else:
                callback = lambda: None     # attempt timed out, don't do a callback

            # even if the attempt timed out, go ahead and process data
            def add(self = self, r = r, callback = callback):
                self.postrequest(r, callback)
            self.externalsched(add)
        except:
            logger.exception('Error occurred performing request to single tracker')
            self.externalsched(callback)


    def postrequest(self, r, callback):
        """Process the returned request from a single tracker.
        
        @type r: C{dictionary}
        @param r: the bdecoded data returned by the tracker
        @type callback: C{method}
        @param callback: the method to call when the request is complete
        
        """
        
        if r.has_key('warning message'):
            logger.warning('warning from tracker - ' + r['warning message'])
            self.errorfunc('warning from tracker - ' + r['warning message'])
        self.announce_interval = r.get('interval', self.announce_interval)
        self.interval = r.get('min interval', self.interval)
        self.trackerid = r.get('tracker id', self.trackerid)
        self.last = r.get('last')
#        ps = len(r['peers']) + self.howmany()
        p = r['peers']
        new_peers = {}
        if type(p) == type(''):
            lenpeers = len(p)/6
        else:
            lenpeers = len(p)
        cflags = r.get('crypto_flags')
        if type(cflags) != type('') or len(cflags) != lenpeers:
            cflags = None
        if cflags is None:
            cflags = [None for i in xrange(lenpeers)]
        else:
            cflags = [ord(x) for x in cflags]
        if type(p) == type(''):
            for x in xrange(0, len(p), 6):
                ip = '.'.join([str(ord(i)) for i in p[x:x+4]])
                port = (ord(p[x+4]) << 8) | ord(p[x+5])
                new_peers[(ip, port)] = (0, cflags[int(x/6)])
        else:
            for i in xrange(len(p)):
                x = p[i]
                new_peers[(x['ip'].strip(), x['port'])] = (x.get('peer id',0),
                                                           cflags[i])
        
        # Now build the list of peers that are unique in the list
        peers = []
        for dns, (id, crypto) in new_peers.items():
            peers.append((dns, id, crypto))
        logger.info('received from tracker: '+str(peers))
        ps = len(peers) + self.howmany()
        if ps < self.maxpeers:
            if self.doneflag.isSet():
                if r.get('num peers', 1000) - r.get('done peers', 0) > ps * 1.2:
                    self.last = None
            else:
                if r.get('num peers', 1000) > ps * 1.2:
                    self.last = None
        if self.seededfunc and r.get('seeded'):
            self.seededfunc()
        elif peers:
            shuffle(peers)
            self.connect(peers)
        callback()


class SuccessLock:
    """Locks to synchronize threaded requests to trackers.
    
    @type lock: C{threading.Lock}
    @ivar lock: lock to ensure no concurrent access to this object
    @type pause: C{threading.Lock}
    @ivar pause: lock to synchronize sending requests to trackers
    @type code: C{long}
    @ivar code: a unique code sent to each setter of the lock
    @type success: C{boolean}
    @ivar success: whether the request was successful
    @type finished: C{boolean}
    @ivar finished: whether the request is complete
    @type first: C{boolean}
    @ivar first: whether the first trip has occurred yet
    
    """
    
    def __init__(self):
        """Initialize the instance."""
        self.lock = Lock()
        self.pause = Lock()
        self.code = 0L
        self.success = False
        self.finished = True

    def reset(self):
        """Reset the instance to it's initial state."""
        self.success = False
        self.finished = False

    def set(self):
        """Set the lock for a request.
        
        @rtype: C{long}
        @return: the unique identifier code for this request
        
        """
        
        self.lock.acquire()
        if not self.pause.locked():
            self.pause.acquire()
        self.first = True
        self.code += 1L
        self.lock.release()
        return self.code

    def trip(self, code, s = False):
        """Interrupt a not yet finished request.
        
        @type code: C{long}
        @param code: the unique identifier code for the request
        @type s: C{boolean}
        @param s: whether to set the request as successful
        
        """
        
        self.lock.acquire()
        try:
            if code == self.code and not self.finished:
                r = self.first
                self.first = False
                if s:
                    self.finished = True
                    self.success = True
                return r
        finally:
            self.lock.release()

    def give_up(self):
        """Terminate a request unsuccessfully."""
        self.lock.acquire()
        self.success = False
        self.finished = True
        self.lock.release()

    def wait(self):
        """Wait for the current request to complete."""
        self.pause.acquire()

    def unwait(self, code):
        """Stop waiting for a request to complete.
        
        @type code: C{long}
        @param code: the unique identifier code for the request
        
        """
        
        if code == self.code and self.pause.locked():
            self.pause.release()

    def isfinished(self):
        """Check if the current request is complete.
        
        @rtype: C{boolean}
        @return: whether the request is complete
        
        """
        
        self.lock.acquire()
        x = self.finished
        self.lock.release()
        return x    
