# Written by Bram Cohen
# Modified by Cameron Dale
# see LICENSE.txt for license information
#
# $Id: PiecePicker.py 266 2007-08-18 02:06:35Z camrdale-guest $

"""Choose pieces to download.

@type logger: C{logging.Logger}
@var logger: the logger to send all log messages to for this module

"""

from random import randrange, shuffle
from DebTorrent.clock import clock
import logging

logger = logging.getLogger('DebTorrent.BT1.Rerequester')

class PiecePicker:
    """Choose pieces to download.
    
    @type rarest_first_cutoff: C{int}
    @ivar rarest_first_cutoff: number of completed piece downloads at
        which to switch from random piece selection to rarest first
    @type rarest_first_priority_cutoff: C{int}
    @ivar rarest_first_priority_cutoff: the number of peers which need to
        have a piece before other partials take priority over rarest first
    @type priority_step: C{int}
    @ivar priority_step: the difference bewteen priority levels 0, 1, and 2
    @type cutoff: C{int}
    @ivar cutoff: the number of peers which need to
        have a piece before other partials take priority over rarest first
    @type numpieces: C{int}
    @ivar numpieces: total number of pieces in the download
    @type started: C{list} of C{int}
    @ivar started: the pieces that have been requested for download
    @type totalcount: C{int}
    @ivar totalcount: the total number of copies of all pieces in non-seeding peers
    @type numhaves: C{list} of C{int}
    @ivar numhaves: the number of copies of each piece in non-seeding peers
    @type priority: C{list} of C{int}
    @ivar priority: the priority of each piece::
        -1 -- do not download
         0 -- highest priority
         1 -- medium priority
         2 -- lowest priority
    @type removed_partials: C{dictionary}
    @ivar removed_partials: keys are the pieces that were started but then disabled
    @type crosscount: C{list} of C{int}
    @ivar crosscount: at each index, the value is the number of pieces that
        have that many copies in the non-seeding peers
    @type crosscount2: C{list} of C{int}
    @ivar crosscount2: at each index, the value is the number of pieces that
        have that many copies in the non-seeding peers (including this peer)
    @type has: C{list} of C{int}
    @ivar has: values are 1 for the pieces that this peer has
    @type numgot: C{int}
    @ivar numgot: the number of pieces this peer has
    @type done: C{boolean}
    @ivar done: whether the download is complete
    @type seed_connections: C{dictionary}
    @ivar seed_connections: connections that have been made in super-seed mode,
        keys are L{Connecter.Connection}, values are the piece have that was
        last sent to the peer
    @type past_ips: C{dictionary}
    @ivar past_ips: the IPs that have previously connected in super-seed mode,
        keys are IP addresses, values are the piece have that was last sent to
        the peer
    @type seed_time: C{float}
    @ivar seed_time: the time when the first peer was seen in super-seed mode
    @type superseed: C{boolean}
    @ivar superseed: whether we are in super-seed mode
    @type seeds_connected: C{int}
    @ivar seeds_connected: the number of connected seeds
    @type interests: C{list} of C{list} of C{int}
    @ivar interests: the interest levels, each level is a list of piece indexes
        that are at that level
    @type level_in_interests: C{list} of C{int}
    @ivar level_in_interests: one per piece, the level that each piece is at
        in the interests
    @type pos_in_interests: C{list} of C{int}
    @ivar pos_in_interests: the position within the interest level that each
        piece is at
    @type seed_got_haves: C{list} of C{int}
    @ivar seed_got_haves: the number of copies of pieces in super-seed mode
    
    """
    
    def __init__(self, numpieces,
                 rarest_first_cutoff = 1, rarest_first_priority_cutoff = 3,
                 priority_step = 20):
        """Initialize the instance and the piece interests.
        
        @type numpieces: C{int}
        @param numpieces: total number of pieces in the download
        @type rarest_first_cutoff: C{int}
        @param rarest_first_cutoff: number of completed piece downloads at
            which to switch from random piece selection to rarest first
            (optional, defaults to 1)
        @type rarest_first_priority_cutoff: C{int}
        @param rarest_first_priority_cutoff: the number of peers which need to
            have a piece before other partials take priority over rarest first
            (optional, defaults to 3)
        @type priority_step: C{int}
        @param priority_step: the difference bewteen priority levels 0, 1, and 2
            (optional, defaults to 20)
        
        """
        
        self.rarest_first_cutoff = rarest_first_cutoff
        self.rarest_first_priority_cutoff = rarest_first_priority_cutoff + priority_step
        self.priority_step = priority_step
        self.cutoff = rarest_first_priority_cutoff
        self.numpieces = numpieces
        self.started = []
        self.totalcount = 0
        self.numhaves = [0] * numpieces
        self.priority = [-1] * numpieces
        self.removed_partials = {}
        self.crosscount = [numpieces]
        self.crosscount2 = [numpieces]
        self.has = [0] * numpieces
        self.numgot = 0
        self.done = False
        self.seed_connections = {}
        self.past_ips = {}
        self.seed_time = None
        self.superseed = False
        self.seeds_connected = 0
        self._init_interests()

    def _init_interests(self):
        """Initialize the interests in pieces to all not interested."""
        self.interests = [[] for x in xrange(self.priority_step)]
        self.level_in_interests = [self.priority_step] * self.numpieces
        interests = range(self.numpieces)
        shuffle(interests)
        self.pos_in_interests = [0] * self.numpieces
        for i in xrange(self.numpieces):
            self.pos_in_interests[interests[i]] = i
        # Init all interest levels to empty
        self.interests.append([])


    def got_have(self, piece):
        """Process a piece that has a new copy found.
        
        @type piece: C{int}
        @param piece: the piece that was copied
        
        """
        
        self.totalcount+=1
        numint = self.numhaves[piece]
        self.numhaves[piece] += 1
        self.crosscount[numint] -= 1
        if numint+1==len(self.crosscount):
            self.crosscount.append(0)
        self.crosscount[numint+1] += 1
        if not self.done:
            numintplus = numint+self.has[piece]
            self.crosscount2[numintplus] -= 1
            if numintplus+1 == len(self.crosscount2):
                self.crosscount2.append(0)
            self.crosscount2[numintplus+1] += 1
            numint = self.level_in_interests[piece]
            self.level_in_interests[piece] += 1
        if self.superseed:
            self.seed_got_haves[piece] += 1
            numint = self.level_in_interests[piece]
            self.level_in_interests[piece] += 1
        elif self.has[piece] or self.priority[piece] == -1:
            return
        if numint == len(self.interests) - 1:
            self.interests.append([])
        self._shift_over(piece, self.interests[numint], self.interests[numint + 1])

    def lost_have(self, piece):
        """Process a piece that lost a copy.
        
        @type piece: C{int}
        @param piece: the piece that lost a copy
        
        """
        
        self.totalcount-=1
        numint = self.numhaves[piece]
        self.numhaves[piece] -= 1
        self.crosscount[numint] -= 1
        self.crosscount[numint-1] += 1
        if not self.done:
            numintplus = numint+self.has[piece]
            self.crosscount2[numintplus] -= 1
            self.crosscount2[numintplus-1] += 1
            numint = self.level_in_interests[piece]
            self.level_in_interests[piece] -= 1
        if self.superseed:
            numint = self.level_in_interests[piece]
            self.level_in_interests[piece] -= 1
        elif self.has[piece] or self.priority[piece] == -1:
            return
        self._shift_over(piece, self.interests[numint], self.interests[numint - 1])

    def _shift_over(self, piece, l1, l2):
        """Remove a piece from an old interest level to a new one.
        
        @type piece: C{int}
        @param piece: the piece that is to be moved
        @type l1: C{list} of C{int}
        @param l1: the old interest level list
        @type l2: C{list} of C{int}
        @param l2: the new interest level list
        
        """
        
        assert self.superseed or (not self.has[piece] and self.priority[piece] >= 0)
        parray = self.pos_in_interests
        p = parray[piece]
        assert l1[p] == piece
        q = l1[-1]
        l1[p] = q
        parray[q] = p
        del l1[-1]
        newp = randrange(len(l2)+1)
        if newp == len(l2):
            parray[piece] = len(l2)
            l2.append(piece)
        else:
            old = l2[newp]
            parray[old] = len(l2)
            l2.append(old)
            l2[newp] = piece
            parray[piece] = newp


    def got_seed(self):
        """Process a newly found seed."""
        self.seeds_connected += 1
        self.cutoff = max(self.rarest_first_priority_cutoff-self.seeds_connected,0)

    def became_seed(self):
        """Process this client becoming a seed."""
        self.got_seed()
        self.totalcount -= self.numpieces
        self.numhaves = [i-1 for i in self.numhaves]
        if self.superseed or not self.done:
            self.level_in_interests = [i-1 for i in self.level_in_interests]
            if self.interests:
                del self.interests[0]
        del self.crosscount[0]
        if not self.done:
            del self.crosscount2[0]

    def lost_seed(self):
        """Process a lost seed."""
        self.seeds_connected -= 1
        self.cutoff = max(self.rarest_first_priority_cutoff-self.seeds_connected,0)


    def requested(self, piece):
        """Add a piece that a request has been sent for to the list of started pieces.
        
        @type piece: C{int}
        @param piece: the piece that is started
        
        """
        
        if piece not in self.started:
            self.started.append(piece)

    def _remove_from_interests(self, piece, keep_partial = False):
        """Remove a piece from the interests.
        
        @type piece: C{int}
        @param piece: the piece that is to be removed
        @type keep_partial: C{boolean}
        @param keep_partial: if the piece has been started, whether to keep a
            note of that so that if it becomes interesting again it can be
            immediately added to the list of started pieces
            (optional, defaults to False)
        
        """
        
        l = self.interests[self.level_in_interests[piece]]
        p = self.pos_in_interests[piece]
        assert l[p] == piece
        q = l[-1]
        l[p] = q
        self.pos_in_interests[q] = p
        del l[-1]
        try:
            self.started.remove(piece)
            if keep_partial:
                self.removed_partials[piece] = 1
        except ValueError:
            pass

    def complete(self, piece):
        """Process a successfully received piece.
        
        @type piece: C{int}
        @param piece: the piece that was received
        
        """
        
        assert not self.has[piece]
        self.has[piece] = 1
        self.numgot += 1
        if self.numgot == self.numpieces:
            self.done = True
            self.crosscount2 = self.crosscount
        else:
            numhaves = self.numhaves[piece]
            self.crosscount2[numhaves] -= 1
            if numhaves+1 == len(self.crosscount2):
                self.crosscount2.append(0)
            self.crosscount2[numhaves+1] += 1
        self._remove_from_interests(piece)


    def next(self, haves, wantfunc, complete_first = False):
        """Choose a piece to request to download next from a peer.
        
        @type haves: C{list} of C{int}
        @param haves: the list of pieces that the peer has
        @type wantfunc: C{method}
        @param wantfunc: a method that returns True if the piece is desired
        @type complete_first: C{boolean}
        @param complete_first: whether to complete a partially dowloaded piece
            before requsting a new one (optional, defaults to False)
        @rtype: C{int}
        @return: the index of the piece to download
        
        """
        
        cutoff = self.numgot < self.rarest_first_cutoff
        complete_first = (complete_first or cutoff) and not haves.complete()
        best = None
        bestnum = 2 ** 30
        for i in self.started:
            if haves[i] and wantfunc(i):
                if self.level_in_interests[i] < bestnum:
                    best = i
                    bestnum = self.level_in_interests[i]
        if best is not None:
            if complete_first or (cutoff and len(self.interests) > self.cutoff):
                logger.info('Next piece to download/request: '+str(best))
                return best
        if haves.complete():
            r = [ (0, min(bestnum,len(self.interests))) ]
        elif cutoff and len(self.interests) > self.cutoff:
            r = [ (self.cutoff, min(bestnum,len(self.interests))),
                      (0, self.cutoff) ]
        else:
            r = [ (0, min(bestnum,len(self.interests))) ]
        for lo,hi in r:
            for i in xrange(lo,hi):
                for j in self.interests[i]:
                    if haves[j] and wantfunc(j):
                        logger.info('Next piece to download/request: '+str(j))
                        return j
        if best is not None:
            logger.info('Next piece to download/request: '+str(best))
            return best
        return None


    def am_I_complete(self):
        """Determine if the download is complete.
        
        @rtype: C{boolean}
        @return: whether the download is complete
        
        """
        
        return self.done
    
    def bump(self, piece):
        """Move a piece to the end of its interest level.
        
        Reduces the priority for downloading this piece, unless it is the only
        one in its level.
        
        @type piece: C{int}
        @param piece: the piece that is to be downgraded
        
        """
        
        l = self.interests[self.level_in_interests[piece]]
        pos = self.pos_in_interests[piece]
        del l[pos]
        l.append(piece)
        for i in range(pos,len(l)):
            self.pos_in_interests[l[i]] = i
        try:
            self.started.remove(piece)
        except:
            pass

    def set_priority(self, piece, p):
        """Adjust the priority for a piece.
        
        @type piece: C{int}
        @param piece: the piece that is to be prioritised
        @type p: C{int}
        @param p: the new priority for the piece
        @rtype: C{boolean}
        @return: whether the downloader should try requesting more
        
        """
        
        if self.superseed:
            return False    # don't muck with this if you're a superseed
        oldp = self.priority[piece]
        if oldp == p:
            return False
        self.priority[piece] = p
        if p == -1:
            # when setting priority -1,
            # make sure to cancel any downloads for this piece
            if not self.has[piece]:
                self._remove_from_interests(piece, True)
            return True
        if oldp == -1:
            level = self.numhaves[piece] + (self.priority_step * p)
            self.level_in_interests[piece] = level
            if self.has[piece]:
                return True
            while len(self.interests) < level+1:
                self.interests.append([])
            l2 = self.interests[level]
            parray = self.pos_in_interests
            newp = randrange(len(l2)+1)
            if newp == len(l2):
                parray[piece] = len(l2)
                l2.append(piece)
            else:
                old = l2[newp]
                parray[old] = len(l2)
                l2.append(old)
                l2[newp] = piece
                parray[piece] = newp
            if self.removed_partials.has_key(piece):
                del self.removed_partials[piece]
                self.started.append(piece)
            # now go to downloader and try requesting more
            return True
        numint = self.level_in_interests[piece]
        newint = numint + ((p - oldp) * self.priority_step)
        self.level_in_interests[piece] = newint
        if self.has[piece]:
            return False
        while len(self.interests) < newint+1:
            self.interests.append([])
        self._shift_over(piece, self.interests[numint], self.interests[newint])
        return False

    def is_blocked(self, piece):
        """Determine whether a piece is disabled.
        
        @type piece: C{int}
        @param piece: the piece that is to be checked
        @rtype: C{boolean}
        @return: whether the priority is to not download the piece
        
        """
        
        return self.priority[piece] < 0


    def set_superseed(self):
        """Switch to super-seeding mode."""
        assert self.done
        self.superseed = True
        self.seed_got_haves = [0] * self.numpieces
        self._init_interests()  # assume everyone is disconnected

    def next_have(self, connection, looser_upload):
        """Determine the next piece to tell a peer we have in super-seed mode.
        
        @type connection: L{Connecter.Connection}
        @param connection: the peer connection the have will be sent on
        @type looser_upload: C{boolean}
        @param looser_upload: whether to be looser in determining the piece
        @rtype: C{int}
        @return: the piece index to send in the have message, or -1 if
            something has gone wrong and the connection is to be closed
        
        """
        
        if self.seed_time is None:
            self.seed_time = clock()
            return None
        if clock() < self.seed_time+10:  # wait 10 seconds after seeing the first peers
            return None                 # to give time to grab have lists
        if not connection.upload.super_seeding:
            return None
        olddl = self.seed_connections.get(connection)
        if olddl is None:
            ip = connection.get_ip()
            olddl = self.past_ips.get(ip)
            if olddl is not None:                           # peer reconnected
                self.seed_connections[connection] = olddl
                if not looser_upload:
                    self.seed_got_haves[olddl] -= 1         # penalize
        if olddl is not None:
            if looser_upload:
                num = 1     # send a new have even if it hasn't spread that piece elsewhere
            else:
                num = 2
            if self.seed_got_haves[olddl] < num:
                return None
            if not connection.upload.was_ever_interested:   # it never downloaded it?
                connection.upload.skipped_count += 1
                if connection.upload.skipped_count >= 3:    # probably another stealthed seed
                    return -1                               # signal to close it
        for tier in self.interests:
            for piece in tier:
                if not connection.download.have[piece]:
                    seedint = self.level_in_interests[piece]
                    self.level_in_interests[piece] += 1  # tweak it up one, so you don't duplicate effort
                    if seedint == len(self.interests) - 1:
                        self.interests.append([])
                    self._shift_over(piece,
                                self.interests[seedint], self.interests[seedint + 1])
                    self.seed_got_haves[piece] = 0       # reset this
                    self.seed_connections[connection] = piece
                    connection.upload.seed_have_list.append(piece)
                    return piece
        return -1       # something screwy; terminate connection

    def lost_peer(self, connection):
        """Process a lost peer in super-seed mode.
        
        @type connection: L{Connecter.Connection}
        @param connection: the peer connection that was lost
        
        """
        
        olddl = self.seed_connections.get(connection)
        if olddl is None:
            return
        del self.seed_connections[connection]
        self.past_ips[connection.get_ip()] = olddl
        if self.seed_got_haves[olddl] == 1:
            self.seed_got_haves[olddl] = 0
