# Written by John Hoffman
# Modified by Cameron Dale
# see LICENSE.txt for license information
#
# $Id: FileSelector.py 266 2007-08-18 02:06:35Z camrdale-guest $

"""Enable the selective downloading of files within a torrent.

@type logger: C{logging.Logger}
@var logger: the logger to send all log messages to for this module

"""

from random import shuffle
import logging

logger = logging.getLogger('DebTorrent.BT1.FileSelector')

class FileSelector:
    """Enable the selective downloading of files within a torrent.
    
    @type files: C{list} of (C{list} of C{string}, C{long})
    @ivar files: the paths and sizes of the files
    @type storage: L{Storage.Storage}
    @ivar storage: the Storage instance to use
    @type storagewrapper: L{StorageWrapper.StorageWrapper}
    @ivar storagewrapper: the StorageWrapper instance to use
    @type sched: C{method}
    @ivar sched: the mehtod to call to schedule a future invocation of a function
    @type failfunc: C{method}
    @ivar failfunc: the method to call to indicate an error has occurred
    @type downloader: unknown
    @ivar downloader: not used
    @type picker: L{PiecePicker.PiecePicker}
    @ivar picker: the PiecePicker instance to use
    @type numfiles: C{int}
    @ivar numfiles: the number of files in the download
    @type priority: C{list} of C{int}
    @ivar priority: the current priority of each file in the download::
        -1 -- do not download
         0 -- highest priority
         1 -- medium priority
         2 -- lowest priority
    @type init_priority: C{list} of C{int}
    @ivar init_priority: the initial unpickled priority of each file in the download
    @type new_priority: C{list} of C{int}
    @ivar new_priority: the new priority of each file in the download
    @type new_partials: C{list} of C{int}
    @ivar new_partials: the new list of partially completed pieces to process
    @type filepieces: C{list} of C{tuple}
    @ivar filepieces: an ordered list, one for each file, containing a tuple of
        the pieces that belong to that file
    @type numpieces: C{int}
    @ivar numpieces: the number of pieces in the download
    @type piece_priority: C{list} of C{int}
    @ivar piece_priority: the current priority for each piece in the download
    @type cancelfunc: C{method}
    @ivar cancelfunc: method to call to cancel piece downloads
    @type requestmorefunc: C{method}
    @ivar requestmorefunc: method to call to request more peers for a
        newly enabled piece download
    @type rerequestfunc: C{method}
    @ivar rerequestfunc: method to call to request more peers
    
    """
    
    def __init__(self, files, piece_lengths, bufferdir,
                 storage, storagewrapper, sched, picker, failfunc):
        """Initialize the instance.
        
        @type files: C{list} of (C{list} of C{string}, C{long})
        @param files: the paths and sizes of the files
        @type piece_lengths: C{list} of C{int}
        @param piece_lengths: the sizes of the pieces
        @type bufferdir: C{string}
        @param bufferdir: the directory to store the buffered pieces in
        @type storage: L{Storage.Storage}
        @param storage: the Storage instance to use
        @type storagewrapper: L{StorageWrapper.StorageWrapper}
        @param storagewrapper: the StorageWrapper instance to use
        @type sched: C{method}
        @param sched: the mehtod to call to schedule a future invocation of a function
        @type picker: L{PiecePicker.PiecePicker}
        @param picker: the PiecePicker instance to use
        @type failfunc: C{method}
        @param failfunc: the method to call to indicate an error has occurred
        
        """
        
        self.files = files
        self.storage = storage
        self.storagewrapper = storagewrapper
        self.sched = sched
        self.failfunc = failfunc
        self.downloader = None
        self.picker = picker
        self.cancelfunc = None
        self.requestmorefunc = None
        self.rerequestfunc = None

        storage.set_bufferdir(bufferdir)
        
        self.numfiles = len(files)
        self.priority = [-1] * self.numfiles
        self.init_priority = None
        self.new_priority = None
        self.new_partials = None
        self.filepieces = []
        total = 0L
        piece_total = 0l
        cur_piece = 0
        for file, length in files:
            if not length:
                self.filepieces.append(())
            else:
                total += length
                start_piece = cur_piece
                for cur_piece in xrange(start_piece,len(piece_lengths)+1):
                    if piece_total >= total:
                        break
                    piece_total += piece_lengths[cur_piece]
                end_piece = cur_piece-1
                if piece_total > total:
                    cur_piece -= 1
                    piece_total -= piece_lengths[cur_piece]
                pieces = range(start_piece,end_piece+1)
                self.filepieces.append(tuple(pieces))
        self.numpieces = len(piece_lengths)
        self.piece_priority = [-1] * self.numpieces
        


    def init_priorities(self, init_priority):
        """Initialize the priorities of all the files from the unpickled state.
        
        @type new_priority: C{list} of C{int}
        @param new_priority: the new file priorities
        @rtype: C{boolean}
        @return: whether the initialization was successful
        
        """
        
        try:
            assert len(init_priority) == self.numfiles
            for v in init_priority:
                assert type(v) in (type(0),type(0L))
                assert v >= -1
                assert v <= 2
        except:
            logger.warning('Initializing the priority failed', exc_info = True)
            return False
        try:
            for f in xrange(self.numfiles):
                if init_priority[f] < 0:
                    self.storage.disable_file(f)
                else:
                    self.storage.enable_file(f)
            self.storage.reset_file_status()
            self.init_priority = init_priority
        except (IOError, OSError), e:
            self.failfunc("can't open partial file for "
                          + self.files[f][0] + ': ' + str(e))
            return False
        return True

    def unpickle(self, d):
        """Unpickle the previously saved state.
        
        Data is in the format::
            d['priority'] = [file #1 priority, file #2 priority, ...]
        
        It is a list of download priorities for each file. The priority may be::
            -1 -- download disabled
             0 -- highest
             1 -- normal
             2 -- lowest

        Also see Storage.pickle and StorageWrapper.pickle for additional keys.

        @type d: C{dictionary}
        @param d: the pickled state data
        
        """
        
        if d.has_key('priority'):
            if not self.init_priorities(d['priority']):
                return
        pieces = self.storage.unpickle(d)
        init_piece_priority = self._get_piece_priority_list(self.init_priority)
        self.storagewrapper.reblock([i == -1 for i in init_piece_priority])
        self.new_partials = self.storagewrapper.unpickle(d, pieces)
        self.piece_priority = self._initialize_piece_priority(self.init_priority)


    def tie_in(self, cancelfunc, requestmorefunc, rerequestfunc):
        """Set some instance variables that weren't available at initialization.
        
        @type cancelfunc: C{method}
        @param cancelfunc: method to call to cancel piece downloads
        @type requestmorefunc: C{method}
        @param requestmorefunc: method to call to request more peers for a
            newly enabled piece download
        @type rerequestfunc: C{method}
        @param rerequestfunc: method to call to request more peers
        
        """
        
        self.cancelfunc = cancelfunc
        self.requestmorefunc = requestmorefunc
        self.rerequestfunc = rerequestfunc

        # Set up the unpickled initial priorities
        if self.init_priority:
            self.priority = self.init_priority
            self.init_priority = None

        # Set up the unpickled list of partially completed pieces
        if self.new_partials:
            shuffle(self.new_partials)
            for p in self.new_partials:
                self.picker.requested(p)
        self.new_partials = None
        
        # Schedule the processing of any early arrivals of new priorities
        if self.new_priority:
            self.sched(self.set_priorities_now)
        

    def _set_files_disabled(self, old_priority, new_priority):
        """Disable files based on a new priority setting.
        
        @type old_priority: C{list} of C{int}
        @param old_priority: the old file priorities
        @type new_priority: C{list} of C{int}
        @param new_priority: the new file priorities
        @rtype: C{boolean}
        @return: whether the disabling was successful
        
        """
        
        old_disabled = [p == -1 for p in old_priority]
        new_disabled = [p == -1 for p in new_priority]
        data_to_update = []
        for f in xrange(self.numfiles):
            if new_disabled[f] != old_disabled[f]:
                data_to_update.extend(self.storage.get_piece_update_list(f))
        buffer = []
        for piece, start, length in data_to_update:
            if self.storagewrapper.has_data(piece):
                data = self.storagewrapper.read_raw(piece, start, length)
                if data is None:
                    return False
                buffer.append((piece, start, data))

        files_updated = False        
        try:
            for f in xrange(self.numfiles):
                if new_disabled[f] and not old_disabled[f]:
                    self.storage.disable_file(f)
                    files_updated = True
                if old_disabled[f] and not new_disabled[f]:
                    self.storage.enable_file(f)
                    files_updated = True
        except (IOError, OSError), e:
            if new_disabled[f]:
                msg = "can't open partial file for "
            else:
                msg = 'unable to open '
            self.failfunc(msg + self.files[f][0] + ': ' + str(e))
            return False
        if files_updated:
            self.storage.reset_file_status()

        changed_pieces = {}
        for piece, start, data in buffer:
            if not self.storagewrapper.write_raw(piece, start, data):
                return False
            data.release()
            changed_pieces[piece] = 1
        if not self.storagewrapper.doublecheck_data(changed_pieces):
            return False

        return True        


    def _get_piece_priority_list(self, file_priority_list):
        """Create a piece priority list from a file priority list.
        
        @type file_priority_list: C{list} of C{int}
        @param file_priority_list: the file priorities
        @rtype: C{list} of C{int}
        @return: the new piece priorities
        
        """
        l = [-1] * self.numpieces
        for f in xrange(self.numfiles):
            if file_priority_list[f] == -1:
                continue
            for i in self.filepieces[f]:
                if l[i] == -1:
                    l[i] = file_priority_list[f]
                    continue
                l[i] = min(l[i],file_priority_list[f])
        return l
        

    def _initialize_piece_priority(self, new_priority):
        """Initialize the piece priorities on startup.
        
        @type new_priority: C{list} of C{int}
        @param new_priority: the new file priorities
        @rtype: C{list} of C{int}
        @return: the new piece priorities
        
        """
        
        new_piece_priority = self._get_piece_priority_list(new_priority)
        pieces = range(self.numpieces)
        shuffle(pieces)
        for piece in pieces:
            self.picker.set_priority(piece,new_piece_priority[piece])
        self.storagewrapper.reblock([i == -1 for i in new_piece_priority])

        return new_piece_priority        


    def _set_piece_priority(self, new_priority):
        """Disable pieces base on a new file priority setting.
        
        @type new_priority: C{list} of C{int}
        @param new_priority: the new file priorities
        @rtype: C{list} of C{int}
        @return: the new piece priorities
        
        """
        
        was_complete = self.storagewrapper.am_I_complete()
        new_piece_priority = self._get_piece_priority_list(new_priority)
        pieces = range(self.numpieces)
        shuffle(pieces)
        new_blocked = []
        new_unblocked = []
        for piece in pieces:
            self.picker.set_priority(piece,new_piece_priority[piece])
            o = self.piece_priority[piece] == -1
            n = new_piece_priority[piece] == -1
            if n and not o:
                new_blocked.append(piece)
            if o and not n:
                new_unblocked.append(piece)
        if new_blocked:
            self.cancelfunc(new_blocked)
        self.storagewrapper.reblock([i == -1 for i in new_piece_priority])
        if new_unblocked:
            self.requestmorefunc(new_unblocked)
        if was_complete and not self.storagewrapper.am_I_complete():
            self.rerequestfunc()

        return new_piece_priority        


    def set_priorities_now(self, new_priority = None):
        """Set the new priorities.
        
        @type new_priority: C{list} of C{int}
        @param new_priority: the new file priorities
            (optional, defaults to using the saved L{new_priority})
        
        """

        if not new_priority:
            new_priority = self.new_priority
            self.new_priority = None    # potential race condition
            if not new_priority:
                return
        old_priority = self.priority
        self.priority = new_priority
        if not self._set_files_disabled(old_priority, new_priority):
            return
        self.piece_priority = self._set_piece_priority(new_priority)

    def set_priorities(self, new_priority):
        """Schedule the future setting of the new priorities.
        
        @type new_priority: C{list} of C{int}
        @param new_priority: the new file priorities
        
        """

        self.new_priority = new_priority
        # Only set the new priorities if the tie_in initialization is complete
        if self.requestmorefunc:
            self.sched(self.set_priorities_now)
        
    def set_priority(self, f, p):
        """Set the priority of a single file.
        
        @type f: C{int}
        @param f: the index of the file to set the priority of
        @type p: C{int}
        @param p: the new priority for the file
        
        """
        
        new_priority = self.get_priorities()
        new_priority[f] = p
        self.set_priorities(new_priority)

    def get_priorities(self):
        """Get the current (or soon to be set) file priorities.
        
        @rtype: C{list} of C{int}
        @return: the current file priorities
        
        """
        
        priority = self.new_priority
        if not priority:
            priority = self.init_priority
            if not priority:
                priority = self.priority    # potential race condition
        return [i for i in priority]

    def __setitem__(self, index, val):
        """Set the priority for the file.
        
        @type index: C{int}
        @param index: the index of the file to set the priority of
        @type val: C{int}
        @param val: the new priority for the file
        
        """
        
        self.set_priority(index, val)

    def __getitem__(self, index):
        """Get the priority for the file.
        
        @type index: C{int}
        @param index: the index of the file to get the priority of
        @rtype: C{int}
        @return: the priority for the file

        """
        
        try:
            return self.new_priority[index]
        except:
            try:
                return self.init_priority[index]
            except:
                return self.priority[index]


    def finish(self):
        """Delete disabled files when the download is being shutdown."""
        for f in xrange(self.numfiles):
            if self.priority[f] == -1:
                self.storage.delete_file(f)

    def pickle(self):
        """Pickle the current state for later.
        
        @see: L{unpickle}
        @rtype: C{dictionary}
        @return: the pickled data
        
        """
        
        d = {'priority': self.priority}
        try:
            s = self.storage.pickle()
            sw = self.storagewrapper.pickle()
            for k in s.keys():
                d[k] = s[k]
            for k in sw.keys():
                d[k] = sw[k]
        except (IOError, OSError):
            pass
        return d
