/***************************************************************************
 *   Copyright (C) 2005 by Adam Treat                                      *
 *   treat@kde.org                                                         *
 *                                                                         *
 *   Copyright (C) 2004 by Scott Wheeler                                   *
 *   wheeler@kde.org                                                       *
 *                                                                         *
 *   Copyright (C) 2000 Trolltech AS.  All rights reserved.                *
 *   info@trolltech.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef PROJECT_H
#define PROJECT_H

#include "datatableinterface.h"

#include <qmap.h>
#include <qstring.h>
#include <qptrlist.h>
#include <qptrdict.h>
#include <qfeatures.h>
#include <qstringlist.h>
#include <qobjectlist.h>

#include <kshortcut.h>
#include <kdirwatch.h>
#include <kconfig.h>
#include <klocale.h>

class QWidgetStack;
class KAction;
class DataTable;
class DataReport;
class DataTableSearch;
class DatabaseConnection;

typedef QValueList<DataTable *> DataTableList;
typedef QValueList<DataReport *> DataReportList;
typedef QValueList<DataTableSearch> DataSearchList;

class Project : public DataTableInterface
{
    friend class DataTable;
    friend class DataTableBox;
    friend class DatabaseConnection;

public:
    Project( QWidgetStack *dataTableStack, const QString &fileName = "" );

    virtual ~Project();

    virtual void initialize();
    //UGLY HACK
    virtual void uglyInitialize();

    QString fileName() const;

    virtual void newProject();
    void openProject();
    void saveProject();
    void saveAsProject();
    virtual void closeProject();
    void editProject();
    virtual void clipboard();

    void setModified( bool b );
    QString makeAbsolute( const QString &f );
    QString makeRelative( const QString &f );

    bool openDatabase( const QString &connection, bool suppressDialog = true );
    void closeDatabase( const QString &connection );

    QPtrList<DataTable> dataTablesByPtr() const;
    QPtrList<DatabaseConnection> databaseConnections() const;
    void setDatabaseConnections( const QPtrList<DatabaseConnection> &lst );
    void addDatabaseConnection( DatabaseConnection *conn );
    void removeDatabaseConnection( const QString &conn );
    void removeAllDatabaseConnections();
    DatabaseConnection *databaseConnection( const QString &name );
    QStringList databaseConnectionList();
    QStringList databaseTableList( const QString &connection );
    QStringList databaseFieldList( const QString &connection, const QString &table );

    bool loadConnections();

    void setDataTables( const QPtrList<DataTable> &lst );
    void addDataTable( DataTable *table );
    void addDataReport( DataReport *report );
    void removeDataTable( const QString &table );
    void removeDataReport( const QString &report );
    DataTable *dataTable( const QString &name );
    QStringList dataTableList();

    bool loadDataTables();

    //    ******************** DataTableCollection ********************
    // The following group of functions implement the DataTableInterface API.
    virtual QString name() const;

    virtual QStringList dataTables() const;
    //     virtual void createDataTable( const QString &name );

    virtual QString dataTable() const;
    virtual void setDataTable( const QString &dataTable );

    virtual void configureTable();
    virtual void addVirtualField();
    virtual void removeTable();
    virtual void moveTableUp();
    virtual void moveTableDown();
    virtual void sortTables();
    virtual void configureReport();
    virtual void removeReport();
    virtual void refreshReport();

    virtual void editSearch();
    virtual void removeSearch();
    DataSearchList searchList();
    virtual void invokeSearch( const QString &search );
    virtual void appendSearch( const DataTableSearch &search );
    virtual void removeSearch( const DataTableSearch &search );

    virtual void advancedSearchRequested();
    virtual void createKugarReport();
    virtual void createKugarReport( const QString &parentTable );

    void createDataTable();
    void createChildDataTable( const QString &name );
    void createSearchDataTable();
    void createForeignDataTable();
    void createDatabaseDataTable();

    QObject *object() const;

    class ActionHandler;

    virtual DataTable *currentDataTable() const;
    virtual DataReport *currentDataReport() const;
    virtual DataTable *rootOfCurrentDataTable();
    virtual DataTableList dataTablesInCurrentTree();
    virtual DataTableList dataTablesInDataTableTree( DataTable *dataTable );
    virtual DataTableList childrenOfDataTable( DataTable *dataTable );
    virtual DataTableList dataTablesInCurrentPeerTree();
    virtual DataTableList dataTablesInDataTablePeerTree( DataTable *dataTable );

    virtual void scrollTableUp();
    virtual void scrollTableDown();
    void scrollTabLeft();
    void scrollTabRight();
    void firstRecord();
    void previousRecord();
    void nextRecord();
    void lastRecord();
    void commit();
    void insertRecord();
    void changeRecord();
    void deleteRecord();

    virtual void setSplashMessage( const QString &txt );
    virtual void setupDataTable( DataTable *dataTable, const QString &iconName );
    virtual void setupDataTable( DataTable *dataTable, const QString &iconName, DataTable* parentTable );
    virtual void setIconName( DataTable* dataTable, const QString &iconName );
    virtual void setupDataReport( DataReport *dataReport, const QString &iconName );
    virtual void setupDataReport( DataReport *dataReport, const QString &iconName, DataTable *parentTable );
    virtual void setIconName( DataReport* dataReport, const QString &iconName );
    QString uniqueDataTableName( const QString &suggest = i18n( "DataTable" ) ) const;
    virtual QWidgetStack *dataTableStack() const;
    DataTable *dataTableByName( const QString &name ) const;

    void addName( const QString &name );
    void removeName( const QString &name );

protected:
    virtual void raise( DataTable *dataTable );
    virtual void raise( DataReport *datareport );

    bool containsDataTableFile( const QString &file ) const;

    QString dataTableNameDialog( const QString &caption = i18n( "Create New DataTable" ),
                                 const QString &suggest = QString::null,
                                 bool forceUnique = true ) const;

    //    ******************** DataTableCollection ********************

private:
    bool m_modified;
    QString m_projectIcon;
    QString m_fileName;
    QPtrList<DatabaseConnection> m_dbConnections;
    QPtrList<DataTable> m_dataTables;
    QPtrList<DataReport> m_dataReports;
    DataSearchList m_searchList;

    //    ******************** DataTableCollection ********************

    QWidgetStack *m_dataTableStack;
    ActionHandler *m_actionHandler;

    QStringList m_dataTableNames;
    //    ******************** DataTableCollection ********************
};

inline QString sanitize( const QString &str )
{
    //Replace special xml characters like '&','<' '>',''','"'
    //with there xml markup equivalent (eg: &amp, &lt, etc)
    QString value = str;
    value = value.replace( '&', "&amp;" );
    value = value.replace( '<', "&lt;" );
    value = value.replace( '>', "&gt;" );
    value = value.replace( '\'', "&apos;" );
    value = value.replace( '"', "&quot;" );
    return value;
}

inline QString deSanitize( const QString &str )
{
    //Replace xml markup characters (eg: &amp, &lt, etc)
    //with there equivalent like '&','<' '>',''','"'
    QString value = str;
    value = value.replace( "&amp;", "&" );
    value = value.replace( "&lt;", "<" );
    value = value.replace( "&gt;", ">" );
    value = value.replace( "&apos;", "'" );
    value = value.replace( "&quot;", "\"" );
    return value;
}

class Project::ActionHandler : public QObject
{
    Q_OBJECT
public:
    ActionHandler( Project *collection );

private:
    KAction *createAction( const QString &text,
                           const char *slot,
                           const char *name,
                           const QString &icon = QString::null,
                           const KShortcut &shortcut = KShortcut() );

private slots:
    void slotConfigureTable()
    {
        m_project->configureTable();
    }
    void slotAddVirtualField()
    {
        m_project->addVirtualField();
    }
    void slotConfigureReport()
    {
        m_project->configureReport();
    }
    void slotRemoveTable()
    {
        m_project->removeTable();
    }
    void slotMoveTableUp()
    {
        m_project->moveTableUp();
    }
    void slotMoveTableDown()
    {
        m_project->moveTableDown();
    }
    void slotRemoveReport()
    {
        m_project->removeReport();
    }
    void slotRefreshReport()
    {
        m_project->refreshReport();
    }
    void slotNewProject()
    {
        m_project->newProject();
    }
    void slotOpenProject()
    {
        m_project->openProject();
    }
    void slotSaveProject()
    {
        m_project->saveProject();
    }
    void slotSaveAsProject()
    {
        m_project->saveAsProject();
    }
    void slotCloseProject()
    {
        m_project->closeProject();
    }
    void slotEditProject()
    {
        m_project->editProject();
    }
    void slotClipboard()
    {
        m_project->clipboard();
    }
    void slotEditSearch()
    {
        m_project->editSearch();
    }
    void slotRemoveSearch()
    {
        m_project->removeSearch();
    }
    void slotCreateDataTable()
    {
        m_project->createDataTable();
    }
    void slotCreateSearchDataTable()
    {
        m_project->createSearchDataTable();
    }
    void slotCreateDatabaseDataTable()
    {
        m_project->createDatabaseDataTable();
    }
    void slotCreateForeignDataTable()
    {
        m_project->createForeignDataTable();
    }
    void slotAdvancedSearchClicked()
    {
        m_project->advancedSearchRequested();
    }
    void slotCreateKugarReport()
    {
        m_project->createKugarReport();
    }
    void slotScrollTableUp()
    {
        m_project->scrollTableUp();
    }
    void slotScrollTableDown()
    {
        m_project->scrollTableDown();
    }
    void slotScrollTabLeft()
    {
        m_project->scrollTabLeft();
    }
    void slotScrollTabRight()
    {
        m_project->scrollTabRight();
    }
    void slotFirstRecord()
    {
        m_project->firstRecord();
    }
    void slotPreviousRecord()
    {
        m_project->previousRecord();
    }
    void slotNextRecord()
    {
        m_project->nextRecord();
    }
    void slotLastRecord()
    {
        m_project->lastRecord();
    }
    void slotCommit()
    {
        m_project->commit();
    }
    void slotInsertRecord()
    {
        m_project->insertRecord();
    }
    void slotChangeRecord()
    {
        m_project->changeRecord();
    }
    void slotDeleteRecord()
    {
        m_project->deleteRecord();
    }
signals:
    void signalSelectedItemsChanged();
    void signalCountChanged();

private:
    Project *m_project;
    QPtrList<KAction*> *m_parentFieldActions;
};

#endif
