/*
 * Danpei -- a GTK+ based Image Viewer
 * Copyright (C) 2001-2003 Shinji Moiino
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/* These codes are mostly taken from:
 * gimageview-0.1.7 Copyright (C) 2001 Takuro Ashie
 */
/* pcx.h */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "config.h"
#include "pcx.h"

/* Static function declarations. */
static gboolean pcx_get_header (const gchar*, PcxInfo*             );
static gboolean pcx_readline   (FILE*       , guchar*, gint, guint8);

/* Function definitions. */
/*
 * @pcx_load
 *
 *
 *
 */
guchar *pcx_load(const gchar *filename, gint *widthp, gint *heightp) {
  FILE *fd;
  PcxHeader pcx_header;
  gint width, height, bytes, x, y, c, t1, t2, pix, ptr;
  guchar *dest, *line, *line0, *line1, *line2, *line3;
  guchar cmap[768];

  /* Initialize local variables. */
  ptr = 0;
  dest = line = line0 = line1 = line2= line3 = NULL;

  if (filename == NULL || widthp == NULL || heightp == NULL) { return NULL; }

  if ((fd = fopen(filename, "rb")) == NULL) { return NULL; }

  if (fread (&pcx_header, 128, 1, fd) == 0) {
     fclose(fd);
     return NULL;
  }

  if(pcx_header.manufacturer != 10) {
     fclose (fd);
     return NULL;
  }
	
  width =  *widthp  = pcx_header.x2 - pcx_header.x1 + 1;
  height = *heightp = pcx_header.y2 - pcx_header.y1 + 1;
  bytes = pcx_header.bytesperline;
	
  dest = g_new0 (guchar, width * height * 3);

  if (pcx_header.planes == 1 && pcx_header.bpp == 1) {
    /* loading 1-bpp pcx */
    dest = (guchar*)g_malloc(width * 3 * sizeof(guchar));
    line = (guchar*)g_malloc(bytes * sizeof(guchar));
    for (y = 0; y < height; y++) {
      if (!pcx_readline(fd, line, bytes, pcx_header.compression)) { break; }
      for (x = 0; x < width; x++) {
        if (line[x / 8] & (128 >> (x % 8))) {
          dest[ptr++]= 0xff;
          dest[ptr++]= 0xff;
          dest[ptr++]= 0xff;
        } 
        else {
          dest[ptr++]= 0;
          dest[ptr++]= 0;
          dest[ptr++]= 0;
        }
      }
    }
    g_free (line);	
  } 
  else if (pcx_header.planes == 4 && pcx_header.bpp == 1) {
    /* loading 4-bpp pcx */
    line0 = (guchar*)g_malloc(bytes * sizeof(guchar));
    line1 = (guchar*)g_malloc(bytes * sizeof(guchar));
    line2 = (guchar*)g_malloc(bytes * sizeof(guchar));
    line3 = (guchar*)g_malloc(bytes * sizeof(guchar));
    for (y = 0; y < height; y++) {
      if (!pcx_readline(fd, line0, bytes, pcx_header.compression)) { break; }
      if (!pcx_readline(fd, line1, bytes, pcx_header.compression)) { break; }
      if (!pcx_readline(fd, line2, bytes, pcx_header.compression)) { break; }
      if (!pcx_readline(fd, line3, bytes, pcx_header.compression)) { break; }
      for (x = 0; x < width; x++) {
	    t1 = x / 8;
	    t2 = (128 >> (x % 8));
	    pix = (((line0[t1] & t2) == 0) ? 0 : 1) +
              (((line1[t1] & t2) == 0) ? 0 : 2) +
              (((line2[t1] & t2) == 0) ? 0 : 4) +
              (((line3[t1] & t2) == 0) ? 0 : 8);
	    pix *= 3;
	    dest[ptr++] = pcx_header.colormap[pix];
	    dest[ptr++] = pcx_header.colormap[pix + 1];
	    dest[ptr++] = pcx_header.colormap[pix + 2];
      }
    }
    g_free(line0);
    g_free(line1);
    g_free(line2);
    g_free(line3);
  } 
  else if (pcx_header.planes == 1 && pcx_header.bpp == 8) {
    /* loading 8-bpp pcx */
    line = (guchar*)g_malloc(bytes * sizeof(guchar));
    fseek(fd, -768L, SEEK_END);
    if (fread(cmap, 768, 1, fd) == 0) { 
      g_free (dest);
      return NULL;
    }
    fseek(fd, 128, SEEK_SET);
    for (y = 0; y < height; y++) {
      if (!pcx_readline(fd, line, bytes, pcx_header.compression)) { break; }
      for (x = 0; x < width; x++) {
	    pix = line[x] * 3;
	    dest[ptr++] = cmap[pix];
	    dest[ptr++] = cmap[pix + 1];
	    dest[ptr++] = cmap[pix + 2];
      }
    }
    g_free(line);
  } 
  else if (pcx_header.planes == 3 && pcx_header.bpp == 8) {
    /* loading 24-bpp pcx(rgb) */
    dest = (guchar*) g_malloc(width * 3 * sizeof(guchar));
    line = (guchar*) g_malloc(bytes * 3 * sizeof(guchar));
    for (y = 0; y < height; y++) {
      for (c= 0; c < 3; c++) {
        if (!pcx_readline (fd, line, bytes, pcx_header.compression)) { break; }
        for (x= 0; x < width; x++) {
          ptr = x * 3 + y * width * 3 + c;
          dest[ptr] = line[x];
        }
      }
    }
    g_free (line);
  } 
  else {
    fclose (fd);
    g_free (dest);
    return NULL;
  }

  /* Normal end. */
  fclose(fd);
  return dest;
}


/* Static function definitions. */
/*
 * @pcx_get_header
 *
 *  Get pcx header. 
 *  If header is valid, return TRUE, otherwise return FALSE.
 *
 */
static gboolean pcx_get_header (const gchar *filename, PcxInfo *info) {
  FILE *fd;
  PcxHeader pcx_header;
	
  if ((fd = fopen(filename, "rb")) == NULL) { return FALSE; }

  if (fread (&pcx_header, 128, 1, fd) == 0) {
     fclose(fd);
     return FALSE;
  }

  if(pcx_header.manufacturer != 10) {
     fclose(fd);
     return FALSE;
  }
	
  info->width = pcx_header.x2 - pcx_header.x1 + 1;
  info->height = pcx_header.y2 - pcx_header.y1 + 1;
  if (pcx_header.planes == 1 && pcx_header.bpp == 1) {
     info->ncolors = 1;
  } else if (pcx_header.planes == 4 && pcx_header.bpp == 1) {
     info->ncolors = 4;
  } else if (pcx_header.planes == 1 && pcx_header.bpp == 8) {
     info->ncolors = 8;
  } else if (pcx_header.planes == 3 && pcx_header.bpp == 8) {
     info->ncolors = 24;
  } else {
     fclose (fd);
     return FALSE;
  }
	
  fclose (fd);
  return TRUE;
}

/*
 * @pcx_readline
 *
 *
 *
 */
static gboolean pcx_readline(FILE   *fd    ,
                             guchar *buffer,
                             gint   bytes  ,
                             guint8 compression ) {
  guchar count;
  gint   value;

  /* Initialize local variables. */
  count = 0;
  value = 0;

  if (compression) {
    while (bytes--) {
      if (count == 0) {
        if ((value = fgetc(fd)) == EOF) { return FALSE; }
        if (value < 0xc0) {
          count= 1;
        } 
        else {
          count= value - 0xc0;
          if ((value = fgetc(fd)) == EOF) { return FALSE; }
        }
      } 
      count--;
      *(buffer++) = (guchar)value;
    }
  } 
  else {
    if (fread(buffer, bytes, 1, fd) == 0) { return FALSE; }
  }
  return TRUE;
}


