=begin rdoc

This program is copyright 2006 by Vincent Fourmond.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
=end

# This program enables one to print out as text the data read by the
# backends. Can come in dead useful for debugging, or if you intend to plot
# with another data

require 'CTioga/backends'
require 'CTioga/log'
require 'optparse'

module CTioga

  class CTable
    include Log
    include Backends

    attr_accessor :block
    attr_accessor :args
    attr_accessor :print_banner

    # This can be used by various scripts to add hooks on the parser.
    # You can alternatively use your own parser beforehand, but you lose
    # the benefits of the online help.
    attr_accessor :parser

    def initialize
      init_logger
      init_backend_structure
      @args = []
      @block = proc { |set,data,*a| 
        puts "# #{set}"
        data.each do |x,y|
          puts "#{x}\t#{y}"
        end
      }

      @print_banner = true

      @parser = OptionParser.new
      prepare_backend_options(@parser)

      @parser.separator ""
      @parser.separator "Code execution"

      @parser.on("-e", "--execute BLOCK",
                 "Executes the given ruby code for each set, " ,
                 "yielding the set name and its data for each." ,
                 "Use 'set' to refer to the set's name, and 'data' " ,
                 "for it's data"
                 ) do |code|
        self.block = eval "proc { |set,data,*args| #{code}}"
      end

      @parser.on("-f", "--file FILE",
                 "Same as -e except it's specifying a file which is ",
                 "read an executed with the same parameters as the BLOCK"
                 ) do |file|
        self.block = eval "proc { |set,data,*args| #{IO.readlines(file).join}}"
      end

      @parser.on("-r", "--require FILE",
                 "Ask ruby to require the file before data processing.",
                 "You can use it to declare some external functions."
                 ) do |file|
        require file
      end
      
      @parser.on("-a", "--arg a",  
                 "Provides additionnal arguments to your code") do |arg|
        @args << arg
      end

    end

    def process_data_set(set)
      data = xy_data_set(set)
      @block.call(set,data,*@args)
    end
    
    # Runs the program with the given command-line arguments. If a block
    # is given, use this block rather than the default one. Be careful,
    # though, as command-line arguments can still override this.
    def run(cmd_line_args,*extra_args, &a)
      $stderr.puts <<"EOBANNER" if @print_banner
This is ctable version #{CTioga.version}, copyright (C) 2006 Vincent Fourmond
ctable comes with absolutely NO WARRANTY. This is free software, you are 
welcome to redistribute it under certain conditions. See the COPYING file
in the original tarball for details. You are also welcome to contribute if
you like it, see in the manual page where to ask.
EOBANNER

      # Use the user-given block
      if block_given?
        @block = a
      end
      @args = extra_args

      @parser.order(cmd_line_args) do |spec|
        # We use Backend#expand to leave room for the backend to
        # interpret the text as many different sets.
        expand_spec(spec).each do |set|
          process_data_set(set)
        end
      end
    end

    # A simple convenience function.
    def self.run(args, &a)
      self.new.run(args,&a)
    end
    
  end
end
