/*
 * The Cryptonit security software suite is developped by IDEALX
 * Cryptonit Team (http://IDEALX.org/ and http://cryptonit.org).
 *
 * Copyright 2003-2006 IDEALX
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version
 * 2 as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301, USA. 
 *
 * In addition, as two special exceptions:
 *
 * 1) IDEALX S.A.S gives permission to:
 *  * link the code of portions of his program with the OpenSSL library under
 *    certain conditions described in each source file
 *  * distribute linked combinations including the two, with respect to the
 *    OpenSSL license and with the GPL
 *
 * You must obey the GNU General Public License in all respects for all of the
 * code used other than OpenSSL. If you modify file(s) with this exception,
 * you may extend this exception to your version of the file(s), but you are
 * not obligated to do so. If you do not wish to do so, delete this exception
 * statement from your version, in all files (this very one along with all
 * source files).

 * 2) IDEALX S.A.S acknowledges that portions of his sourcecode uses (by the
 * way of headers inclusion) some work published by 'RSA Security Inc.'. Those
 * portions are "derived from the RSA Security Inc. PKCS #11Cryptographic
 * Token Interface (Cryptoki)" as described in each individual source file.
 */

#ifndef _AddressBook_hh
#define _AddressBook_hh



#include <vector>
#include <string>
#include <iostream>

#include "Contact.hh"

#include "DirectoryService.hh"


#define ADDRESSBOOK_DEFAULT_FILENAME "AddressBook.conf"


namespace Cryptonit 
{

   /** 
    * AddressBook
    * This class represents an address book which contains a list of contacts.
    *
    */
   class AddressBook
   {
	
      private :
	
	 /** The AddressBook itself */
	 DirectoryService* addressBook;

	 /** Conatains the current AddressBook URI */
	 std::string currentURI;
    
      public :


	 /** Constructs an empty AddressBook
	  *
	  */
	 AddressBook();
	

	 /** Constructs an AddressBook from a given URI.
	  *  The backend is determined by the 'scheme' part of the URI.
	  */
	 AddressBook( const std::string uri );


	 /** 
	  * Destructs this AddressBook
	  */
	 ~AddressBook();
    

	 /** Write the address book on disk.
	  *  If no filename is specified write to the file which
	  *  it was loaded.
	  *
	  *  @param fileName : the address book filename
	  *  @return true if the operation succeed.
	  */
	 bool save( const std::string fileName = "" );


	 /** Add a contact in the Address Book.
	  *  @return true if the operation succeed.
	  *
	  * @param contact : the contact to add.
	  *
	  */
	 bool addContact( Contact& contact );
    

	 /** Add a contact in the Address Book.
	  *  @return true if the operation succeed.
	  *
	  * @param contact : a pointer on the contact to be added.
	  *
	  */
	 bool addContact( Contact* contact );


	 /** Removes a contact in the Address Book.
	  *  @return true if the operation succeed.
	  *
	  * @param name : the contact to remove.
	  *
	  */
	 bool removeContact( const std::string name );
    

	 /** Return a contact desgigned by its name.
	  *  @return the corresponding contact, NULL if it 
	  *          does not exists.
	  *
	  *  @param name : contact name
	  */
	 Contact* getContact( const std::string name );


	 /** Return a contact desgigned by its name.
	  *  @return the corresponding contact, NULL if it 
	  *          does not exists.
	  *
	  *  @param name : contact name
	  */
	 Entry* getContactDSE( const std::string name );



	 /** Get a vector of the names of the addressBook 
	  * 
	  * 
	  * 
	  * @return a vector of name
	  */	
	 std::vector<std::string>* getAllContactName();


	 /** Returns a specified contact information, if there is more than
	  *  one value, the first is returned.
	  *  @return contact information if it exists.
	  *
	  *  @param name : contact name
	  *  @param infoName : information name
	  */
	 std::string getContactInfo( const std::string name, const std::string infoName );


	 /** Returns a specified contact informations, all values are returned
	  *  in form of a vector.
	  *  @return contact informations if it exists.
	  *
	  *  @param name : contact name
	  *  @param infoName : information name
	  */
	 std::vector<std::string> getContactInfos( const std::string name, const std::string infoName );

   };
}
#endif
