/*
 *  Device-mapper utilities for cryptmount
 *  (C)Copyright 2005-2011, RW Penney
 */

/*
    This file is part of cryptmount

    cryptmount is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    cryptmount is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <config.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#if HAVE_NANOSLEEP
#  include <time.h>
#endif

#if defined(HAVE_LIBDEVMAP)
#  include <libdevmapper.h>
#else
#  error libdevmapper headers are needed to build cryptmount
#endif

#include "cryptmount.h"
#include "dmutils.h"

#ifndef UDEV_QUEUE_DIR
#  define UDEV_QUEUE_DIR "/dev/.udev/queue"
#endif
#ifndef UDEV_QUEUE_FILE
#  define UDEV_QUEUE_FILE "/dev/.udev/queue.bin"
#endif
int udev_queue_size();


struct dm_task *devmap_prepare(int type, const char *ident)
    /* prepare device-mapper task structure */
{   struct dm_task *dmt=NULL;

    dmt = dm_task_create(type);
    if (dmt != NULL) {
        if (!dm_task_set_name(dmt, ident)) {
            dm_task_destroy(dmt);
            dmt = NULL;
        }
    }

    return dmt;
}


int devmap_path(char **buff, const char *ident)
    /* create device-mapper full pathname from target description */
{   size_t pfxlen, sfxlen;

    pfxlen = strlen(dm_dir());
    sfxlen = strlen(ident);
    *buff = (char*)realloc((void*)(*buff), (pfxlen + sfxlen + 2));

    snprintf(*buff, (pfxlen + sfxlen + 2), "%s/%s", dm_dir(), ident);

    return (int)(pfxlen + sfxlen + 1);
}


int devmap_create(const char *ident, uint64_t blk0, uint64_t blklen,
                const char *tgttype, const char *params)
    /* create new device-mapper target & associated device node: */
{   struct dm_task *dmt=NULL;
    struct dm_info dmi;
    char *devpath=NULL;
    struct stat sbuff;
    mode_t mode;
    dev_t dev;

    /* create device-mapper target: */
    if ((dmt = devmap_prepare(DM_DEVICE_CREATE, ident)) == NULL) {
        fprintf(stderr, "failed to initialize device-mapper task\n");
        return ERR_DMSETUP;
    }
    if (!dm_task_add_target(dmt, blk0, blklen, tgttype, params)) {
        fprintf(stderr, "failed to add device-mapper target \"%s\" { %s }\n",
                tgttype, params);
        return ERR_DMSETUP;
    }
    if (!dm_task_run(dmt)) {
        fprintf(stderr, "device-mapper task failed\n");
        return ERR_DMSETUP;
    }
    if (!dm_task_get_info(dmt, &dmi)) {
        fprintf(stderr, "device-mapper info not available\n");
        return ERR_DMSETUP;
    }
    dm_task_destroy(dmt);

    /* create device node (below /dev?): */
    mode = S_IFBLK | S_IRUSR | S_IWUSR;
    dev = makedev(dmi.major, dmi.minor);
    devmap_path(&devpath, ident);
    if (stat(devpath, &sbuff) != 0 && mknod(devpath, mode, dev) != 0) {
        fprintf(stderr, "device \"%s\" (%u,%u) creation failed\n",
                devpath, dmi.major, dmi.minor);
        return ERR_BADDEVICE;
    }

    if (devpath != NULL) free((void*)devpath);

    return ERR_NOERROR;
}


int devmap_dependencies(const char *ident, unsigned *count, dev_t **devids)
{   struct dm_task *dmt=NULL;
    struct dm_deps *deps;
    unsigned i;
    int eflag=ERR_NOERROR;

    if ((dmt = devmap_prepare(DM_DEVICE_DEPS, ident)) == NULL) {
        fprintf(stderr, "failed to initialize device-mapper task\n");
        eflag = ERR_DMSETUP;
        goto bail_out;
    }
    if (!dm_task_run(dmt)) {
        fprintf(stderr, "device-mapper task failed\n");
        eflag = ERR_DMSETUP;
        goto bail_out;
    }

    if ((deps = dm_task_get_deps(dmt)) == NULL) {
        eflag = ERR_DMSETUP;
        goto bail_out;
    }

    /* copy device info into fresh array: */
    *count = deps->count;
    *devids = (dev_t*)malloc((size_t)(deps->count * sizeof(dev_t)));
    for (i=0; i<deps->count; ++i) (*devids)[i] = (dev_t)deps->device[i];

  bail_out:

    if (dmt != NULL) dm_task_destroy(dmt);

    return eflag;
}


int devmap_remove(const char *ident)
    /* remove device-mapper target and associated device */
{   struct dm_task *dmt=NULL;
    struct dm_info dmi;
    struct stat sbuff;
    char *devpath=NULL;
    int eflag = ERR_NOERROR;

    /* check device-mapper target is configured & get info: */
    if (!is_configured(ident, &dmi)) {
        eflag = ERR_BADDEVICE;
        goto bail_out;
    }

    /* remove device node (below /dev?): */
    devmap_path(&devpath, ident);
    if (stat(devpath, &sbuff) != 0) {
        fprintf(stderr, "unable to stat() device node\n");
        eflag = ERR_DMSETUP;
        goto bail_out;
    }
    if ((uint32_t)major(sbuff.st_rdev) == dmi.major
      && (uint32_t)minor(sbuff.st_rdev) == dmi.minor) {
        unlink(devpath);
    } else {
        fprintf(stderr,"device \"%s\" doesn't match device-mapper info (%d,%d)\n", devpath, dmi.major, dmi.minor);
        eflag = ERR_BADDEVICE;
        goto bail_out;
    }

    /* remove device-mapper target: */
    if ((dmt = devmap_prepare(DM_DEVICE_REMOVE, ident)) == NULL) {
        fprintf(stderr, "failed to initialize device-mapper task\n");
        eflag = ERR_DMSETUP;
        goto bail_out;
    }
    if (!dm_task_run(dmt)) {
        fprintf(stderr, "device-mapper task failed\n");
        eflag = ERR_DMSETUP;
        goto bail_out;
    }

  bail_out:

    if (dmt != NULL) dm_task_destroy(dmt);
    if (devpath != NULL) free((void*)devpath);

    return eflag;
}


int is_configured(const char *ident, struct dm_info *dminfo)
    /* check if device-mapper target has been setup & (optionally) get info */
{   struct dm_task *dmt=NULL;
    struct dm_info *dmi, dmi_local;
    int config=1;

    dmi = (dminfo != NULL ? dminfo : &dmi_local);

    /* create device-mapper target: */
    if (ident == NULL
      || (dmt = devmap_prepare(DM_DEVICE_INFO, ident)) == NULL
      || !dm_task_run(dmt)
      || !dm_task_get_info(dmt, dmi)) {
        config = 0;
    }
    if (dmt != NULL) dm_task_destroy(dmt);

    return config;
}


int udev_settle()
    /* Allow time for udev events to be processed */
{   double totdelay = 0.0, inc;
    time_t starttime;
    struct stat sbuff;
#if HAVE_NANOSLEEP
    struct timespec delay;
#endif
    int settling;
    const double timeout = 5.0;

    /* This routine mitigates apparent race-conditions
     * between udev events which may temporarily take ownership of
     * and rename newly created devices, thereby causing
     * other processes to fail if they try to destroy
     * or reconfigure those devices at the same time.
     * Whether this is the responsibilty of kernel-level functions
     * to resolve, or for user applications to mitigate,
     * is open to debate. */

    time(&starttime);

#if HAVE_NANOSLEEP
    delay.tv_sec = 0.0;
    delay.tv_nsec = 2e7;
    inc = delay.tv_sec + delay.tv_nsec * 1e-9;
#else
    inc = 1.0;
#endif

    /* Keep waiting until there are no more queued udev events: */
    do {
#if HAVE_NANOSLEEP
        nanosleep(&delay, NULL);
#else
        sleep((unsigned)floor(inc + 0.5));
#endif
        totdelay += inc;

        settling = 0;

        /* Older versions of udev place events in separate directory */
        if (stat(UDEV_QUEUE_DIR, &sbuff) == 0) {
            settling |= ((starttime - sbuff.st_mtime) < 100 * timeout);
        }

        /* Newer versions of udev place events in a queue file */
        settling |= (udev_queue_size() > 0);
    } while (settling && totdelay < timeout);

    return settling;
}


int udev_queue_size()
    /* Count number of unprocessed udev events in queue.bin file */
{   FILE *fp;
    unsigned long long seqnum;
    unsigned short skiplen;
    int nqueued = 0;

    fp = fopen(UDEV_QUEUE_FILE, "rb");
    if (fp == NULL) return 0;
    if (fread((void*)&seqnum, sizeof(seqnum), (size_t)1, fp) != 1) return 0;

    for (;;) {
        skiplen = 0;
        if (fread((void*)&seqnum, sizeof(seqnum), (size_t)1, fp) != 1
          || fread((void*)&skiplen, sizeof(skiplen), (size_t)1, fp) != 1) break;

        if (skiplen > 0) {
            void *buff = malloc((size_t)skiplen);
            nqueued += (fread(buff, (size_t)skiplen, (size_t)1, fp) == 1);
            free(buff);
        } else {
            --nqueued;
        }
    }
    fclose(fp);

    return nqueued;
}

/*
 *  (C)Copyright 2005-2011, RW Penney
 */
