#
# Copyright (C) 2005  Robert Collins  <robertc@squid-cache.org>
# 
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import os
import re
from shutil import rmtree
import sys
import urllib
from config_manager.optparse import (OptionParser, Sequence, 
                                     StringArgument, TerminalElement,
                                     ArgumentValueError, Optional)

fake_builds = []
fake_updates = []

class Config(object):
    """A configurationof recipe."""

    _re_config_line = re.compile("^(#.*)|(([^\s]+)\s+([^\s]+)|\s*)$")
    # comment is 0, path 2, source 3

    def __init__(self, fromStream=None):
        self._entries = {}
        if fromStream is not None:
            # parsing might be better in a factory ?
            for line in fromStream.readlines():
                groups = self._re_config_line.match(line).groups()
                if groups[2] and groups[3]:
                    self.add_entry(ConfigEntry(groups[2], groups[3]))

    def add_entry(self, entry):
        assert (entry.path not in self._entries)
        self._entries[entry.path] = entry

    def build(self, dir):
        """Perform a build of this config with base dir dir, which must exist
        already.
        """
        to_process = self._entries.items()
        to_process.sort(key=lambda x:len(x[0]))
        for path, entry in to_process:
            entry.build(dir)

    def get_entries(self):
        """Return a diction of ConfigEntries."""
        # this returns a copy of the internal dict to prevent
        # garbage being put in the internal dict.
        # it might be nice to return a readonly reference to the
        # internal one though ...
        return self._entries.copy()

    def update(self, dir):
        """Perform a update of this config with base dir dir, which must exist
        already.
        """
        to_process = self._entries.items()
        to_process.sort(key=lambda x:len(x[0]))
        for path, entry in to_process:
            entry.update(dir)


class ConfigEntry(object):
    """A single element in a configuration."""

    def __eq__(self, other):
        try:
            return self.path == other.path and self.url == other.url
        except AttributeError:
            return False
        
    def __init__(self, relative_path, url):
        """Construct a ConfigEntry for the provided path and url."""
        self.path = relative_path
        self.url = url

    def build(self, path):
        """Build this element from root path."""
        ## FIXME: the C++ version uses a ConfigSource to abstract out
        ## the url types, we should do that once we have a couple of
        ## types implemented
        target = os.path.abspath(os.path.join(path, self.path))
        if not os.path.isdir(os.path.dirname(target)):
            raise KeyError("Cannot build to target %s"
                           ", its parent does not exists" % target)
        # protocol specific urls
        if self.url.startswith("arch://"):
            import pybaz
            pybaz.get(self.url[7:], os.path.join(path, self.path))
        elif self.url.startswith("fake://"):
            os.mkdir(os.path.join(path, self.path))
            fake_builds.append((self.url, os.path.join(path, self.path)))
        else:
            # autodetect urls
            #
            try:
                import pybaz
                lastslash = self.url.rfind('/')
                url = self.url[:lastslash]
                version = self.url[lastslash:]
                archive = str(pybaz.register_archive(None, url))
                pybaz.get(archive + version, os.path.join(path, self.path))
            except pybaz.errors.ExecProblem:
                rmtree(os.path.join(path, self.path), ignore_errors=True)
                try:
                    import errno
                    from bzrlib.merge import merge
                    from bzrlib.branch import DivergedBranches, NoSuchRevision, \
                         find_cached_branch, Branch
                    from bzrlib.meta_store import CachedStore
                    import tempfile
                    cache_root = tempfile.mkdtemp()
            
                    revision = [None]
                    from_location = self.url
                    to_location = os.path.join(path, self.path)

                    if from_location.startswith("file://"):
                        from_location = from_location[7:]

                    try:
                        try:
                            br_from = find_cached_branch(from_location, cache_root)
# we should catch all and report this type as not supported on that url.
                        except OSError, e:
                            if e.errno == errno.ENOENT:
                                raise BzrCommandError('Source location "%s" does not'
                                                      ' exist.' % from_location)
                            else:
                                raise
            
                        os.mkdir(to_location)
                        br_to = Branch(to_location, init=True)
                        br_to.set_root_id(br_from.get_root_id())
                        br_to.update_revisions(br_from, stop_revision=br_from.revno())
                        merge((to_location, -1), (to_location, 0), this_dir=to_location,
                              check_clean=False, ignore_zero=False)
                        from_location = pull_loc(br_from)
                        br_to.controlfile("x-pull", "wb").write(from_location + "\n")
                    finally:
                        rmtree(cache_root)
                except Exception, e:
                    print "%r" % e, e
# bare except because I havn't tested a bad url yet.
                    raise ValueError("unknown url type '%s'" % self.url)

    def update(self, path):
        """Update this entry (from root path)."""
        ## FIXME: the C++ version uses a ConfigSource to abstract out
        ## the url types, we should do that once we have a couple of
        ## types implemented
        target = os.path.abspath(os.path.join(path, self.path))
        if not os.path.isdir(os.path.dirname(target)):
            raise KeyError("Cannot build to target %s"
                           ", its parent does not exists" % target)
        if not os.path.exists(os.path.join(path, self.path)):
            return self.build(path)
        # protocol specific urls
        if self.url.startswith("arch://"):
            import pybaz
            tree = pybaz.WorkingTree(os.path.join(path, self.path))
            # if not tree.version == self.url[7:] ... wrong version
            tree.update()
        elif self.url.startswith("fake://"):
            fake_updates.append((self.url, os.path.join(path, self.path)))
        else:
            raise ValueError("unknown url type '%s'" % self.url)


class CommandArgument(StringArgument):

    def rule(self):
        return "COMMAND"

class UrlArgument(StringArgument):

    def rule(self):
        return "url"

def main(argv):
    """The entry point for main()."""
    parser = OptionParser(prog="cm")
    parser.set_arguments(Sequence((CommandArgument(), Optional(UrlArgument()),
                                   TerminalElement())))
    parser.set_usage("cm [options] %s" % parser.arguments.rule())
    try:
        (options, args) = parser.parse_args(argv[1:])
    except ArgumentValueError:
        parser.print_usage()
        return 0
    if len(args) == 1:
        config = Config(fromStream=sys.stdin)
    else:
        stream = urllib.urlopen(args[1])
        config = Config(fromStream=stream)
    if args[0] == "build":
        config.build(os.path.abspath(os.curdir))
        return 0
    if args[0] == "show":
        entries = config.get_entries().items()
        entries.sort(key=lambda x:len(x[0]))
        for path, entry in entries:
            print "%s\t%s" % (path, entry.url)
        return 0
    if args[0] == "update":
        config.update(os.path.abspath(os.curdir))
        return 0
        

