#!/usr/bin/python
# -*- coding: Latin-1 -*-
entete = """
###############################################################################
#
#    This file is part of COLLATINVS.
#
#    COLLATINVS is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    COLLATINVS is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with COLLATINVS; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
###############################################################################
"""
# BOGUES :
#  - Mauvais calcul de la conjugaison de sum
#  - les formes irrgulires ne sont pas prises en compte
#    
#
# A FAIRE :
#  - Vite : ajouter un champ <exclusif>[0, 1]</exclusif> dans 
#    les items <irreg> de irregs.xml, afin de savoir si la forme
#    irrgulire inhibe la reconnaissance de la forme rgulire 
#    correspondante, ou si elle reprsente une possibilit supplmentaire
#    Exemple de forme irrgulire exclusive : optimus pour le superlatif de bonus
#    Exemple de forme irrgulire non exclusive : consuerint pour consueuerint.
#  - La suite -ii- devient trs souvent -i- en latin. Et multiplier
#    les radicaux (peto -> pet,peti,petiu) peut gnrer de nombreuses
#    erreurs d'analyse. Il faut envisager une conversion i -> ii dans
#    l'algo plutt que dans les donnes.
#  - insolens, entis, et tous les adj. dverbaux part. prs : pas de modle !
#    le modle acer est une approximation,  cause du neutre acre (vs insolens).
#  - Terminer la gestion du lexique personnel :
#      . prvoir l'ajout de formes irrgulires
#      . Dans lisEntreeP(x), rationaliser l'ajout des radicaux r2 et r3 
#        (code dupliqu pour l'instant)
#      . Lexique personnel : au chargement d'un nouveau lexique, prvoir la destruction des 
#        doublons lexique officiel/lexique personnel, aprs confirmation. En cas 
#        de conflit, le lexique personnel prime.
#  - Ajouter un champ degr  tirreg, et aux adverbes !
#  - prsenter la lemmatisation ordonne par frquence d'apparition du lemme dans
#    la littrature latine 

# collat.py $Revision: 7.5 $
# dernire mise  jour : $Date: 2005/12/25 08:15:28 $ par $Author: collatinus $

import os, string, re
# la fonction frequences.freq(l) donne la frqence du lemme l dans la littrature latine.
from frequences import freq 

def revision():
    return '$Revision: 7.5 $'.split()[1]

print "Latinus pour python, rvision "+revision()+"\n  chargement du lexique..."

# =====================================================
# constantes lies au systme d'exploitation
# =====================================================

Linux = os.name == "posix"
# viaD : Rpertoire de config de collatinus ; 
# viaP : fichier lexique utilisateur
# viaL : fichier lemmata utilisateur
if Linux:
    viaD = os.path.expanduser("~/.collatinus/")
else:
    viaD = os.path.expanduser("c:\Program Files\collatinus\\")
viaP = viaD + "canons.xml"
viaL = viaD + "lemmata"
# crer le rpertoire de conf
if not os.path.exists(viaD):
    os.mkdir(viaD, 0777)
    import shutil
    shutil.copy2('/usr/share/collatinus/canons.xml.vacuum',viaP)
    shutil.copy2('/usr/share/collatinus/lemmata',viaL)
    shutil.copy2('/usr/share/collatinus/conf.py.debian-orig',viaD+'conf.py')

# =====================================================
# constantes morphologiques et manipulation de chanes
# =====================================================

def Nombre(i):
   Lnombres = [' ', 'singulier', 'pluriel']
   return Lnombres[i]
    
Lmodeles = ['uita', 'amicus', 'puer', 'ager', 'templum', #4 
      'miles', 'ciuis', 'corpus', 'mare', 'manus', 'res', # 10 
      'bonus', 'miser', 'pulcher', 'fortis', 'uetus', 'acer', # 16
      'amo', 'moneo', 'lego', 'capio', 'audio', 'sum', 'eo', # 23
      'imitor', 'uereor', 'sequor', 'patior', 'potior', # 28
      'pronoms', 'invaria']
  
def Modele(i): 
   global Lmodeles
   return Lmodeles[i]

def numero_du_modele(m):
   global Lmodeles
   return Lmodeles.index(m)
   
def MorphoK(m):
   if m < 11:
      return "nominatif singulier"
   elif m < 17:
      return "nominatif masculin singulier"
   elif m < 29:
      return "1re singulier prsent indicatif actif"
   else: 
      return ""
      
def DesK(m):
   """ Fournit la dsinence canonique du lemme de modle m.           
       Cette fonction n'est videmment pas applicable pour 
       les modles sans dsinence canonique stable (essentiellement la 
       troisime dclinaison des noms et adjectifs)."""
   if m == 0:
      return 'a'
   elif m in [1, 9, 11]:
      return 'us'
   elif 1 < m < 4 or 11 < m < 14:
      return 'er'
   elif m == 4:
      return 'um'
   elif m == 10:
      return 'es'
   elif m == 17 or m == 19:
      return 'o'
   elif m == 18 or m == 23:
      return 'eo'
   elif 19 < m < 22:
      return 'io'
   elif m == 22:
      return 'sum'
   elif m == 24 or m == 26:
      return 'or'
   elif m == 25:
      return 'eor'
   else:
      return 'ior'
   
def Cas(i):   
   Lcas = ['', 'nominatif', 'vocatif', 'accusatif', 'gnitif', 'datif', 'ablatif']
   return Lcas[i]

def Genre(i):
   Lgenres = ['', 'masculin', 'fminin', 'neutre']
   return Lgenres[i]
   
def Personne(i):
   Lpersonnes = ['', '1re', '2me', '3me']
   return Lpersonnes[i]
   
def Temps(i):
   Ltemps = ['', 'prsent', 'futur', 'imparfait', 'parfait',
      'futur antrieur', 'plus-que-parfait']
   return Ltemps[i]
   
def Mode(i):
   Lmodes = ['', 'indicatif', 'subjonctif', 'impratif', 'infinitif',
     'participe', 'grondif',  'adjectif verbal']
   return Lmodes[i]
   
def Voix(i):
   Lvoix = ['', 'actif', 'passif']
   return Lvoix[i]
   
# =====================================================   
# classe tdes : les dsinences
# =====================================================
class tdes:   
   def __init__(self, gr, c=0, gn=0, n=0, d=0, p=0, t=0, mde=0, v=0, mdl=0, r=0):
      self.graphie = gr
      self.cas = int(c)
      self.genre = int(gn)
      self.nombre = int(n)
      self.degre = int(d)
      self.personne = int(p)
      self.temps = int(t)
      self.mode = int(mde)
      self.voix = int(v)
      self.modele = int(mdl)
      self.radnum = int(r)
         
   def morpho(self):  
      """retourne la morphologie de l'instance en utilisant les listes ci-dessus. """
      #return '%s %s %s %s %s %s %s (modle %s) ' % (Cas(self.cas), 
      return '%s %s %s %s %s %s %s' % (Cas(self.cas), 
         Genre(self.genre), Personne(self.personne), 
         Nombre(self.nombre), Temps(self.temps), Mode(self.mode), 
         Voix(self.voix)) #, Modele(self.modele))   
         
   def affiche(self):
      print self.morpho()
            
# =====================================================
#  classe tirregs : les formes irrgulires
# =====================================================
class tirreg(tdes):
   def __init__(self, gr, k, c, gn, n, p, t, mde, v, mdl):
      # graphie, canon, cas, genre, nombre, personne, temps, mode, voix, modele
      self.graphie = gr
      self.cas = int(c)
      self.genre = int(gn)
      self.nombre = int(n)
      self.personne = int(p)
      self.temps = int(t)
      self.mode = int(mde)
      self.voix = int(v)
      self.modele = int(mdl)
      self.canon = k


# =====================================================
# tentree : ses instances composent le dictionnaire *lexique*
# =====================================================

# filtreG permet de calculer le genre des noms
filtreG = re.compile("^.*([mfn])\.\s:")

class tentree:
   def __init__(self, m, g, t, r2, r3):
      # modle, graphie, texte, radicaux 2 et 3
      self.modele = int(m)
      self.graphie = g
      self.texte = t
      #~ intgration des radicaux...
      self.r2 = r2
      self.r3 = r3
      # calcul du genre d'aprs le champ texte (peccatum est, id scio.)
      s = filtreG.search(self.texte)
      if s: 
         e = s.group(1) 
         if e == 'm' : self.genre = 1
         elif e == 'f' : self.genre =  2
         else : self.genre = 3
      else: self.genre = 0

   def classe_gr(self):
      if self.modele < 11: return "nom"
      if self.modele < 17: return "adj"
      if self.modele < 28: return "verbe" 
      if self.modele == 29: return "pron" 
      return "inv"

   def r1(self):
      """ donne le radical 1 du lemme 
      'uita', 'amicus', 'puer', 'ager', 'templum',            #4 
      'miles', 'ciuis', 'corpus', 'mare', 'manus', 'res',     # 10 
      'bonus', 'miser', 'pulcher', 'fortis', 'uetus', 'acer', # 16
      'amo', 'moneo', 'lego', 'capio', 'audio', 'sum', 'eo',  # 23
      'imitor', 'uereor', 'sequor', 'patior', 'potior',       # 28
      'pronoms', 'invaria']
      """
      # nettoyer le (2)
      if self.graphie[-3:] == "(2)" :
         self.graphie = self.graphie[:-4]
      if [5,6,7,8].count(self.modele): return self.graphie
      elif [0,8,17,19].count(self.modele): return self.graphie[:-1]
      elif [1,2,3,4,9,10,11,12,13,15,18,20,21,23,24,26].count(self.modele):
          return self.graphie[:-2]
      elif [22,25,27].count(self.modele): return self.graphie
      return false

   def declineN(self):
       """ renvoie le tableau de dclinaison du lemme de graphie g """
       # recherche du lemme dans lexique
       # pour chaque nombre
       retour = ""
       for n in range(1,3):
           retour += Nombre(n) + "\n"
           #retour += "="*9 + "\n" # pour souligner
           # pour chaque cas
           for c in range(1,7):
               # forme canonique
               if c == 1 and n == 1:
                  retour += "  %s %s\n" % ("nominatif".ljust(12), self.graphie)
               # recherche de la dsinence
               elif c == 3 and n == 1 and self.genre == 3:
                  retour += "  %s %s\n" % ("accusatif".ljust(12), self.graphie)
               else:
                   mesDes = desN(self, n, c)
                   # pour chaque dsinence trouve
                   for des in mesDes:
                       # radical voulu par la dsinence
                       if des.radnum == 2:
                           retour += "  %s %s%s\n" % (Cas(c).ljust(12), self.r2, des.graphie)
                       else:
                           retour +=  "  %s %s%s\n" % (Cas(c).ljust(12), self.r1(), des.graphie)
       return retour


   def declineA(self):
       """ renvoie le tableau de dclinaison du lemme de graphie g """
       # recherche du lemme dans lexique
       # pour chaque nombre
       retour = ""
       for g in range(1,4):
           for n in range(1,3):
               retour += "="*9 + "\n" 
               retour += "%s %s\n" % (Genre(g), Nombre(n))
               # pour chaque cas
               for c in range(1,7):
                   retour += "   %s" % Cas(c).ljust(12)
                   # forme canonique
                   if g == 1 and c == 1 and n == 1:
                      retour += self.graphie
                   # recherche de la dsinence
                   elif g == 3 and c == 3 and n == 1:
                      retour += self.graphie
                   else:
                       mesDes = desA(self, g, n, c)
                       # pour chaque dsinence trouve
                       for des in mesDes:
                           # radical voulu par la dsinence
                           if des.radnum == 2:
                               retour += "%s%s " % (self.r2, des.graphie)
                           else:
                               retour += "%s%s " % (self.r1(), des.graphie)
                   retour += "\n"
       return retour

   def conjugue(self):
       """'', 'indicatif', 'subjonctif', 'impratif', 'infinitif',
       'participe', 'grondif',  'adjectif verbal']
       Ltemps = ['', 'prsent', 'futur', 'imparfait', 'parfait',
      'futur antrieur', 'plus-que-parfait']
       """
       retour = ""
       soul   = "-"*10
       for v in range(1,3):            # voix
           m = 1                       # indicatif
           retour += "%s\n%s\n" % (soul, Voix(v).upper())
           for t in range(1,7):
               retour += "%s %s\n" % (Mode(m).upper(), Temps(t))
               for n in range(1,3):
                   for p in range(1,4):
                       mesDes = desV(self.modele, p,n,t,m,v,0,0)
                       for des in mesDes:
                           if des.radnum == 1:
                               retour += "   %s%s " % (self.r1(), des.graphie)
                           else:
                               retour += "   %s%s " % (self.r2, des.graphie)
                       retour += '\n'

           m = 2                       # subjonctif
           for t in (1, 3,4,6):
               retour += "%s %s\n" % (Mode(m).upper(), Temps(t))
               for n in range(1,3):
                   for p in range(1,4):
                       mesDes = desV(self.modele, p,n,t,m,v,0,0)
                       for des in mesDes:
                           if des.radnum == 1:
                               retour += "   %s%s " % (self.r1(), des.graphie)
                           else:
                               retour += "   %s%s " % (self.r2, des.graphie)
                       retour += '\n'

           m = 3                       # impratif
           for t in (1,2):
               retour += "%s %s\n" % (Mode(m).upper(), Temps(t))
               for n in range(1,3):
                   for p in range(1,4):
                       mesDes = desV(self.modele, p,n,t,m,v,0,0)
                       for des in mesDes:
                           retour += "   %s%s " % (self.r1(), des.graphie)
                       retour += '\n'

           m = 4                       # infinitif
           for t in (1,4): 
               retour += "%s %s\n" % (Mode(m).upper(), Temps(t))
               mesDes = desV(self.modele, 0,0,t,m,v,0,0)
               for des in mesDes:
                   if des.radnum == 1:
                       retour += "   %s%s " % (self.r1(), des.graphie)
                   else:
                       retour += "   %s%s " % (self.r2, des.graphie)
               retour += '\n'
          
           m = 5 
           if v == 1: t = 1
           else: t = 4
           retour += "%s %s\n" % (Mode(m).upper(), Temps(t))
           for g in (1,2,3):
               for n in (1,2):
                   retour += "%s\n%s %s\n" % (soul, Genre(g), Nombre(n))
                   for c in range(1,7):
                       mesDes = desV(self.modele, 0,n,t,m,v,c,g)
                       for des in mesDes:
                           if des.radnum == 1:
                               retour += "  %s\t%s%s " % (Cas(c), self.r1(), des.graphie)
                           else:
                               retour += "  %s\t%s%s " % (Cas(c), self.r3, des.graphie)
                       retour += '\n'
               retour += '\n'
           
           if v == 1:
               m = 6                       # grondif
               retour += "%s\n" % Mode(m).upper()
               for c in (3,4,6):
                                    # modele,p n t m v c g 
                   mesDes = desV(self.modele,0,0,0,m,0,c,0)
                   for des in mesDes:
                       retour += "%s%s " % (self.r1(),des.graphie)
           else: 
               m = 7                       # adjectif verbal
               retour += "%s\n" % Mode(m).upper()
                                # modele,p n t m v c g 
               mesDes = desV(self.modele,0,1,0,m,0,1,1)
               for des in mesDes:
                 retour += "%s%s" % (self.r1(),des.graphie) 
               retour += ', a, um'
           retour += "\n"
       return retour

   def flechis(self):        
       if self.classe_gr() == "nom":
           return self.declineN()
       elif self.classe_gr() == "adj":
           return self.declineA()
       elif self.classe_gr() == "verbe":
           return self.conjugue()
       else: return ''
      
   def docLemm(self):
       return "%s, (%s)\n  %s" % (self.graphie, Lmodeles[self.modele], self.texte) 

   def doc(self):
      """ retourne le lemme suivi de ses donnes 
      morphologiques, et des traductions franaises """
      return '%s %s' % (self.graphie, self.texte)

   def dochtml(self):
      """ comme doc, mais le lemme est en <b> gras</b> 
          prvoit l'intgration dans un environnement <ol></ol> ou <ul></ul>"""
      return '<li><b>%s</b> %s</li>' % (self.graphie, self.texte)

   def docLaTeX(self):
      """ Nunc carmen nostimus 
          prvoit l'intgration dans un environnement \begin{itemize} ou {enumerate}"""
      return "\item \\textbf{%s} %s" % (self.graphie, self.texte)

   def affiche(self):
       print self.doc()
      
   def ecris(self, f):
      """ crit les donnes du lemme dans le fichier f """
      f.write('<canon>\n')
      f.write('   <graphie>' + self.graphie + '</graphie>\n')
      f.write('   <modele>' + str(self.modele) + '</modele>\n')
      try: 
         f.write('   <R2>' + self.r2 + '</R2>\n')
      except: pass
      try:
         f.write('   <R3>' + self.r3 + '</R3>\n')
      except: pass 
      # couper le \n final du champ texte
      t = self.texte
      while t[-1] == '\n':
         t = t[:-1]
      f.write('   <texte>%s</texte>\n' % t)
      f.write('</canon>\n')

# =====================================================
#  tradical : les radicaux
# =====================================================
class tradical:  
   def __init__(self, g, k, m):
      self.graphie = g
      self.canon = k
      self.modele = m
    
   def kanon(self):
       """ retourne l'entre de lexique correspondant au radical """  
       return lexique[self.canon]
         
   def affiche(self, i):
      print "%s : radical %d de %s" % (self.graphie, i, self.canon)

# Peupler les radicaux
radicaux = {}
radicaux[2] = {}
radicaux[3] = {}
def peupleRadicaux():
    # Rinitialiser les radicaux en cas de modification par lexicumP
    global radicaux
    radicaux = {}
    radicaux[2] = {}
    radicaux[3] = {}
    for cle in lexique.keys():
        e = lexique[cle]
        if e.r2 > "":
            # prise en compte des radicaux multiples
            rad2 = e.r2.split(',')
            for r in rad2:
                if not radicaux[2].has_key(r):
                    radicaux[2][r] = {}
                    radicaux[2][r][0] = tradical(r, e.graphie, e.modele)
                else: 
                    radicaux[2][r][len(radicaux[2][r])] = tradical(r, e.graphie, e.modele)
        if e.r3 > "":
            rad3 = e.r3.split(',')
            for r in rad3: 
                if not radicaux[3].has_key(r):
                    radicaux[3][r] = {}
                    radicaux[3][r][0] = tradical(r, e.graphie, e.modele)
                else:         
                    radicaux[3][r][len(radicaux[3][r])] = tradical(r, e.graphie, e.modele)
    return 1
       


# =====================================================
# chargement du lexique
# =====================================================

def datede(file):
   """ retourne la date du fichier file, et la chaine 'nil' 
   en cas d'absence du fichier."""
   if os.path.isfile(file):
      return os.stat(file).st_mtime
   else:
      return 'nil'

# lexicorum creatio :
lexique = {}
desinences = {}
irregs = {}

def a_lemmatibus():
   """ Chargement rapide  partir du fichier lemmata"""
   m = 0
   t = g = r2 = r3 = ""
   if os.path.exists(viaL): fdata = open(viaL) 
   else:
       if os.path.exists('lemmata'): fdata = open('lemmata')
       else: fdata = open('/usr/share/collatinus/lemmata')
   linea = fdata.readline()
   print 'lemmata leguntur...'
   while string.find(linea, '---desinentiae---') < 0:
      eclats = re.split('[|]', linea)
      g = eclats[0]
      m = int(eclats[1])
      r2 = eclats[2]
      r3 = eclats[3]
      # tentree(m, g, t, r2, r3):
      # miles|5|milit||itis, m. : soldat
      lexique[g] = tentree(m, g, eclats[4], r2, r3) 
      linea = fdata.readline()
   
   # dsinences
   print 'desinentiae leguntur...'
   linea = fdata.readline ()
   while string.find(linea, '---irregulares---') < 0:
      eclats = re.split('[|]', linea)
      g = eclats[0]
      # graphie,cas,genre,nombre,degr,personne,temps,mode,voix,modle,radical
      des = tdes(g, eclats[1], eclats[2], eclats[3], eclats[4],
         eclats[5], eclats[6], eclats[7], eclats[8], eclats[9], eclats[10])
      if not desinences.has_key(g):
         #~ ajout de la cl et de la dsinence
         desinences[g] = {}
         desinences[g][0] = des
      #~ ajout de la dsinence dans la cl      
      else :
         desinences[g][len(desinences[g])] = des
      linea = fdata.readline()

   # irrguliers
   print 'irregulares leguntur...'
   linea = fdata.readline()
   while string.find(linea, '---finis---') < 0:
      eclats = re.split('[|]', linea)
      g = eclats[0]
      # graphie, canon, cas, genre, nombre, temps, mode, voix, modele
      # afferte|adfero|0|0|2|afferte|adfero|0|0|22|1|3|1|0
      irreg = tirreg(g, eclats[1], eclats[2], eclats[3], eclats[4],\
         eclats[5], eclats[6], eclats[7], eclats[8], eclats[9]) #, eclats[10])  
      if not irregs.has_key(g):
         #~ ajout de la cl et de la forme
         irregs[g] = [] # {}
         #irregs[g][0] = irreg
         irregs[g].append(irreg)
      else:
         #~ ajout de la forme dans la cl      
         #irregs[g][len(irregs[g])] = irreg
         irregs[g].append(irreg)
      linea = fdata.readline()
   fdata.close()

#================================================================
# Chargement du lexique  partir du fichier lemmata 
# Le chargement  partir de xml a t conserve dans collat.py
#================================================================
a_lemmatibus()

#==============================================   
# LECTURE DU LEXIQUE PERSONNEL (LEXIQUEP) ET  
# CORRECTION DE LEXIQUE PAR LEXIQUEP
#==============================================   
from xml.dom import minidom
from os.path import *

class tentreeP(tentree):
    """Deux diffrences avec tentree : 
       1. un champ status qui dfinit
           comment altrer le lexique officiel. Trois valeurs possibles
           pour status :
           0 = ajouter 
           1 = modifier
           2 = supprimer 
       2. un champ com qui indique si l'entre a t communiqu  un serveur collatinus.  """

    def __init__(self, s, m, g, t, r2="", r3="", com=0):
        tentree.__init__(self, m, g, t, r2, r3)
        self.status = s
        self.com = com

    def ecris(self, f):
        """ crit les donnes du lemme dans le fichier f """
        f.write('<canon>\n')
        f.write('<status>%d</status>\n' % self.status)
        f.write('   <graphie>%s</graphie>\n' % self.graphie)
        f.write('   <modele>%d</modele>\n' % self.modele)
        if self.r2 > '':
            f.write('   <R2>%s</R2>\n' % self.r2)
        if self.r3 > '':
            f.write('   <R3>%s</R3>\n' % self.r3)
        # couper le \n final du champ texte
        t = self.texte
        while t[-1] == '\n': t = t[:-1]
        f.write('   <texte>%s</texte>\n' % t)
        f.write('   <com>%s</com>\n' % self.com)
        f.write('</canon>\n')

    def egale(self, entree):
        """ renvoie True si entree (type tentree) a les mmes valeurs
        que self,  part le champ com."""
        return entree.graphie == self.graphie \
           and self.modele == entree.modele \
           and self.r2 == entree.r2 \
           and self.r3 == entree.r3 \
           and self.texte == entree.texte


def lisEntreeP(x):
    """ renvoie une instance de tentreeP  partir d'un Element DOM"""
    s = int(x.getElementsByTagName('status')[0].childNodes[0].data)
    m = x.getElementsByTagName('modele')[0].childNodes[0].data
    m = m.encode('latin-1')
    g = x.getElementsByTagName('graphie')[0].childNodes[0].data
    g = g.encode('latin-1')
    try:
        t = x.getElementsByTagName('texte')[0].childNodes[0].data
        t = t.encode('latin-1')
    except: t = ""
    try:
        r2 = x.getElementsByTagName('R2')[0].childNodes[0].data
        r2 = r2.encode('latin-1')
    except: r2 = ""
    try:
        r3 = x.getElementsByTagName('R3')[0].childNodes[0].data 
        r3 = r3.encode('latin-1')
    except: r3 = ""
    try: com = x.getElementsByTagName('com')[0].childNodes[0].data 
    except: com = 0 
    return tentreeP(s, m, g, t, r2, r3, com) 

# cration du lexique personnel
import UserDict
class tlexicumP(UserDict.UserDict):
    def communiquees(self):
       n = 0
       for i in self.keys():
          if int(self.data[i].com) > 0: n += 1
       return n

    def nonCommuniquees(self):
       return len(self.keys()) - self.communiquees()

    def setEnvoye(self):
       for i in self.keys():
           self.data[i].com = 1

    def ecris(self):
        """ Ecrit le lexique personnel """ 
        f = open(viaP, 'w')
        f.write('<?xml version="1.0" encoding="iso-8859-1" ?>\n<!--') 
        f.write(entete)
        f.write('-->\n')
        f.write('<collatinus>')
        for e in self.data.keys():
            self.data[e].ecris(f)
        f.write('</collatinus>')
        f.close()
        print "lexique personnel mis  jour"

    def dedoublonne(self):
        """ Compare, aprs un chargement de lemmata, le lexique personnel 
        afin de
        - supprimer dans le xml perso les doublons xml-lemmata ;
        - laisser le choix en cas de diffrence entre les deux """
        for cle in lexicumP.keys() :
            entreeP = lexicumP[cle]
            if lexique.has_key(cle)\
                and entreeP.egale(lexique[cle]):
                del(lexicumP[cle])
        self.ecris()

lexicumP = tlexicumP()

# construction de l'arbre DOM  partir du fichier lexique personnel
def ecrisP():
    """ Ecrit le lexique personnel """ 
    f = open(viaP, 'w')
    f.write('<?xml version="1.0" encoding="iso-8859-1" ?>\n<!--') 
    f.write(entete)
    f.write('-->\n')
    f.write('<collatinus>')
    for e in lexicumP.keys():
        lexicumP[e].ecris(f)
    f.write('</collatinus>')
    f.close()
    print "lexique personnel mis  jour"

if not os.path.exists(viaP):
    # cration du fichier 
    ecrisP()

doc = minidom.parse(viaP)
canonsP = doc.getElementsByTagName('canon')
# constitution du lexique personnel lexicumP
for i in canonsP:
    gr = i.childNodes[3].firstChild.data 
    lexicumP[gr] = lisEntreeP(i)
# correction du lexique officiel
for e in lexicumP.keys():
    entreeP = lexicumP[e]
    if entreeP.status < 2: 
        entreeP.texte += '\n'
        lexique[e] = entreeP
    else: del(lexique[e])

# =====================================================
# Peupler les radicaux :
# =====================================================
peupleRadicaux()

# =====================================================
#  Rsolution des renvois cf.
# =====================================================
for e in lexique.keys():
   if lexique[e].texte[:4] == 'cf. ':
       entreeR = lexique[e].texte[4:]
       while entreeR[-1] == '\n': entreeR = entreeR[:-1]
       if lexique.has_key(entreeR):
           lexique[e].graphie = lexique[entreeR].graphie
           lexique[e].texte   = lexique[entreeR].texte

#==============================================   
#  Affichage des stats du chargement
#==============================================   
def stats():
    print "  ", len(lexique), " entrees ;"
    print len(desinences), " desinences ;"
    print len(radicaux[2]), "radicaux 2 ;"
    print len(radicaux[3]), "radicaux 3 ;"
    print len(irregs), "irreguliers."
    # affichage des stats du chargementP
    print "%d corrections personnelles du lexique" % len(lexicumP)
    print "   dont %d communiques au serveur" % lexicumP.communiquees()

# Afficher les stats
stats()


# =====================================================
#              FIN DE LA PARTIE CHARGEMENT. 
# =====================================================

# =====================================================
# consultation du lexique 
# =====================================================

def dico():
   m = "dico"
   while m != "q":
      if m > "":
         if lexique.has_key(m):
            lexique[m].affiche()
         else:
            print "introuvable"
         m = raw_input("entre (q pour quitter) : ")

# =====================================================
# consultation des dsinences
# ===================================================== 
def dicodes():
   mot = ""
   while mot != "q":
      if mot != "":   
         if desinences.has_key(mot):
            for i in range(len(desinences[mot])):
               desinences[mot][i].affiche()
         else:
            print "introuvable"           
      mot = raw_input("dsinence ? (q pour quitter) ")


def desN(l, n, c):
    """ renvoie les dsinences de nombre n et de cas c pour l'entre l) """
    retour = [] 
    m = l.modele
    for i in desinences.keys(): 
        for cd in desinences[i].keys():
           d = desinences[i][cd]
           if d.modele == m and d.nombre == n and d.cas == c:
               retour.append(d)
    return retour

def desA(l, g, n, c):
    """ renvoie les dsinences de genre g, de nombre n et de cas c pour l'entre l) """
    retour = [] 
    m = l.modele
    for i in desinences.keys(): 
        for cd in desinences[i].keys():
           d = desinences[i][cd]
           if d.modele == m and d.genre == g and d.nombre == n and d.cas == c:
               retour.append(d)
    return retour
    

def desV(modele, p,n,t,m,v,c,g):
    """ renvoie les dsinences de
        modele
        personne p
        nombre   n
        temps    t
        mode     m
        voix     v
        cas      c
        genre    g         pour l'entre l """
    retour = [] 
    for i in desinences.keys():
        for cd in desinences[i].keys():
           d = desinences[i][cd]
           if d.modele == modele \
                   and d.voix == v \
                   and d.personne == p \
                   and d.nombre == n \
                   and d.temps == t \
                   and d.mode == m \
                   and d.cas == c \
                   and d.genre == g:
               retour.append(d)
    return retour

# =====================================================
#~ consultation des radicaux
# =====================================================
def dicorad():
   mot = ""
   while mot != "q":
      if mot > "":
         for i in [2, 3]:
            if radicaux[i].has_key(mot):
               for iRad in range(len(radicaux[i][mot])):
                  radicaux[i][mot][iRad].affiche(i)
      mot = raw_input("radical ? (q pour quitter) ")      

# =====================================================
# recherche des dsinences possibles d'une forme
# =====================================================
def desforme():
   mot = ""
   while not mot == "q":
      if mot > '':
         for i in range(len(mot)):
            if desinences.has_key(mot[i:]):
               Des = desinences[mot[i:]]
               for iMorph in range(len(Des)):
                   print Des[iMorph].graphie,
                   Des[iMorph].affiche()
      mot = raw_input("forme ? (q pour quitter) ")

# =====================================================
# analyses morphologiques d'une forme
# =====================================================
class morpho:
    """  Dans cette classe, j'ai ajout
              - lemme : le lemme ;
              - traduc : la traduction ;
         Prennent un autre sens :
              - gr[aphie] est le mot flchi pass  l'initialisation
         Pour les autres paramtres, qui sont morphologiques et paradigmatiques,
            voir la classe tdes : graphie cas genre nombre degre personne
                                  temps mode voix modele radnum
    """
    # doc tdes :
    # self, gr, c=0, gn=0, n=0, d=0, p=0, t=0, mde=0, v=0, mdl=0, r=0
    # graphie, cas, genre, nombre, degr, personne, temps, mode, voix, modle, radical

    def __init__(self, k, mdl, tr, gr, c=0, gn=0, n=0, d=0, p=0, t=0, mde=0, v=0, r=0):
         self.des = tdes('', c, gn, n, d, p, t, mde, v, mdl, r)
         # print '******', k, self.des.genre, self.des.morpho(), '*****'
         self.lemme = lexique[k]
         self.traduc = tr

    def humain(self):
        #return "%s %s %s" % (self.lemme, self.traduc, self.des.morpho())
        return self.des.morpho()

#   Conversion des chiffres romains. Merci  DiveIntoPython

romanNumeralPattern = \
    re.compile('^M?M?M?M?(CM|CD|D?C?C?C?)(XC|XL|L?X?X?X?)(IX|IV|V?I?I?I?)$') 

#Define digit mapping
romanNumeralMap = (('M',  1000),
                   ('CM', 900),
                   ('D',  500),
                   ('CD', 400),
                   ('C',  100),
                   ('XC', 90),
                   ('L',  50),
                   ('XL', 40),
                   ('X',  10),
                   ('IX', 9),
                   ('V',  5),
                   ('IV', 4),
                   ('I',  1))    

def fromRoman(m):
    """convert Roman numeral to integer"""
    result = 0
    index = 0
    for numeral, integer in romanNumeralMap:
        while m[index:index+len(numeral)] == numeral:
            result += integer
            index += len(numeral)
    return result
        

class morphos:
    """ morphos est la liste de toutes les analyses possibles d'un mot
        ... classe  complter ...
    """
    def __init__(self, mot):
           self.items = {} 
           self.forme = mot
           # chiffre romain ?
           if mot > '' and romanNumeralPattern.search(mot):
               eRomain = tentree(29, mot, 'inv. : %d\n' % fromRoman(mot), '', '') 
               lexique[mot] = eRomain
               
           #~ chercher les formes canoniques
           for uox in [mot, mot + ' (2)']:
               if lexique.has_key(uox):
                   lm = lexique[uox]
                   if not self.items.has_key(uox):
                       self.items[uox] = []
                   m = lexique[uox].modele 
                   if m < 17: # nom ou adj 
                       if m > 10: # adj
                           self.items[uox].append(morpho(uox, m, lm.texte, uox, 1, 1, 1))
                       else: # nom
                           self.items[uox].append(morpho(uox, m, lm.texte, uox, 1, lm.genre, 1))
                   elif m < 29: # verbe
                       self.items[uox].append(morpho(uox,lm.modele, 
                             lm.texte, uox,0, 0, 1, 0, 1, 1, 1, 1))
                   else:
                       self.items[uox].append(morpho(uox, m, lm.texte, uox))

           #~ chercher les formes irrgulires
           if irregs.has_key(mot):
               for iIrr in range(len(irregs[mot])):
                  im = irregs[mot][iIrr]
                  lk = lexique[im.canon]
                  if not self.items.has_key(im.canon):
                      self.items[im.canon] = []
                  self.items[im.canon].append(morpho(im.canon, im.modele, lk.texte, mot,
                      im.cas, im.genre, im.nombre, 0,  # tirreg n'a pas de degr de sign. !
                      im.personne, im.temps, im.mode, im.voix)) #, im.radnum))

           # chercher toutes les dsinences
           for i in range(len(mot)):
              grDes = mot[i:]
              grRad = mot[:i]
              if desinences.has_key(grDes):
                 lesDes = desinences[grDes]
                 for iDes in range(len(lesDes)):
                    Des = lesDes[iDes]
                    if Des.radnum > 1:
                       if radicaux[Des.radnum].has_key(grRad):
                          radix = radicaux[Des.radnum][grRad]
                          for iRad in range(len(radix)):
                             if Des.modele == radix[iRad].modele and lexique.has_key(radix[iRad].canon)\
                                and  radix[iRad].modele == lexique[radix[iRad].canon].modele:
                                K = radix[iRad].canon
                                # si la cl n'existe pas dans les items
                                if not self.items.has_key(K):
                                    self.items[K] = []
                                lk = lexique[K]
                                self.items[K].append(morpho(lk.graphie, lk.modele, lk.texte, 
                                        mot, Des.cas, Des.genre, Des.nombre, Des.degre, 
                                        Des.personne, Des.temps, Des.mode, Des.voix, Des.radnum))

                    else: # cas plus simple : la dsinence canonique est extrapolable
                       K = grRad + DesK(Des.modele) # reconstitution d'une forme canonique hypothtique    
                       if lexique.has_key(K) and Des.modele == lexique[K].modele:
                             if not self.items.has_key(K):
                                self.items[K] = []
                             lk = lexique[K]
                             self.items[K].append(morpho(K, lk.modele, lk.texte, mot, 
                                 Des.cas, Des.genre, Des.nombre, Des.degre, Des.personne,
                                 Des.temps, Des.mode, Des.voix, Des.radnum))
                       if lexique.has_key(K + ' (2)')\
                          and Des.modele == lexique[K + ' (2)'].modele:
                             K = K + ' (2)'
                             if not self.items.has_key(K):
                                self.items[K] = []
                             lk = lexique[K]
                             self.items[K].append(morpho(K, lk.modele, lk.texte, mot,
                                 Des.cas, Des.genre, Des.nombre, Des.degre, Des.personne,
                                 Des.temps, Des.mode, Des.voix, Des.radnum))
    # fin de def __init__(mot)

    def humain(self):
       res = ''
       for K in self.items.keys():
          res = "%s\n%s %s" % (res, K, self.items[K][0].traduc)
          for ik in range(len(self.items[K])):
             res = "%s    %s\n" % (res, self.items[K][ik].humain())
       return res

def analysesDe(mot):
   """ Pour assurer la compatibilit avec les nombreuses 
       fonctions qui appellent encore analysesDe """ 
   return morphos(mot)

# =====================================================
#  Chane de retour d'une analyse
# =====================================================

def ijuv(forme):
   # transformation de v et j et u et i
   r = string.replace(forme, 'v', 'u')
   r = string.replace(r, 'j', 'i')
   return r

def chaineAnalyse(forme):
    f = ijuv(forme)
    m = morphos(f)
    # majuscules
    lf = ijuv(string.lower(f))
    if lf != f:
        m.items.update(morphos(lf).items)
    # suffixes
    if m == {}:
       return '-'
    if lf[-3:] == 'que':
        m.items.update(morphos(lf[:-3]).items)
    elif lf[-2:] in ['ue', 'ne']: 
        m.items.update(morphos(lf[:-2]).items)
    return m.humain()


# =====================================================
# dialogue d'analyse morphologique
# =====================================================
def dialogueAnalyse():
   forme = ""
   arep = ''
   while forme != "q": 
       if forme != "":
           #print chaineAnalyse(forme)
           print lemmatiseMBrut(forme).humain()
       forme = raw_input("forme ? (q pour quitter) ")

# =====================================================
#  Vestige du module Jeanneau, mais qui peut servir : 
#  traduction de l'orthographe u/i en orthographe v/j 
# =====================================================
def ui2vj(f):
        """ renvoie f orthographi en -v- et -j- """
        f = re.sub('^ua', 'va', f)
        f = re.sub('^ue', 've', f)
        f = re.sub('^ui', 'vi', f)
        f = re.sub('^uo', 'vo', f)
        f = re.sub('^uu', 'vu', f)
        f = re.sub('^ia', 'ja', f)
        f = re.sub('^ie', 'je', f)
        f = re.sub('^ii', 'ji', f)
        f = re.sub('^io', 'jo', f)
        f = re.sub('^iu', 'ju', f)        
        f = re.sub('[aeiou]ua', '[aeiou]va', f)
        f = re.sub('[aeiou]ue', '[aeiou]ve', f)
        f = re.sub('[aeiou]ui', '[aeiou]vi', f)
        f = re.sub('[aeiou]uo', '[aeiou]vo', f)
        f = re.sub('[aeiou]uu', '[aeiou]vu', f)
        f = re.sub('[aeiou]ia', '[aeiou]ja', f)
        f = re.sub('[aeiou]ie', '[aeiou]je', f)
        f = re.sub('[aeiou]ii', '[aeiou]ji', f)
        f = re.sub('[aeiou]io', '[aeiou]jo', f)
        f = re.sub('[aeiou]iu', '[aeiou]ju', f)
        return f

# =====================================================
# lemmatisation d'un texte, retour type liste
# =====================================================

def listelemmes(t):
   """renvoie dans une liste de chaines 
      le lexique utilis par le texte t."""
   retour = []
   # quivalences j->i et v->u
   t = ijuv(t)
   # requete d'analyse morpho :
   # suppression de la ponctuation et du texte entre [], mise en tableau
   t = re.sub('\[.*\]','', t)
   # ajout d'espaces en dbut et fin pour assurer la mise en tableau
   # des premier et dernier mots.
   t = " " + t + " "
   tableau = re.split('[^a-zA-Z]', t)
   # lemmatisation de chaque mot
   for iTab in range(len(tableau)):
       f = tableau[iTab]
       lignes = analysesDe(f)
       if lignes == {}:              
           # essayer en minuscules
           lignes = analysesDe(ijuv(string.lower(f)))
       # essais suffixes
       if lignes == {}:
           L = f + ' : ?'
           try:
              retour.index(L)
           except:
              retour.append(L)
       else:
           #for iLin in range(len(lignes)):
           for iLin in range(len(lignes.items)):
               L = lexique[lignes.items.keys()[iLin]].doc()
               try:
                   retour.index(L)
               except:
                   retour.append(L)
   # tri et ajout au rsultat
   retour.sort()
   # liminer le : ? en premire ligne.
   del retour[0]
   for i in range(len(retour)):
         retour.append(retour[i])
   return retour

def lemmatise_fichier(f):
   """lemmatisation d'un fichier, retour texte."""
   fichier = open(f)
   t = fichier.read()
   liste = listelemmes(t)
   #for l in range(len(liste)):
   for l in range(liste.len):
      print liste[l]

def lemmatiseMBrut(mot):
      """ lemmatisation d'un seul mot, retour liste d'objets morphos """
      mot = ijuv(mot)
      m = morphos(mot) 
      # essayer en minuscules
      lmot = ijuv(string.lower(mot))
      if lmot != mot:
          m.items.update(morphos(lmot).items)
      # suffixes
      if lmot[-3:] == 'que':
          m.items.update(morphos(mot[:-3]).items)
      elif lmot[-2:] in ['ue', 'ne']: 
          m.items.update(morphos(mot[:-2]).items)
      return m

def lemmatisem(mot, format):
      """ lemmatisation d'un seul mot, retour fomat """
      m = lemmatiseMBrut(mot)
      dico = []
      if m != {}:
         for k in m.items.keys():
             if format == 'texte':
                  L = lexique[k].doc()
             elif format == 'LaTeX':
                  L = lexique[k].docLaTeX()
             elif format == 'html':
                  L = lexique[k].dochtml()
             try:
                  dico.index(L)
             except:
                  dico.append(L)
      dico.sort()
      retour = ''
      #for i in range(len(dico)):
      for i in dico:
         retour = retour + i #+ '\n'
      return retour

def lemmatise(t, format, filtre=-1, tri=1):
   """renvoie dans un tableau html de deux colonnes
      le lexique utilis par la chane t. tri=0
      renvoie les formes dans l'ordre de leur apparition
      dans le texte. 
      C'est la fonction appele par collatinus.py pour lemmatiser
      tout le texte.
   """

   def filtreM(nm):
       """ Sous-procdure de lemmatise().
           Si une frquence est indique, te de la lemmatisation nm
           tous les lemmes dont la frquence dpasse la frquence *filtre*
       """
       # avec map() a irait plus vite, m'enfin.
       if filtre > -1:
           for l in nm.items.keys():
                f = freq(l) 
                if f < filtre:
                    del nm.items[l]
       return nm

   def NonReconnu(mot):
      if format == 'texte': return '%s ?\n' % mot
      elif format == 'LaTeX': return "\item \\textbf{%s} : ?\n" % mot
      else: return '<li>%s : ?</li>\n' % mot
      
   # quivalences j->i et v->u
   t = ijuv(t)
   # suppression de la ponctuation et du texte entre [], mise en tableau
   t = re.sub('\[.*\]','', t)
   # ajout d'espaces en dbut et fin pour assurer la mise en tableau
   # des premier et dernier mots.
   t = " %s " % (t)
   tableau = re.split('[\W]+', t) # 15/05/04 : ajout le + aprs W
   # pourquoi des items vides  chaque bout ?
   dico = []
   # lemmatisation de chaque mot
   for iTab in tableau:
      # ne pas traiter une chane vide
      if string.strip(iTab) == '':
         continue
      # correction orthographique
      mot = ijuv(iTab)
      # lemmatisation ordinaire
      m = filtreM(morphos(mot)) 
      # essayer en minuscules
      lm = string.lower(mot)
      if lm != mot:
          lm = ijuv(lm)
          nov = filtreM(morphos(lm))
          m.items.update(nov.items)
      # suffixes
      if mot[-3:] == 'que':
          nov = filtreM(morphos(mot[:-3]))
          m.items.update(nov.items)
          # en minuscules
          m.items.update(morphos(lm[:-3]).items)
      elif lm[-2:] in ['ue', 'ne']: 
          nov = filtreM(morphos(mot[:-2]))
          m.items.update(nov.items)
          # en minuscules
          m.items.update(morphos(lm[:-2]).items)
      # mise en format
      if len(m.items) > 0:
         for k in m.items.keys():
             if format == 'texte':
                  L = lexique[k].doc()
             elif format == 'LaTeX':
                  L = lexique[k].docLaTeX()
             elif format == 'html':
                  L = lexique[k].dochtml()
             try:
                  dico.index(L)
             except:
                  dico.append(L)
      # formes non reconnues : seulement si FiltreM < 0
      else: 
          if filtre < 0:
              dico.append(NonReconnu(mot))
   # tri et ajout au rsultat
   if tri: dico.sort()
   retour = ''
   for lin in dico:
       retour += lin
   # print "%d mots lemmatiss" % len(dico)
   return retour

# test du module 
if __name__ == '__main__':
    dialogueAnalyse()
