(*                              collatinus.ml

    This file is part of COLLATINUS

    COLLATINUS is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    COLLATINUS is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with COLLATINUS; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    (c) Yves Ouvrard, Angoulme, 2005 - 2006

    compilation : 
	ocamlopt -o collatinus -I +lablgtk2 lablgtk.cmxa guibase.ml str.cmxa latinus.ml collatinus.ml
*)

open GMain
open Gdk
open Latinus

(* aliquot conuersiones *)

let to_utf8 msg =
    Glib.Convert.convert msg ~to_codeset:"UTF-8" ~from_codeset:"ISO-8859-1"

let to_latin msg =
    Glib.Convert.convert msg ~to_codeset:"ISO-8859-1" ~from_codeset:"UTF-8"

let is_word_char c = 
  Glib.Unichar.isalnum c 

let prerr_endline s =
  prerr_endline s;
  flush stderr

(* documentorum capita et pedes *)

let en_tete_LaTeX = 
    "\\documentclass[12pt]{article}\n"
   ^"\\usepackage[T1]{fontenc}\n"
   ^"\\usepackage[latin1]{inputenc}\n"
   ^"\\usepackage{geometry}\n"
   ^"\\usepackage[frenchb]{babel}\n"
   ^"\\geometry{a4paper, left=1in, right=1in, top=1in, bottom=1in}\n"
   ^"\\begin{document}\n"

let debut_lemm_LaTeX = "\\begin{itemize}\n"

let pied_LaTeX = "\\end{itemize} \n \\end{document}\n"

let en_tete_html = "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 3.2 Final//EN\">\n"
   ^"<html>\n"
   ^"<head>\n"
   ^"<title>Collatinus - Sine nomine</title>\n"
   ^"</head>\n"
   ^"<body>\n"

(* uox sub mure *)
let rec find_word_start it =
  if not it#nocopy#backward_char then it
  else if is_word_char it#char
  then find_word_start it
  else (it#nocopy#forward_char; it) 
let find_word_start (it:GText.iter) = find_word_start it#copy

let rec find_word_end it =
  if let c = it#char in c<>0 && is_word_char c
  then begin 
    ignore (it#nocopy#forward_char);
    find_word_end it
  end else 
    it
let find_word_end it = find_word_end it#copy

(* capsarum dialogon *)
let file_dialog ~title ~callback ?filename () =
  let sel =
    GWindow.file_selection ~title ~modal:true ?filename () in
      ignore (sel#cancel_button#connect#clicked ~callback:sel#destroy);
      ignore (sel#ok_button#connect#clicked ~callback:
        begin fun () ->
          let name = sel#filename in
          sel#destroy ();
          callback name
        end);
sel#show ()

let with_file name ~f =
  let ic = open_in name in
  try f ic; close_in ic with exn -> close_in ic; raise exn

let input_channel b ic =
  let buf = String.create 1024 and len = ref 0 in
  while len := input ic buf 0 1024; !len > 0 do
    Buffer.add_substring b buf 0 !len
  done

(* Gtk initializations *)
let _ = GMain.Main.init ()

class gui ()=

  let tt = GText.tag_table () in
  let magnitudo = GText.tag ~name: "magnitudo" () in  
  let _ = tt#add magnitudo#as_tag in

  object (self)
      inherit Guibase.main ()

      val mutable nomen = None
      val mutable mag = 10


      method caput =
         match wnote#current_page with
             |0 -> "" 
             |1 -> en_tete_LaTeX 
             |2 -> en_tete_html 
             |_ -> "" 

      method inc_item  =            
         match wnote#current_page with
             |0 -> "\n-o0o-\n" 
             |1 -> "\\begin{itemize}\n"
             |2 -> "<ul>\n" 
             |_ -> "" 

      method pes =
         match wnote#current_page with
             |0 -> "" 
             |1 -> "\\end{itemize}\n\\end{document}"
             |2 -> "</ul>\n</body>\n</html>" 
             |_ -> ""

      method alteratum =
           wt_lat#buffer#modified
        || wt_txt#buffer#modified
        || wt_tex#buffer#modified
        || wt_html#buffer#modified

      method integrum =
          wt_lat#buffer#set_modified false;
          wt_txt#buffer#set_modified false;
          wt_tex#buffer#set_modified false;
          wt_html#buffer#set_modified false;
          ()

       method cautio () =
          if not self#alteratum then true
          else begin
          match (GToolbox.question_box ~title:"Noniam in discum scripsisti quae fecisti."
               ~buttons:["scribere nunc";
                         "Pergere neque scribere";
                         "Nil agere"] 
               ~default:1
               ~icon:
               (let img = GMisc.image () in
            img#set_stock `DIALOG_WARNING;
            img#set_icon_size `DIALOG;
            img#coerce)
               ("Quae fecisti noniam scripta sunt.")
            )
          with 1 -> self#save_f (nomen); false
            |  2 -> true 
            |  _ -> false 
          end

      method  noua = if self#cautio () then 
         begin
         self#wt_lat#buffer#set_text "";
         self#tabula#buffer#set_text "";
         self#integrum
         end

      method load_file name =
        try
          let b = Buffer.create 1024 in
          with_file name ~f:(input_channel b);
          let s = Glib.Convert.locale_to_utf8 (Buffer.contents b) in
          let n_buff = GText.buffer ~text:s () ~tag_table:tt in
          wt_lat#set_buffer n_buff;
          (* hoc erasi : non optandum est ut originalis capsa modificetur. *)
          (* nomen <- Some name; *)
          n_buff#place_cursor n_buff#start_iter;
          self#integrum
        (*with _ -> prerr_endline name^" lectum."*)
        with _ -> ()

      method save name =
        try
          let oc = open_out name in 
          output_string oc (self#caput); 
          output_string oc (wt_lat#buffer#get_text ()); 
          output_string oc (self#inc_item); 
          output_string oc (to_latin (self#tabula#buffer#get_text ()));
          output_string oc (self#pes);
          close_out oc;
          nomen <- Some name;
          self#integrum 
        (*with _ -> prerr_endline name^" scriptum" *)
        with _ -> ()

      method save_f fn = 
         (match nomen with 
             | None -> self#save_file ()
             | Some fn -> try self#save fn with _ -> ());

      method save_as f =
        if Sys.file_exists f then 
          match (GToolbox.question_box ~title:"Alter capsa sic iam nuncupatur"
               ~buttons:["super eam scribere";
                 "Antiquo";] 
               ~default:1
               ~icon:
               (let img = GMisc.image () in
            img#set_stock `DIALOG_WARNING;
            img#set_icon_size `DIALOG;
            img#coerce)
               ("Alter capsa "^f^"sic nuncupatur.")
            )
          with 1 -> self#save f
            | _ -> () 
        else self#save f

      method open_file () = file_dialog ~title:"Capsam legere" ~callback: self#load_file ()
      method save_file () = file_dialog ~title:"Capsam scribere" ~callback: self#save_as ()


      method omnia_l () = 
          let t = wt_lat#buffer#get_text () in 
          (* let r = Latinus.lemmatise_texte t in *)
          let r = Latinus.lemmatise_txt_frq t bfreq#active in
          self#tabula#buffer#set_text (to_utf8 r)

      method lemm_radere () = 
         self#tabula#buffer#set_text ""

      method modus =
         match wnote#current_page with
            |0 -> "tex"
            |1 -> "html"
            |2 -> "flexio"
            |_ -> "txt"

      method maiores =
         mag <- (mag + 2);
         magnitudo#set_property (`SIZE (mag*Pango.scale));
         let start,stop = wt_lat#buffer#bounds in
         wt_lat#buffer#apply_tag_by_name "magnitudo" start stop;
         let start,stop = wt_txt#buffer#bounds in
         wt_txt#buffer#apply_tag_by_name "magnitudo" start stop;
         let start,stop = wt_tex#buffer#bounds in
         wt_tex#buffer#apply_tag_by_name "magnitudo" start stop;
         let start,stop = wt_html#buffer#bounds in
         wt_html#buffer#apply_tag_by_name "magnitudo" start stop;
         ()

      method exiliores  =
         mag <- (mag - 2);
         magnitudo#set_property (`SIZE (mag*Pango.scale));
         let start,stop = wt_lat#buffer#bounds in
         wt_lat#buffer#apply_tag_by_name "magnitudo" start stop;
         let start,stop = wt_txt#buffer#bounds in
         wt_txt#buffer#apply_tag_by_name "magnitudo" start stop;
         let start,stop = wt_tex#buffer#bounds in
         wt_tex#buffer#apply_tag_by_name "magnitudo" start stop;
         let start,stop = wt_html#buffer#bounds in
         wt_html#buffer#apply_tag_by_name "magnitudo" start stop;
         ()

      method tabula: GText.view =
        match wnote#current_page with
            |0 -> wt_txt
            |1 -> wt_tex
            |_ -> wt_html
            (*
            |_ -> wt_flexio
            *)

      initializer

      (* format de sortie (texte, LaTeX, html) *)
      ignore (wnote#connect#switch_page ~callback: 
         (fun n -> Latinus.lemmes#set_modus self#modus));

      (* Capsa *)
      ignore (itemNoua#connect#activate (fun () -> self#noua));
      ignore (bnoua#connect#clicked (fun() -> self#noua));
      ignore (itemLegere#connect#activate (fun () -> self#open_file (); ));
      ignore (blege#connect#clicked (fun () -> self#open_file (); ));
      ignore (itemScribere#connect#activate (fun () -> self#save_f (nomen); ));
      ignore (bscribe#connect#clicked (fun () -> self#save_f (nomen); ));
      ignore (itemScribereSub#connect#activate (fun () -> self#save_file (); ));
      ignore (itemQuit#connect#activate (fun () ->  if self#cautio () then GMain.Main.quit ()));
      (* deleta fenestra, ex euentuum circulo  exeundum est. *)
      (* ignore (fenestra#connect#destroy GMain.Main.quit); *)
      ignore (fenestra#connect#destroy (fun () -> if self#cautio () then GMain.Main.quit ()));

      (* editio *)
      ignore (itemDeponere#connect#activate (fun () -> wt_lat#buffer#paste_clipboard GMain.clipboard; ));
      ignore (itemSeponere#connect#activate (fun () -> wt_lat#buffer#copy_clipboard GMain.clipboard; ));
      ignore (itemAlin#connect#toggled (fun () -> self#tabula#set_wrap_mode 
           (if itemAlin#active then `WORD else `NONE)));
      ignore (itemMaiores#connect#activate (fun () -> self#maiores ));
      ignore (itemExiliores#connect#activate (fun () -> self#exiliores )); 

      (* lemmatizatio *)
      ignore (itemOmniaLemm#connect#activate (fun () -> self#omnia_l (); ));
      ignore (blemmata#connect#clicked (fun () -> self#omnia_l (); ));
      ignore (itemLemmDele#connect#activate (fun () -> self#lemm_radere (); ));
      ignore (bnullum#connect#clicked (fun () -> self#lemm_radere (); ));
      ignore (bmaiores#connect#clicked (fun () -> self#maiores));
      ignore (bexiliores#connect#clicked (fun () -> self#exiliores )); 
      ignore (itemGallice#connect#activate (fun () -> (
          Latinus.lemmata_lege "lemmata.fr")));
      ignore (itemGermanice#connect#activate (fun () -> (
          Latinus.lemmata_lege "lemmata.de")));
      ignore (itemAnglice#connect#activate (fun () -> (
          Latinus.lemmata_lege "lemmata.uk")));
      ignore (wt_lat#event#connect#button_release ~callback:
	    (fun ev -> 
          GdkEvent.Button.button ev = 1 &&
          begin
	      let it = wt_lat#buffer#get_iter_at_mark `INSERT in
          let start = find_word_start it in
	      let stop = find_word_end start in
	      let f =wt_lat#buffer#get_text ~slice:true ~start ~stop () in
          let r = "\n" ^ Latinus.lemmatise_u f in
          self#tabula#buffer#insert (to_utf8 r) ;
          (* drouler en bas de page *)
          self#tabula#scroll_to_mark `INSERT;
          (*wt_lat#misc#grab_focus ();*)
          false
          end
          ));

      (* Auxilium *)
      ignore (itemDe#connect#activate (fun () ->
            let scripta = "Collatinus - latinae linguae lemmatizatio"
              ^ "\n Gratias tibi Jacques Julien, "
              ^ "\n et François Marmèche, "
              ^ "\n et Philippe Remacle, "
              ^ "\n et Georges Khaznadar. "
              ^ "\n Licentia GPL " 
              ^ "\n © Yves Ouvrard, 2004 - 2006" in
             let md = GWindow.message_dialog
                ~message: scripta
                ~message_type: `INFO
                ~buttons: GWindow.Buttons.ok
                ~modal: true
                ~show:true 
                ()
             in ignore (md#run());
             md#destroy ()
             ));

      ignore (itemAuxilium#connect#activate (fun () ->
         prerr_endline "inter agenda"));
          

      let tooltips = GData.tooltips () in
      let _ = GtkBase.Widget.add_events wt_lat#as_widget
	       [`ENTER_NOTIFY;`POINTER_MOTION; `LEAVE_NOTIFY] in

      ignore (wt_lat#event#connect#enter_notify 
          ~callback: (fun e -> tooltips#enable (); true ));
      
      ignore (wt_lat#event#connect#leave_notify 
          ~callback: (fun e -> tooltips#disable (); tooltips#set_tip ~text:"" wt_lat#coerce; true ));

      ignore (wt_lat#event#connect#motion_notify
	      ~callback:
             (* Calculer le mot sous la souris *)
		     (fun e -> 
             tooltips#enable ();
		     let win = match wt_lat#get_window `WIDGET with
		       | None -> assert false
		       | Some w -> w
		     in
		     let x,y = Gdk.Window.get_pointer_location win in
		     let b_x,b_y = wt_lat#window_to_buffer_coords 
				     ~tag:`WIDGET 
				     ~x 
				     ~y 
		     in
		     let it = wt_lat#get_iter_at_location ~x:b_x ~y:b_y in
             let start = find_word_start it in
             let stop = find_word_end start in
             let f =wt_lat#buffer#get_text ~slice:true ~start ~stop () in
             (* et demander l'analyse morpho du mot sous le curseur *)
             let r = "\n" ^ Latinus.analyse_u f in
             tooltips#set_tip ~text: (to_utf8 r) wt_lat#coerce ;
		     true));


      self#load_file "lucretia.txt";

      let bt = GText.buffer ~text:" " ()  ~tag_table: tt in
      wt_txt#set_buffer bt;
      let bl = GText.buffer ~text:" " ()  ~tag_table: tt in
      wt_tex#set_buffer bl;
      let bh = GText.buffer ~text:" " ()  ~tag_table: tt in
      wt_html#set_buffer bh;
      (*
      let bf = GText.buffer ~text:"Haec scripturus sum." () ~tag_table: tt in
      wt_flexio#set_buffer bf;
      *)

      (* fenestra apparet *)
      fenestra#show ();
      (* signum nullae modificationis *)
      self#integrum
      end

(* Fenestrae aedificatur *)
let c = new gui ()

(* euentuum ciculum intremus *)
let _ = GMain.Main.main ()
