/* test_contrib.cpp: implementation file for clipper contrib self-test */
//C Copyright (C) 2000-2006 Kevin Cowtan and University of York
//L
//L  This library is free software and is distributed under the terms
//L  and conditions of version 2.1 of the GNU Lesser General Public
//L  Licence (LGPL) with the following additional clause:
//L
//L     `You may also combine or link a "work that uses the Library" to
//L     produce a work containing portions of the Library, and distribute
//L     that work under terms of your choice, provided that you give
//L     prominent notice with each copy of the work that the specified
//L     version of the Library is used in it, and that you include or
//L     provide public access to the complete corresponding
//L     machine-readable source code for the Library including whatever
//L     changes were used in the work. (i.e. If you make changes to the
//L     Library you must distribute those, but you do not need to
//L     distribute source or object code to those portions of the work
//L     not covered by this licence.)'
//L
//L  Note that this clause grants an additional right and does not impose
//L  any additional restriction, and so does not affect compatibility
//L  with the GNU General Public Licence (GPL). If you wish to negotiate
//L  other terms, please contact the maintainer.
//L
//L  You can redistribute it and/or modify the library under the terms of
//L  the GNU Lesser General Public License as published by the Free Software
//L  Foundation; either version 2.1 of the License, or (at your option) any
//L  later version.
//L
//L  This library is distributed in the hope that it will be useful, but
//L  WITHOUT ANY WARRANTY; without even the implied warranty of
//L  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//L  Lesser General Public License for more details.
//L
//L  You should have received a copy of the CCP4 licence and/or GNU
//L  Lesser General Public License along with this library; if not, write
//L  to the CCP4 Secretary, Daresbury Laboratory, Warrington WA4 4AD, UK.
//L  The GNU Lesser General Public can also be obtained by writing to the
//L  Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
//L  MA 02111-1307 USA


#include "test_contrib.h"
#include "mapfilter.h"
#include "convolution_search.h"
#include "fffear.h"
#include "sfcalc.h"
#include "sfcalc_obs.h"
#include "sfweight.h"
#include "sfscale.h"
#include "../core/test_data.h"
#include "../core/hkl_compute.h"


namespace clipper {

namespace data {

  float contrib_vals[] = {62085,0,0.000489426,-537.853,3.6676e-05,271.468,313.756,-7.91723e-06,88.9389,7.16639e-06,6.76159e-05,137.227,-1020.36,-230.467,258.784,19.4909,-21.0351,168.158,82.8908,82.9012,0.000325024,-106.365,-4.12623,311.855,-182.047,291.716,-473.216,24.3526,58.3351,-90.6147,-72.7842,46.5315,123.021,-21.1225,-156.35,110.062,-277.138,-442.304,-142.18,89.8284,3.18781e-05,-73.6002,271.811,-64.1525,-270.657,-265.163,273.196,779.432,256.913,276.635,226.079,544.524,644.946,-244.673,3.80937e-05,-504.567,4.02741,-451.095,27.3692,95.8539,-1049.43,-0.000875712,332.786,-326.619,-120.613,-233.546,374.933,-125.7,-1.43747e-05,328.855,67.4085,-117.594,-579.966,-250.85,-286.214,421.978,-407.023,-0.000291126,379.259,20.3695,-546.954,33.0032,638.675,-649.005,70.5813,-146.142,224.169,43.4661,302.437,49.827,390.892,-184.531,215.482,-101.874,545.265,474.695,0.000136025,364.171,0.0011328,-1244.89,0.000452873,-497.683,0.99961,9.55408e-06,70.7172,-1.10399e-05,-81.7153,0.964763,622.421,-1.5706e-05,189.423,-4.77983e-06,0.999997,58.2484,4.69345e-06,-5.21254,-4.20008e-07,0.485547,8.05844e-05,163.547,1.16586e-05,23.6612,0.821406,-637.729,-144.043,168.498,38.0584,0.981985,111.524,8.39966,-55.5248,-4.18197,0.878919,6.41568,-51.2879,11.2749,-90.1328,0.505131,187.321,187.344,61.8825,61.8902,0.839203,0.000369899,-121.051,0.000148723,-48.67,0.206272,-6.79966,513.909,-1.7877,135.112,0.938164,-159.46,255.522,-13.3331,21.3652,0.827869,-470.603,24.2181,-58.7401,3.02288,0.962895,65.199,-101.277,8.33814,-12.952,0.643657,-26.3343,4.52155,-36.3731,6.2452,0.105881,33.7553,-23.7619,76.6316,-53.9445,0.601312,-204.495,-326.368,10.7737,17.1945,0.962109,-105.289,66.5212,2.94677,-1.86175,0.708906,0.000155346,-358.663,6.5296e-05,-150.755,0.703103,185.297,-43.7336,-24.722,5.83487,0.917471,-151.123,-148.055,33.0658,32.3945,0.918929,218.346,622.944,-0.55324,-1.5784,0.983466,104.308,112.315,-47.282,-50.9117,0.933225,136.292,328.267,-29.8151,-71.8115,0.97852,604.09,-229.174,41.5473,-15.7618,0.978951,2.67642e-05,-354.502,-1.48389e-06,19.6547,0.99935,3.55235,-397.885,0.086151,-9.64941,0.967002,12.8526,45.013,-3.58072,-12.5406,0.475785,-623.255,-0.000520084,136.345,0.000113775,1,231.224,-226.939,-9.57715,9.39968,0.965113,-77.3065,-149.69,7.53276,14.5858,0.863165,335.14,-112.359,13.1663,-4.41414,0.961459,-3.89951e-06,89.2104,3.66052e-06,-83.743,0.963529,-28.856,50.3391,-40.7842,71.1478,0.334988,-453.601,-196.194,-0.238624,-0.103211,0.974458,-181.776,268.001,20.2216,-29.8136,0.979734,-17.8819,-1.27901e-05,140.521,0.000100509,0.984483,81.2454,4.36357,-100.431,-5.39399,0.933014,-466.079,28.1232,-3.16235,0.190817,0.976128,597.613,-607.279,28.6369,-29.1001,0.990703,200.062,-414.238,74.1539,-153.539,0.86706,528.692,102.512,173.657,33.6718,0.946721,231.947,38.2136,-14.2682,-2.35072,0.940218,334.414,-157.869,24.0072,-11.3332,0.955347,171.91,-81.2741,1.67601,-0.792372,0.89157,285.505,248.554,-89.1148,-77.5814,0.987069,1.27172e-05,34.0467,-4.87792e-05,-130.593,0.997212};
  float contrib_tols[] = {0.31,1e-09,0.0013,0.0029,3.2e-05,0.0014,0.0016,3.3e-05,0.00048,5.8e-05,0.00018,0.00075,0.0055,0.0034,0.0013,0.00016,0.00044,0.001,0.00048,0.00048,0.00089,0.00083,0.00047,0.0017,0.0014,0.0021,0.0025,0.00035,0.00035,0.00049,0.00051,0.00039,0.00069,0.00024,0.001,0.00088,0.0022,0.0026,0.001,0.00065,0.00011,0.00049,0.0015,0.0004,0.002,0.0014,0.0017,0.0042,0.0014,0.0015,0.0013,0.0028,0.0034,0.0013,1.2e-09,0.0026,7.5e-05,0.0023,0.0004,0.00053,0.0055,0.0021,0.0017,0.0018,0.0011,0.0014,0.0019,0.00072,7.8e-05,0.0017,0.00051,0.00069,0.0033,0.0022,0.0022,0.0028,0.0021,0.00071,0.0021,0.00018,0.0029,0.0011,0.0033,0.0033,0.00048,0.0009,0.0012,0.00031,0.0016,0.00033,0.002,0.00099,0.0012,0.00064,0.0029,0.0025,0.0003,0.0019,0.003,0.0062,0.0012,0.0025,5e-06,8.4e-06,0.00038,9.7e-06,0.00043,4.8e-06,0.0031,6.6e-05,0.00098,2e-05,5e-06,0.00031,3.8e-05,3.4e-05,3.4e-06,2.6e-06,0.00021,0.00082,3.1e-05,0.00014,4.3e-06,0.0036,0.0021,0.0011,0.00058,4.9e-06,0.00059,7.1e-05,0.0003,3.4e-05,4.4e-06,0.00013,0.00032,0.00023,0.00055,2.8e-06,0.0011,0.0011,0.00036,0.00036,4.3e-06,0.001,0.00095,0.00041,0.00037,1.6e-06,0.00077,0.0027,0.0002,0.00074,4.7e-06,0.0014,0.0015,0.00021,0.00024,4.3e-06,0.0024,0.00035,0.00033,4.5e-05,4.8e-06,0.00039,0.00054,5e-05,7.3e-05,3.4e-06,0.00015,5e-05,0.0002,6.9e-05,6e-07,0.00024,0.00019,0.00051,0.00043,3.2e-06,0.0015,0.0021,0.00015,0.00017,4.8e-06,0.00065,0.00055,7.4e-05,3.9e-05,3.8e-06,0.00051,0.0021,0.00022,0.00088,4.2e-06,0.001,0.00031,0.00026,5.5e-05,4.6e-06,0.00081,0.0011,0.0004,0.0003,4.7e-06,0.0014,0.0033,5.6e-05,0.00016,5e-06,0.00057,0.00059,0.00026,0.00029,4.7e-06,0.00074,0.0018,0.00021,0.00047,4.9e-06,0.0032,0.0012,0.00037,0.00014,4.9e-06,1.1e-09,0.0019,1e-09,0.0002,5e-06,6.5e-05,0.002,1.6e-06,0.00014,4.8e-06,0.00018,0.00025,5.6e-05,8.2e-05,2.6e-06,0.0033,0.0012,0.0009,0.00027,5e-06,0.0013,0.0012,0.00013,0.00013,4.8e-06,0.0007,0.00094,7.9e-05,0.00011,4.4e-06,0.0018,0.00062,0.00012,4e-05,4.8e-06,2.1e-05,0.00049,2e-05,0.00046,4.8e-06,0.00022,0.0003,0.00031,0.00042,1.9e-06,0.0026,0.0017,7.7e-05,3.4e-05,4.9e-06,0.0015,0.0017,0.00014,0.00026,4.9e-06,0.00017,3.1e-05,0.00077,0.00024,4.9e-06,0.00054,3.5e-05,0.00063,5.2e-05,4.7e-06,0.0025,0.00092,0.0001,1e-05,4.9e-06,0.0031,0.0032,0.0002,0.00021,5e-06,0.0014,0.0023,0.00052,0.00082,4.5e-06,0.0027,0.0007,0.00095,0.00023,4.8e-06,0.0013,0.00024,0.00016,2.6e-05,4.8e-06,0.0018,0.00084,0.00016,7.2e-05,4.8e-06,0.0009,0.00053,3.2e-05,1.6e-05,4.5e-06,0.0015,0.0014,0.00052,0.00043,4.9e-06,2.8e-05,0.00021,0.00011,0.00069,5e-06};
  int contrib_size = sizeof(contrib_vals)/sizeof(contrib_vals[0]);

}

bool Test_contrib::operator() () {
  data_val = std::vector<float>( data::contrib_vals, 
				 data::contrib_vals+data::contrib_size );
  data_tol = std::vector<float>( data::contrib_tols, 
				 data::contrib_tols+data::contrib_size );

  typedef HKL_info::HKL_reference_index HRI;
  data::Test_data data;
  const HKL_data<data32::F_sigF>& fsig = data.hkl_data_f_sigf();
  const HKL_data<data32::ABCD>&   abcd = data.hkl_data_abcd();
  const clipper::Atom_list&       xyzb = data.atom_list();
  const Spacegroup spgr = fsig.hkl_info().spacegroup();
  const Cell       cell = fsig.hkl_info().cell();
  const Resolution reso = fsig.hkl_info().resolution();

  typedef HKL_info::HKL_reference_index HRI;
  typedef Xmap<float>::Map_reference_index MRI;

  // test sfcalc objects
  {
    // select spacegroup
    std::vector<String> hallsymbols;
    for ( int i = 0; i < data::sgdata_size; i += 10 )
      hallsymbols.push_back( data::sgdata[i].hall );
    // build model
    Atom_list atoms;
    Atom atom = Atom::null();
    atom.set_occupancy( 1.0 );
    atom.set_u_iso( 0.5 );
    atom.set_element( "C" );
    atom.set_coord_orth( Coord_orth( 12, 8, 5 ) );
    atoms.push_back( atom );
    atom.set_element( "N" );
    atom.set_coord_orth( Coord_orth( 11, 6, 4 ) );
    atoms.push_back( atom );
    atom.set_element( "O" );
    atom.set_coord_orth( Coord_orth( 13, 5, 4 ) );
    atoms.push_back( atom );
    // calc cell
    Cell cellc( Cell_descr( 37, 37, 37 ) );
    Cell cellha( Cell_descr( 37, 37, 37, 120, 90, 90 ) );
    Cell cellhb( Cell_descr( 37, 37, 37, 90, 120, 90 ) );
    Cell cellhc( Cell_descr( 37, 37, 37, 90, 90, 120 ) );
    Cell cg;
    String symbol;
    Spacegroup sg;
    for ( int s = 0; s < hallsymbols.size(); s++ ) {
      try {
	symbol = hallsymbols[s];
	sg = Spacegroup( Spgr_descr( symbol, Spgr_descr::Hall ) );
	// identify trigonal/hexagonal groups
	cg = cellc;
	for ( int sym = 1; sym < sg.num_symops(); sym++ ) {
	  if ( ( sg.symop(sym).rot()(1,1) * sg.symop(sym).rot()(1,2) == -1 ) ||
	       ( sg.symop(sym).rot()(2,1) * sg.symop(sym).rot()(2,2) == -1 ) )
	    cg = cellha;
	  if ( ( sg.symop(sym).rot()(0,0) * sg.symop(sym).rot()(0,2) == -1 ) ||
	       ( sg.symop(sym).rot()(2,0) * sg.symop(sym).rot()(2,2) == -1 ) )
	    cg = cellhb;
	  if ( ( sg.symop(sym).rot()(0,0) * sg.symop(sym).rot()(0,1) == -1 ) ||
	       ( sg.symop(sym).rot()(1,0) * sg.symop(sym).rot()(1,1) == -1 ) )
	    cg = cellhc;
	}
	HKL_info hkl_info( sg, cg, Resolution( 5.0 ), true );
	HKL_data<data32::F_phi> fp1( hkl_info );
	HKL_data<data32::F_phi> fp2( hkl_info );
	SFcalc_iso_sum<float>( fp1, atoms );
	SFcalc_iso_fft<float>( fp2, atoms, 2.5, 2.5, 0.25 );
	// begin extra fft tests
	Grid_sampling gg( sg, cg, hkl_info.resolution() );
	Xmap<float> xg( sg, cg, gg );
	FFTmap fftmap( sg, cg, gg );
	xg.fft_from( fp2, Xmap_base::Normal );
	xg.fft_to( fp2, Xmap_base::Sparse );
	fftmap.fft_rfl_to_map( fp2, xg );
	xg.fft_to( fp2, Xmap_base::Normal );
	xg.fft_from( fp2, Xmap_base::Sparse );
	fftmap.fft_map_to_rfl( xg, fp2 );
	// end extra fft tests
	float tol = 0.005 * fp1[ HKL( 0, 0, 0 ) ].f();
	for ( HRI ih = fp1.first(); !ih.last(); ih.next() ) {
	  std::complex<float> ab1(0.0,0.0), ab2(0.0,0.0);
	  if ( !fp1[ih].missing() ) ab1 = fp1[ih];
	  if ( !fp2[ih].missing() ) ab2 = fp2[ih];
	  test( "SF-A", ab1.real(), ab2.real(), tol );
	  test( "SF-B", ab1.imag(), ab2.imag(), tol );
	}
      } catch ( Message_base ) {
	test( "SFSG "+symbol, sg.spacegroup_number(), -1 );
      }
    }
  }

  // test sfcalc_obs and sfweight
  {
    // sfcalc_obs
    HKL_data<data32::F_phi> fcal( fsig.hkl_info() );
    SFcalc_obs_bulk<float> sfcb;
    sfcb( fcal, fsig, xyzb );

    // sfweight
    HKL_data<data32::F_phi> fb( fsig.hkl_info() ), fd( fsig.hkl_info() );
    HKL_data<data32::Phi_fom> phiw( fsig.hkl_info() );
    HKL_data<data32::Flag> flag( fsig.hkl_info() );
    for ( HRI ih = flag.first(); !ih.last(); ih.next() )
      if ( !fsig[ih].missing() )
	flag[ih].flag() = SFweight_spline<float>::BOTH;
      else
	flag[ih].flag() = SFweight_spline<float>::NONE;
    clipper::SFweight_spline<float> sfw( 600, 12 );
    bool fl = sfw( fb, fd, phiw, fsig, fcal, flag );
    if ( !fl ) test( "SFW-FAIL", 1, 0 );

    // sfcalc_obs results
    for ( HRI ih = fcal.first(); !ih.last(); ih.next() )
      if ( ih.index() % 20 == 0 ) {
	std::complex<float> ab( fcal[ih] );
	test( "SFO-A", ab.real() );
	test( "SFO-B", ab.imag() );
      }
    // sfweight results
    for ( HRI ih = phiw.first(); !ih.last(); ih.next() )
      if ( !fsig[ih].missing() )
	if ( ih.index() % 20 == 0 ) {
	  std::complex<float> ab_b( fb[ih] );
	  std::complex<float> ab_d( fd[ih] );
	  test( "SFWB-A", ab_b.real() );
	  test( "SFWB-B", ab_b.imag() );
	  test( "SFWD-A", ab_d.real() );
	  test( "SFWD-B", ab_d.imag() );
	  test( "SFW-W", phiw[ih].fom() );
	}
  }

  // test map filter objects
  {
    HKL_data<data32::Phi_fom> pw( fsig.hkl_info() );
    HKL_data<data32::F_phi> fp( fsig.hkl_info() );
    pw.compute( abcd, data32::Compute_phifom_from_abcd() );
    fp.compute( fsig, pw, data32::Compute_fphi_from_fsigf_phifom() );
    Grid_sampling grid( spgr, cell, reso, 2.5 );
    Xmap<float> xmap( spgr, cell, grid );
    xmap.fft_from( fp );

    MapFilterFn_step step( 2.5 );
    MapFilter_slow<float> fltr1( step, 1.0, MapFilter_slow<float>::Relative );
    MapFilter_fft<float>  fltr2( step, 1.0, MapFilter_fft<float>::Relative );
    Xmap<float> f1, f2;
    fltr1( f1, xmap );
    fltr2( f2, xmap );

    for ( MRI ix = xmap.first(); !ix.last(); ix.next() )
      test( "MAPFILTER", f1[ix], f2[ix], 0.0001 );
  }

  // test the convolution and fffear objects
  {
    HKL_data<data32::Phi_fom> pw( fsig.hkl_info() );
    HKL_data<data32::F_phi> fp( fsig.hkl_info() );
    pw.compute( abcd, data32::Compute_phifom_from_abcd() );
    fp.compute( fsig, pw, data32::Compute_fphi_from_fsigf_phifom() );
    Grid_sampling grid( spgr, cell, reso, 2.5 );
    Xmap<float> xmap( spgr, cell, grid );
    xmap.fft_from( fp );

    Xmap<float> r1( Spacegroup::p1(), xmap.cell(), xmap.grid_sampling() );
    Xmap<float> r2( Spacegroup::p1(), xmap.cell(), xmap.grid_sampling() );
    int irad = 3;
    clipper::Grid_range tg( clipper::Coord_grid(-irad,-irad,-irad),
			    clipper::Coord_grid( irad, irad, irad) );
    NXmap<float> target( xmap.cell(), xmap.grid_sampling(), tg );
    NXmap<float> weight( xmap.cell(), xmap.grid_sampling(), tg );
    target = weight = 0.0;
    for ( Coord_grid c = tg.min(); !c.last(tg); c.next(tg) ) {
      if ( c*c <= 5 ) {
	target.set_data(c-tg.min(),xmap.get_data(c));
	weight.set_data(c-tg.min(),1.0);
      }
    }

    Convolution_search_slow<float> conv1( xmap );
    Convolution_search_fft<float>  conv2( xmap );
    conv1( r1, target );
    conv2( r2, target );
    for ( MRI ix = r1.first(); !ix.last(); ix.next() )
      test( "CONVOL", r1[ix], r2[ix], 0.0001 );

    FFFear_slow<float> srch1( xmap );
    FFFear_fft<float>  srch2( xmap );
    srch1( r1, target, weight );
    srch2( r2, target, weight );

    for ( MRI ix = r1.first(); !ix.last(); ix.next() )
      test( "FFFEAR", r1[ix], r2[ix], 0.0001 );
  }

  // test anisotropic scaling
  {
    // expand to P1
    Spacegroup spgrp1( Spacegroup::P1 );
    HKL_info hkl1( spgrp1, cell, fsig.hkl_info().resolution(), true );
    HKL_data<data32::F_sigF> fs( hkl1 );
    for ( HRI ih = hkl1.first(); !ih.last(); ih.next() )
      fs[ih] = fsig[ih.hkl()];
    // make data objects
    HKL_data<data32::F_phi>  fp( fs.hkl_info() );
    HKL_data<data32::F_sigF> fs1 = fs;
    HKL_data<data32::F_sigF> fs2 = fs;
    U_aniso_orth u_ref( 0.10, 0.13, 0.17, -0.02, 0.03, -0.04 );
    // simulate aniso data
    for ( HRI ih = fs.first(); !ih.last(); ih.next() )
      if ( !fs[ih].missing() ) {
	Coord_reci_orth c = ih.hkl().coord_reci_orth(cell);
	double s = exp( Util::twopi2() * 0.5 * u_ref.quad_form( c ) );
	fs1[ih].scale(s);
	fs2[ih].scale(1.0/s);
	fp[ih] = data32::F_phi( fs[ih].f(), 0.0 );
      }
    // now attempt scaling
    clipper::SFscale_aniso<float> sfscl;
    sfscl( fs1, fp );
    U_aniso_orth u_wrk1 = sfscl.u_aniso_orth();
    sfscl( fp, fs2 );
    U_aniso_orth u_wrk2 = sfscl.u_aniso_orth();
    test( "ANISO-O-00", u_ref.mat00(), u_wrk1.mat00(), 1.0e-6 );
    test( "ANISO-O-11", u_ref.mat11(), u_wrk1.mat11(), 1.0e-6 );
    test( "ANISO-O-22", u_ref.mat22(), u_wrk1.mat22(), 1.0e-6 );
    test( "ANISO-O-01", u_ref.mat01(), u_wrk1.mat01(), 1.0e-6 );
    test( "ANISO-O-02", u_ref.mat02(), u_wrk1.mat02(), 1.0e-6 );
    test( "ANISO-O-12", u_ref.mat12(), u_wrk1.mat12(), 1.0e-6 );
    test( "ANISO-C-00", u_ref.mat00(), u_wrk2.mat00(), 1.0e-6 );
    test( "ANISO-C-11", u_ref.mat11(), u_wrk2.mat11(), 1.0e-6 );
    test( "ANISO-C-22", u_ref.mat22(), u_wrk2.mat22(), 1.0e-6 );
    test( "ANISO-C-01", u_ref.mat01(), u_wrk2.mat01(), 1.0e-6 );
    test( "ANISO-C-02", u_ref.mat02(), u_wrk2.mat02(), 1.0e-6 );
    test( "ANISO-C-12", u_ref.mat12(), u_wrk2.mat12(), 1.0e-6 );
  }

  return ( error_count == 0 );
}


} // namespace clipper
