using Gtk;

namespace Clinica {
    
    public class MedicineSearchPage : Alignment, Page {
    
        /**
         * @brief Entry used to insert search keys for the 
         * medicines.
         */
        private FindEntry find_entry;
        
        /**
         * @brief Treeview used to display the list of the medicines
         * that have been found by the search engine.
         */
        public MedicineTreeView treeview;
        
        private string key;
        
        public ResourceManager resource_manager { get; set; }
        
        private unowned MedicineSearchEngine? selected_search_engine = null;
        
        /**
         * @brief Button that brings back to the main page or stop searches if they
         * are running. 
         */
        private Button back_button;
        
        public MedicineSearchPage (ResourceManager resources) {
            resource_manager = resources;
            connect_signal ();
            
            var main_vbox = new VBox (false, 5);
            
            /* Add a search bar on top */
            find_entry = new FindEntry ();
            find_entry.activate.connect (on_find_entry_activate);

            /* And the button to go back */
            back_button = new Button.with_label (_("Back"));
            back_button.clicked.connect (on_back_button_clicked);
            
            /* Creating the search box */
            var search_box = new HBox (false, 5);
            search_box.pack_start (find_entry);
            search_box.pack_start (back_button, false);
            main_vbox.pack_start (search_box, false);
            
            /* And the treeview just below */
            treeview = new MedicineTreeView (resource_manager);
            var scrolled_window = new ScrolledWindow (null, null);
            scrolled_window.set_shadow_type (ShadowType.IN);
            scrolled_window.add (treeview);
            main_vbox.pack_start (scrolled_window);
            
            /* Show all the items added to this VBox */
            add (main_vbox);
            show_all ();
        }
        
        /**
         * @brief Callback called when the used click on the back button.
         */
        private void on_back_button_clicked (Button back_button) {
            if (find_entry.is_sensitive ())
                start_page_requested ();
            else {
                selected_search_engine.abort_search ();
                find_entry.set_sensitive (true);
                (back_button.get_child () as Label).set_text (_("Back"));
            }
        }
        
        /**
         * @brief Callback called when the user want to search for a Medicine.
         */
        private void on_find_entry_activate (Editable e) {
            /* Select the right search engine */
            selected_search_engine = resource_manager.settings.get_medicine_search_engine ();
            if (selected_search_engine == null) {
                debug ("No search engine available");
                
                /* Tell to the user that a plugin is needed */
                var dialog = new MessageDialog.with_markup (null, 
                    DialogFlags.MODAL,
                    MessageType.INFO,
                    ButtonsType.OK,
                    _("It's not possible to perform a search for medicine
since no search engine is available.
You should enable a search engine in the plugins
to solve this issue."));
                dialog.set_title (_("No search engine available"));
                dialog.set_transient_for (resource_manager.user_interface.window);
                dialog.run ();
                dialog.destroy ();    
                return;
            }
            
            key = find_entry.get_text ();
            find_entry.set_text (_("Searching for %s...").printf (key));
            find_entry.set_sensitive (false);
            (back_button.get_child () as Label).set_text (_("Stop"));
            treeview.clear ();
            /* And start the research in another thread */
            try {
                Thread<void>.create<void> (this.do_search, false);
            } catch (GLib.Error e) {
                // FIXME: Add the error handler.
            }
        }
        
        /**
         * @brief Function that performs the search, spawned in another thread.
         */
        private void do_search () {
            selected_search_engine.search_medicine (key, treeview);
                
            Gdk.threads_enter ();                
            find_entry.set_sensitive (true);
            find_entry.set_text ("");
            Gdk.threads_leave ();
        }
        
        public void setup () {
            new_title (_("Medicine search"));
        }

    }
}
