#!/usr/bin/env python

########################################################################
# Chris's Lame Filebrowser 4 (System Quick Browser)
# Copyright 2004, Gabe Ginorio <gabe@zevallos.com.br>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
######################################################################

import commands, os, os.path, string, sys, shutil, re
import gtk, gtk.gdk, pygtk, pango
import overwrite_dialog, file_mover, xml_reader, db
from xml.dom import minidom

IMAGE_DIRECTORY= os.path.abspath(string.replace(sys.argv[0],"claw4.py","") + "Images")

class create:
	def __init__(self, Notebook):
		'''This class creates the bookmarks window, the system quick browser and the
		preferences dialog in a notebook'''
		self.main_notebook = gtk.Notebook()
		self.Notebook = Notebook

		# Get the theme from the preferences.xml file
		preferences_file = minidom.parse(os.environ['HOME'] + "/.claw/preferences.xml")
		node = preferences_file.getElementsByTagName("theme")
		self.theme = node[0].childNodes[0].data

		main_tree_view = self.__systemQuickBrowser__()
		tree_view_pic = gtk.Image()
		pixbuf = gtk.gdk.pixbuf_new_from_file(IMAGE_DIRECTORY +"/Icons/" + self.theme +"/folder.png")
		pixbuf = pixbuf.scale_simple(16, 16, gtk.gdk.INTERP_HYPER)
		tree_view_pic.set_from_pixbuf(pixbuf)
		tree_scroll = gtk.ScrolledWindow()

		tree_scroll.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
		tree_scroll.add_with_viewport(main_tree_view)

		self.main_notebook.append_page(tree_scroll, tree_view_pic)

		main_list = self.__bookmarkWindow__()
		list_pic = gtk.Image()
		list_pic.set_from_file(IMAGE_DIRECTORY + "/Sidebar/bookmark.png")
		self.main_notebook.append_page(main_list ,list_pic)

		preference_dialog = self.__preferenceDialog__()
		preference_pic = gtk.Image()
		preference_pic.set_from_file(IMAGE_DIRECTORY + "/Sidebar/preferences.png")
		self.main_notebook.append_page(preference_dialog, preference_pic)

		look_dialog = self.__lookDialog__()
		look_pic = gtk.Image()
		look_pic.set_from_file(IMAGE_DIRECTORY + "/Sidebar/look.png")
		self.main_notebook.append_page(look_dialog, look_pic)

		self.main_notebook.show_all()
	

	def __systemQuickBrowser__(self):
		'''This function creates the system quick browser tree view and model'''
		main_tree_view = gtk.TreeView()

		main_tree_store = gtk.TreeStore(gtk.gdk.Pixbuf, str, str)
		main_tree_view = gtk.TreeView(main_tree_store)

		main_tree_view.set_events(gtk.gdk.BUTTON_PRESS_MASK)
		main_tree_view.connect("button-press-event", self.__directoryClicked__)

		tree_selection = main_tree_view.get_selection()
		tree_selection.set_mode(gtk.SELECTION_BROWSE)
		tree_selection.connect("changed", self.__selectionChanged__, main_tree_view)

		icon_column = gtk.TreeViewColumn("")
		icon_column.set_sizing(gtk.TREE_VIEW_COLUMN_AUTOSIZE)
		name_column = gtk.TreeViewColumn("File Name")

		main_tree_view.append_column(icon_column)
		main_tree_view.append_column(name_column)

		main_tree_view.set_search_column(0)

		pixbuf_renderer = gtk.CellRendererPixbuf()
		name_renderer = gtk.CellRendererText()

		icon_column.pack_start(pixbuf_renderer, False)
		name_column.pack_start(name_renderer, True)

		icon_column.add_attribute(pixbuf_renderer, 'pixbuf', 0)
		name_column.add_attribute(name_renderer, "text", 1)

		main_tree_view.show()

		self.__initializeTree__(main_tree_store)

		main_tree_view.enable_model_drag_dest([("text/plain", 0, 80)], gtk.gdk.ACTION_COPY)		
		main_tree_view.connect("drag-data-received", self.__fileDropped__)		

		return main_tree_view


	def __initializeTree__(self, main_tree_store):
		'''This function populates the main tree store'''

		pixbuf = gtk.gdk.pixbuf_new_from_file(IMAGE_DIRECTORY +"/Icons/" + self.theme +"/folder.png")
		pixbuf = pixbuf.scale_simple(16, 16, gtk.gdk.INTERP_HYPER)

		files = os.listdir("/")
		files.sort(lambda x,y : cmp (string.lower(x), string.lower(y)))

		for afile in files:
			if os.path.isdir("/" + afile) and string.find(afile, ".", 0, 1) == -1:
				next_iter = main_tree_store.append(None, [pixbuf, os.path.basename(afile),"/" + afile])
			#main_tree_store.self.treestore.append(next_iter, ['child %i of parent %i' %(child, parent)])


	def __fileDropped__(self, main_tree_view, drag_context, x, y, selection_data, info, timestamp):
		'''This function is the callback when a file is dropped into the system quick browser'''

		main_tree_store = main_tree_view.get_model()

		# Make sure the files were dropped to a folder and that something valid was dropped
		if  main_tree_view.get_dest_row_at_pos(x, y):
			files = string.split(selection_data.data, "   ")
			files.pop()

			if files == []:
				self.Notebook.info_bar.set_text("No Valid Files To Drop")
				self.Notebook.info_bar.modify_fg(gtk.STATE_NORMAL, gtk.gdk.Color(40000, 0, 0, 0))
				return 0

			path, position =  main_tree_view.get_dest_row_at_pos(x, y)
			drop_iter = main_tree_store.get_iter(path)
			destination = main_tree_store.get_value(drop_iter, 2)

			if os.path.dirname(files[0]) == destination:
				self.Notebook.info_bar.set_text("Can't Drop Into Current Directory")
				self.Notebook.info_bar.modify_fg(gtk.STATE_NORMAL, gtk.gdk.Color(40000, 0, 0, 0))
				return 0

			old_files = os.listdir(destination)
			for afile in files: 
				afile = os.path.realpath(afile)
				if os.path.basename(afile) in old_files:
					x,y = self.__getMousePosition__()
					answer = overwrite_dialog.create(x, y, "Overwrite " + os.path.basename(afile))

					if answer == "yes":
						file_mover.move( "copy", x, y, "Copying " + os.path.basename(afile), afile, destination)
				else:
					file_mover.move( "copy", x, y, "Copying " + os.path.basename(afile), afile, destination)


	def __selectionChanged__(self, tree_selection, main_tree_view):
		'''The function handles expanding directories in the tree store'''

		main_tree_store, selected_iter = tree_selection.get_selected()

		if not selected_iter or  main_tree_store.get_value(selected_iter, 2) == "":
			return 0
		else:
			while main_tree_store.iter_has_child(selected_iter):
				main_tree_store.remove(main_tree_store.iter_children(selected_iter))

			location = main_tree_store.get_value(selected_iter, 2)
			self.__expandDirectory__(location, selected_iter, main_tree_store)
			main_tree_view.expand_row(main_tree_store.get_path(selected_iter), False)


	def __directoryClicked__(self, main_tree_view, event):
		'''This function handles mouse clicks on the folders (controls expansion of directories)'''

		tree_selection =  main_tree_view.get_selection()
		main_tree_store, selected_iter = tree_selection.get_selected()
		if tree_selection == None:
			return 0

		# A right click opens
		if event.button == 3:
			location =  main_tree_store.get_value(selected_iter, 2)
			self.Notebook._populate_(os.path.realpath(location))

	
	def __expandDirectory__(self, location, selected_iter, main_tree_store):
		'''This function adds child nodes (child directories) to a parent node (parent directory)'''

		pixbuf = gtk.gdk.pixbuf_new_from_file(IMAGE_DIRECTORY +"/Icons/" + self.theme +"/folder.png")
		pixbuf = pixbuf.scale_simple(16, 16, gtk.gdk.INTERP_HYPER)

		if os.listdir(location) != []:

			files = os.listdir(location)
			files.sort(lambda x,y : cmp (string.lower(x), string.lower(y)))

			for afile in files:
				if os.path.isdir(location + "/" + afile) and string.find(afile, ".", 0, 1) == -1:
					main_tree_store.append(selected_iter, [pixbuf, afile, location + "/" + afile])


########################## BOOKMARKS WINDOW ###############################

	def __bookmarkWindow__(self):
		'''This function creates the bookmark list'''
		self.bookmarks = {}

		main_list_scroll = gtk.ScrolledWindow()
		main_list_scroll.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)

		main_list_store = gtk.ListStore(str, str)
		main_list = gtk.TreeView(main_list_store)
		main_list.set_reorderable(False)

		main_list.set_events(gtk.gdk.BUTTON_PRESS_MASK)
		main_list.connect("button-press-event", self. __bookmarkSelected__)
		
		# This should handle key presses without the annoying search box popping up
		main_list.set_events(gtk.gdk.KEY_PRESS_MASK)
		main_list.connect("key-press-event", self.__buttonPressed__)
		main_list.set_enable_search(False)

		column = gtk.TreeViewColumn("Press Here To Save")
		column.set_clickable(True)
		column.connect("clicked", self.__saveBookmarks__)

		hidden_column = gtk.TreeViewColumn("")
		main_list.append_column(column)
		main_list.append_column(hidden_column)

		text_renderer = gtk.CellRendererText()
		column.pack_start(text_renderer, False)
		column.add_attribute(text_renderer, "text", 0)

		text_renderer = gtk.CellRendererText()
		hidden_column.pack_start(text_renderer, False)
		hidden_column.add_attribute(text_renderer, "text", 1)

		main_list.enable_model_drag_dest([("text/plain", 0, 80)], gtk.gdk.ACTION_COPY)
		main_list.connect("drag-data-received", self.__addBookmark__)

		self.__readBookmarks__()
		if self. bookmarks != {}:
			for name in self.bookmarks:
				new_iter = main_list_store.insert_before(None, None)
				main_list_store.set_value(new_iter, 1, self.bookmarks[name])
				main_list_store.set_value(new_iter, 0, name)

		main_list_scroll.add_with_viewport(main_list)
				
		main_list.show()
		return main_list_scroll


	def __addBookmark__(self, main_list, drag_context, x, y, selection_data, info, timestamp):
		'''This is the callback when an icon is dropped in the bookmarks window'''

		main_list_store = main_list.get_model()
		print selection_data.data.split("\t")
		if selection_data.data != "" and selection_data.data.split("\t") != []:
			for location in  string.split(selection_data.data, "\t"):
				if os.path.exists(location) and os.path.isdir(location) and location not in self.bookmarks:
					main_list_store.insert_after(None ,[os.path.basename(location), location])
					self.bookmarks[os.path.basename(location)] = location


	def __readBookmarks__(self):
		'''This function reads the bookmarks folder and returns the bookmarks'''
			
		bookmarks_file =  os.environ['HOME'] + "/" + ".claw/bookmarks.xml"

		# If a bookmarks file exists ...
		if os.path.exists(bookmarks_file):

			# Create a DOM tree from the file
			dom_file = minidom.parse(bookmarks_file)

			# Get the bookmark tags
			listTag = dom_file.getElementsByTagName("list")
			bookmark_tags = listTag[0].getElementsByTagName("bookmark")

			# Get the name and locations from the tags
			for tag in bookmark_tags:	
				name = tag.attributes["name"]
				location = tag.childNodes[0].data
				self.bookmarks[name.value] = location

		else:
			# If there no file, return some empty values
			self.bookmarks =  {}


	def __saveBookmarks__(self, column_header_button):
		'''This saves the bookmarks to the bookmarks file'''

		bookmarks_file =  os.environ['HOME'] + "/" + ".claw/bookmarks.xml"

		# First we create a bookmarks file (and directory if one doesn't exist)
		if not os.path.exists(os.environ['HOME'] + "/" + ".claw"):
			os.mkdir(os.environ['HOME'] + "/" + ".claw")
		bookmarkFile = file(bookmarks_file, 'w')

		# Add a opening header
		bookmarkFile.write("<!-- This file contains the bookmarks for the CLAW file browser -->")
		bookmarkFile.write("\n<list>")

		# Get the information from the bookmarks list, then add it to the file
		for name in self.bookmarks:
			location = self.bookmarks[name]
			bookmarkFile.write("\n\t<bookmark name =\"" + name + "\">" + location +"</bookmark>")

		# Add a closing header
		bookmarkFile.write("\n</list>")
		bookmarkFile.close()

		# Let the user know what we are up to
		self.Notebook.info_bar.set_text("Bookmarks Saved")
		self.Notebook.info_bar.modify_fg(gtk.STATE_NORMAL, gtk.gdk.Color(0, 40000, 0, 0))

			
	def __bookmarkSelected__(self, main_list, event):
		'''This handles is the bookmarks bar callback for opening bookmarks'''

		main_selection =  main_list.get_selection()
		main_list_store, selected_iter = main_selection.get_selected()
		if main_selection == None:
			return 0
			
		# A right click opens
		if selected_iter and event.button == 3:
			location = os.path.realpath( main_list_store.get_value(selected_iter, 1))
			self.Notebook._populate_(location)
		elif event.button == 1 and event.get_state() == gtk.gdk.SHIFT_MASK:
			self.__deleteBookmark__(main_list_store, selected_iter)

	def __buttonPressed__(self, main_list, event):
		'''This function handles key presses for the bookmarks (This is a hack)'''
		# Check for Delete Key
		if event.keyval == 65535:
			self.__deleteBookmark__(main_list.get_model(), main_list.get_selection().get_selected()[1])
		# Check for Enter Key
		elif event.keyval == 65293:
			main_list_store, selected_iter = main_list.get_selection().get_selected()
			location = os.path.realpath(main_list_store.get_value(selected_iter, 1))
			self.Notebook._populate_(location)

	def __deleteBookmark__(self,  main_list_store, selected_iter):
		'''This function handles the callback for deleting a bookmark'''

		name = main_list_store.get_value(selected_iter, 0)
		del self.bookmarks[name]
		main_list_store.remove(selected_iter)


############################# PREFERENCE DIALOG ##################################

	def __preferenceDialog__(self):
		'''This function creates the preferences dialog to put in the notebook'''

		dialog_frame = gtk.Frame("Preferences")
		
		dialog_scroll = gtk.ScrolledWindow()
		dialog_scroll.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)		
		dialog_frame.add(dialog_scroll)

		dialog_box = gtk.VBox()
		dialog_scroll.add_with_viewport(dialog_box)

		preferences = xml_reader.read()

		terminal_frame = gtk.Frame()
		self.__makeFrameLabel__(terminal_frame, "Terminal Command")
		self.terminal_entry = gtk.Entry()
		self.terminal_entry.set_text(preferences[0])
		terminal_frame.add(self.terminal_entry)
		dialog_box.pack_start(terminal_frame, False, True)

		archive_frame = gtk.Frame()
		self.__makeFrameLabel__(archive_frame, "Archive Command")
		self.archive_combo = gtk.combo_box_new_text()
		for archiver in ["bzip2","gzip","zip"]:
			if commands.getstatusoutput("which " + archiver)[0] == 0:
				if preferences[8] == archiver:
					self.archive_combo.insert_text(0, archiver)
				else:
					self.archive_combo.append_text(archiver)

		self.archive_combo.set_active(0)
		archive_frame.add(self.archive_combo)
		dialog_box.pack_start(archive_frame, False, True, 5)	

		single_click_frame = gtk.Frame("Mouse")
		self.__makeFrameLabel__(single_click_frame, "Mouse")			
		dialog_box.pack_start(single_click_frame, False, True)

		self.single_click_button = gtk.CheckButton("Single Click Activation", use_underline=False)			
		self.single_click_button.get_child().modify_fg(gtk.STATE_ACTIVE, gtk.gdk.Color(0,40000,0,10))
		self.single_click_button.get_child().modify_fg(gtk.STATE_NORMAL, gtk.gdk.Color(40000,0,0,10))		
		single_click_frame.add(self.single_click_button)

		if preferences[10] == "True":
			self.single_click_button.set_active(True)
		else:
			self.single_click_button.set_active(False)

		image_preview_frame = gtk.Frame()
		self.__makeFrameLabel__(image_preview_frame, "Images")	
		dialog_box.pack_start(image_preview_frame, False, True)

		image_preview_box = gtk.VBox()
		image_preview_frame.add(image_preview_box)

		self.image_preview_button = gtk.CheckButton("Show Popups", use_underline=False)
		self.image_preview_button.get_child().modify_fg(gtk.STATE_ACTIVE, gtk.gdk.Color(0,40000,0,10))
		self.image_preview_button.get_child().modify_fg(gtk.STATE_NORMAL, gtk.gdk.Color(40000,0,0,10))
		if preferences[2] == "True":
			self.image_preview_button.set_active(True)
		else:
			self.image_preview_button.set_active(False)
		image_preview_box.pack_start(self.image_preview_button, False, True)

		self.inline_image_preview_button = gtk.CheckButton("Show Previews", use_underline=False)
		self.inline_image_preview_button.get_child().modify_fg(gtk.STATE_ACTIVE, gtk.gdk.Color(0,40000,0,10))
		self.inline_image_preview_button.get_child().modify_fg(gtk.STATE_NORMAL, gtk.gdk.Color(40000,0,0,10))

		if preferences[3] == "True":
			self.inline_image_preview_button.set_active(True)
		else:
			self.inline_image_preview_button.set_active(False)
		image_preview_box.pack_start(self.inline_image_preview_button, False, True)		
		
		image_preview_max_size_label = gtk.Label("Max Size (Megs)")
		image_preview_box.pack_start(image_preview_max_size_label, False, True)
		
		self.image_preview_max_size_adjustment = gtk.Adjustment(int(preferences[4]), lower=0, upper=100, step_incr=1, page_incr=10, page_size=1)
		image_preview_max_size_scale = gtk.HScale(self.image_preview_max_size_adjustment)
		image_preview_box.pack_start(image_preview_max_size_scale, False, True)		
		
		audio_preview_frame = gtk.Frame()
		self.__makeFrameLabel__(audio_preview_frame, "Audio")			
		dialog_box.pack_start(audio_preview_frame, False, True)
		
		audio_preview_box = gtk.VBox()
		audio_preview_frame.add(audio_preview_box)
		
		self.audio_preview_button = gtk.CheckButton("Hear Previews", use_underline=False)
		self.audio_preview_button.get_child().modify_fg(gtk.STATE_ACTIVE, gtk.gdk.Color(0,40000,0,10))
		self.audio_preview_button.get_child().modify_fg(gtk.STATE_NORMAL, gtk.gdk.Color(40000,0,0,10))

		if preferences[1] == "True":
			self.audio_preview_button.set_active(True)
		else:
			self.audio_preview_button.set_active(False)
		audio_preview_box.pack_start(self.audio_preview_button, False, True)

		functions_frame = gtk.Frame()
		self.__makeFrameLabel__(functions_frame, "Functions")			
		functions_box = gtk.VBox()
		functions_frame.add(functions_box)	
		dialog_box.pack_start(functions_frame, False, True)

		clear_database_button = gtk.Button("Empty Trash")
		clear_database_button.connect("clicked", self.__emptyTrash__)
		functions_box.pack_start(clear_database_button, False, True)

		empty_trash_button = gtk.Button("Clear File Cache")
		empty_trash_button.connect("clicked", self.__clearDatabase__)
		functions_box.pack_start(empty_trash_button, False, True)

		save_frame = gtk.Frame()
		set_button = gtk.Button("Save")
		set_button.connect("clicked", self.__setPreferences__)
		save_frame.add(set_button)
		dialog_box.pack_start(save_frame, False, True)

		return dialog_frame

	def __emptyTrash__(self, button):
		for path in os.listdir(os.environ['HOME']+"/..Wastebasket/"):
			if os.path.isdir(os.environ['HOME']+"/..Wastebasket/" + path):
				shutil.rmtree(os.environ['HOME']+"/..Wastebasket/" + path)
			else:
				os.remove(os.environ['HOME']+"/..Wastebasket/" + path)
		self.Notebook.info_bar.modify_fg(gtk.STATE_NORMAL, gtk.gdk.Color(0, 40000, 0, 0))
		self.Notebook.info_bar.set_text( "Trashcan Empty")	


############################# LOOK DIALOG ##################################

	def __lookDialog__(self):
		'''This function creates the look dialog to put in the notebook'''

		dialog_frame = gtk.Frame("Look")

		dialog_scroll = gtk.ScrolledWindow()
		dialog_scroll.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)		
		dialog_frame.add(dialog_scroll)

		dialog_box = gtk.VBox()
		dialog_scroll.add_with_viewport(dialog_box)

		preferences = xml_reader.read()

		font_frame = gtk.Frame()
		self.__makeFrameLabel__(font_frame, "Font")			
		dialog_box.pack_start(font_frame, False, True)

		font_event_box = gtk.EventBox()
		font_box = gtk.VBox()
		font_event_box.add(font_box)
		font_frame.add(font_event_box)
		#font_event_box.modify_bg(gtk.STATE_NORMAL, gtk.gdk.Color(40000,50000,65000,0))
		
		font_example_label = gtk.Label("Claw 1234")
		font_example_label.modify_font(pango.FontDescription(preferences[5]))
		font_box.pack_start(font_example_label, False, True)

		font_name = preferences[5]
		self.font_selection_button =  gtk.FontButton(font_name)
		self.font_selection_button.connect("font-set", self.__setFont__, font_example_label)
		font_box.pack_start(self.font_selection_button, False, True)
		
		colors = string.split(preferences[6], ",")
		self.font_color_button = gtk.ColorButton(gtk.gdk.Color(int(colors[0]), int(colors[1]), int(colors[2]), int(colors[3])))
		font_box.pack_start(self.font_color_button, False, True)

		bg_color_frame = gtk.Frame()
		self.__makeFrameLabel__(bg_color_frame, "Background Color")			
		dialog_box.pack_start(bg_color_frame, False, True)

		bg_color_box = gtk.VBox()
		bg_color_frame.add(bg_color_box)

		colors = string.split(preferences[7], ",")
		self.bg_color_button = gtk.ColorButton(gtk.gdk.Color(int(colors[0]), int(colors[1]), int(colors[2]), int(colors[3])))
		bg_color_box.pack_start(self.bg_color_button, False, True)

		theme_frame = gtk.Frame()
		self.__makeFrameLabel__(theme_frame, "Icon Theme")			
		self.theme_combo = gtk.combo_box_new_text()
		for a_file in os.listdir(IMAGE_DIRECTORY + "/Icons"):
			if preferences[9] == a_file:
				self.theme_combo.insert_text(0, a_file)
			else:
				self.theme_combo.append_text(a_file)

		self.theme_combo.set_active(0)
		theme_frame.add(self.theme_combo)
		dialog_box.pack_start(theme_frame, False, True)

		save_frame = gtk.Frame()
		set_button = gtk.Button("Save")
		set_button.connect("clicked", self.__setPreferences__)
		save_frame.add(set_button)
		dialog_box.pack_start(save_frame, False, True)

		return dialog_frame

	def __setFont__(self, font_button, font_example_label):
		'''This function is the callback for the font selection button'''

		selected_font =  font_button.get_font_name()
		font_example_label.modify_font(pango.FontDescription(selected_font))		


############################# SET PREFERENCES ##################################

	def __setPreferences__(self, button):
		'''This function saves the preferences to file and sets the preferences for all the tabs'''

		# Get the information from the widget
		terminal = self.terminal_entry.get_text()

		audio = self.audio_preview_button.get_property("active")
		if audio == True:
			audio = "True"
		else:
			audio = "False"

		image = self.image_preview_button.get_property("active")
		if image == True:
			image = "True"
		else:
			image = "False"	

		inline_image = self.inline_image_preview_button.get_property("active")
		if inline_image == True:
			inline_image = "True"
		else:
			inline_image = "False"				

		single_click = self.single_click_button.get_property("active")
		if single_click == True:
			single_click = "True"
		else:
			single_click = "False"	
		
		archive_iter = self.archive_combo.get_active_iter()
		archive_model = self.archive_combo.get_model()
		archive_command = archive_model.get_value(archive_iter, 0)

		theme_iter = self.theme_combo.get_active_iter()
		theme_model = self.theme_combo.get_model()
		theme = theme_model.get_value(theme_iter, 0)

						
		# Open the preferences file and read it
		preferences_file = file(os.environ['HOME'] + "/.claw/preferences.xml", 'r')
		lines = preferences_file.readlines()
		preferences_file.close()

		# Open the file for writing
		preferences_file = file(os.environ['HOME'] + "/.claw/preferences.xml", 'w')

		# Replace the relevant lines
		for line in lines:
			if re.findall("termcmd", line):
				line = "\t<termcmd>"+terminal+"</termcmd>\n"
			elif re.findall("<image_previews>", line):
				line = "\t<image_previews>"+image+"</image_previews>\n"
			elif re.findall("inline_image_previews", line):
				line = "\t<inline_image_previews>"+inline_image+"</inline_image_previews>\n"
			elif re.findall("image_preview_size", line):
				line = "\t<image_preview_size>" +str(int(self.image_preview_max_size_adjustment.get_value()))  + "</image_preview_size>\n"
			elif re.findall("audio_previews", line):
				line = "\t<audio_previews>"+audio+"</audio_previews>\n"
			elif re.findall("<font>", line):
				line = "\t<font>" + self.font_selection_button.get_font_name() + "</font>\n"
			elif re.findall("<font_color>", line):
				color = self.font_color_button.get_color()
				line = "\t<font_color>"+str(color.red) + "," + str(color.green) + "," + str(color.blue) + ",0</font_color>\n"
			elif re.findall("bg_color", line):
				color = self.bg_color_button.get_color()
				line = "\t<bg_color>"+str(color.red) + "," + str(color.green) + "," + str(color.blue) + ",0</bg_color>\n"
			elif re.findall("archive", line) and archive_command != None:
				line = "\t<archive>" + archive_command + "</archive>\n"
			elif re.findall("theme", line) and archive_command != None:
				line = "\t<theme>" + theme + "</theme>\n"
			elif re.findall("single_click", line):
				line = "\t<single_click>" + single_click+ "</single_click>\n"							
			# Write the line		
			preferences_file.write(line)

		# Close the File
		preferences_file.close()

		self.Notebook.setPreferences()

	def __clearDatabase__(self, button):
		'''This function clears the ~/.claw/gadfly/  database to get a fresh start'''

		# Erase all the files
		for a_file in os.listdir(os.environ['HOME'] + "/.claw/gadfly/"):
			os.remove(os.environ['HOME'] + "/.claw/gadfly/" + a_file)

		# Reinitilize the database
		db.initialize("Claw4")

	def __getMousePosition__(self):
		'''This functions returns the current mouse location
		relative to the upper left corner of the main window.'''
		
		main_window = self.main_notebook.get_toplevel()
		x, y = main_window.get_position()
		offset_x ,offset_y = main_window.get_pointer()
		return x + offset_x, y+offset_y

	def __makeFrameLabel__(self, frame, text):
		'''This functions creates a label, adds markup to the label text and adds the 
		label to the frame'''
		
		label = gtk.Label()
		label.set_use_markup(True)
		markup = '<span size="10000"><b>' + text + '</b></span>' 
		label.set_markup(markup)
		frame.set_label_widget(label)
		
		frame.set_border_width(5)
		frame.modify_bg(gtk.STATE_NORMAL, gtk.gdk.Color(40000,50000,65000,0))
