# Copyright (C) 2005-2009 Jelmer Vernooij <jelmer@samba.org>
 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Post-commit hook to submit the commit to CIA (http://cia.navi.cx/)

Requires bzr 0.15 or higher.

Copyright (C) 2005-2008 Jelmer Vernooij <jelmer@samba.org>
Published under the GNU GPLv2 or later

Installation:
Copy this directory to ~/.bazaar/plugins/cia/*

Configuration:
In ~/.bazaar/bazaar.conf, set 'cia_user' to your username on CIA.

Other options that can be set are 'cia_server' and 'cia_quiet', and
'cia_send_revno'

To enable CIA reporting for a particular branch, run:
$ bzr cia-project <name> 
inside that branch
"""

version_info = (1, 0, 0, 'dev', 0)


import bzrlib
from bzrlib.branch import Branch
from bzrlib.commands import Command, register_command
from bzrlib.option import Option
from bzrlib.trace import info, warning


def branch_commit_hook(local, master, old_revno, old_revid, 
                                      new_revno, new_revid):
    if local is None:
        return cia_submit(master, new_revid, new_revno)
    else:
        return cia_submit(local, new_revid, new_revno)


class BacklogList:
    def __init__(self, filename):
        self.filename = filename

    def add_revision(self, revid):
        pass

    def read_revisions(self):
        pass


def generate_cia_xml(branch, revid, project, revname=None, author=None):
    from xml.sax import saxutils
    revision = branch.repository.get_revision(revid)
    delta = branch.repository.get_revision_delta(revid)

    if revname is None:
        revname = revid

    if author is None:
        authors = revision.get_apparent_authors()
    else:
        authors = [author]

    files = []
    [files.append(f) for (f,_,_) in delta.added]
    [files.append(f) for (f,_,_) in delta.removed]
    [files.append(f) for (_,f,_,_,_,_) in delta.renamed]
    [files.append(f) for (f,_,_,_,_) in delta.modified]

    authors_xml = "".join(["      <author>%s</author>\n" % saxutils.escape(author) for author in authors])

    return """
<message>
  <generator> 
    <name>bzr</name> 
    <version>%s</version> 
    <url>http://samba.org/~jelmer/bzr/cia_bzr.py</url>
  </generator>
  <source>
    <project>%s</project>
    <module>%s</module>
  </source>
  <timestamp>%d</timestamp>
  <body>
    <commit>
      <revision>%s</revision>
      <files>%s</files>
      %s
      <log>%s</log>
    </commit>
  </body>
</message>
""" % (bzrlib.version_string, project, branch.nick, revision.timestamp, revname,
        "\n".join(["<file>%s</file>" % saxutils.escape(f) for f in files]), 
        authors_xml, 
        saxutils.escape(revision.message))


def cia_cache_dir():
    from bzrlib.config import config_dir
    import os
    return os.path.join(config_dir(), "cia")


def store_failed_message(revid, message):
    import os
    cache_dir = cia_cache_dir()
    if not os.path.isdir(cache_dir):
        os.mkdir(cache_dir)
    cache_file = os.path.join(cache_dir, revid)
    f = open(cache_file, 'w')
    f.write(message)
    f.close()
    return cache_file


def cia_connect(config):
    import xmlrpclib
    server = config.get_user_option('cia_server')
    if server is None:
        server = "http://cia.navi.cx"

    return xmlrpclib.ServerProxy(server)


class CIADeliverError(Exception):

    def __init__(self, message):
        self.message = message


def cia_deliver(server, msg):
    import socket, xmlrpclib
    try:
        server.hub.deliver(msg)
    except xmlrpclib.ProtocolError, e:
        raise CIADeliverError(e.errmsg)
    except socket.gaierror, (_, errmsg):
        raise CIADeliverError(errmsg)
    except socket.error, (_, errmsg):
        raise CIADeliverError(errmsg)


def cia_submit(branch, revid, revno, dry_run=False):
    config = branch.get_config()

    project = config.get_user_option('cia_project')
    if project is None:
        return

    author = config.get_user_option('cia_user')
    quiet = config.get_user_option('cia_quiet')

    use_revno = config.get_user_option('cia_send_revno')
    if use_revno is not None and use_revno.lower()[:1] in ('t', '1'):
        revspec = revno
    else:
        revspec = revid

    msg = generate_cia_xml(branch, revid, project, revspec, author)

    if not quiet:
        info("Submitting revision to CIA.")
    if not dry_run:
        try:
            cia_deliver(cia_connect(config), msg)
        except CIADeliverError, (error, ):
            warning("Unable to submit revision to CIA: %s" % error)
            store_failed_message(revid, msg)
            info("To retry submit later, run 'bzr cia-submit --pending'.")
    else:
        print msg


class cmd_cia_project(Command):
    """Print or set project to submit changes to on CIA.

    """
    takes_args = ['project?']
    takes_options = []

    def run(self, project=None):
        br = Branch.open('.')
        
        config = br.get_config()
        if project:
            config.set_user_option('cia_project', project)
        else:
            print config.get_user_option('cia_project')


def submit_pending(config):
    import os
    server = cia_connect(config)
    cache_dir = cia_cache_dir()
    for revid in os.listdir(cache_dir):
        path = os.path.join(cache_dir, revid)
        msg = open(path, 'r').read()
        try:
            cia_deliver(server, msg)
        except CIADeliverError, (error, ):
            warning("Submitting %s failed: %s" % (revid, error))
        else:
            os.remove(path)


class cmd_cia_submit(Command):
    """Submit a revision to CIA

    """
    takes_args = ['branch?']
    takes_options = ['revision', 
        Option('pending', 
            help="Submit all pending CIA submissions."),
        Option('dry-run', 
            help="Show what would be done, but don't actually do anything.")]

    def run(self, branch=None, revision=None, dry_run=False, pending=False):
        if pending and (branch is not None or revision is not None):
            raise BzrCommandError("--pending and specifying a branch are mutually exclusive")
        if pending:
            from bzrlib.config import GlobalConfig
            submit_pending(GlobalConfig())
        else:
            if branch is not None:
                branch = "."
            br = Branch.open(branch)

            if revision is None:
                revs = [br.last_revision()]
            elif len(revision) == 1:
                revs = [revision[0].in_history(br).rev_id]
            elif len(revision) == 2:
                if revision[0].spec is None:
                    from_revno = 1
                else:
                    from_revno = revision[0].in_history(br).revno

                if revision[1].spec is None:
                    to_revno = br.revno()
                else:
                    to_revno = revision[1].in_history(br).revno
                revs = br.revision_history()[from_revno:to_revno]
            else:
                raise BzrCommandError('bzr submit-cia --revision takes one or two arguments')

            for rev_id in revs:
                cia_submit(br, rev_id, br.revision_id_to_revno(rev_id), dry_run)


register_command(cmd_cia_project)
register_command(cmd_cia_submit)

def install_hooks():
    """Install CommitSender to send after commits with bzr >= 0.15 """
    if getattr(Branch.hooks, 'install_named_hook', None):
        Branch.hooks.install_named_hook('post_commit', branch_commit_hook, 
                                        'CIA')
    else:
        Branch.hooks.install_hook('post_commit', branch_commit_hook)
        if getattr(Branch.hooks, 'name_hook', None):
            Branch.hooks.name_hook(branch_commit_hook, "cia-bzr")


install_hooks()

def test_suite():
    from unittest import TestSuite, TestLoader
    from bzrlib.plugins.cia import tests
    suite = TestSuite()
    suite.addTest(tests.test_suite())
    return suite
