#ifndef H_CDW_CONFIG
#define H_CDW_CONFIG

#include <stdbool.h>

#include "main.h"
#include "cdw_iso9660.h"

/** \brief Maximal length of some char tables storing long option values */
#define OPTION_FIELD_LEN_MAX 950


int cdw_config_module_init(void);
void cdw_config_module_clean(void);
cdw_rv_t cdw_config_write_to_file(void);
const char *cdw_config_get_config_dir(void);


/* volume size - maximum size of selected files, used
   in "selected files info" view in main ui window; it is tempting
   to use "disc type" from cdw_disc.h, but it's not that simple to
   unify the two sets */
enum {
	CDW_CONFIG_VOLUME_SIZE_CD74 = 0,
	CDW_CONFIG_VOLUME_SIZE_CD80,
	CDW_CONFIG_VOLUME_SIZE_DVD_GENERIC, /* smallest common denominator for DVDs */
	CDW_CONFIG_VOLUME_SIZE_DVD_R,
	CDW_CONFIG_VOLUME_SIZE_DVD_RP,
	CDW_CONFIG_VOLUME_SIZE_DVD_RW,
	CDW_CONFIG_VOLUME_SIZE_DVD_RWP,
	CDW_CONFIG_VOLUME_SIZE_DVD_R_DL,
	CDW_CONFIG_VOLUME_SIZE_DVD_RP_DL,
	CDW_CONFIG_VOLUME_SIZE_CUSTOM,
	CDW_CONFIG_VOLUME_SIZE_AUTO };



/** \brief Pair of name-value strings that make an option */
typedef struct {
	char *name; /**< \brief Name of option, string that stands on left-hand side of '=' char in configuration file*/
	char *value; /**< \brief Value of option, string that stands on right-hand side of '=' char in configuration file */
} cdw_option_t;

/* this is in header file only because unit testing code requires prototype */
cdw_option_t cdw_config_split_options_line(char *line);



/** \brief Main cdw configuration variable, storing all major options used by cdw */
typedef struct {

	/* Options related to writing or erasing */
	bool write_pad;            /**< \brief Pad written track with blanks (cdrecord) */
	int write_pad_size;
	cdw_id_t erase_mode;     /**< \brief Select mode of blanking a disc: fast / all */
	bool eject;          /**< \brief Should cdw eject drive tray after finished writing or blanking? */
	bool burnproof;      /**< \brief Should a tool use technique that helps avoiding buffer underrun? */
	char *other_cdrecord_options;     /**< \brief Place for other cdrecord options, specified manually */
	char *other_growisofs_options; /**< \brief Place for other growisofs options, specified manually */
	char *other_xorriso_burn_options;


	/* Hardware configuration */

	/* device name in form of /dev/xxx - it is used as default device
	   name in cdrecord, but SCSI descriptor can be also used if needed;
	   custom_device is also used in other tools code - these tools use
	   only this entry to access reading/writing device (I don't mention
	   here cddb nor legacy cdda2wav code, because it is currently
	   unmaintained) */
	char custom_drive[OPTION_FIELD_LEN_MAX + 1]; /**< \brief Path to CD/DVD reader/burner */
	/* state of "drive" dropdown in configuration window */
	char selected_drive[OPTION_FIELD_LEN_MAX + 1];
	/* SCSI device (reader/writer) descriptor in form of X:Y:Z;
	   it should be used only in cdrecord code, and user should set this
	   string only in cases when cdrecord can't work with /dev/xxx device */
	char scsi[OPTION_FIELD_LEN_MAX + 1];        /**< \brief Variable specifying scsi device in form preferred by cdrecord: bus:target:lun */


	/* Audio configuration */
	char *audiodir;


	/* ISO9660 filesystem options **/
	char volume_id[CDW_ISO9660_VOLI_LEN + 1];   /**< \brief Label of disc visible in file managers */
	char volume_set_id[CDW_ISO9660_VOLS_LEN + 1];
	char preparer[CDW_ISO9660_PREP_LEN + 1];
	char publisher[CDW_ISO9660_PUBL_LEN + 1];
	char system_id[CDW_ISO9660_SYSI_LEN + 1];
	char copyright[CDW_ISO9660_COPY_LEN + 1];
	char abstract[CDW_ISO9660_ABST_LEN + 1];
	cdw_id_t iso_level;          /**< \brief Level of conformance with ISO9660; valid values are 1 to 4 */
	bool joliet;            /**< \brief Turn on creting Joliet filenames (-J, mkisofs option) */
	cdw_id_t rock_ridge;         /* 0 = none (no option), 1 = useful only (-r), 2 = full (-R) */
	bool joliet_long;       /**< \brief "Allow  Joliet  filenames  to  be  up to 103 Unicode characters, instead of 64." */
	bool follow_symlinks;   /**< \brief Follow symbolic links when generating the filesystem */

	char *mkisofs_root_dir;
	char *iso_image_full_path;     /**< \brief Current full path to ISO9660 image file */
	char *boot_disc_options;       /**< \brief Options for creating a bootable disc */
	char *other_mkisofs_options;   /**< \brief Place for other mkisofs options, specified manually */
	char *other_xorriso_iso_options;   /**< \brief Place for other xorriso options, specified manually */


	/* Configuration of external tools */
	/* external tools have their own configuration variable, and there
	   are no settings (yet) that should be stored in config file */


	/* Log file settings, other setting  */
	char *log_full_path; /**< \brief Full path to cdw log file */
	bool showlog;        /**< \brief Show cdw log after finishing operation */

        cdw_id_t volume_size_id;             /**< \brief ID of preferred standard size of target media/iso file */
	long int volume_size_custom_value;   /**< \brief Custom size of target media/iso file, MB */

	/* Options not displayed in configuration window and not saved to config file */
	bool support_dvd_rp_dl;
	bool show_dvd_rw_support_warning;

	struct {
		bool display_hidden_files;
	} fs;

} cdw_config_t;



cdw_rv_t cdw_config_var_copy(cdw_config_t *dest, cdw_config_t *src);
cdw_rv_t cdw_config_var_validate(cdw_config_t *config, cdw_id_t *page, int *field);


cdw_rv_t cdw_config_var_init_fields(cdw_config_t *config);
cdw_rv_t cdw_config_var_free_fields(cdw_config_t *config);

bool cdw_config_has_scsi_device(void);
const char *cdw_config_get_custom_drive(void);
const char *cdw_config_get_scsi_drive(void);

void cdw_config_debug_print_config(cdw_config_t *config);

long int cdw_config_get_volume_size_mb_by_id(cdw_id_t id);
long int cdw_config_get_current_volume_size_value(void);
cdw_id_t cdw_config_volume_id_by_label(const char *label);

bool cdw_config_support_dvd_rp_dl(void);
bool cdw_config_follow_symlinks(void);



/* this is mostly for cdw_config_window.h, but also for
   cdw_config.c/cdw_config_var_validate() */
#define CONFIG_PAGE_ID_AUDIO   0
#define CONFIG_PAGE_ID_HW      1
#define CONFIG_PAGE_ID_TOOLS   2
#define CONFIG_PAGE_ID_OTHER   3


/* unit tests */

void cdw_config_run_tests(void);

#endif /* H_CDW_CONFIG */
