/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2010 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#define _BSD_SOURCE /* strdup() */
#define _GNU_SOURCE /* asprintf() */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdbool.h>
#include <ctype.h>
#include <math.h> /* floor(), log10() */

#include <ncursesw/ncurses.h>
#include <ncursesw/form.h>
#include <ncursesw/panel.h>

#include "cdw_config_ui.h"
#include "cdw_config_ui_internals.h"
#include "gettext.h"
#include "cdw_string.h"
#include "cdw_main_window.h"
#include "cdw_widgets.h"
#include "cdw_ncurses.h"
#include "cdw_debug.h"
#include "cdw_ext_tools.h"
#include "cdw_file_picker.h"
#include "cdw_form.h"
#include "cdw_erase_disc.h"
#include "cdw_drive.h"

/**
   I formalized concept of configuration window pages. There are currently
   six pages, grouping related options:
   \li options related to writing and blanking: page A, titled "Writing"
   \li options related to hardware: page B, titled "Hardware"
   \li options related to audio CDs: page C, titled "Audio"
   \li options related to ISO filesystem: page D, titled "ISO filesystem"
   \li options related to external tools: page E, titled "Tools"
   \li options related to logging facility: page F, titled "Log and other"

   I'm referring to these pages by letter, not by ordering number, because
   these pages can be put in conf window in any order. Position of pages
   can be modified by editing "#define PAGE_X_INDEX" lines in "cdw_config_ui.h"
   file. Code in this file was modified so it doesn't depend on order of
   pages. This was achieved by using page indexes: PAGE_X_INDEX.

   This design should make adding new pages easier, since there is less
   probability that there are some hidden interactions or relations
   between pages.

   Another factor that should simplify adding new page is page_u_driver()
   function which handles user's key events in pages. If new page has only
   simple input fields and checkboxes, the function can be used (I believe)
   without any customization. However if there is some dropdown in the page,
   there will be some more hacking required.

   One note on naming convention: "pi" stands for "page index",
   "fi" is "field index".
*/


//#define CDW_ENABLE_AUTO_VOLUME /* controls content of dropdown in "log and other" page */


#ifdef CDW_ENABLE_AUTO_VOLUME
	#define CDW_CONFIG_VOLUME_ITEMS_MAX 6 /* 74min CD, 80min CD, DVD, DVD+R DL, custom, auto */
#else
	#define CDW_CONFIG_VOLUME_ITEMS_MAX 5 /* 74min CD, 80min CD, DVD, DVD+R DL, custom */
#endif

cdw_id_label_t cdw_config_volume_size_items[CDW_CONFIG_VOLUME_ITEMS_MAX] = {
	/* 2TRANS: this is dropdown item label: 650MB CD */
	{ CDW_CONFIG_VOLUME_SIZE_CD74,               gettext_noop("74 min CD (650 MB)") },
	/* 2TRANS: this is dropdown item label: 700MB CD */
	{ CDW_CONFIG_VOLUME_SIZE_CD80,               gettext_noop("80 min CD (700 MB)") },
	/* 2TRANS: this is dropdown item label: DVD */
	{ CDW_CONFIG_VOLUME_SIZE_DVD_GENERIC,        gettext_noop("Generic DVD (4.7 GB)") },
	/* 2TRANS: this is dropdown item label: DVD+R DL */
	{ CDW_CONFIG_VOLUME_SIZE_DVD_RP_DL,          gettext_noop("DVD+R DL (8.5 GB)") },
	/* 2TRANS: this is dropdown item label: custom value of
	   ISO volume; user can enter arbitrary natural number */
	{ CDW_CONFIG_VOLUME_SIZE_CUSTOM,             gettext_noop("Custom value") }
#ifdef CDW_ENABLE_AUTO_VOLUME
	,
	/* 2TRANS: this is dropdown item label:
	   "automatic" = automatic detection/resolution of size of ISO volume */
	{ CDW_CONFIG_VOLUME_SIZE_AUTO,               gettext_noop("Automatic") }
#endif
};


static cdw_rv_t cdw_config_ui_draw_additional_widgets(cdw_config_t *config);
static cdw_rv_t cdw_config_ui_window_populate_pages(cdw_config_t *config);
static cdw_rv_t cdw_config_ui_window_prebuild_pages(bool tools_page_is_visible);
static cdw_rv_t cdw_config_ui_window_build_page(int pi, const char *window_title, const char *window_tip);

static cdw_rv_t cdw_config_ui_tab_labels_create(void);

static void cdw_config_ui_draw_tabs(int pi);

static cdw_rv_t page_a_build_fields(cdw_form_t *cdw_form, cdw_config_t *config);
static cdw_rv_t page_b_build_fields(cdw_form_t *cdw_form, cdw_config_t *config);
static cdw_rv_t page_c_build_fields(cdw_form_t *cdw_form, cdw_config_t *config);
static cdw_rv_t page_d_build_fields(cdw_form_t *cdw_form, cdw_config_t *config);
static cdw_rv_t page_e_build_fields(cdw_form_t *cdw_form, cdw_config_t *config);
static cdw_rv_t page_f_build_fields(cdw_form_t *cdw_form, cdw_config_t *config);

static int cdw_config_ui_window_pages_driver_sub(cdw_config_t *config, int key, int *current_page, int fi);
static void cdw_config_ui_page_u_driver_handle_enter(cdw_form_t *cdw_form, int fi, void *data);
static void cdw_config_ui_page_e_driver_toggle_checkbox(cdw_form_t *cdw_form, int fi);
static void cdw_config_ui_page_e_driver_toggle_manual_selection(void);

static void page_c_handle_enter_on_audiodir_button(cdw_config_t *config);
static void page_d_handle_enter_on_iso_path_button(cdw_config_t *config);
//static void page_d_handle_enter_on_boot_path_button(cdw_config_t *config);
static void page_f_handle_enter_on_log_path_button(cdw_config_t *config);

static void option_fields_save_changes(cdw_config_t *config);
static void cdw_config_ui_refresh_page_windows(int pi);

static CDW_DROPDOWN *cdw_config_ui_volume_size_dropdown(WINDOW *window, int begin_y, int begin_x, int width, cdw_config_t *config);
static CDW_DROPDOWN *cdw_config_ui_iso_level_dropdown(WINDOW *window, int begin_y, int begin_x, int width, cdw_config_t *config);
static CDW_DROPDOWN *cdw_config_ui_speed_range_dropdown(WINDOW *window, int begin_y, int begin_x, int width, cdw_config_t *config);

/* values of some options can be selected only from dropdown */
static CDW_DROPDOWN *speed_range_dropdown;
static CDW_DROPDOWN *iso_level_dropdown;
static CDW_DROPDOWN *volume_size_dropdown;

struct cdw_config_ext_tools_dropdowns_t ext_tools_dropdowns;

static CDW_BUTTON *iso_image_path_button;
static CDW_BUTTON *audio_dir_button;
static CDW_BUTTON *log_path_button;



/* ****************************** */
/* some size and layout constants */
/* ****************************** */

/* width and height of main options window can't be smaller
   than size of minimal supported terminal */
/** \brief Width of configuration window */
#define CONF_WIDTH 80
/** \brief Height of configuration window */
#define CONF_HEIGHT 24
/** \brief Width of right-hand area with tabs */
#define TABS_WIDTH 23


/** \brief Width of (most) labels in configuration window */
#define LABEL_WIDTH 30
/** \brief Width of some fields that should be as wide as possible: input
    fields storing file path, some dropdowns, "other options" fields,
    txt subwins, long labels */
#define WIDE_FIELD_WIDTH (CONF_WIDTH - TABS_WIDTH - 4)
/** \brief Window column of first column of items in a configuration page  */
#define FIRST_COL 1
/** \brief Window column of second column of items in a configuration page  */
#define SECOND_COL (FIRST_COL + LABEL_WIDTH + 2)

/* Number of form fields on page {A,B,C,D,E,F} of options; ending field == NULL not included */
#define N_FIELDS_A 21
#define N_FIELDS_B 6  /* hardware */
#define N_FIELDS_C 3
#define N_FIELDS_D 23 /* ISO 9660 */
#define N_FIELDS_E 14
#define N_FIELDS_F 9

static WINDOW *page_B_txt_subwin = (WINDOW *) NULL;
static WINDOW *page_E_txt_subwin = (WINDOW *) NULL;

/* number of pages existing in configuration window */
#define N_CONFIG_PAGES 6

static FIELD *page_a_fields[N_FIELDS_A + 1]; /* +1 for last field == NULL */
static FIELD *page_b_fields[N_FIELDS_B + 1]; /* +1 for last field == NULL */
static FIELD *page_c_fields[N_FIELDS_C + 1]; /* +1 for last field == NULL */
static FIELD *page_d_fields[N_FIELDS_D + 1]; /* +1 for last field == NULL */
static FIELD *page_e_fields[N_FIELDS_E + 1]; /* +1 for last field == NULL */
static FIELD *page_f_fields[N_FIELDS_F + 1]; /* +1 for last field == NULL */


/* currently only few fields are used, but most probably I will
   use all of the fields in future */
typedef struct {
	cdw_form_t *cdw_form;
	bool page_visible;
	PANEL *panel;
	/* name of tab that group options - without shortcut key in <> */
	char *tab_label;
	size_t tab_label_len;

	cdw_rv_t (*fields_builder_function)(cdw_form_t *cdw_form, cdw_config_t *config);
} cdw_config_page_t;

static cdw_config_page_t c_pages[N_CONFIG_PAGES];


/* robwoj44 has informed me that in some setups F1 is caught by desktop
   environment or terminal emulator, so hotkeys for configuration window
   panels can't start with F1. */
#define CDW_CONFIG_FX_START_KEY 2
/* "Safe and Close Key"; it can't be F12 because it is caught by Puppy
   menu in Puppy Linux (information from robwoj44) */
#define CDW_CONFIG_SnC_KEY 10


/**
   \brief Initialize windows and subwindows of cdw configuration window

   The function initializes main N windows and subwindows (where N is
   number of pages in configuration window). It does not initialize
   any text subwindows in any of pages

   \return CDW_GEN_ERROR if some resource was not initialized properly
   \return CDW_OK on success
*/
cdw_rv_t cdw_config_ui_window_create(cdw_config_t *config, bool tools_page_is_visible)
{
	cdw_rv_t crv = cdw_config_ui_window_prebuild_pages(tools_page_is_visible);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to prebuild pages\n");
		return CDW_GEN_ERROR;
	}

	/* 2TRANS: this is title of cdw configuration main window */
	const char *window_title = _("Configuration");
	char *window_tip = (char *) NULL;
	/* 2TRANS: this is message at the bottom of configuration window;
	   cancel means: quit without saving; '%d' is an integer used to
	   create label of function key, e.g. "F10" */
	int rv = asprintf(&window_tip, _("Press F%d key to save changes or ESC to cancel"), CDW_CONFIG_SnC_KEY);
	if (window_tip == (char *) NULL || rv == -1) {
		cdw_vdm ("ERROR: failed to create window tip\n");
		return CDW_GEN_ERROR;
	}

	bool success = true;
	for (int i = 0; i < N_CONFIG_PAGES; i++) {
		crv = cdw_config_ui_window_build_page(i, window_title, window_tip);
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to build page #%d\n", i);
			success = false;
			break;
		}
	}

	free(window_tip);
	window_tip = (char *) NULL;

	if (success) {
		crv = cdw_config_ui_window_populate_pages(config);
		if (crv == CDW_OK) {
			update_panels();
			return CDW_OK;
		} else {
			cdw_vdm ("ERROR: failed to populate pages of config window\n");
		}
	}
	cdw_config_ui_window_destroy();
	return CDW_GEN_ERROR;
}





cdw_rv_t cdw_config_ui_window_prebuild_pages(bool tools_page_is_visible)
{
	c_pages[PAGE_A_INDEX].cdw_form = cdw_form_new(N_FIELDS_A);
	c_pages[PAGE_B_INDEX].cdw_form = cdw_form_new(N_FIELDS_B);
	c_pages[PAGE_C_INDEX].cdw_form = cdw_form_new(N_FIELDS_C);
	c_pages[PAGE_D_INDEX].cdw_form = cdw_form_new(N_FIELDS_D);
	c_pages[PAGE_E_INDEX].cdw_form = cdw_form_new(N_FIELDS_E);
	c_pages[PAGE_F_INDEX].cdw_form = cdw_form_new(N_FIELDS_F);

	for (int i = 0; i < N_CONFIG_PAGES; i++) {
		if (c_pages[i].cdw_form == (cdw_form_t *) NULL) {
			cdw_vdm ("ERROR: cdw form #%d is NULL\n", i);
			for (int j = 0; j < N_CONFIG_PAGES; j++) {
				if (c_pages[j].cdw_form != (cdw_form_t *) NULL) {
					cdw_form_delete(&(c_pages[j].cdw_form));
				}
			}
			cdw_vdm ("ERROR: failed to build cdw forms\n");
			return CDW_GEN_ERROR;
		}
	}

	for (int i = 0; i < N_CONFIG_PAGES; i++) {
		cdw_form_add_return_char(c_pages[i].cdw_form, KEY_F(PAGE_A_INDEX + CDW_CONFIG_FX_START_KEY));
		cdw_form_add_return_char(c_pages[i].cdw_form, KEY_F(PAGE_B_INDEX + CDW_CONFIG_FX_START_KEY));
		cdw_form_add_return_char(c_pages[i].cdw_form, KEY_F(PAGE_C_INDEX + CDW_CONFIG_FX_START_KEY));
		cdw_form_add_return_char(c_pages[i].cdw_form, KEY_F(PAGE_D_INDEX + CDW_CONFIG_FX_START_KEY));
		cdw_form_add_return_char(c_pages[i].cdw_form, KEY_F(PAGE_E_INDEX + CDW_CONFIG_FX_START_KEY));
		cdw_form_add_return_char(c_pages[i].cdw_form, KEY_F(PAGE_F_INDEX + CDW_CONFIG_FX_START_KEY));
		cdw_form_add_return_char(c_pages[i].cdw_form, KEY_F(CDW_CONFIG_SnC_KEY));

		c_pages[i].cdw_form->handle_enter = cdw_config_ui_page_u_driver_handle_enter;
		c_pages[i].page_visible = true;
		c_pages[i].cdw_form->form_id = i;
	}

	/* for this one page replace generic "toggle checkbox" function
	   (provided by cdw form) with custom "toggle checkbox" function */
	c_pages[PAGE_E_INDEX].cdw_form->toggle_checkbox = cdw_config_ui_page_e_driver_toggle_checkbox;
	c_pages[PAGE_E_INDEX].page_visible = tools_page_is_visible;

	c_pages[PAGE_A_INDEX].cdw_form->fields = page_a_fields;
	c_pages[PAGE_B_INDEX].cdw_form->fields = page_b_fields;
	c_pages[PAGE_C_INDEX].cdw_form->fields = page_c_fields;
	c_pages[PAGE_D_INDEX].cdw_form->fields = page_d_fields;
	c_pages[PAGE_E_INDEX].cdw_form->fields = page_e_fields;
	c_pages[PAGE_F_INDEX].cdw_form->fields = page_f_fields;

	c_pages[PAGE_A_INDEX].fields_builder_function = page_a_build_fields;
	c_pages[PAGE_B_INDEX].fields_builder_function = page_b_build_fields;
	c_pages[PAGE_C_INDEX].fields_builder_function = page_c_build_fields;
	c_pages[PAGE_D_INDEX].fields_builder_function = page_d_build_fields;
	c_pages[PAGE_E_INDEX].fields_builder_function = page_e_build_fields;
	c_pages[PAGE_F_INDEX].fields_builder_function = page_f_build_fields;

	return CDW_OK;
}





cdw_rv_t cdw_config_ui_window_build_page(int pi, const char *window_title, const char *window_tip)
{
	c_pages[pi].cdw_form->window = cdw_ncurses_window_new((WINDOW *) NULL,
							      CONF_HEIGHT, CONF_WIDTH,
							      (LINES - CONF_HEIGHT) / 2, (COLS - CONF_WIDTH) / 2,
							      CDW_COLORS_DIALOG, window_title, window_tip);
	if (c_pages[pi].cdw_form->window == (WINDOW *) NULL) {
		cdw_vdm ("ERROR: failed to create window #%d\n", pi);
		return CDW_GEN_ERROR;
	}

	c_pages[pi].cdw_form->subwindow = cdw_ncurses_window_new(c_pages[pi].cdw_form->window,
								 CONF_HEIGHT - 2, CONF_WIDTH - 2, 1, 1,
								 CDW_COLORS_DIALOG, (char *) NULL, (char *) NULL);
	if (c_pages[pi].cdw_form->subwindow == (WINDOW *) NULL) {
		cdw_vdm ("ERROR: failed to create subwindow #%d\n", pi);
		return CDW_GEN_ERROR;
	}

	c_pages[pi].panel = new_panel(c_pages[pi].cdw_form->window);
	if (c_pages[pi].panel == (PANEL *) NULL) {
		cdw_vdm ("ERROR: failed to create panel #%d with new_panel()\n", pi);
		return CDW_GEN_ERROR;
	}

	return CDW_OK;
}




/**
   \brief Initialize pages in cdw configuration window

   Create all subelements in all pages (subwindows) of cdw config window,
   create ncurses panels, associate them with ncurses windows.

   Call this function after creating windows and subwindows of cdw
   configuration window.

   \p config should be a copy of current config variable, values from
   \p config will be used to initialize fields in configuration window
   or, more precisely, in forms in every page of cdw configuration window.

   \param config - config variable with current values of options

   \return CDW_GEN_ERROR if something goes wrong
   \return CDW_GEN_ERROR on success
*/
cdw_rv_t cdw_config_ui_window_populate_pages(cdw_config_t *config)
{
	cdw_rv_t crv = cdw_config_ui_tab_labels_create();
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to create tab labels\n");
		return CDW_GEN_ERROR;
	}

	bool success = true;
	for (int i = 0; i < N_CONFIG_PAGES; i++) {
		crv = c_pages[i].fields_builder_function(c_pages[i].cdw_form, config);
		if (crv != CDW_OK) {
			success = false;
			cdw_vdm ("ERROR: failed to build fields in page #%d\n", i);
			break;
		}

		c_pages[i].cdw_form->form = cdw_form_new_form(c_pages[i].cdw_form->window,
							      c_pages[i].cdw_form->subwindow,
							      c_pages[i].cdw_form->fields);
		if (c_pages[i].cdw_form->form == (FORM *) NULL) {
			success = false;
			cdw_vdm ("ERROR: failed to create form on page #%d\n", i);
			break;
		}

		/* tabs on the right side of configuration window */
		cdw_config_ui_draw_tabs(i);

		wrefresh(c_pages[i].cdw_form->subwindow);
	}

	/* draw additional stuff in configuration window */
	crv = cdw_config_ui_draw_additional_widgets(config);
	if (crv != CDW_OK) {
		success = false;
		cdw_vdm ("ERROR: failed to draw additional widgets\n");
	}

	if (success) {
		return CDW_OK;
	} else {
		cdw_vdm ("ERROR: failed to populate configuration window\n");
		/* displaying error dialog window is performed by caller */
		return CDW_GEN_ERROR;
	}
}





/**
   \brief Function reacting to keys pressed when cdw config window is displayed

   Function catches all keys pressed when cdw configuration window is displayed,
   and reacts to them either by switching pages, or by passing the keys to
   page driver(s).

   F(CDW_CONFIG_FX_START_KEY) to F(CDW_CONFIG_FX_START_KEY + 5) keys switch
   between pages.
   F(CDW_CONFIG_SnC_KEY) key is interpreted as "save and close" - values of
   all option fields are stored in \p config and validated. Function then
   returns CDW_OK if validation is successful, or displays incorrect field
   to user.
   ESCAPE key causes the function to return with CDW_CANCEL.

   \param config - variable in which function saves values of all option fields

   \return CDW_CANCEL if user pressed ESCAPE key in cdw config window
   \return CDW_OK if user pressed F10 and validation of \p tmp config was successful
*/
cdw_rv_t cdw_config_ui_window_pages_driver(cdw_config_t *config)
{
	int fi = 0;
	/* weak FIXME: make sure that page A is visible; it's
	   "writing page" so there is a reasonable
	   assumption that it will be always visible */
	int pi = PAGE_A_INDEX;
	int key = KEY_F(pi + CDW_CONFIG_FX_START_KEY);
	top_panel(c_pages[pi].panel);

	while (1) {
		key = cdw_config_ui_window_pages_driver_sub(config, key, &pi, fi);
		if (key == CDW_KEY_ESCAPE) {
			break;
		} else if (key == KEY_F(CDW_CONFIG_SnC_KEY)) { /* SnC = Save and Close */

			/* flush */
			for (int i = 0; i < N_CONFIG_PAGES; i++) {
				form_driver(c_pages[i].cdw_form->form, REQ_VALIDATION);
			}

			/* this function also saves indexes of selected
			   tool paths into cdw_tools[] */
			option_fields_save_changes(config);
#ifndef NDEBUG
			cdw_config_debug_print_config(config);
#endif
			/* we are in driver and in theory we shouldn't
			   validate config here, but depending on result
			   of validation we may have to jump to one of pages;
			   moving validation outside of driver may be too
			   complicated */
			pi = -1; /* -1 = validate fields from all pages */
			cdw_rv_t valid = cdw_config_var_validate(config, &pi, &fi);

			if (valid == CDW_NO) { /* some option field is invalid */
				/* 2TRANS: this is title of dialog window */
				cdw_buttons_dialog(_("Error"),
						   /* 2TRANS: this is message in dialog window */
						   _("One of option fields is incorrect or contains character that is not allowed. Please fix it."),
						   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

				key = KEY_F(pi + CDW_CONFIG_FX_START_KEY);
				/* loop */
			} else { /* valid == CDW_OK */
				/* ok - values from config can be saved
				   to global config variable */
				return CDW_OK;
			}
		} else {
			/* loop */
			/* reset field index; it may have here some specific
			   value that is valid in page X, but may be invalid
			   in page Y, and that may lead to form_driver()
			   trying to act on invalid field; '-1' may trigger
			   warning debug message in cdw form driver, but
			   don't worry about that */
			fi = -1;
		}
	} /* while (1) */

	return CDW_CANCEL;
}





int cdw_config_ui_window_pages_driver_sub(cdw_config_t *config, int key, int *current_page, int fi)
{
	for (int pi = 0; pi < N_CONFIG_PAGES; pi++) {
		if (key == KEY_F(pi + CDW_CONFIG_FX_START_KEY)) {
			if (c_pages[pi].page_visible) {
				top_panel(c_pages[pi == 0 ? 1 : 0].panel);
				top_panel(c_pages[pi].panel);
				/* this line probably should be removed
				   completely; refreshing main ui window that
				   lies below configuration window causes
				   flickering in slower terminals */
				//cdw_main_ui_main_window_wrefresh();
				update_panels();
				key = cdw_form_driver(c_pages[pi].cdw_form, fi, config);

				*current_page = pi;
			} else {
				/* page specified by 'key' is invisible,
				   don't switch to that page; return key
				   corresponding to current, visible page */
				key = KEY_F(*current_page + CDW_CONFIG_FX_START_KEY);
			}
			break;
		}
	}

	return key;
}





/**
   \brief Create fields that are shown on page A of configuration window

   Create fields that are shown on page A of configuration window, including
   initialization of size and position (all N_FIELDS_A of them, not
   including last one which is of course NULL), set their types, options
   and appearance.

   Fields will be initialized with values from \p config.
   Basic check for correctness of creating fields is performed, but it may
   not be perfect.

   \param cdw_form - form page for which the fields will be built
   \param config - variable holding current configuration

   \return CDW_OK when all fields were created (probably) without problems
   \return CDW_GEN_ERROR if one of fields was created incorrectly
*/
cdw_rv_t page_a_build_fields(cdw_form_t *cdw_form, cdw_config_t *config)
{
	size_t sd_width = 10;
	char pad_size[PADSIZE_FIELD_LEN_MAX + 1];
	snprintf(pad_size, PADSIZE_FIELD_LEN_MAX + 1, "%d", config->pad_size);

	cdw_form_descr_t descr[] = {
		/*     type         begin_y   begin_x       n_cols       n_lines   field enum                   data1                           data2 */

		/* 2TRANS: this is a separator label, below the label there will be options "common" for group of tools */
		{ CDW_WIDGET_LABEL,     1,    FIRST_COL,   LABEL_WIDTH,       0,  f_general_l,         _("  ---  Common options  ---  "),         0 },
		/* 2TRANS: this is checkbox label: keep no longer than original; discs can be erased in 'fast' or 'all' mode */
		{ CDW_WIDGET_LABEL,     2,    FIRST_COL,   LABEL_WIDTH,       0,  f_blank_l,           _("Erase fast"),                           0 },
		{ CDW_WIDGET_CHECKBOX,  2,    SECOND_COL,  1,                 1,  f_blank_cb,          (void *) NULL,    config->erase_mode == CDW_ERASE_MODE_FAST ? 1 : 0 },
		/* 2TRANS: this is checkbox label: keep no longer than original; "eject" refers to ejecting cd drive tray */
		{ CDW_WIDGET_LABEL,     3,    FIRST_COL,   LABEL_WIDTH,       0,  f_eject_l,           _("Eject when done"),                      0 },
		{ CDW_WIDGET_CHECKBOX,  3,    SECOND_COL,  1,                 1,  f_eject_cb,          (void *) NULL,         config->eject ? 1 : 0 },
		/* 2TRANS: this is input field label: keep no longer than original */
		{ CDW_WIDGET_LABEL,     4,    FIRST_COL,   LABEL_WIDTH,       0,  f_speed_l,           _("Preferred writing speed"),              0 },
		{ CDW_WIDGET_DROPDOWN,  4,    SECOND_COL,  sd_width,          1,  f_speed_dd,          (void *) NULL,                             0 },
		/* 2TRANS: this is checkbox label: keep no longer than original; "Dummy write" means attempting to write
		   to cd and performing all normal operations (just for a try), but stopping just before writing */
		{ CDW_WIDGET_LABEL,     5,    FIRST_COL,   LABEL_WIDTH,       0,  f_dummy_l,           _("Dummy write"),                          0 },
		{ CDW_WIDGET_CHECKBOX,  5,    SECOND_COL,  1,                 1,  f_dummy_cb,          (void *) NULL,         config->dummy ? 1 : 0 },
		/* 2TRANS: this is a separator label, below the label there will be options specific for cdrecord */
		{ CDW_WIDGET_LABEL,     8,    FIRST_COL,   WIDE_FIELD_WIDTH,  0,  f_cdrecord_l,        _(" --- Options for cdrecord --- "),       0 },
		/* 2TRANS: this is checkbox label: keep no longer than original; "Pad" refers to adding empty data at the end of track */
		{ CDW_WIDGET_LABEL,     9,    FIRST_COL,   LABEL_WIDTH,       0,  f_pad_l,             _("Pad (recommended)"),                    0 },
		{ CDW_WIDGET_CHECKBOX,  9,    SECOND_COL,  1,                 1,  f_pad_cb,            (void *) NULL,           config->pad ? 1 : 0 },
		/* the "150" value is selected after small tests with DVD; originally it was 63, which
		   worked just fine for CDs, but for DVDs it was insufficient;
		   TODO: the same value is used in code initializing config variable,
		   so it should be a constant defined in some header */
		/* 2TRANS: this is checkbox label: keep no longer than original; "Pad" refers to adding empty data at the end of track */
		{ CDW_WIDGET_LABEL,    10,    FIRST_COL,   WIDE_FIELD_WIDTH,  0,  f_pad_size_l,        _("Pad size (number >= 0, '150' is recommended)"), 0 },
		{ CDW_WIDGET_INPUT,    11,    SECOND_COL,  PADSIZE_FIELD_LEN_MAX,  1,  f_pad_size_i,   pad_size,              PADSIZE_FIELD_LEN_MAX },
		/* 2TRANS: this is checkbox label: keep no longer than original */
		{ CDW_WIDGET_LABEL,    12,    FIRST_COL,   LABEL_WIDTH,       0,  f_burnproof_l,       _("Burnproof"),                             0 },
		{ CDW_WIDGET_CHECKBOX, 12,    SECOND_COL,  1,                 1,  f_burnproof_cb,      (void *) NULL,      config->burnproof ? 1 : 0 },
		/* 2TRANS: this is input field label: keep no longer than original; other options passed to program writing data to cd */
		{ CDW_WIDGET_LABEL,    13,    FIRST_COL,   WIDE_FIELD_WIDTH,  0,  f_oco_l,             _("Other \"cdrecord\" options:"),           0 },
		{ CDW_WIDGET_INPUT,    14,    FIRST_COL,   WIDE_FIELD_WIDTH,  1,  f_oco_i,             config->other_cdrecord_options,             0 },
		/* 2TRANS: this is a separator label, below the label there will be options specific for growisofs */
		{ CDW_WIDGET_LABEL,    17,    FIRST_COL,   WIDE_FIELD_WIDTH,  0,  f_growisofs_l,       _(" --- Options for growisofs --- "),       0 },
		/* 2TRANS: this is input field label: keep no longer than original; other options passed to program creating iso image */
		{ CDW_WIDGET_LABEL,    18,    FIRST_COL,   WIDE_FIELD_WIDTH,  0,  f_ogo_l,             _("Other \"growisofs\" options:"),          0 },
		{ CDW_WIDGET_INPUT,    19,    FIRST_COL,   WIDE_FIELD_WIDTH,  1,  f_ogo_i,             config->other_growisofs_options,            0 },
		/* guard */
		{ -1,                  0,     0,           0,                 0,  0,                   (void *) NULL,                              0 }};


	/* NOTE: dropdown must be wider than underlying field and no
	   printed part of dropdown can be placed in input field area */
	speed_range_dropdown = cdw_config_ui_speed_range_dropdown(cdw_form->subwindow,
								  4, SECOND_COL - 1, (int) sd_width + 5,
								  config);
	if (speed_range_dropdown == (CDW_DROPDOWN *) NULL) {
		cdw_vdm ("ERROR: failed to create speed range dropdown\n");
		return CDW_GEN_ERROR;
	} else {
		descr[f_speed_dd].data1 = (void *) speed_range_dropdown;
	}

	/* the function adds guard at the end of fields */
	cdw_rv_t crv = cdw_form_description_to_fields(descr, cdw_form);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to convert form description to form\n");
		return CDW_GEN_ERROR;
	}

	field_opts_on(cdw_form->fields[f_pad_size_i], O_STATIC);
	set_field_just(cdw_form->fields[f_ogo_l], JUSTIFY_LEFT);

	return CDW_OK;
}





/**
   \brief Create fields that are shown on page B of configuration window

   Create fields that are shown on page B of configuration window,
   including initialization of size and position (all N_FIELDS_B
   of them, last one is of course NULL), set their types, options and
   appearance.

   Fields will be initialized with values from \p config.
   Basic check for correctness of creating fields is performed, but it may
   not be perfect.

   \param cdw_form - form page for which the fields will be built
   \param config - variable holding current configuration

   \return CDW_OK when all fields were created (probably) without problems
   \return CDW_GEN_ERROR if one of fields was created incorrectly
*/
cdw_rv_t page_b_build_fields(cdw_form_t *cdw_form, cdw_config_t *config)
{
	cdw_form_descr_t descr[] = {
		/*     type        begin_y   begin_x      n_cols              n_lines   field enum        data1                             data2 */
		/* 2TRANS: this is label in configuration window, "drive" is an optical drive */
		{ CDW_WIDGET_LABEL,    1,    FIRST_COL,   LABEL_WIDTH,           0,  f_selected_drive_l,  _("cdw should use this drive:"),    0 },
		{ CDW_WIDGET_DROPDOWN, 2,    FIRST_COL,   WIDE_FIELD_WIDTH - 3,  1,  f_selected_drive_dd, (void *) NULL,                      0 },
		/* 2TRANS: this is input field label: keep no longer than original;
		   input field holds path to cd reader/writer device */
		{ CDW_WIDGET_LABEL,    4,    FIRST_COL,   LABEL_WIDTH,           0,  f_custom_drive_l,    _("Custom path to drive:"),         0 },
		{ CDW_WIDGET_INPUT,    5,    FIRST_COL,   WIDE_FIELD_WIDTH,      1,  f_custom_drive_i,    config->custom_drive,               0 },
		/* 2TRANS: this is input field label: keep no longer than original;
		   input field holds address of scsi writing device */
		{ CDW_WIDGET_LABEL,    6,    FIRST_COL,   LABEL_WIDTH,           0,  f_scsi_l,            _("SCSI device (for cdrecord):"),   0 },
		{ CDW_WIDGET_INPUT,    7,    FIRST_COL,   WIDE_FIELD_WIDTH,      1,  f_scsi_i,            config->scsi,                       0 },
		/* guard */
		{ -1,                  0,    0,           0,                     0,  0,                   (void *) NULL,                      0 }};



	CDW_DROPDOWN *dropdown = cdw_drive_make_drives_dropdown(cdw_form->subwindow,
								2, FIRST_COL, WIDE_FIELD_WIDTH);
	if (dropdown == (CDW_DROPDOWN *) NULL) {
		cdw_vdm ("ERROR: failed to create \"drives\" dropdown\n");
		return CDW_GEN_ERROR;
	} else {
		descr[f_selected_drive_dd].data1 = (void *) dropdown;
	}

	/* the function adds guard at the end of fields */
	cdw_rv_t crv = cdw_form_description_to_fields(descr, cdw_form);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to convert form description to form\n");
		return CDW_GEN_ERROR;
	}
	/* set_field_type(fields[f_scsi_i], TYPE_REGEXP, "^(ATAPI:|REMOTE:){0,1}[0-9],[0-9],[0-9]"); */

	return CDW_OK;
}





/**
   \brief Create fields that are shown on page C of configuration window

   Create fields that are shown on page C of configuration window,
   including initialization of size and position (all N_FIELDS_C
   of them, not including last one which is of course NULL), set their
   types, options and appearance.

   Fields will be initialized with values from \p config.
   Basic check for correctness of creating fields is performed, but it may
   not be perfect.

   \param cdw_form - form page for which the fields will be built
   \param config - variable holding current configuration

   \return CDW_OK when all fields were created (probably) without problems
   \return CDW_GEN_ERROR if one of fields was created incorrectly
*/
cdw_rv_t page_c_build_fields(cdw_form_t *cdw_form, cdw_config_t *config)
{
	size_t audio_dir_button_col =  FIRST_COL + WIDE_FIELD_WIDTH - 3;
	cdw_form_descr_t descr[] = {
		/*      type        begin_y   begin_x       n_cols          n_lines   field enum       data1               data2 */
		/* 2TRANS: this is input field label: keep no longer than original;  input field
		   holds path to directory, into which audio tracks will be ripped */
		{ CDW_WIDGET_LABEL,    1,    FIRST_COL,   LABEL_WIDTH,           0,  f_audiodir_l,  _("Audio output dir"),  0 },
		{ CDW_WIDGET_INPUT,    2,    FIRST_COL,   WIDE_FIELD_WIDTH - 3,  1,  f_audiodir_i,  config->audiodir,       0 },
		/* this fields is created only because there will be a button
		   here, that I need to be able to navigate to, so string can be empty */
		{ CDW_WIDGET_BUTTON,   2,    audio_dir_button_col,  1,           1,  f_audiodir_b,  (void *) NULL,          0 },
		/* guard */
		{ -1,                  0,    0,           0,                     0,  0,             (void *) NULL,          0 } };

	audio_dir_button = cdw_button_new(cdw_form->subwindow, 2, audio_dir_button_col, ">", CDW_COLORS_DIALOG);
	if (audio_dir_button == (CDW_BUTTON *) NULL) {
		cdw_vdm ("ERROR: failed to create \"audio dir more\" button\n");
		return CDW_GEN_ERROR;
	} else {
		descr[f_audiodir_b].data1 = (void *) audio_dir_button;
	}

	/* the function adds guard at the end of fields */
	cdw_rv_t crv = cdw_form_description_to_fields(descr, cdw_form);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to convert form description to form\n");
		return CDW_GEN_ERROR;
	}

	return CDW_OK;
}





/**
   \brief Create fields that are shown on page D of configuration window

   Create fields that are shown on page D of configuration window,
   including initialization of size and position (all N_FIELDS_D
   of them, not including last one which is of course NULL), set their
   types, options and appearance.

   Fields will be initialized with values from \p config.
   Basic check for correctness of creating fields is performed, but it may
   not be perfect.

   \param cdw_form - form page for which the fields will be built
   \param config - variable holding current configuration

   \return CDW_OK when all fields were created (probably) without problems
   \return CDW_GEN_ERROR if one of fields was created incorrectly
*/
cdw_rv_t page_d_build_fields(cdw_form_t *cdw_form, cdw_config_t *config)
{
	size_t ild_width = 5;
	size_t iso_image_button_col = FIRST_COL + WIDE_FIELD_WIDTH - 3;
	cdw_form_descr_t descr[] = {
		/*     type        begin_y   begin_x       n_cols           n_lines  field enum              data1                                       data2 */

		/* 2TRANS: this is input field label: keep no longer than original; volume id is name of cd visible e.g. in file manager */
		{ CDW_WIDGET_LABEL,    1,    FIRST_COL,       LABEL_WIDTH - 10,  0,  f_volumeid_l,         _("Volume ID label"),                          0 },
		{ CDW_WIDGET_INPUT,    1,    SECOND_COL - 10, VOLUME_ID_LEN_MAX, 1,  f_volumeid_i,         config->volumeid,              VOLUME_ID_LEN_MAX },
		/* 2TRANS: this is checkbox label: keep no longer than original; volume id is name of cd visible e.g. in file manager */
		{ CDW_WIDGET_LABEL,    2,    FIRST_COL,   LABEL_WIDTH,           0,  f_showvol_l,          _("Ask for volume ID"),                        0 },
		{ CDW_WIDGET_CHECKBOX, 2,    SECOND_COL,  0,                     0,  f_showvol_cb,         (void *) NULL,           config->showvol ? 1 : 0 },
		/* 2TRANS: this is dropdown label: keep no longer than original */
		{ CDW_WIDGET_LABEL,    3,    FIRST_COL,   LABEL_WIDTH,           0,  f_iso_level_l,        _("ISO level"),                                0 },
		{ CDW_WIDGET_DROPDOWN, 3,    SECOND_COL,  ild_width,             1,  f_iso_level_dd,       (void *) NULL,                                 0 },
		/* 2TRANS: this is checkbox label: keep no longer than original */
		{ CDW_WIDGET_LABEL,    4,    FIRST_COL,   LABEL_WIDTH,           0,  f_joliet_l,           _("Joliet information"),                       0 },
		{ CDW_WIDGET_CHECKBOX, 4,    SECOND_COL,  0,                     0,  f_joliet_cb,          (void *) NULL,            config->joliet ? 1 : 0 },
		/* 2TRANS: this is checkbox label: keep no longer than original; you probably want to keep it untranslated */
		{ CDW_WIDGET_LABEL,    5,    FIRST_COL,   LABEL_WIDTH,           0,  f_rockridge_l,        _("Rock Ridge"),                               0 },
		{ CDW_WIDGET_CHECKBOX, 5,    SECOND_COL,  0,                     0,  f_rockridge_cb,       (void *) NULL,         config->rockridge ? 1 : 0 },
		/* 2TRANS: this is checkbox label: keep no longer than original */
		{ CDW_WIDGET_LABEL,    6,    FIRST_COL,   LABEL_WIDTH,           0,  f_useful_rr_l,        _("Useful RR attributes"),                     0 },
		{ CDW_WIDGET_CHECKBOX, 6,    SECOND_COL,  0,                     0,  f_useful_rr_cb,       (void *) NULL,         config->useful_rr ? 1 : 0 },
		/* 2TRANS: this is checkbox label: keep no longer than original */
		{ CDW_WIDGET_LABEL,    7,    FIRST_COL,   LABEL_WIDTH,           0,  f_joliet_long_l,     _("Long Joliet names"),                         0 },
		{ CDW_WIDGET_CHECKBOX, 7,    SECOND_COL,  0,                     0,  f_joliet_long_cb,     (void *) NULL,       config->joliet_long ? 1 : 0 },
		/* 2TRANS: this is checkbox label: keep no longer than original */
		{ CDW_WIDGET_LABEL,    8,    FIRST_COL,   LABEL_WIDTH,           0,  f_follow_symlinks_l,  _("Follow symbolic links"),                    0 },
		{ CDW_WIDGET_CHECKBOX, 8,    SECOND_COL,  0,                     0,  f_follow_symlinks_cb, (void *) NULL,   config->follow_symlinks ? 1 : 0 },
		/* 2TRANS: this is input field label: keep no longer than original */
		{ CDW_WIDGET_LABEL,   10,    FIRST_COL,   WIDE_FIELD_WIDTH,      0,  f_iso_image_full_path_l,  _("Path to ISO image file:"),               0 },
		{ CDW_WIDGET_INPUT,   11,    FIRST_COL,   WIDE_FIELD_WIDTH - 3,  1,  f_iso_image_full_path_i,  config->iso_image_full_path,                0 },
		/* this fields is created only because there will be a button
		   here, that I need to be able to navigate to, so string can be empty */
		{ CDW_WIDGET_BUTTON,  11,    iso_image_button_col, 1,            1,  f_iso_image_full_path_b,  (void *) NULL,                              0 },
		/* 2TRANS: this is input field label: keep no longer than original;
		   input field holds options related to boot image for bootable disc */
		{ CDW_WIDGET_LABEL,   13,    FIRST_COL,   WIDE_FIELD_WIDTH,      0,  f_boot_disc_options_l,   _("Boot disc options (not tested):"),       0 },
		{ CDW_WIDGET_INPUT,   14,    FIRST_COL,   WIDE_FIELD_WIDTH,      1,  f_boot_disc_options_i,   config->boot_disc_options,                  0 },
		/* 2TRANS: this is input field label: keep no longer than original; other options passed to program creating ISO image */
		{ CDW_WIDGET_LABEL,   16,    FIRST_COL,   WIDE_FIELD_WIDTH,      0,  f_omo_l,                 _("Other \"mkisofs\" options:"),            0 },
		{ CDW_WIDGET_INPUT,   17,    FIRST_COL,   WIDE_FIELD_WIDTH,      1,  f_omo_i,                 config->other_mkisofs_options,              0 },
		/* guard */
		{ -1,                  0,    0,           0,                     0,  0,                       (void *) NULL,                              0 } };


	/* NOTE: dropdown must be wider than underlying field and no
	   printed part of dropdown can be placed in input field area */
	cdw_assert (0 < config->iso_level && config->iso_level < 5,
		    "ERROR: iso level value out of range: %d\n", config->iso_level);
	iso_level_dropdown = cdw_config_ui_iso_level_dropdown(cdw_form->subwindow,
							      3, SECOND_COL - 1, (int) ild_width + 3,
							      config);
	if (iso_level_dropdown == (CDW_DROPDOWN *) NULL) {
		cdw_vdm ("ERROR: failed to create iso level dropdown\n");
			return CDW_GEN_ERROR;
	} else {
		descr[f_iso_level_dd].data1 = (void *) iso_level_dropdown;
	}

	iso_image_path_button = cdw_button_new(cdw_form->subwindow,
					       11, iso_image_button_col, ">", CDW_COLORS_DIALOG);
	if (iso_image_path_button == (CDW_BUTTON *) NULL) {
		cdw_vdm ("ERROR: failed to create \"iso image path\" button\n");
		return CDW_GEN_ERROR;
	} else {
		descr[f_iso_image_full_path_b].data1 = (void *) iso_image_path_button;
	}

	/* the function adds guard at the end of fields */
	cdw_rv_t crv = cdw_form_description_to_fields(descr, cdw_form);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to convert form description to form\n");
		return CDW_GEN_ERROR;
	}

	return CDW_OK;
}





/**
   \brief Create fields that are shown on page E of configuration window

   Create fields that are shown on page E of configuration window,
   including initialization of size and position (all N_FIELDS_E
   of them, not including last one which is of course NULL), set their
   types, options and appearance.

   Basic check for correctness of creating fields is performed, but it may
   not be perfect.

   \param cdw_form - form page for which the fields will be built
   \param config - variable holding current configuration

   \return CDW_OK when all fields were created (probably) without problems
   \return CDW_GEN_ERROR if one of fields was created incorrectly
*/
cdw_rv_t page_e_build_fields(cdw_form_t *cdw_form, __attribute__((unused)) cdw_config_t *config)
{
	cdw_form_descr_t descr[] = {
		/*     type        begin_y   begin_x          n_cols              n_lines   field enum                data1                               data2 */
		/* 2TRANS: this is checkbox label, it means that user will perform manual
		   selection of values in dropdowns representing available tools */
		{ CDW_WIDGET_LABEL,    3,    FIRST_COL + 3,   WIDE_FIELD_WIDTH - 3,  0,  f_manual_selection_l,       _("Configure tools manually"),         0 },
		{ CDW_WIDGET_CHECKBOX, 3,    FIRST_COL,       1,                     1,  f_manual_selection_cb,      (void *) NULL,  cdw_ext_tools_get_manual_selection_state() ? 1 : 0 },
		/* 2TRANS: this is label of input field; */
		{ CDW_WIDGET_LABEL,    5,    FIRST_COL,       WIDE_FIELD_WIDTH,      0,  f_dvd_family_l,             _("Tools family for handling DVDs:"),  0 },
		{ CDW_WIDGET_DROPDOWN, 6,    FIRST_COL + 2,   WIDE_FIELD_WIDTH - 3,  1,  f_dvd_family_dd,            (void *) NULL,                         0 },
		/* 2TRANS: this is label of input field; "path" is path in filesystem */
		{ CDW_WIDGET_LABEL,    7,    FIRST_COL,       LABEL_WIDTH,           0,  f_mkisofs_path_l,           _("Path to mkisofs:"),                 0 },
		{ CDW_WIDGET_DROPDOWN, 8,    FIRST_COL + 2,   WIDE_FIELD_WIDTH - 3,  1,  f_mkisofs_path_dd,          (void *) NULL,                         0 },
		/* 2TRANS: this is label of input field; "path" is path in filesystem */
		{ CDW_WIDGET_LABEL,    9,    FIRST_COL,       LABEL_WIDTH,           0,  f_cdrecord_path_l,          _("Path to cdrecord:"),                0 },
		{ CDW_WIDGET_DROPDOWN,10,    FIRST_COL + 2,   WIDE_FIELD_WIDTH - 3,  1,  f_cdrecord_path_dd,         (void *) NULL,                         0 },
		/* 2TRANS: this is label of input field; "path" is path in filesystem */
		{ CDW_WIDGET_LABEL,   11,    FIRST_COL,       LABEL_WIDTH,           0,  f_growisofs_path_l,         _("Path to growisofs:"),               0 },
		{ CDW_WIDGET_DROPDOWN,12,    FIRST_COL + 2,   WIDE_FIELD_WIDTH - 3,  1,  f_growisofs_path_dd,        (void *) NULL,                         0 },
		/* 2TRANS: this is label of input field; "path" is path in filesystem */
		{ CDW_WIDGET_LABEL,   13,    FIRST_COL,       LABEL_WIDTH,           0,  f_dvd_rw_mediainfo_path_l,  _("Path to dvd+rw-mediainfo:"),        0 },
		{ CDW_WIDGET_DROPDOWN,14,    FIRST_COL + 2,   WIDE_FIELD_WIDTH - 3,  1,  f_dvd_rw_mediainfo_path_dd, (void *) NULL,                         0 },
		/* 2TRANS: this is label of input field; "path" is path in filesystem */
		{ CDW_WIDGET_LABEL,   15,    FIRST_COL,       LABEL_WIDTH,           0,  f_dvd_rw_format_path_l,     _("Path to dvd+rw-format:"),           0 },
		{ CDW_WIDGET_DROPDOWN,16,    FIRST_COL + 2,   WIDE_FIELD_WIDTH - 3,  1,  f_dvd_rw_format_path_dd,    (void *) NULL,                         0 },
		/* guard */
		{ -1,                  0,    0,               0,                     0,  0,                          (void *) NULL,                         0 } };


	/* NOTE for all dropdowns: dropdown must be wider than underlying field
	   and "[" + "]" parts of dropdown must be placed in input field area */

	cdw_rv_t crv = cdw_ext_tools_create_config_dropdowns(cdw_form->subwindow, 6, FIRST_COL + 1, WIDE_FIELD_WIDTH);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to create ext tools dropdowns\n");
		return CDW_GEN_ERROR;
	}

	descr[f_mkisofs_path_dd].data1 = (void *) ext_tools_dropdowns.mkisofs;
	descr[f_cdrecord_path_dd].data1 = (void *) ext_tools_dropdowns.cdrecord;
	descr[f_growisofs_path_dd].data1 = (void *) ext_tools_dropdowns.growisofs;
	descr[f_dvd_rw_mediainfo_path_dd].data1 = (void *) ext_tools_dropdowns.dvd_rw_mediainfo;
	descr[f_dvd_rw_format_path_dd].data1 = (void *) ext_tools_dropdowns.dvd_rw_format;
	descr[f_dvd_family_dd].data1 = (void *) ext_tools_dropdowns.dvd_family;

	/* the function adds guard at the end of fields */
	crv = cdw_form_description_to_fields(descr, cdw_form);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to convert form description to form\n");
		return CDW_GEN_ERROR;
	}

	return CDW_OK;
}





/**
   \brief Create fields that are shown on page F of configuration window

   Create fields that are shown on page F of configuration window, including
   initialization of size and position (all N_FIELDS_F of them, not
   including last one which is of course NULL), set their types, options
   and appearance.

   Fields will be initialized with values from \p config.
   Basic check for correctness of creating fields is performed, but it may
   not be perfect.

   \param cdw_form - form page for which the fields will be built
   \param config - variable holding current configuration

   \return CDW_OK when all fields were created (probably) without problems
   \return CDW_GEN_ERROR if one of fields was created incorrectly
*/
cdw_rv_t page_f_build_fields(cdw_form_t *cdw_form, cdw_config_t *config)
{

#define CDW_CVF_WIDTH 6 /* CVF = custom value field */

	char cvf_buffer[CDW_CVF_WIDTH + 1];
	if (config->volume_size_custom_value <= 0) {
		config->volume_size_custom_value = 0;
	} else {
		if ((floor(log10((double) config->volume_size_custom_value)) + 1) > CDW_CVF_WIDTH) {
			config->volume_size_custom_value = 0;
		}
	}
	snprintf(cvf_buffer, CDW_CVF_WIDTH + 1, "%ld", config->volume_size_custom_value);

	size_t log_path_button_col = FIRST_COL + WIDE_FIELD_WIDTH - 3;
	cdw_form_descr_t descr[] = {
		/*    type             begin_y   begin_x         n_cols           n_lines  field enum                  data1                         data2 */
		/* 2TRANS: this is input field label: keep no longer than original;
		   input field holds full path to log file */
		{ CDW_WIDGET_LABEL,       1,    FIRST_COL,     WIDE_FIELD_WIDTH,      0,  f_log_fp_l,            _("Log file path (obligatory):"),     0 },
		{ CDW_WIDGET_INPUT,       2,    FIRST_COL,     WIDE_FIELD_WIDTH - 3,  1,  f_log_fp_i,            config->log_full_path,                0 },
		/* this fields is created only because there will be a button
		   here, that I need to be able to navigate to, so string can be empty */
		{ CDW_WIDGET_BUTTON,      2,    log_path_button_col,   3,             1,  f_log_fp_b,            (void *) NULL,                        0 },
		/* 2TRANS: this is checkbox label: keep no longer than original */
		{ CDW_WIDGET_LABEL,       3,    FIRST_COL,     LABEL_WIDTH,           0,  f_showlog_l,           _("Show log after writing"),          0 },
		{ CDW_WIDGET_CHECKBOX,    3,    SECOND_COL,    1,                     1,  f_showlog_cb,          (void *) NULL,  config->showlog ? 1 : 0 },
		/* 2TRANS: this is label above a dropdown, a maximal size for selected files, that will be put into ISO file system */
		{ CDW_WIDGET_LABEL,       6,    FIRST_COL,     LABEL_WIDTH,           0,  f_volume_size_l,       _("Maximal ISO volume size"),         0 },
		{ CDW_WIDGET_DROPDOWN,    7,    FIRST_COL + 2, WIDE_FIELD_WIDTH - 3,  1,  f_volume_size_dd,      (void *) NULL,                        0 },
		/* 2TRANS: this is input field label, user can enter into the field a custom value of max. size of ISO volume; "MB" is megabytes */
		{ CDW_WIDGET_LABEL,       8,    FIRST_COL,     LABEL_WIDTH,           1,  f_cust_volume_size_l,  _("Custom ISO volume size (MB):"),    0 },
		{ CDW_WIDGET_INPUT,       8,    SECOND_COL + 5,  CDW_CVF_WIDTH,       1,  f_cust_volume_size_i,  cvf_buffer,               CDW_CVF_WIDTH },

		/* guard */
		{ -1,                     0,    0,             0,                     0,  0,                 (void *) NULL,                        0 } };

	log_path_button = cdw_button_new(cdw_form->subwindow, 2, log_path_button_col, ">", CDW_COLORS_DIALOG);
	if (log_path_button == (CDW_BUTTON *) NULL) {
		cdw_vdm ("ERROR: failed to create \"log path\" button\n");
		return CDW_GEN_ERROR;
	} else {
		descr[f_log_fp_b].data1 = (void *) log_path_button;
	}

	volume_size_dropdown = cdw_config_ui_volume_size_dropdown(cdw_form->subwindow,
								  7, FIRST_COL, WIDE_FIELD_WIDTH, config);
	if (volume_size_dropdown == (CDW_DROPDOWN *) NULL) {
		cdw_vdm ("ERROR: failed to create \"volume size\" dropdown\n");
		return CDW_GEN_ERROR;
	} else {
		descr[f_volume_size_dd].data1 = (void *) volume_size_dropdown;
	}

	/* the function adds guard at the end of fields */
	cdw_rv_t crv = cdw_form_description_to_fields(descr, cdw_form);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to convert form description to form\n");
		return CDW_GEN_ERROR;
	}

	return CDW_OK;
}





/**
   \brief Draw widgets in configuration window

   Function draws missing square brackets around checkboxes, displays current
   items of dropdowns, creates text subwindows and fills them with text, and
   does other useful stuff that should be done to show nice configuration
   window, but could not be done in loops. This function does not draw tabs
   that should be visible on right side of config widow, you have to call
   cdw_config_ui_draw_tabs() to do this.

   \param config - config variable with current configuration

   \return CDW_OK if there were no problems
   \return CDW_GEN_ERROR if there was a problem when creating some widget
*/
cdw_rv_t cdw_config_ui_draw_additional_widgets(cdw_config_t *config)
{
	/* *** page A - writing *** */
	WINDOW *window = c_pages[PAGE_A_INDEX].cdw_form->subwindow;
	cdw_put_ckbox(window, 2, SECOND_COL); /* blank fast */
	cdw_put_ckbox(window, 3, SECOND_COL); /* eject */
	cdw_put_ckbox(window, 5, SECOND_COL); /* dummy write */
	cdw_put_ckbox(window, 9, SECOND_COL); /* pad */
	mvwprintw(window, 11, SECOND_COL + PADSIZE_FIELD_LEN_MAX, "s"); /* padsize, Xs) */
	cdw_put_ckbox(window, 12, SECOND_COL); /* burnproof */

	cdw_dropdown_display_current_item(speed_range_dropdown);


	/* *** page B - hardware *** */
	/* first -2 is because subwindow sizes are (CONF_* - 2);
	   rest of values are chosen so that this subwindow 'looks good' */
	page_B_txt_subwin = derwin(c_pages[PAGE_B_INDEX].cdw_form->subwindow,
				   CONF_HEIGHT - 2 - 9, WIDE_FIELD_WIDTH, 9, 1);
	if (page_B_txt_subwin == (WINDOW *) NULL) {
		cdw_vdm ("ERROR: failed to create txt subwindow in page B\n");
		return CDW_GEN_ERROR;
	}

	cdw_drive_print_help_message(page_B_txt_subwin);

	wrefresh(page_B_txt_subwin);
	form_driver(c_pages[PAGE_B_INDEX].cdw_form->form, REQ_END_LINE);


	/* *** page C - audio *** */
	cdw_button_unfocus(audio_dir_button);


	/* *** page D - ISO file system *** */
	window = c_pages[PAGE_D_INDEX].cdw_form->subwindow;
	cdw_put_ckbox(window, 2, SECOND_COL); /* showvol */
	cdw_put_ckbox(window, 4, SECOND_COL); /* rock ridge */
	cdw_put_ckbox(window, 5, SECOND_COL); /* joliet */
	cdw_put_ckbox(window, 6, SECOND_COL); /* useful RR */
	cdw_put_ckbox(window, 7, SECOND_COL); /* long joliet names */
	cdw_put_ckbox(window, 8, SECOND_COL); /* follow symbolic links */

	cdw_assert (config->iso_level > 0 && config->iso_level < 5,
		    "ERROR: invalid value of iso level: %d\n", config->iso_level);
	cdw_dropdown_set_current_item_by_id(iso_level_dropdown, (size_t) config->iso_level);
	cdw_dropdown_display_current_item(iso_level_dropdown);
	cdw_button_unfocus(iso_image_path_button);


	/* *** page E - tools *** */
	cdw_put_ckbox(c_pages[PAGE_E_INDEX].cdw_form->subwindow, 3, FIRST_COL); /* manual selection of tools */

	page_E_txt_subwin = derwin(c_pages[PAGE_E_INDEX].cdw_form->subwindow,
				   4, WIDE_FIELD_WIDTH, 1, 1);
	if (page_E_txt_subwin == (WINDOW *) NULL) {
		cdw_vdm ("ERROR: failed to create txt subwindow in page E\n");
		return CDW_GEN_ERROR;
	}

	cdw_textarea_print_message(page_E_txt_subwin,
				   /* 2TRANS: this is a description in one of
				      configuration window panels; a series of
				      dropdown widgets is displayed below the label */
				   _("Changes made here won't be saved after closing cdw."),
				   CDW_ALIGN_LEFT);
	wrefresh(page_E_txt_subwin);

	cdw_dropdown_display_current_item(ext_tools_dropdowns.mkisofs);
	cdw_dropdown_display_current_item(ext_tools_dropdowns.cdrecord);
	cdw_dropdown_display_current_item(ext_tools_dropdowns.growisofs);
	cdw_dropdown_display_current_item(ext_tools_dropdowns.dvd_rw_mediainfo);
	cdw_dropdown_display_current_item(ext_tools_dropdowns.dvd_rw_format);
	cdw_dropdown_display_current_item(ext_tools_dropdowns.dvd_family);

	cdw_config_ui_page_e_driver_toggle_manual_selection();


	/* *** page F - log and other *** */
	cdw_put_ckbox(c_pages[PAGE_F_INDEX].cdw_form->subwindow, 3, SECOND_COL); /* show log */
	cdw_button_unfocus(log_path_button);
	cdw_dropdown_display_current_item(volume_size_dropdown);

	return CDW_OK;
}





/**
   \brief Save options in temporary config variable

   \param config - temporary variable to which all field values will be saved
*/
void option_fields_save_changes(cdw_config_t *config)
{
	cdw_rv_t ss;
	FIELD **fields = (FIELD **) NULL;


	/* *** WRITING *** */
	fields = c_pages[PAGE_A_INDEX].cdw_form->fields;
	bool erase_fast = cdw_form_get_field_bit(*(fields + f_blank_cb));
	config->erase_mode = erase_fast ? CDW_ERASE_MODE_FAST : CDW_ERASE_MODE_ALL;
	config->eject = cdw_form_get_field_bit(*(fields + f_eject_cb));
	config->speed_range = cdw_dropdown_get_current_item_ind(speed_range_dropdown);
	config->dummy = cdw_form_get_field_bit(*(fields + f_dummy_cb));
	config->pad = cdw_form_get_field_bit(*(fields + f_pad_cb));

	char *pad_size_buf = cdw_form_get_field_string(*(fields + f_pad_size_i));
	char *pad_size_rest = (char *) NULL;
	config->pad_size = (int) strtol(pad_size_buf, &pad_size_rest, 10);
	cdw_vdm ("INFO: pad size: buf = \"%s\", rest = \"%s\"\n", pad_size_buf, pad_size_rest);
	if (strlen(pad_size_rest)) {
		/* "-1" will be recognized as indicator of invalid buffer
		   content at time of validation of config variable */
		config->pad_size = -1;
		cdw_vdm ("WARNING: invalid substring \"%s\" in pad size string \"%s\"\n", pad_size_rest, pad_size_buf);
	} else {
		cdw_vdm ("INFO: pad size string \"%s\" converted to int %d\n", pad_size_buf, config->pad_size);
	}

	config->burnproof = cdw_form_get_field_bit(*(fields + f_burnproof_cb));
	ss = cdw_ncurses_get_field_buffer(*(fields + f_oco_i), &(config->other_cdrecord_options), 0);
	if (ss != CDW_OK) {
		cdw_vdm ("ERROR: failed to save \"other cdrecord options\"\n");
	}
	ss = cdw_ncurses_get_field_buffer(*(fields + f_ogo_i), &(config->other_growisofs_options), 0);
	if (ss != CDW_OK) {
		cdw_vdm ("ERROR: failed to save \"other growisofs options\"\n");
	}


	/* *** HARDWARE *** */
	fields = c_pages[PAGE_B_INDEX].cdw_form->fields;
	snprintf(config->custom_drive, OPTION_FIELD_LEN_MAX + 1, "%s", cdw_form_get_field_string(*(fields + f_custom_drive_i)));
	snprintf(config->scsi, OPTION_FIELD_LEN_MAX + 1, "%s", cdw_form_get_field_string(*(fields + f_scsi_i)));
	cdw_drive_save_config(config);


	/* *** AUDIO *** */
	fields = c_pages[PAGE_C_INDEX].cdw_form->fields;
	cdw_string_set((&config->audiodir), cdw_form_get_field_string(*(fields + f_audiodir_i)));


	/* *** ISO FILE SYSTEM *** */
	fields = c_pages[PAGE_D_INDEX].cdw_form->fields;
	snprintf(config->volumeid, VOLUME_ID_LEN_MAX + 1, "%s", cdw_form_get_field_string(*(fields + f_volumeid_i)));
	config->showvol = cdw_form_get_field_bit(*(fields + f_showvol_cb));

	config->iso_level = cdw_dropdown_get_current_item_id(iso_level_dropdown);
	if (config->iso_level < 1 || config->iso_level > 4) {
		/* safe default value */
		config->iso_level = 3;
		/* iso level is represented by a dropdown, so user can't
		   enter invalid value; if there is an invalid value, then
		   this is programmer's error */
		cdw_assert (0, "ERROR: field stores invalid value of iso level: %d\n", config->iso_level);
	}

	config->joliet = cdw_form_get_field_bit(*(fields + f_joliet_cb));
	config->rockridge = cdw_form_get_field_bit(*(fields + f_rockridge_cb));
	config->useful_rr = cdw_form_get_field_bit(*(fields + f_useful_rr_cb));
	config->joliet_long = cdw_form_get_field_bit(*(fields + f_joliet_long_cb));
	config->follow_symlinks = cdw_form_get_field_bit(*(fields + f_follow_symlinks_cb));
	cdw_string_set(&(config->iso_image_full_path), cdw_form_get_field_string(*(fields + f_iso_image_full_path_i)));
	cdw_string_set(&(config->boot_disc_options), cdw_form_get_field_string(*(fields + f_boot_disc_options_i)));
	cdw_string_set(&(config->other_mkisofs_options), cdw_form_get_field_string(*(fields + f_omo_i)));


	/* *** TOOLS *** */
	fields = c_pages[PAGE_E_INDEX].cdw_form->fields;
	cdw_ext_tools_save_manual_selection_state(cdw_form_get_field_bit(*(fields + f_manual_selection_cb)));
	cdw_ext_tools_save_dropdown_states();


	/* *** LOGGING AND OTHER *** */
	fields = c_pages[PAGE_F_INDEX].cdw_form->fields;
	ss = cdw_string_set(&(config->log_full_path), cdw_form_get_field_string(*(fields + f_log_fp_i)));
	config->showlog = cdw_form_get_field_bit(*(fields + f_showlog_cb));

	config->volume_size_id = cdw_dropdown_get_current_item_id(volume_size_dropdown);
	char *cvs_buffer = cdw_form_get_field_string(*(fields + f_cust_volume_size_i));
	char *cvs_buffer_rest = (char *) NULL;
	config->volume_size_custom_value = strtol(cvs_buffer, &cvs_buffer_rest, 10);
	cdw_vdm ("INFO: volume size: buf = \"%s\", rest = \"%s\"\n", cvs_buffer, cvs_buffer_rest);
	if (strlen(cvs_buffer_rest)) {
		/* "-1" will be recognized as indicator of invalid buffer
		   content at time of validation of config variable */
		config->volume_size_custom_value = -1;
		cdw_vdm ("WARNING: invalid substring \"%s\" in custom volume size string \"%s\"\n", cvs_buffer_rest, cvs_buffer);
	} else {
		cdw_vdm ("INFO: volume size string \"%s\" converted to int %ld\n", cvs_buffer, config->volume_size_custom_value);
	}


	/* TODO: not a best place to set config->volume_size_value,
	   because if volume_size_id == AUTO then it will be hard to
	   do disc check here */
	if (config->volume_size_id == CDW_CONFIG_VOLUME_SIZE_CUSTOM) {
		config->volume_size_value = config->volume_size_custom_value;
	} else {
		config->volume_size_value = cdw_config_volume_size_by_id(config->volume_size_id);
	}

	return;
}





/**
   \brief Create strings used as labels in configuration window tabs

   The function assigns values to c_pages[].tab_label fields using
   gettext(), so no additional memory is allocated. The function also
   assigns proper values to c_pages[].tab_label_len fields.

   \return CDW_OK
*/
cdw_rv_t cdw_config_ui_tab_labels_create(void)
{
	/* we can't use loop here because values of PAGE_X_INDEX aren't ordered */

	/* 2TRANS: this is name of tab of notepad-style widget:
	   "Writing" refers to writing / burning CD or DVD */
	c_pages[PAGE_A_INDEX].tab_label = _("Writing");
	/* 2TRANS: this is name of tab of notepad-style widget:
	   "Hardware" refers to CD/DVD drives */
	c_pages[PAGE_B_INDEX].tab_label = _("Hardware");
	/* 2TRANS: this is name of tab of notepad-style widget:
	   "Audio" means 'audio cd ripping configuration' */
	c_pages[PAGE_C_INDEX].tab_label = _("Audio");
	/* 2TRANS: this is name of tab of notepad-style widget:
	   "ISO" is ISO9660 filesystem */
	c_pages[PAGE_D_INDEX].tab_label = _("ISO filesystem");
	/* 2TRANS: this is name of tab of notepad-style widget: "tools" are
	   external programs (like cdrecord or dvd+rw-mediainfo) used by cdw */
	c_pages[PAGE_E_INDEX].tab_label = _("Tools");
	/* 2TRANS: this is name of tab of notepad-style widget: "log" is a
	   file where the program writes various stuff that can be later
	   examined by user; "other" means "other settings" */
	c_pages[PAGE_F_INDEX].tab_label = _("Log and other");

	for (int i = 0; i < N_CONFIG_PAGES; i++) {
		/* we need to know lengths of labels to create tabs
		   of proper size */
		c_pages[i].tab_label_len = strlen(c_pages[i].tab_label);
	}

	return CDW_OK;
}





void cdw_config_ui_window_destroy(void)
{
	/* DON'T free c_pages[].tab_label strings, they are "allocated" (?)
	   by gettext() */

	cdw_dropdown_delete(&speed_range_dropdown);
	cdw_assert (speed_range_dropdown == (CDW_DROPDOWN *) NULL, "delete() didn't set pointer to NULL\n");
	cdw_dropdown_delete(&iso_level_dropdown);
	cdw_assert (iso_level_dropdown == (CDW_DROPDOWN *) NULL, "delete() didn't set pointer to NULL\n");
	cdw_dropdown_delete(&volume_size_dropdown);
	cdw_assert (volume_size_dropdown == (CDW_DROPDOWN *) NULL, "delete() didn't set pointer to NULL\n");

	cdw_dropdown_delete(&ext_tools_dropdowns.cdrecord);
	cdw_assert (ext_tools_dropdowns.cdrecord == (CDW_DROPDOWN *) NULL, "delete() didn't set pointer to NULL\n");
	cdw_dropdown_delete(&ext_tools_dropdowns.mkisofs);
	cdw_assert (ext_tools_dropdowns.mkisofs == (CDW_DROPDOWN *) NULL, "delete() didn't set pointer to NULL\n");
	cdw_dropdown_delete(&ext_tools_dropdowns.growisofs);
	cdw_assert (ext_tools_dropdowns.growisofs == (CDW_DROPDOWN *) NULL, "delete() didn't set pointer to NULL\n");
	cdw_dropdown_delete(&ext_tools_dropdowns.dvd_rw_mediainfo);
	cdw_assert (ext_tools_dropdowns.dvd_rw_mediainfo == (CDW_DROPDOWN *) NULL, "delete() didn't set pointer to NULL\n");
	cdw_dropdown_delete(&ext_tools_dropdowns.dvd_rw_format);
	cdw_assert (ext_tools_dropdowns.dvd_rw_format == (CDW_DROPDOWN *) NULL, "delete() didn't set pointer to NULL\n");
	cdw_dropdown_delete(&ext_tools_dropdowns.dvd_family);
	cdw_assert (ext_tools_dropdowns.dvd_family == (CDW_DROPDOWN *) NULL, "delete() didn't set pointer to NULL\n");


	cdw_button_delete(&iso_image_path_button);
	cdw_assert (iso_image_path_button == (CDW_BUTTON *) NULL, "delete() didn't set pointer to NULL\n");
	cdw_button_delete(&audio_dir_button);
	cdw_assert (audio_dir_button == (CDW_BUTTON *) NULL, "delete() didn't set pointer to NULL\n");
	cdw_button_delete(&log_path_button);
	cdw_assert (log_path_button == (CDW_BUTTON *) NULL, "delete() didn't set pointer to NULL\n");

	if (page_B_txt_subwin != (WINDOW *) NULL) {
		delwin(page_B_txt_subwin);
		page_B_txt_subwin = (WINDOW *) NULL;
	}

	if (page_E_txt_subwin != (WINDOW *) NULL) {
		delwin(page_E_txt_subwin);
		page_E_txt_subwin = (WINDOW *) NULL;
	}

	for (int i = 0; i < N_CONFIG_PAGES; i++) {

		if (c_pages[i].cdw_form != (cdw_form_t *) NULL) {
			cdw_form_delete_form_objects(c_pages[i].cdw_form);
		}

		if (c_pages[i].panel != (PANEL *) NULL) {
			del_panel(c_pages[i].panel);
			c_pages[i].panel = (PANEL *) NULL;
		}

		if (c_pages[i].cdw_form->subwindow != (WINDOW *) NULL) {
			delwin(c_pages[i].cdw_form->subwindow);
			c_pages[i].cdw_form->subwindow = (WINDOW *) NULL;
		}

		if (c_pages[i].cdw_form->window != (WINDOW *) NULL) {
			delwin(c_pages[i].cdw_form->window);
			c_pages[i].cdw_form->window = (WINDOW *) NULL;
		}

		if (c_pages[i].cdw_form != (cdw_form_t *) NULL) {
			cdw_form_delete(&(c_pages[i].cdw_form));
		}
	}

	return;
}





/**
 * \brief Draw tabs that are visible on right side of config window
 *
 * Function draws tabs (with labels) visible on right side of window,
 * that make it easier to navigate configuration window.
 *
 * \param pi - page index, number of tab (0-based) that should be drawn as "active"
 */
void cdw_config_ui_draw_tabs(int pi)
{
	WINDOW *window = c_pages[pi].cdw_form->subwindow;
	cdw_assert (pi >= 0 && pi < N_CONFIG_PAGES, "incorrect number of tabs");

	int start_col = CONF_WIDTH - TABS_WIDTH;
	mvwvline(window, 0, start_col, ACS_VLINE, CONF_HEIGHT);

	/* draw horizontal lines and tab labels */
	for (int i = 0; i < 6; i++) {
		if (! c_pages[i].page_visible) {
			/* "disable" tab X; if "tools page visible" == false
			   user can't enter the tab, but still can see it; let's
			   at least erase tab's label to indicate that tab X
			   can't be visited */
			mvwhline(window, i * 2, start_col + 1, '-', TABS_WIDTH - 2);
		} else {
			mvwprintw(window, 2 * i, start_col + 1, "<F%1d>%s",
				  CDW_CONFIG_FX_START_KEY + i, c_pages[i].tab_label);
		}
		mvwhline(window, 2 * i + 1, start_col + 1, ACS_HLINE, CONF_HEIGHT);
	}

	/* draw of corners, specific for values of tab_number */
	if (pi == 0) {
		mvwaddch(window, 0, start_col, ' ');
		mvwaddch(window, 1, start_col, ACS_ULCORNER);
	} else {
		mvwaddch(window, (pi * 2) - 1, start_col, ACS_LLCORNER);
		mvwaddch(window, (pi * 2), start_col, ' ');
		mvwaddch(window, (pi * 2) + 1, start_col, ACS_ULCORNER);
	}

	wrefresh(window);

	return;
}




/**
 * \brief Check which dropdown has currently keyboard focus, and call ENTER handler for this dropdown
 *
 * Function checks which dropdown (identified by page index \p pi and
 * field index \p fi) has currently keyboard focus and a proper function
 * (cdw_dropdown_driver()) is called for this dropdown.
 * When cdw_dropdown_driver() returns, some other functions are called
 * to clean things up.
 *
 * Call this function only after checking (using is_dropdown_field_index())
 * that current field is related with dropdown. Otherwise the function will
 * call assert(0).
 *
 * \param pi - page index, index of page that is currently visible
 * \param fi - field index, index of field that has keyboard focus
 */
void cdw_config_ui_page_u_driver_handle_enter(cdw_form_t *cdw_form, int fi, void *data)
{
	cdw_config_t *config = (cdw_config_t *) data;
	if (cdw_form->is_button_field_index(cdw_form, fi)) {
		/* there is some function associated with a button;
		   call the function */
		if (cdw_form->form_id == PAGE_D_INDEX && fi == f_iso_image_full_path_b) {
			page_d_handle_enter_on_iso_path_button(config);
		} else if (cdw_form->form_id == PAGE_C_INDEX && fi == f_audiodir_b) {
			page_c_handle_enter_on_audiodir_button(config);
		} else if (cdw_form->form_id == PAGE_F_INDEX && fi == f_log_fp_b) {
			page_f_handle_enter_on_log_path_button(config);
		} else {
			;
		}
	} else if (cdw_form->is_dropdown_field_index(cdw_form, fi)) {
		/* "Enter" on dropdown means "expand dropdown"; call
		   dropdown driver, and after that - if needed - update
		   label displayed in corresponding field of form */
		CDW_DROPDOWN *dropdown = cdw_form->get_dropdown_from_form(cdw_form, fi);
		if (dropdown != (CDW_DROPDOWN *) NULL) {
			cdw_rv_t crv = cdw_dropdown_driver(dropdown);
			if (crv == CDW_OK) {
				const char *label = cdw_dropdown_get_current_item_label(dropdown);
				FIELD **fields = form_fields(cdw_form->form);
				/* setting buffer makes field to display
				   string, but slightly moved to left */
				set_field_buffer(*(fields + fi), 0, label);

				/* call to cdw_dropdown_focus() fixes it */
				cdw_dropdown_focus(dropdown);

				form_driver(cdw_form->form, REQ_VALIDATION);
			} /* else field buffer is not updated and nothing changes */
		} else {
			cdw_assert (0, "ERROR: dropdown in field #%d is NULL\n", fi);
		}
	} else {
		;
	}

	return;
}





/**
   \brief Toggle state of checkbox

   Toggle state of checkbox that is placed on page \p pi with field
   index \p fi. Make sure that this is really a checkbox before calling
   the function.

   Additionally the function shows/hides widgets in "tools" page as
   state of related checkbox changes.

   \param pi - page index, index of page on which the checkbox is placed
   \param fi - field index, index of field of form, that is related to the checkbox
*/
void cdw_config_ui_page_e_driver_toggle_checkbox(cdw_form_t *cdw_form, int fi)
{
	cdw_assert (cdw_form->field_widget_types[fi] == CDW_WIDGET_CHECKBOX, "ERROR: called the function for non-checkbox\n");
	cdw_assert (cdw_form->form_id == PAGE_E_INDEX, "ERROR: this function should be connected to page E\n");

	if (cdw_form_get_field_bit(cdw_form->fields[fi])) {
		set_field_buffer(cdw_form->fields[fi], 0, " ");
	} else {
		set_field_buffer(cdw_form->fields[fi], 0, "X");
	}
	if (cdw_form->form_id == PAGE_E_INDEX && fi == f_manual_selection_cb) {
		cdw_config_ui_page_e_driver_toggle_manual_selection();
	}

	return;
}




void cdw_config_ui_page_e_driver_toggle_manual_selection(void)
{
	cdw_form_t *cdw_form = c_pages[PAGE_E_INDEX].cdw_form;
	bool ms = cdw_form_get_field_bit(*(cdw_form->fields + f_manual_selection_cb));

	int (* field_opts_toggle)(FIELD *, Field_Options) = ms ? field_opts_on : field_opts_off;
	void (* dd_toggle)(CDW_DROPDOWN *) = ms ? cdw_dropdown_make_visible : cdw_dropdown_make_invisible;

	/* "manual selection" input field is the last field that should
	   be always visible, so don't toggle it here;
	   hence "i = f_manual_selection_cb + 1" in for () loops */
	for (int i = f_manual_selection_cb + 1; i < cdw_form->n_fields; i++) {
		field_opts_toggle(*(cdw_form->fields + i), O_VISIBLE);
	}

	dd_toggle(ext_tools_dropdowns.dvd_family);
	dd_toggle(ext_tools_dropdowns.mkisofs);
	dd_toggle(ext_tools_dropdowns.cdrecord);
	dd_toggle(ext_tools_dropdowns.growisofs);
	dd_toggle(ext_tools_dropdowns.dvd_rw_mediainfo);
	dd_toggle(ext_tools_dropdowns.dvd_rw_format);

	cdw_config_ui_refresh_page_windows(PAGE_E_INDEX);
	form_driver(cdw_form->form, REQ_FIRST_FIELD);

	return;
}





/* function called when user pressed Enter key on button next
   to ISO file path, so we need to call file picker, and after closing
   the picker we need to update content of "iso file path" field if necessary */
void page_d_handle_enter_on_iso_path_button(cdw_config_t *config)
{
	/* 2TRANS: this is title of window with file picker */
	cdw_rv_t crv = cdw_fs_ui_file_picker(_("ISO file"),
					     /* 2TRANS: this is message in file picker
						window, explaining what file should be
						selected */
					     _("Select ISO file"),
					     &(config->iso_image_full_path),
					     CDW_FS_FILE,
					     R_OK | W_OK,
					     CDW_FS_NEW | CDW_FS_EXISTING);
	if (crv == CDW_OK) {
		int rv = set_field_buffer(c_pages[PAGE_D_INDEX].cdw_form->fields[f_iso_image_full_path_i], 0, config->iso_image_full_path);
		if (rv != E_OK) {
			cdw_vdm ("ERROR: failed to set field buffer with string = \"%s\"\n", config->iso_image_full_path);
		} else {
			form_driver(c_pages[PAGE_D_INDEX].cdw_form->form, REQ_END_LINE);
		}
	}

	cdw_main_ui_main_window_wrefresh();
	cdw_config_ui_refresh_page_windows(PAGE_D_INDEX);

	return;
}




#if 0
/* function called when user pressed Enter key on button next
   to boot image file path, so we need to call file picker, and after closing
   the picker we need to update content of "boot file path" field if necessary */
void page_d_handle_enter_on_boot_path_button(cdw_config_t *config)
{
	/* 2TRANS: this is title of window with file picker */
	cdw_rv_t crv = cdw_fs_ui_file_picker(_("boot image file"),
					     /* 2TRANS: this is message in file picker
						window, explaining what file should be
						selected */
					     _("Select boot image file"),
					     &(config->boot_image_path),
					     CDW_FS_FILE,
					     R_OK | W_OK,
					     CDW_FS_NEW | CDW_FS_EXISTING);
	if (crv == CDW_OK) {
		int rv = set_field_buffer(c_pages[PAGE_D_INDEX].cdw_form->fields[f_boot_image_path_i], 0, config->boot_image_path);
		if (rv != E_OK) {
			cdw_vdm ("ERROR: failed to set field buffer with string = \"%s\"\n", config->boot_image_path);
		} else {
			form_driver(c_pages[PAGE_D_INDEX].cdw_form->form, REQ_END_LINE);
		}
	}

	cdw_main_ui_main_window_wrefresh();
	cdw_config_ui_refresh_page_windows(PAGE_D_INDEX);

	return;
}
#endif




void page_f_handle_enter_on_log_path_button(cdw_config_t *config)
{
	/* 2TRANS: this is title of window with file picker */
	cdw_rv_t crv = cdw_fs_ui_file_picker(_("Log file"),
					     /* 2TRANS: this is message in file picker
						window, explaining what file should be
						selected */
					     _("Select new or existing log file"),
					     &(config->log_full_path),
					     CDW_FS_FILE,
					     R_OK | W_OK,
					     CDW_FS_NEW | CDW_FS_EXISTING);
	if (crv == CDW_OK) {
		int rv = set_field_buffer(c_pages[PAGE_F_INDEX].cdw_form->fields[f_log_fp_i], 0, config->log_full_path);
		if (rv != E_OK) {
			cdw_vdm ("ERROR: failed to set field buffer with string = \"%s\"\n", config->log_full_path);
		} else {
			form_driver(c_pages[PAGE_F_INDEX].cdw_form->form, REQ_END_LINE);
		}
	}

	cdw_main_ui_main_window_wrefresh();
	cdw_config_ui_refresh_page_windows(PAGE_F_INDEX);

	return;
}





void page_c_handle_enter_on_audiodir_button(cdw_config_t *config)
{
	/* 2TRANS: this is title of window with file picker */
	cdw_rv_t crv = cdw_fs_ui_file_picker(_("Audio output dir"),
					     /* 2TRANS: this is message in file picker
						window, explaining what file should be
						selected */
					     _("Select existing output dir"),
					     &(config->audiodir),
					     CDW_FS_DIR,
					     R_OK | W_OK | X_OK,
					     CDW_FS_EXISTING);
	if (crv == CDW_OK) {
		int rv = set_field_buffer(c_pages[PAGE_C_INDEX].cdw_form->fields[f_audiodir_i], 0, config->audiodir);
		if (rv != E_OK) {
			cdw_vdm ("ERROR: failed to set field buffer with string = \"%s\"\n", config->audiodir);
		} else {
			form_driver(c_pages[PAGE_C_INDEX].cdw_form->form, REQ_END_LINE);
		}
	}

	cdw_main_ui_main_window_wrefresh();
	cdw_config_ui_refresh_page_windows(PAGE_C_INDEX);

	return;
}





void cdw_config_ui_refresh_page_windows(int pi)
{
	redrawwin(c_pages[pi].cdw_form->subwindow);
	wrefresh(c_pages[pi].cdw_form->subwindow);
	redrawwin(c_pages[pi].cdw_form->window);
	wrefresh(c_pages[pi].cdw_form->window);

	return;
}





CDW_DROPDOWN *cdw_config_ui_volume_size_dropdown(WINDOW *window, int begin_y, int begin_x, int width, cdw_config_t *config)
{
	CDW_DROPDOWN *dropdown = cdw_dropdown_new(window, begin_y, begin_x, width, CDW_CONFIG_VOLUME_ITEMS_MAX, CDW_COLORS_DIALOG);
	if (dropdown == (CDW_DROPDOWN *) NULL) {
		cdw_vdm ("ERROR: failed to create new dropdown\n");
		return (CDW_DROPDOWN *) NULL;
	}

	cdw_id_label_t *items = cdw_config_volume_size_items;

	for (int i = 0; i < CDW_CONFIG_VOLUME_ITEMS_MAX; i++) {
		if (items[i].id == CDW_CONFIG_VOLUME_SIZE_DVD_RP_DL
		    && !config->support_dvd_rp_dl) {
			/* skipping this item won't introduce empty
			   item in dropdown */
			continue;
		}
		cdw_rv_t crv = cdw_dropdown_add_item(dropdown, (int) items[i].id, _(items[i].label));
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to add item #%d (id = %ld, label = \"%s\"\n",
				 i, items[i].id, items[i].label);
			cdw_dropdown_delete(&dropdown);
			return (CDW_DROPDOWN *) NULL;
		}
	}

	cdw_rv_t crv = cdw_dropdown_finalize(dropdown);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to finalize dropdown\n");
		cdw_dropdown_delete(&dropdown);
		return (CDW_DROPDOWN *) NULL;
	}

	if (config->volume_size_id == CDW_CONFIG_VOLUME_SIZE_DVD_RP_DL
	    && !config->support_dvd_rp_dl) {
		/* this may happen when value saved in config file is
		   "DVD+R DL" (because previously cdw was called with
		   enabled DVD+R DL support, and user has selected DVD+R DL
		   in this dropdown, but this time cdw is called with support
		   for DVD+R DL disabled; in such case try with CD-R */
		config->volume_size_id = CDW_CONFIG_VOLUME_SIZE_CD74;
	}

	crv = cdw_dropdown_set_current_item_by_id(dropdown, config->volume_size_id);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set current item in dropdown\n");
		cdw_dropdown_delete(&dropdown);
		return (CDW_DROPDOWN *) NULL;
	}

	return dropdown;
}





CDW_DROPDOWN *cdw_config_ui_iso_level_dropdown(WINDOW *window, int begin_y, int begin_x, int width, cdw_config_t *config)
{
#define N_ISO_LEVEL_ITEMS_MAX 4 /* iso level = 1, 2, 3, 4; id == level */
	CDW_DROPDOWN *dropdown = cdw_dropdown_new(window, begin_y, begin_x, width, N_ISO_LEVEL_ITEMS_MAX, CDW_COLORS_DIALOG);
	if (dropdown == (CDW_DROPDOWN *) NULL) {
		cdw_vdm ("ERROR: failed to create new dropdown\n");
		return (CDW_DROPDOWN *) NULL;
	}

	cdw_dropdown_item_t items[N_ISO_LEVEL_ITEMS_MAX] = {
		/* 2TRANS: do not translate */
		{ 1,       _("1") },
		/* 2TRANS: do not translate */
		{ 2,       _("2") },
		/* 2TRANS: do not translate */
		{ 3,       _("3") },
		/* 2TRANS: do not translate */
		{ 4,       _("4") } };

	for (int i = 0; i < N_ISO_LEVEL_ITEMS_MAX; i++) {
		cdw_rv_t crv = cdw_dropdown_add_item(dropdown, items[i].id, items[i].label);
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to add item #%d (id = %d, label = \"%s\"\n",
				 i, items[i].id, items[i].label);
			cdw_dropdown_delete(&dropdown);
			return (CDW_DROPDOWN *) NULL;
		}
	}

	cdw_rv_t crv = cdw_dropdown_finalize(dropdown);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to finalize dropdown\n");
		cdw_dropdown_delete(&dropdown);
		return (CDW_DROPDOWN *) NULL;
	}

	crv = cdw_dropdown_set_current_item_by_id(dropdown, config->iso_level);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set current item in dropdown\n");
		cdw_dropdown_delete(&dropdown);
		return (CDW_DROPDOWN *) NULL;
	}

	return dropdown;
}





CDW_DROPDOWN *cdw_config_ui_speed_range_dropdown(WINDOW *window, int begin_y, int begin_x, int width, cdw_config_t *config)
{
#define N_SPEED_RANGE_ITEMS_MAX 4 /* auto, lowest, middle, highest */

	CDW_DROPDOWN *dropdown = cdw_dropdown_new(window, begin_y, begin_x, width, N_SPEED_RANGE_ITEMS_MAX, CDW_COLORS_DIALOG);
	if (dropdown == (CDW_DROPDOWN *) NULL) {
		cdw_vdm ("ERROR: failed to create new dropdown\n");
		return (CDW_DROPDOWN *) NULL;
	}

	cdw_dropdown_item_t items[N_SPEED_RANGE_ITEMS_MAX] = {
		/* 2TRANS: it means 'speed of writing set automatically' */
		{ SPEED_RANGE_AUTO,       _("auto") },
		/* 2TRANS: it means 'lowest writing speed available' */
		{ SPEED_RANGE_LOWEST,     _("lowest") },
		/* 2TRANS: it means 'medium speed of writing' */
		{ SPEED_RANGE_MIDDLE,     _("middle") },
		/* 2TRANS: it means 'highest writing speed available' */
		{ SPEED_RANGE_HIGHEST,    _("highest") }};

	for (int i = 0; i < N_SPEED_RANGE_ITEMS_MAX; i++) {
		cdw_rv_t crv = cdw_dropdown_add_item(dropdown, items[i].id, items[i].label);
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to add item #%d (id = %d, label = \"%s\"\n",
				 i, items[i].id, items[i].label);
			cdw_dropdown_delete(&dropdown);
			return (CDW_DROPDOWN *) NULL;
		}
	}

	cdw_rv_t crv = cdw_dropdown_finalize(dropdown);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to finalize dropdown\n");
		cdw_dropdown_delete(&dropdown);
		return (CDW_DROPDOWN *) NULL;
	}

	crv = cdw_dropdown_set_current_item_by_id(dropdown, config->speed_range);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set current item in dropdown\n");
		cdw_dropdown_delete(&dropdown);
		return (CDW_DROPDOWN *) NULL;
	}

	return dropdown;
}



