/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2010 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <stdbool.h>
#include <stddef.h>

#include <ncursesw/ncurses.h>

#include "cdw_config.h"
#include "cdw_config_ui.h"
#include "cdw_config_ui_internals.h"
#include "gettext.h"
#include "cdw_main_window.h"
#include "cdw_widgets.h"
#include "cdw_debug.h"
#include "cdw_ncurses.h"
#include "cdw_string.h"
#include "cdw_file_manager.h"


extern cdw_config_t global_config;

/* is set to true if something went wrong during module setup -
   cdw works without reading/writing configuration to disk file */
extern bool failsafe_mode;

static cdw_rv_t cdw_config_ui_edit_volume_id(void);



/**
 * \brief Top level function for configuration window
 *
 * \return CDW_GEN_ERROR
 * \return CDW_OK
 */
cdw_rv_t cdw_config_ui_window(bool tools_page_visible)
{
	bool follow_symlinks = global_config.follow_symlinks;

	cdw_config_t tmp_config;
	cdw_config_var_init_fields(&tmp_config);
	/* DON'T read configuration from disk file to config variable:
	   this is done already in cdw_config_module_init(); and between
	   calling cdw_config_module_init() and opening configuration
	   window user could modify configuration variable in other
	   places; reading config from file again could overwrite user's
	   changes  */
	/**** read_conf(); ****/

	/* copy current option values from main config variable to
	   temporary config variable and use the temporary variable to
	   initialize option fields. */
	cdw_config_var_copy(&tmp_config, &global_config);
	cdw_config_debug_print_config(&tmp_config);

	cdw_rv_t crv = cdw_config_ui_window_create(&tmp_config, tools_page_visible);
	if (crv != CDW_OK) {
		/* no need to call cdw_config_ui_window_destroy(),
		   create() function should do necessary clean up */
		cdw_vdm ("ERROR: failed to create config ui window\n");
		cdw_config_var_free_fields(&tmp_config);
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("Failed to create configuration window"),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return CDW_GEN_ERROR;
	}
	cdw_main_ui_main_window_wrefresh();
	crv = cdw_config_ui_window_pages_driver(&tmp_config);
	if (crv == CDW_OK) { /* "save changes and exit from cdw config window" */
		cdw_config_var_copy(&global_config, &tmp_config);
		cdw_config_debug_print_config(&global_config);

		if (failsafe_mode) {
			/* cannot work with file system because of previous
			   errors detected in cdw_conifg.c */
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Config file error"),
					   /* 2TRANS: this is message in dialog window */
					   _("Cannot write to config file. Configuration will not be saved to file."),
					   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

			/* cannot save changed values to file, but they were
			   saved to config variable and will be used only in
			   this run of cdw */
		} else {
			/* write current configuration (from config) to disk file */
			cdw_config_write_to_file();
		}
	} else { /* crv == CDW_CANCEL */
		; /* current values in config are without changes, they aren't
		     replaced by those from tmp_config */
	}

	cdw_config_var_free_fields(&tmp_config);
	cdw_config_ui_window_destroy();
	if (follow_symlinks != global_config.follow_symlinks) {
		/* user changed the way how symlinks should be treated,
		   and this may affect size of selected files */
		cdw_vdm ("INFO: changed \"follow symlinks\" to %s, recalculating files size\n",
			 global_config.follow_symlinks ? "true" : "false");
		cdw_main_ui_files_info_view_update(true);
		cdw_file_manager_regenerate_selected_files_view();
	} else {
		cdw_vdm ("INFO: \"follow symlinks\" is stil %s\n", global_config.follow_symlinks ? "true" : "false");
		cdw_main_ui_files_info_view_update(false);
	}

	cdw_main_ui_main_window_wrefresh();

	return CDW_OK;
}






/**
 * \brief Display dialog window in which user can edit string used as volume ID
 *
 * Volume ID is a label that can be seen e.g. next to disc icon
 * in file managers. Length of this label is limited to 32 chars.
 */
cdw_rv_t cdw_config_ui_edit_volume_id(void)
{
	cdw_safe_input_data_t data;

	/* how many attempts? */
	data.attempts_max = 5;
	/* title of input window */
	/* 2TRANS: this is title of dialog window */
	data.window_title = _("Volume ID");
	/* message in input window */
	/* 2TRANS: this is message in dialog window, below is input field
	   for user to enter label for optical disc or ISO image */
	data.prompt_message = _("Enter volume label:");
	data.input_type = CDW_NCURSES_INPUT_NONE;
	/* maximal length of string */
        data.chars_max = VOLUME_ID_LEN_MAX;
	data.buffer = &(global_config.volumeid);
	/* error message window displayed at the end in case of failure */
	/* 2TRANS: this is message in dialog window
	   "volume id" is a name/label of ISO volume or optical disc */
	data.error_message = _("Could not get correct volume ID.");

	cdw_rv_t crv = cdw_safe_input_dialog(&data);
	if (crv == CDW_OK) {
		cdw_vdm ("INFO: \"%s\" dialog: string saved as \"%s\"\n", data.window_title, global_config.volumeid);
	} else if (crv == CDW_CANCEL) {
		;
	} else {
		cdw_vdm ("ERROR: \"%s\" dialog returns ERROR\n", data.window_title);
	}
	return crv;
}




/**
   \brief Ask user for volume label

   Check if user has to be asked for volume label. Show input dialog if
   this is the case. Save volume label in config variable.

   \return CDW_OK if user entered volume label (or accepted default value) and pressed enter;
   \return CDW_OK if user should not be asked for volume label; (default/stored value should be used)
   \return CDW_CANCEL if user pressed escape when asked for volume label
*/
cdw_rv_t cdw_config_ui_conditional_volume_label_dialog(void)
{
	if (global_config.showvol) { /* ask for disk label */
		cdw_rv_t crv = cdw_config_ui_edit_volume_id();
		cdw_main_ui_main_window_wrefresh();
		return crv; /* CDW_OK or CDW_CANCEL - user decision */
	} else {
		/* use default value or value stored in configuration file */
		return CDW_OK;
	}
}



