/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * A few small modifications (C) 2007 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <pthread.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <ncursesw/ncurses.h>
#include <regex.h>
#include <nl_types.h>
#include <time.h>
#include <libintl.h>
// #include <signal.h> // kill

#include "cdw_widgets.h"
#include "options.h"
#include "gettext.h"
#include "isosize.h"
#include "commands.h"
#include "iso9660.h"
#include "log.h"

#define BUFSIZE 1000
#define IMAGE_BUF 2048

int stdin_pipe[2];
int stdout_pipe[2];
int stderr_pipe[2];
char cmd[200], *window;
WINDOW *processwin;
extern WINDOW *mainwin; // main application window

int total_size = 0;
pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;
static FILE *logf;
struct conf config;
int size, lasttrack = 0;
time_t time0;
time_t time1;
int prev_percent = 0;

extern media_erasable;


// small helper functions
void update_processwin_eta(WINDOW *processwin, int eta);
void eta_calculations(int eta, char *seta_hour, char *seta_min, char *seta_sec);
void draw_progress_bar(int value, int color1, int color2, char c);


// functions used for output handling in print_stdout
void handle_image_writing(regex_t *regex, regmatch_t *matches);
void handle_direct_writing(regex_t *regex, regmatch_t *matches);
void handle_fixating(void);
void handle_is_erasable(regex_t *regex, regmatch_t *matches);
void handle_blanking_not_supported(regex_t *regex, regmatch_t *matches);


// functions used for output handling in print_stderr
void handle_audio_grab(char *c5);
void handle_image_creation(regex_t *regex, regmatch_t *matches);
void handle_image_creation_100percent(regex_t *regex, regmatch_t *matches);



char buf[BUFSIZE], buffer[BUFSIZE + 1];
char buf_err[BUFSIZE], buffer_err[BUFSIZE + 1];




/*
 * print_stdout and print_stderr are two functions reading from
 * stdout (stdout_pipe[0]) and stderr (stderr_pipe[0])
 * of other process - this process creates image, writes disc or sth else.
 * Purpose of this two functions is to intercept output of this process, extract
 * useful information about state and progress (using regular expressions) and
 * print this information in UI window. This interception is done by reading from
 * sockets that are connected to stdout and stderr of process. Creation of process
 * and connection of sockets is done in run_command().
 *
 */


// I see CDrecorder version 0.1 source, and I write this code after this!!!
// THX to Sharad Mittal
void *print_stdout(void *dummy)
{
	regex_t *regex = NULL, *regex2 = NULL, *regex3 = NULL, *regex4 = NULL, *regex5 = NULL;
	regmatch_t *matches = NULL, *matches2 = NULL, *matches3 = NULL, *matches4 = NULL, *matches5 = NULL;
	int rv;
	int data_processed;

	// regular expressions setup code

	// Image writing...
	regex = (regex_t *) calloc(1, sizeof(regex_t));
	rv = regcomp(regex, "Track ([0-9]+): ([ ]*)([0-9]+) of ([ ]*)([0-9]+) MB written ([(])fifo([ ]*)([0-9]+)", REG_EXTENDED);
	matches = (regmatch_t *) calloc(1, (regex->re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		regerror(rv, NULL, buf, sizeof(buf));
		fprintf(stderr, _("ERROR regcomp: %s\n"), buf);
		exit(1);
	}
	// Direct writing...
	regex3 = (regex_t *) calloc(1, sizeof(regex_t));
	rv = regcomp(regex3, "Track ([0-9]+): ([ ]*)([0-9]+) MB written ([(])fifo([ ]*)([0-9]+)", REG_EXTENDED);
	matches3 = (regmatch_t *) calloc(1, (regex3->re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		regerror(rv, NULL, buf, sizeof(buf));
		fprintf(stderr, _("ERROR regcomp: %s\n"), buf);
		exit(1);
	}
	// Fixating...
	regex2 = (regex_t *) calloc(1, sizeof(regex_t));
	rv = regcomp(regex2, _("Fixating..."), REG_EXTENDED);
	matches2 = (regmatch_t *) calloc(1, (regex2->re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		regerror(rv, NULL, buf, sizeof(buf));
		fprintf(stderr, _("ERROR regcomp: %s\n"), buf);
		exit(1);
	}
	// Is erasable...
	regex4 = (regex_t *) calloc(1, sizeof(regex_t));
	rv = regcomp(regex4, _("Is erasable"), REG_EXTENDED);
	matches4 = (regmatch_t *) calloc(1, (regex4->re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		regerror(rv, NULL, buf, sizeof(buf));
		fprintf(stderr, _("ERROR regcomp: %s\n"), buf);
		exit(1);
	}
	// this media does not support blanking
	regex5 = (regex_t *) calloc(1, sizeof(regex_t));
	rv = regcomp(regex5, _("this media does not support blanking"), REG_EXTENDED);
	matches5 = (regmatch_t *) calloc(1, (regex5->re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		regerror(rv, NULL, buf, sizeof(buf));
		fprintf(stderr, _("ERROR regcomp: %s\n"), buf);
		exit(1);
	}


	// regular expressions prepared, let's get some data

	while (1) {
		memset(buffer, '\0', sizeof(buffer));
		// read console output of some child process
		data_processed = read(stdout_pipe[0], buffer, BUFSIZE);
		pthread_mutex_lock(&mutex);
		if (data_processed != -1) {
			int rv1, rv2, rv3, rv4, rv5, rv6, rv7;
			fprintf(logf, buffer);
			fflush(logf);

			// case 1: Image writing...
			rv1 = regexec(regex, buffer, regex->re_nsub + 1, matches, 0);
			if (rv1 == 0) {
				// image writing
				handle_image_writing(regex, matches);
			}

			// Case 3: Direct writing
			rv3 = regexec(regex3, buffer, regex3->re_nsub + 1, matches3, 0);
			if (rv3 == 0) {
				// direct writing
				handle_direct_writing(regex3, matches3);
			}

			// Case 2: Fixating
			rv2 = regexec(regex2, buffer, regex2->re_nsub, matches2, 0);
			if ((rv2 == 0) && (rv1 != 0)) {
				// fixating
				handle_fixating();
			}

			// Case 4: Is erasable
			rv4 = regexec(regex4, buffer, regex4->re_nsub + 1, matches4, 0);
			if (rv4 == 0) {
				// media is erasable
				handle_is_erasable(regex4, matches4);
			}

			// Case 5: this media does not support blanking
			rv5 = regexec(regex5, buffer, regex5->re_nsub + 1, matches5, 0);
			if (rv5 == 0) {
				// media is not erasable
				handle_blanking_not_supported(regex5, matches5);
			}
		} else if (data_processed == 0) { // EOF
			break;
		}
		pthread_mutex_unlock(&mutex);
		if (data_processed == 0)
			break;
		usleep(10);

	}
	regfree(regex);
	regfree(regex2);
	regfree(regex3);

	free(regex);
	free(regex2);
	free(regex3);


	free(matches);
	free(matches2);
	free(matches3);

	return NULL;
}



/*
 * please see comments before print_stdout()
 */
void *print_stderr(void *dummy)
{
	int data_processed_err, rv;
	regex_t *regex, *regex2, *regex3, *regex4;
	regmatch_t *matches, *matches2, *matches3, *matches4;

	// Mkisofs process during image cration (0 - 99,9%)
	regex = (regex_t *) calloc(1, sizeof(regex_t));
	rv = regcomp(regex, "([ ]*)([0-9]+).([0-9]+)% done,", REG_EXTENDED);
	matches = (regmatch_t *) calloc(1, (regex->re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		regerror(rv, NULL, buf_err, sizeof(buf_err));
		fprintf(stderr, _("ERROR regcomp: %s\n"), buf_err);
		exit(1);
	}

	// Mkisofs 100% (image created)
	regex2 = (regex_t *) calloc(1, sizeof(regex_t));
	rv = regcomp(regex2, "Total translation table size", REG_EXTENDED);
	matches2 = (regmatch_t *) calloc(1, (regex2->re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		regerror(rv, NULL, buf_err, sizeof(buf_err));
		fprintf(stderr, _("ERROR regcomp: %s\n"), buf_err);
		exit(1);
	}
	// cannot write medium - incompatible format
	regex3 = (regex_t *) calloc(1, sizeof(regex_t));
	rv = regcomp(regex3, _("cannot write medium - incompatible format"), REG_EXTENDED);
	matches3 = (regmatch_t *) calloc(1, (regex3->re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		regerror(rv, NULL, buf, sizeof(buf));
		fprintf(stderr, _("ERROR regcomp: %s\n"), buf);
		exit(1);
	}
	// Cannot blank disk
	regex4 = (regex_t *) calloc(1, sizeof(regex_t));
	rv = regcomp(regex4, _("Cannot blank disk"), REG_EXTENDED);
	matches4 = (regmatch_t *) calloc(1, (regex4->re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		regerror(rv, NULL, buf, sizeof(buf));
		fprintf(stderr, _("ERROR regcomp: %s\n"), buf);
		exit(1);
	}

	while (1) {
		memset(buffer_err, '\0', sizeof(buffer_err));
		data_processed_err = read(stderr_pipe[0], buffer_err, BUFSIZE);
		pthread_mutex_lock(&mutex);
		if (data_processed_err != -1) {
			int rv1, rv2, rv3, rv4;

			char c1[25], c2[25], c3[25], c4[25], c5[25], tmp[25];

			fprintf(logf, buffer_err);

			// Cdda2wav process
			// ??/??/??/???????   0%
			if (sscanf(buffer_err, "%s %s %s %s %s %s", c1, c2, c3, c4, c5, tmp) == 5) {
				handle_audio_grab(c5);
			}

			rv1 = regexec(regex, buffer_err, regex->re_nsub + 1, matches, 0);
			if ((rv1 == 0) && (strcmp(window, _("Write direct")) != 0)) {
				handle_image_creation(regex, matches);
			}

			rv2 = regexec(regex2, buffer_err, regex2->re_nsub + 1, matches2, 0);
			if ((rv2 == 0) && (strcmp(window, _("Write direct")) != 0)) {
				handle_image_creation_100percent(regex2, matches2);
			}
			// Case 3: cannot write medium - incompatible format
			rv3 = regexec(regex3, buffer_err, regex3->re_nsub, matches3, 0);
			if (rv3 == 0) {
				// media is not erasable
				handle_blanking_not_supported(regex3, matches3);
			}
			// Case 4: Cannot blank disk
			rv4 = regexec(regex4, buffer_err, regex4->re_nsub, matches4, 0);
			if (rv4 == 0) {
				// media is not erasable
				handle_blanking_not_supported(regex4, matches4);
			}

		} else if (data_processed_err == 0) { // EOF
			break;
		}
		pthread_mutex_unlock(&mutex);
		if (data_processed_err == 0)
			break;
		usleep(10);
/*
	buffer_err[BUFSIZE]='\0';
	usleep(1);*/
	}
	regfree(regex);
	regfree(regex2);

	free(regex);
	free(regex2);

	free(matches);
	free(matches2);

	return NULL;
}






/*
 * Create UI window in which process progress is shown
 *
 * \param int is_prcess
 * \param char *cmd
 */
void create_process_win(int is_process, char *cmd)
{
	processwin = newwin(11, 30, (LINES - 11) / 2, (COLS - 30) / 2);
	wbkgd(processwin, COLOR_PAIR(2));
	werase(processwin);

	/*
	box(processwin, 0, 0);

	/ * process window title bar;
	 * 'window' is string with title * /
	mvwaddch(processwin, 0, 2, ACS_RTEE);
	wprintw(processwin, " %s ", window);
	waddch(processwin, ACS_LTEE);
	*/
	nice_box(processwin, window, "");

	if (is_process != 0)
		box(derwin(processwin, 3, 22, 4, 4), 0, 0);
	else
		mvwprintw(processwin, 5, 15 - (strlen(window) / 2), "%s", _(window));
	mvwprintw(processwin, 2, 15 - (strlen(cmd) / 2), "%s", cmd);
	wrefresh(processwin);

	return;
}



/*
 * Destroy UI window in which process progress is shown
 *
 * This is just simple wrapper
 */
void destroy_process_win(void)
{
	wborder(processwin, ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ');
	wrefresh(processwin);
	delwin(processwin);

	return;
}




int copy_image(char *window_name)
{
	long imagesize, status = 0;
	int cdimage, imagefile, current_done, c;
	void *cpy_buf;
	size_t n;
	char percent[30], bottomstat[30];

	window = window_name;
	create_process_win(1, "");
	if ((cpy_buf = (void *) malloc(IMAGE_BUF)) == NULL) {
		return -1;
	}

	if ((cdimage = open(config.cdrom, O_RDWR | O_NONBLOCK)) == -1) {
		return -1;
	}
	if ((imagefile = open(config.tempdir, O_CREAT | O_WRONLY | O_TRUNC, S_IRWXU)) == -1) {
		return -1;
	}
	imagesize = isosize(cdimage);
	while (status < imagesize) {
		n = read(cdimage, cpy_buf, IMAGE_BUF);

		/* FIXME
		 * Sometimes, when readahead is turned on for CDROM device (hdparm -a /dev/hdx),
		 * read() returns -1 before reaching end of data on disk - it reads 98% or so.
		 * This can be easily detected and proper message should be shown to teh user
		 */
		if (n == 0) {
			break; /* EOF */
		} else if (n == -1) {
			dialogbox(_("CD drive read error"), _("Please check your cdrom settings and try to disable readahead."), 0);
			/// wrefresh(derwin(mainwin,LINES-1,COLS-1,0,0));
			/// select_window(FALSE);
			// wrefresh(mainwin);
			break;
		}
		write(imagefile, cpy_buf, n);
		status += n;
		current_done = status / (imagesize / 99);

		sprintf(bottomstat, _("%ld / %ld MB done"), status / 1024 / 1024, imagesize / 1024 / 1024);
		mvwprintw(processwin, 2, 15 - (strlen(bottomstat) / 2), "%s", bottomstat);
		sprintf(percent, _("%d%% done"), current_done);
		mvwprintw(processwin, 3, 15 - (strlen(percent) / 2), "%s", percent);
		wrefresh(processwin);
		mvwprintw(processwin, 5, 5, "");
		draw_progress_bar( (current_done / 5), 5, 2, ACS_BLOCK);
		mvwprintw(processwin, 9, 29, "");
		wrefresh(processwin);
	}
	close(imagefile);
	close(cdimage);

	// wrefresh(mainwin);
	// wrefresh(processwin);
	nice_box(processwin, window_name, "");
	mvwprintw(processwin, 9, 4, _("     PRESS any key     "));
	wrefresh(processwin);
	c = wgetch(processwin);


	destroy_process_win();

	free(cpy_buf);
	return 0;
}



/*
 * Run command: create sockets to new process and
 * launch given program as new process
 *
 * \param char commnad[] - program name + it's arguments
 * \param char *window_name - string that will appear in process information window
 * \param int dsize
 * \param int wait
 * \param int is_process
 * \param char *msg
 *
 * \returns 1 on success
 */
int run_command(char command[200], char *window_name, int dsize, int wait, int is_process, char *msg)
{

	int c;

	window = window_name;
	size = dsize / 1024 / 1024;
	if ((logf = fopen(config.logfile, "w")) == NULL) {
		endwin();
		fprintf(stderr, _("Cannot create logfile...\n"));
		exit(-1);
	}
	lasttrack = 0;
	//track=0;
	time0 = time(NULL);
	fprintf(logf, _("Time: %d\n"), (int)time0);
	fprintf(logf, "%s\n", command);
	if ((socketpair(AF_UNIX, SOCK_STREAM, 0, stdin_pipe) == 0)
	    && (socketpair(AF_UNIX, SOCK_STREAM, 0, stdout_pipe) == 0)
	    && (socketpair(AF_UNIX, SOCK_STREAM, 0, stderr_pipe) == 0)) {

		int fork_result;

		fcntl(stdout_pipe[1], F_SETFL, O_ASYNC);
		fcntl(stderr_pipe[1], F_SETFL, O_ASYNC);
		fcntl(stderr_pipe[0], F_SETFL, O_NONBLOCK);

		fork_result = fork();

		if (fork_result == -1) {
			fprintf(stderr, _("Fork failure\n"));
			exit(-1);
		} else if (fork_result == 0) { // we are child process
			char *argv[4];
			char *shell = "sh";

			/* we are child process - let's don't write to stdout and stderr;
			 * instead we should write to socked inherited from parent process,
			 * so parent process can know what we are doing and what's going on
			 *
			 * dup() will copy it's argument descriptors to lowest available
			 * descriptors; let's make stdin, stdout and stderr
			 * 'lowest avaliable' by closing them before calling dup();
			 *
			 * let's close original descriptor after copying it:
			 * we have copy, original is no longer needed; we don't need
			 * the other end of pipe here either (that is why we close both
			 * *_pipe[0] and *_pipe[1]), but parent will keep the other end
			 * open to read from it as we write to our new stdout and stderr
			 */
			close(0);
			dup(stdin_pipe[0]);
			close(stdin_pipe[0]);
			close(stdin_pipe[1]);

			close(1);
			dup(stdout_pipe[1]);
			close(stdout_pipe[0]);
			close(stdout_pipe[1]);

			close(2);
			dup(stderr_pipe[1]);
			close(stderr_pipe[0]);
			close(stderr_pipe[1]);

			/* here a copy of cdw turns into e.g. mkisofs */
			strcpy(cmd, command);
			argv[0] = shell;
			argv[1] = "-c";
			argv[2] = cmd;
			argv[3] = NULL;
			execvp(argv[0], argv);

			/* exec*() functions don't return */

			fprintf(stderr, _("Error: return of exec() function\n"));

			exit(0);
		} else { // we are still in parent process
			int rv;
			pthread_t pout, perr;

			/* it is the parent process that intercepts output messages from
			 * mkisofs or cdrecord, so parent process must aso create process
			 * info window, so it can write to it */
			create_process_win(is_process, msg);

			close(stdin_pipe[0]);
			close(stdin_pipe[1]);
			close(stderr_pipe[1]);
			close(stdout_pipe[1]);

			rv = pthread_create(&pout, NULL, &print_stdout, NULL);
			if (rv != 0) {
				fprintf(stderr, _("Thread stdout creation error"));
			}
			rv = pthread_create(&perr, NULL, &print_stderr, NULL);
			if (rv != 0) {
				fprintf(stderr, _("Thread stderr creation error"));
			}

			/* threads are created; now we must wait for them to complete
			 * their job and return (both of them);
			 * when thead functions end we can execute pthread_join()s */
			pthread_join(perr, NULL);
			pthread_join(pout, NULL);

			/* nothing more to read from these two */
			close(stderr_pipe[0]);
			close(stdout_pipe[0]);

			fclose(logf);

			/* say "goodbye" to user */
			mvwprintw(processwin, 3, 2, "                           ");
			mvwprintw(processwin, 3, 15 - (strlen(_("Done")) / 2), _("Done"));
			wrefresh(processwin);
			mvwprintw(processwin, 5, 5, "");

			/* do we need to erase or change progress bar?
			draw_progress_bar(5, 5, 2, ACS_BLOCK); */

			mvwprintw(processwin, 9, 29, "");
			wrefresh(processwin);
			if (wait != 0) {
				mvwprintw(processwin, 10, 4, _("     Press any key     "));
				c = wgetch(processwin);
			}

			destroy_process_win();
			/* erase remainings of process window before
			 * cd tray will open after writing */
			wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
			select_window(FALSE);
		}
	} else {
		exit(-1);
	}

	return 1;
}






/**
 * Calculate estimated time of accomplishment for a task
 *
 * \param int eta - input eta value
 * \param char *seta_hour - outpus string for hours value
 * \param char *seta_min - output string for mins value
 * \param char *seta_sec - output string of seconds value
 *
 */
void eta_calculations(int eta, char *seta_hour, char *seta_min, char *seta_sec)
{

	int eta_hour = (eta / 60) / 60;
	int eta_min = (eta - ((eta_hour * 60) * 60)) / 60;
	int eta_sec = eta - (((eta_hour * 60) * 60) + (eta_min * 60));

	if (eta_hour < 10)
		sprintf(seta_hour, "0%d", eta_hour);
	else
		sprintf(seta_hour, "%d", eta_hour);

	if (eta_min < 10)
		sprintf(seta_min, "0%d", eta_min);
	else
		sprintf(seta_min, "%d", eta_min);

	if (eta_sec < 10)
		sprintf(seta_sec, "0%d", eta_sec);
	else
		sprintf(seta_sec, "%d", eta_sec);


	return;
}





/**
 * Update ETA information in process progress information window
 *
 * \param WINDOW *processwin
 * \param int eta
 *
 * FIXME - processwin is global variable
 */
void update_processwin_eta(WINDOW *processwin, int eta)
{
	char etime[30];
	char seta_hour[3];
	char seta_min[3];
	char seta_sec[3];

	eta_calculations(eta, seta_hour, seta_min, seta_sec);

	mvwprintw(processwin, 9, 1, "                            ");
	sprintf(etime, _("ETA: %s:%s.%s"), seta_hour, seta_min, seta_sec);
	mvwprintw(processwin, 9, 15 - (strlen(etime) / 2), "%s", etime);

	return;
}





void handle_image_writing(regex_t *regex, regmatch_t *matches)
{
	char done[10] = "\0";
	char topstat[30], bottomstat[30];
	char fifostr[30];

	int i;
	for (i = 0; i < (regex->re_nsub + 1); ++i) {
		int len = matches[i].rm_eo - matches[i].rm_so;
		strncpy(buf, (char *) &buffer[matches[i].rm_so], len);
		buf[len] = '\0';
		if ((i == 8)) {
			int fifo = atoi((char *) &buf);
			sprintf(fifostr, _("  Fifo: %d%%  "), fifo);
			if (fifo <= 25)
				wattrset(processwin, COLOR_PAIR(7));
			mvwprintw(processwin, 8, 15 - (strlen(fifostr) / 2), "%s", fifostr);
			wattrset(processwin, COLOR_PAIR(2));
		}
		if ((i == 5)) {
			int new_total = atoi((char *) &buf);
			if (total_size < new_total) {
				total_size = new_total;
			}
		}
		if ((i == 3) && (total_size)) {
			int current_done = atoi((char *) &buf);
			float per_done = ((float) current_done) / ((float) total_size); // KAMIL
			if ((per_done >= 0.0) && (per_done <= 1.0)) {
				if (prev_percent != per_done) {
					int eta = (difftime(time(NULL), time0) / current_done) * (total_size - current_done);
					update_processwin_eta(processwin, eta);
					prev_percent = per_done;
				}
				sprintf(bottomstat, _("%d / %d MB done"), current_done, total_size);
				mvwprintw(processwin, 2, 15 - (strlen(bottomstat) / 2), "%s", bottomstat);
				sprintf(topstat, _("%2.1f%% done"), per_done * 100);
				mvwprintw(processwin, 3, 15 - (strlen(topstat) / 2), "%s", topstat);
				mvwprintw(processwin, 5, 5, "");

				draw_progress_bar( ((per_done * 99) / 5), 5, 2, ACS_BLOCK);

				mvwprintw(processwin, 9, 29, "");
				wrefresh(processwin);
			}
		}
		if (i == 3) {
			int current_done = atoi((char *) &buf);
			sprintf(done, _("%d MB"), current_done);
		}
		if ((i == 1)) {
			int tracknum = atoi((char *) &buf);
			if (tracknum != lasttrack) {
				mvwprintw(processwin, 5, 5, "");
				draw_progress_bar(19, 2, 2, ' ');
			}
			sprintf(bottomstat, _("Track: %d"), tracknum);
			mvwprintw(processwin, 7, 15 - (strlen(bottomstat) / 2), "%s", bottomstat);
			wrefresh(processwin);
			lasttrack = tracknum;
		}
	}

	return;
}





void handle_direct_writing(regex_t *regex, regmatch_t *matches)
{
	// direct writing
	char topstat[30], bottomstat[30];
	char fifostr[30];
	int i;
	for (i = 0; i < (regex->re_nsub + 1); ++i) {
		int len = matches[i].rm_eo - matches[i].rm_so;
		strncpy(buf, (char *) &buffer[matches[i].rm_so], len);
		buf[len] = '\0';
		if (i == 6) {
			int fifo = atoi((char *) &buf);
			sprintf(fifostr, _("  Fifo: %d%%  "), fifo);
			if (fifo <= 25) {
				wattrset(processwin, COLOR_PAIR(7));
			}

			mvwprintw(processwin, 8, 15 - (strlen(fifostr) / 2), "%s", fifostr);
			wattrset(processwin, COLOR_PAIR(2));

		}
		if ((i == 3) && (size)) {
			int current_done = atoi((char *) &buf);
			float per_done = ((float) current_done) / ((float) size);
			if ((per_done >= 0.0) && (per_done <= 1.0)) {
				if (prev_percent != per_done) {
					int eta = (difftime(time(NULL), time0) / current_done) * (size - current_done);
					update_processwin_eta(processwin, eta);

					prev_percent = per_done;
				}
				sprintf(bottomstat, _("%d / %d MB done"), current_done, size);
				mvwprintw(processwin, 2, 15 - (strlen(bottomstat) / 2), "%s", bottomstat);
				sprintf(topstat, _("%2.1f%% done"), per_done * 100);
				mvwprintw(processwin, 3, 15 - (strlen(topstat) / 2), "%s", topstat);
				mvwprintw(processwin, 5, 5, "");

				draw_progress_bar( ((per_done * 99) / 5), 5, 2, ACS_BLOCK);

				mvwprintw(processwin, 9, 29, "");
				wrefresh(processwin);
			}
		}
		if ((i == 3) && (!size)) {
			int current_done = atoi((char *) &buf);
			sprintf(bottomstat, _("%d MB done"), current_done);
			mvwprintw(processwin, 3, 15 - (strlen(bottomstat) / 2), "%s", bottomstat);
			wrefresh(processwin);
		}
		if ((i == 1)) {
			int tracknum = atoi((char *) &buf);
			if (tracknum != lasttrack) {
				mvwprintw(processwin, 5, 5, "");
				draw_progress_bar(19, 2, 2, ACS_BLOCK);
			}
			sprintf(bottomstat, _("Track: %d"), tracknum);
			mvwprintw(processwin, 7, 15 - (strlen(bottomstat) / 2), "%s", bottomstat);
			wrefresh(processwin);
			lasttrack = tracknum;
		}
	}

	return;
}





void handle_fixating(void)
{
	// fixating
	char topstat[30];
	mvwprintw(processwin, 2, 1, "                            ");
	sprintf(topstat, _("Fixating..."));
	mvwprintw(processwin, 3, 15 - (strlen(topstat) / 2), "%s", topstat);
	mvwprintw(processwin, 5, 5, "");

	draw_progress_bar(19, 5, 2, ACS_BLOCK);

	wrefresh(processwin);

	return;
}





void handle_audio_grab(char *c5)
{
	// audio_grab
	int perc = atoi(c5);
	char percent[30];

	if ((strcmp(window, _("Grab Audio CD")) == 0) && (perc <= 100)) {
		if (prev_percent != perc) {
			int eta = (difftime(time(NULL), time0) / perc) * (100 - perc);
			update_processwin_eta(processwin, eta);
			prev_percent = perc;
		}
		mvwprintw(processwin, 5, 5, "");
		draw_progress_bar( (perc / 5), 5, 2, ACS_BLOCK);
		/*
		for (pos = 0; pos <= ((perc) / 5) - 1; pos++) {
			wattrset(processwin, COLOR_PAIR(5));
			waddch(processwin, ACS_BLOCK);
			wattrset(processwin, COLOR_PAIR(2));
		}
		*/
		sprintf(percent, _("  %d%% done  "), perc);
		mvwprintw(processwin, 3, 15 - (strlen(percent) / 2), "%s", percent);
		//mvwprintw(processwin, 5, 5, "");
		//mvwprintw(processwin, 9, 29, "");
		//mvwprintw(processwin, 7, 5, "T: %d P: %d L: %d   ", track, perc, lasttrack);
		wrefresh(processwin);
		//fprintf(log, "\nT: %d P: %d L: %d\n", track, perc, lasttrack);
		fflush(logf);
		buffer_err[0] = '\0';
		//track=perc;
		//sleep(1);
	}

	return;
}





/**
 * Handle normal mkisofs output during image creation
 *
 * Normal mkisofs output durnig image creation is follownig:
 * '58.65% done, estimate finish Sat Mar  3 21:20:06 2007'.
 * mkisofs doesn't write 100% value therefore handle_image_creation_100percent()
 * is needed.
 */
void handle_image_creation(regex_t *regex, regmatch_t *matches)
{
	int i;
	char percent[30];
	for (i = 0; i < (regex->re_nsub + 1); ++i) {
		int len = matches[i].rm_eo - matches[i].rm_so;
		strncpy(buf_err, (char *) &buffer_err[matches[i].rm_so], len);
		buf_err[len] = '\0';
		if (i == 2) {
			int current_done = atoi((char *) &buf_err);
			if (prev_percent != current_done) {
				int eta = (difftime(time(NULL), time0) / current_done) * (100 - current_done);
				update_processwin_eta(processwin, eta);

				prev_percent = current_done;
			}
			sprintf(percent, _("%d%% done"), current_done);
			mvwprintw(processwin, 3, 15 - (strlen(percent) / 2), "%s", percent);
			wrefresh(processwin);
			mvwprintw(processwin, 5, 5, "");

			draw_progress_bar( (current_done / 5), 5, 2, ACS_BLOCK);

			mvwprintw(processwin, 9, 29, "");
			wrefresh(processwin);
		}
	}

	return;
}





/*
 * Handle normal mkisofs output after image creation
 *
 * Normal mkisofs output durnig image creation is follownig:
 * '58.65% done, estimate finish Sat Mar  3 21:20:06 2007'.
 * mkisofs doesn't write 100% value, but it prints following after completion:
 * '
 * Total translation table size: 0
 * Total rockridge attributes bytes: 0
 * Total directory bytes: 0
 * Path table size(bytes): 10
 * Max brk space used 0
 * 127880 extents written (249 MB)
 * '
 *
 * This function is called after capturing
 * 'Total translation table size: 0'
 */
void handle_image_creation_100percent(regex_t *regex, regmatch_t *matches)
{
	char percent[30];
	int i;
	for (i = 0; i < (regex->re_nsub + 1); ++i) {
		int len = matches[i].rm_eo - matches[i].rm_so;
		int current_done = 99;
		strncpy(buf_err, (char *) &buffer_err[matches[i].rm_so], len);
		buf_err[len] = '\0';
		if (prev_percent != current_done) {
			int eta = (difftime(time(NULL), time0) / current_done) * (100 - current_done);
			update_processwin_eta(processwin, eta);

			prev_percent = current_done;
		}
		sprintf(percent, _("%d%% done"), current_done);
		mvwprintw(processwin, 3, 15 - (strlen(percent) / 2), "%s", percent);
		wrefresh(processwin);
		mvwprintw(processwin, 5, 5, "");

		draw_progress_bar( (current_done / 5), 5, 2, ACS_BLOCK);

		mvwprintw(processwin, 9, 29, "");
		wrefresh(processwin);
	}
	return;

}





/*
 * Draw progress bar in (global variable) processwin
 *
 * \param int value - length of progress bar to draw
 * \param int color1
 * \param int color2
 * \param char c - character to be used to draw bar
 */
void draw_progress_bar(int value, int color1, int color2, char c)
{
	int i;

	for (i = 0; i <= value; i++) {
		wattrset(processwin, COLOR_PAIR(color1));
		waddch(processwin, 'o');
		wattrset(processwin, COLOR_PAIR(color2));
	}

	return;
}






void handle_is_erasable(regex_t *regex, regmatch_t *matches)
{
	/*
	int len = matches[0].rm_eo - matches[0].rm_so;
	strncpy(buf, (char *) &buffer[matches[0].rm_so], len);
	buf[len] = '\0';

	dialogbox("ATIP message", buf, 0);
	wrefresh(derwin(mainwin,LINES-1,COLS-1,0,0));
	select_window(FALSE);
	*/
	media_erasable = MEDIA_ERASABLE_YES;

	return;
}





void handle_blanking_not_supported(regex_t *regex, regmatch_t *matches)
{

	media_erasable = MEDIA_ERASABLE_NO;

	return;
}


