/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2012 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#ifndef INTERFACEBITMAP_H
#define INTERFACEBITMAP_H

// -- vtk stuff
#include <vtkSmartPointer.h>

// -- vtk stuff Classes
class vtkImageActor;
class vtkActor;
class vtkImageData;
class vtkWindowLevelLookupTable;


namespace camitk
{
// -- Core stuff Classes
class Component;

/** This class describes what are the methods to implement
 * for a BitMap.
 * An InterfaceBitMap is a kind of simplifier/wrapper for vtkImageData.
 *
 * This class defines an "interface" (in the OOP/java meaning of the term).
 * See the introduction of GoF: "Program to an interface, not an implementation."
 * To see what Erich Gamma has to say about it: http://www.artima.com/lejava/articles/designprinciplesP.html
 * To see what Bjarne Stroustrup has to say about it: http://www.artima.com/intv/modern.html
 *
 * @see Slice
 */

class InterfaceBitMap {

  public:

    /** Common slices orientation: axial, sagittal, coronal or anything else.
     * The slices organisation will depend on how the original volume image has been read
     * (generally: RAI):
     * RAI: Right Anterior  Inferior,
     * LAI: Left  Anterior  Inferior,
     * LAS: Left  Anterior  Superior,
     * RAS: Right Anterior  Superior,
     * LPS: Left  Posterior Superior,
     * RPS: Right Posterior Superior,
     * LPI: Left  Posterior Inferior,
     * RPI: Right Posterior Inferior
     *
     * AXIAL_ORIENTATION:     from feet  to head of the patient in case of RAI, LAI, LPI and RPI
     *                        from head  to feet of the patient in case of LAS, RAS, LPS and RPS
     *
     * CORONAL_ORIENTATION:   from front to back  of the patient in case of RAI, LAI, LAS and RAS
     *                        from back  to front of the patient in case of LPI, RPI, LPS and RPS
     *
     * SAGITTAL_ORIENTATION:  from right to left facing the patient in case of RAI, RPI, RAS and RPS
     *                        from left to right facing the patient in case of LAI, LAS, LPI and LPS
     *
     * ARBITRARY_ORIENTATION: well give the orientation you want!
     */
    enum PossibleOrientation {
      AXIAL_ORIENTATION,
      CORONAL_ORIENTATION,
      SAGITTAL_ORIENTATION,
      ARBITRARY_ORIENTATION
    };

    /// virtual destructor
    virtual ~InterfaceBitMap() {};

    /** Return the displayed images as a vtkImageData. */
    virtual vtkSmartPointer<vtkImageData> getImageData() const = 0;

    /// set the original volume image data (the source vtkImageData before any reslice) and refresh the vtk pipeline
    virtual void setOriginalVolume(vtkSmartPointer<vtkImageData>) = 0;

    /** Return the vtkImageActor (vtkProp) representing a slice to be displayed in 2D. */
    virtual vtkSmartPointer<vtkImageActor> get2DImageActor() const = 0;

	/** Return the vtkImageActor (vtkProp) representing a slice to be displayed in 2D. */
    virtual vtkSmartPointer<vtkImageActor> get3DImageActor() const = 0;


    /** Return the vtkActor used to pick pixels in the slices. */
    virtual vtkSmartPointer<vtkActor> getPickPlaneActor() const = 0;

    /** Return the vtkActor used to pick pixels in the slices. */
    virtual vtkSmartPointer<vtkActor> getPixelActor() = 0;

    /** This method is called when the associated plane has been picked in the InteractiveViewer,
     *  the given coordinates is position where the plane was picked.
     */
    virtual void pixelPicked(double, double, double) = 0;

    virtual void updatePickPlane() = 0;

    /** Return the number of slices in the image data set. */
    virtual int getNumberOfSlices() const = 0;

    /** Return the index of the current displayed slice. */
    virtual int getSlice() const = 0;

    /** Set the current slice index.
     * If the slice index is less than the first slice index, the first slice is displayed.
     * If the slice index is more than the last slice index, the last slice is displayed.
     * @param s the index of the slice to display (base 0). */
    virtual void setSlice(int s) = 0;

    /// Set the slice corresponding to the given world coordinates
    virtual void setSlice(double x, double y, double z) = 0;
    
      /** Return the current angle applied to the arbitrary slice */
    virtual double getRotationX() const = 0;
    virtual double getRotationY() const = 0;
    virtual double getRotationZ() const = 0;
    
    /** Set the current angle to the arbitrary slice */
    virtual void setRotationX(double angle) = 0;
    virtual void setRotationY(double angle) = 0;
    virtual void setRotationZ(double angle) = 0;
    
    
    /** Return the number of colors in the images.
     * If color is coded on 1 byte, the images are on 256 grey level.
     * If color is coded on 2 bytes, the images are on 4096 grey level (not 65536). */
    virtual int getNumberOfColors() const = 0;

    /// move the pixel selection green indicator (pixelActor) to the given real position
    virtual void setPixelRealPosition(double, double, double) = 0;

};

}

#endif
