/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2012 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#ifndef PMManagerDC_H
#define PMManagerDC_H

#include <pml/StructuralComponent.h>
#include <pml/MultiComponent.h>
#include <pml/PhysicalModel.h>

// Qt
class QMenu;
class QPixmap;
#include <QProgressDialog>
class QWidget;

// PML
class PhysicalModel;
class Cell;
class Atom;


// vtk
class vtkPoints;
class vtkCell;
#include <vtkCell.h>
#include <vtkPoints.h>
#include <vtkSmartPointer.h>

// physicalmodel component
#include "PMComponentAPI.h"
class AtomDC;
class ComponentDC;
class MultiComponentDC;
class StructuralComponentDC;
class PMManagerDCPopup;
class CellDC;
class AtomDC;
class AtomDCWidget;

class LoadsManager;

// camitk
#include <MeshComponent.h>
#include <InterfaceGeometry.h>
#include <Geometry.h>
using namespace camitk;


namespace std {
/** definition of a couple (=STL pair) [Component *, ComponentDC *]
 * this associates a Component to its DC
  */
typedef std::pair< ::Component *, ComponentDC *> ComponentDCPair;
/** definition of the association STL map ComponentDCMap.
  * ComponentDCMap associate all the ComponentDC with their Physical Model Components.
  * The key is the Component, so that it is simple to retrieve a DC from the Component
  * (which is supposed to be the most often used functionnality).
  * This map is used in three ways : [Cell,DC], [SC,DC] and [MC,DC]. (yes, it is a shame there are not AC/DC map... Let there be rock
  )
  */
typedef std::map < ::Component *, ComponentDC *> ComponentDCMap;
/** the iterator corresponding to the ComponentDC map */
typedef std::map < ::Component *, ComponentDC *>::iterator ComponentDCMapIterator;

/** definition of a couple (=STL pair) [Atom *, AtomDC *]
 * this associates an atom to its DC
  */
typedef std::pair<Atom *, AtomDC *> AtomDCPair;
/** definition of the association set (=map in STL) AtomDCMap.
  * AtomDCMap associate all the DC with their atom.
  * The key is the atom, so that it is simple to retrieve a DC from its atom
  * (which is supposed to be the most often used functionnality).
  */
typedef std::map <Atom *, AtomDC *> AtomDCMap;
/** the iterator corresponding to the AtomDCMap map */
typedef std::map <Atom *, AtomDC *>::iterator AtomDCMapIterator;
}

/**The manager of the physical model data.
 *
 * NOTE: each cell has one and only one CellDC (see StructuralComponentDC constructor code).
 *
 * NOTE: each atom has one and only one AtomDC (see StructuralComponentDC constructor code).
 *
 * 

 */
class PHYSICALMODEL_COMPONENT_API PMManagerDC : public MeshComponent {
  Q_OBJECT
public:
    /// Default constructor: give it the name of the file containing the data (either a pml file or scn file)
    PMManagerDC(const QString &) throw(AbortException);

    /// Create a manager directly from the PhysicalModel
    PMManagerDC(PhysicalModel *, const QString &);

    /// Default destructor
    virtual ~PMManagerDC();

    /// overloaded method (Also check for Loads modifications)
    bool getModified() const;

    /// set the name of the physical model as well
    virtual void setName(const QString &);

    /// Overriden method so that we actually can build a popup menu with different actions
    virtual QMenu * getPopupMenu(QWidget* parent);

    /// tell the PMDC that another step is done (and set the progress bar correspondingly)
    void progressOneStep();

    /// get the pixmap for physical model DC
    virtual QPixmap getIcon();

    /** create all the point data (scalar value) to display specific information,
     *  for physical model component, a point data is associated to each atom (hence
     *  outside the camitk::Component subclasses, this is renamed "atom data").
     *
     *
     *  SCs that have a surface representation show the atom information as a color,
     *  automatic interpolation in VTK (thanks to point data) allows for nice
     *  color scale display.
     *
     *  Note: this will automatically fills up the AtomDC point data pointers.
     */
    void createPointData();

    /// destroy all the atom data
    void destroyPointData();

    /*
    // /@name Input/Output
    // /@see save (for vtk export)

    // /@{
    // / Transform an geometry to a physical model
    static PhysicalModel * geometryToPhysicalModel(Geometry *);

    / ** Read the file given in parameter as a VTK file and create a physical model from it .
    * As no assumpution could be made on a basic VTK file, the resulting physical model
    * object will have only the list of atoms and one exclusive component containing all the vtk cells.
      * As this method use the VtkMeshUtil static methods, it can import from any vtk file that
      * are supported by the VtkMeshUtil class.
      * Beware to remember to delete this physical model in your own method.
    * @return the new physical model or NULL if an error occurs.
    * /
    static PhysicalModel * vtkToPhysicalModel(const QString&);

    / ** Create a StructuralComponent from a list of vtkPoints
       * (i.e. the resulting structural component is simply a list of atoms).
      * @param n name to give to the structural component
      * /
    static StructuralComponent *vtkToPhysicalModel(PhysicalModel *, vtkSmartPointer<vtkPoints>, const QString& n);

    / ** Create a Cell Structure from a vtkCell and the atom StructuralComponent describing the
       * list where to find the atoms to use. This StructuralComponent has to be complete.
       * This method is mostly used in conjonction with vtkToPhysicalModel(vtkPoints*, std:string) method
       * (the latter giving the proper StructuralComponent to use here).
     * /
    static StructuralComponent *vtkToPhysicalModel(PhysicalModel *, vtkSmartPointer<vtkCell>, StructuralComponent *atomSC);

    / ** Export the structure to a vtk file.
      * Write all the list of atoms as the vtkPoints and the exclusive cells as the vtkCells.
      *  @return tells if the method worked properly (or not!).
      * /
    static bool physicalModelToVtk(PhysicalModel *, const QString& fileName);

    / ** same as physicalModelToVtk but considering all the nodes.
      * Directly write all the atoms and cell, even if some atoms are not used
      * This is the method to call to export the PM for Ansys, and be able to use
      * the informative components.
      * /
    static bool physicalModelToVtkAllNodes(PhysicalModel *myPM, const QString&);
    // /@}
    */

    /// @name Physical Model <-> Component
    /// to get a Component corresponding to a PhysicalModel component or structure
    ///@{

    /// get the current managed PhysicalModel
    PhysicalModel *getPhysicalModel();

    /// get the bounding sphere radius (calculated when loaded)
    double getBoundingRadius();

    /// convert PML rendering mode to Camitk rendering mode
    InterfaceGeometry::RenderingModes toDCRenderingMode(::RenderingMode::Mode);

    /// convert Camitk rendering mode to PML rendering mode
    ::RenderingMode::Mode toPMRenderingMode(InterfaceGeometry::RenderingModes);

    /// get the DC of a particular Component
    ComponentDC * getDC(::Component *);

    /// get the DC of a particular MC
    MultiComponentDC * getDC(MultiComponent *);

    /// get the DC of a particular SC
    StructuralComponentDC * getDC(StructuralComponent *);

    /// get the DC of a particular Cell
    CellDC * getDC(Cell *);

    /// get the DC of a particular atom
    AtomDC * getDC(Atom *);

    /// add the MC/DC pair
    void addMultiComponentDCPair(std::ComponentDCPair);

    /// add the SC/DC pair
    void addStructuralComponentDCPair(std::ComponentDCPair);

    /// add the C/DC pair
    void addCellDCPair(std::ComponentDCPair);

    /// add the A/DC pair
    void addAtomDCPair(std::AtomDCPair);
    ///@}

    ///@name Misc
    ///@{
    /** get the AtomDCWidget, create one if needed
    * @param adc the AtomDC calling the method (needed for creation or update)
    * @param parent parent widget (needed the first time)
    */
    QWidget *getAtomDCWidget(AtomDC *adc = NULL, QWidget *parent = NULL);

    /// get the load manager
    LoadsManager * getLoadsManager();

    ///@}

private:

    /// build the Physical model dcs
    void buildPhysicalModelDCs();

    /** the method that build the Geometry (this method actually creates an empty geometry as there are no Geometry that
     *    represents a whole Physical Model).
     */
    virtual void initRepresentation();
    
    /// Number of Step done yet
    unsigned int nrOfDoneSteps;

    /// Total Number of Step
    unsigned int nrOfSteps;

    /// the popup menu
    PMManagerDCPopup * myPopupMenu;

    /// the PMManagerDC icon
    static QPixmap * myPixmap;

    /// the MC / DC map
    std::ComponentDCMap myMCDCMap;

    /// the SC / DC map
    std::ComponentDCMap mySCDCMap;

    /// the Cell / DC map
    std::ComponentDCMap myCDCMap;

    /// the A / DC map
    std::AtomDCMap myADCMap;

    /// the atomdc widget
    AtomDCWidget * myAtomDCWidget;

    /// the physical model managed here
    PhysicalModel * myPM;

    /// the load manager
    LoadsManager * myLoadsManager;

    /// compute the bounding radius using the PhysicalModel atoms' position
    void computeBoundingRadius();

    /// the bounding sphere radius calculated just after the PML is loaded
    double initialBoundingRadius;
};


// -------------------- getPhysicalModel --------------------
inline PhysicalModel *PMManagerDC::getPhysicalModel() {
  return myPM;
}

// -------------------- getBoundingRadius --------------------
inline double PMManagerDC::getBoundingRadius() {
  return initialBoundingRadius;
}

// -------------------- getLoadsManager --------------------
inline LoadsManager * PMManagerDC::getLoadsManager() {
  return myLoadsManager;
}

// -------------------- C / DC Map --------------------
inline ComponentDC * PMManagerDC::getDC(::Component *mc) {
  if (mc->isInstanceOf("StructuralComponent"))
    return (ComponentDC *) getDC(dynamic_cast<StructuralComponent *>(mc));
  else if (mc->isInstanceOf("MultiComponent"))
    return (ComponentDC *) getDC(dynamic_cast<MultiComponent *>(mc));
  else
    return NULL;
}

// -------------------- MC / DC Map --------------------
inline void PMManagerDC::addMultiComponentDCPair(std::ComponentDCPair p) {
  myMCDCMap.insert(p);
}

// -------------------- SC / DC Map --------------------
inline void PMManagerDC::addStructuralComponentDCPair(std::ComponentDCPair p) {
  mySCDCMap.insert(p);
}

// -------------------- Atom / DC Map --------------------
inline void PMManagerDC::addAtomDCPair(std::AtomDCPair p) {
  myADCMap.insert(p);
}

inline AtomDC * PMManagerDC::getDC(Atom *a) {
  if (!a)
    return NULL;
  std::AtomDCMapIterator result = myADCMap.find(a);
  return (result == myADCMap.end()) ? NULL : (result->second);
}


#endif
