/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2012 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#ifndef DICOM_COMPONENT_PLUGIN_H
#define DICOM_COMPONENT_PLUGIN_H

#include <map>
#include <QObject>
#include <QCheckBox>
#include <itkGDCMSeriesFileNames.h>

#include <ImageComponentExtension.h>

using namespace camitk;
/** This vtkRawData ComponentExtension allows you to manipulate any files handled by vtk
 *
 * 
 */

typedef itk::GDCMSeriesFileNames NamesGeneratorType;

typedef std::vector<std::string> FileNamesContainerType;

typedef struct 
{
  /// Modality (generally MR or CT)
  QString modality;
  /// How was the volume acquired
  QString protocolName;
  /// Description of the series given by clinician
  QString seriesDescription;
  /// Description of the exam given by clinician
  QString studyDescription;

  ///@{ Number of width, height and depth of the volume
  QString rows;
  QString columns;
  QString slices;
  ///@}
  /** In case of temporal series, the volume are stored interlaced
   *  The temporal series volume are stored in only one volume in which
   *   the first slice is the first slice of the first temporal volume, 
   *   the second slice is the first slice of the second temporal volume,
   *   the third slice is the first slice of the third temporal volume and so on
   *   the numberOfTemporalPositionsTh slice is the second slice of the first temporal series
   *   and so on...
   *  This component will de-interlace the temporal series volume to create 
   *   numberOfTemporalPositions temporal volumes.
  **/
  QString numberOfTemporalPositions;
  /// Used to know if we open char, short or int
  QString bitsAllocated;
  /// Big and Little endian information
  QString highBit;

  /// Check box to ask the user wether he/she wants to open this series or not
  QCheckBox * openIt;
  /// Check box to ask the user if he wants the temporal series volume de-interlaced or not
  QCheckBox * desInterlace;

}SomeDicomInfo;

typedef struct
{ 
  bool operator()(const SomeDicomInfo i1, const SomeDicomInfo i2) const
  {
    int nb1 = i1.numberOfTemporalPositions.toInt();
    int nb2 = i2.numberOfTemporalPositions.toInt();
    
    return nb1 < nb2;
  }

}CompareInfo;

class DicomComponentExtension : public ImageComponentExtension {
    Q_OBJECT
    Q_INTERFACES(camitk::ComponentExtension);

  public:
    /// the constructor (do nothing really)
    DicomComponentExtension() : ImageComponentExtension() {};

    /// get the plugin name
    virtual QString getName() const;

    /// get the plugin description (can be html)
    virtual QString getDescription() const;

    /// get the list of managed extensions (each file with an extension in the list can be loaded by this Extension
    virtual QStringList getFileExtensions() const;

    /// this method returns true as DICOM do not use file extension but directory
    virtual bool hasDataDirectory() const;

    /// get a new instance from data stored in a file (this is the most important method to redefine in your subclass)
    virtual Component * open(const QString &) throw (AbortException);

  protected:
    /// the destructor
    virtual ~DicomComponentExtension() {};

    /// Explores the dicom directory to find all dicom volumes and needed info
    bool findAllDicomSeries(const QString &);

    /// Shows a dialog to the user to choose which volume should be read
    bool chooseDicomSeriesIds() throw (AbortException);


    // Map storing  series
    // For each series, it contains:
    //      - Key:  some infos read in the dicom headers (SomeDicomInfo)
    //      - Data: set of filenames needed to build the 3D Volume
    std::map<SomeDicomInfo *, FileNamesContainerType> theSeries;

};

#endif // DICOM_COMPONENT_PLUGIN_H
