/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2012 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/
#include "ComponentGenerator.h"

// includes from std
#include <iostream>

// Includes from Qt
#include <QDate>
#include <QTime>
#include <QPair>
#include <QMapIterator>
#include <QTextStream>


bool ComponentGenerator::generateComponentFiles(QString xmlFileName, QString devDirectoryName, QString *elementClassName) {
    try {
        ComponentGenerator * generator = new ComponentGenerator(xmlFileName, devDirectoryName);    
		generator->generateComponentFiles();

        if (elementClassName != NULL)
			(*elementClassName) = generator->className;
    
    
    } catch (QString msg) {
        std::cout << msg.toStdString() << std::endl;
        std::cout << "Could not generate action files, sorry..." << std::endl;
        return false;
    }


    return true;
    
}
ComponentGenerator::ComponentGenerator(QString xmlFilename, QString devDirectoryName) {
	
	className = QString("");
    hasParameters = false;
    setXmlFileName(xmlFilename);
    setDevDirectoryName(devDirectoryName);

    createElement();
}

void ComponentGenerator::setXmlFileName(QString xmlFileName) throw (QString) {
    QFileInfo xmlFile(xmlFileName);

    if ((! xmlFile.exists()) || (! xmlFile.isFile())) {
        QString msg = "Exception from Action generation: \n     The file " + xmlFileName +  " does not exist or is not a file...\n";
        throw (msg);
    }
    this->xmlFileName = xmlFile;
}

void ComponentGenerator::setDevDirectoryName(QString devDirectoryName) throw (QString) {
    QFileInfo devDir(devDirectoryName);
    if ( ! devDir.isDir()) {
        QString msg = "Exception from action generation: \n     The path " + devDirectoryName + " is not a directory\n";
        throw (msg);
    }

    this->devDirectoryName.cd(devDirectoryName);
}

void ComponentGenerator::createElement() throw (QString) {
    std::string xmlFileStr = this->xmlFileName.canonicalFilePath().toStdString();

    // TODO manage exception (despite Qt...).
    // TODO find how to read an xml file with an absolute path !!!
    this->theComponent = coreschema::component(xmlFileStr, xml_schema::flags::dont_validate);
	this->className = QString(this->theComponent->classNames().componentClass().c_str());

	this->hasParameters = theComponent->properties().present();
    if (hasParameters) {
		Parameters::parameter_sequence theParams = theComponent->properties()->parameter();		
		for (Parameters::parameter_const_iterator it = theParams.begin(); it < theParams.end(); it++) {
			QString parameterName = QString((*it).name().c_str());
			QString parameterType = QString((*it).type().c_str());
			QString parameterValue ="";
			if ((*it).defaultValue().present()) {
				parameterValue = QString((*it).defaultValue()->c_str());
			}
			else {
				if (parameterType == "int")
					parameterValue =	"0";
				else if (parameterType == "bool")
					parameterValue = "false";
				else if (parameterType == "double")
					parameterValue = "0.0";
				else if (parameterType == "QString")
					parameterValue = "\"Hello World\"";
				else if (parameterType == "QDate") {
					QDate today = QDate::currentDate();
					QString defValue;
					QTextStream in(&defValue);
					in << "QDate(" << today.year() << ", " << today.month() << ", " << today.day() << ")";
					parameterValue = defValue;
				}
				else if (parameterType == "QTime") {
					QTime now = QTime::currentTime();
					QString defValue;
					QTextStream in(&defValue);
					in << "QTime(" << now.hour() << ", " << now.minute() << ", " << now.second() << ")";
					parameterValue = defValue;
				}
				else if (parameterType == "QColor")
					parameterValue = "QColor(255, 255, 255, 255)";
				else if (parameterType == "QPoint")
					parameterValue = "QPoint(0, 0)";
				else if (parameterType == "QPointF")
					parameterValue = "QPointF(0.0, 0.0)";
				else if (parameterType == "QVector3D")
					parameterValue = "QVector3D(0.0, 0.0, 0.0)";
			}

			// Create a table with name and default value
			parameters.insert(parameterName, QPair<QString, QString>(parameterType, parameterValue));

            // Select the additional includes
            if ((parameterType != "int") && (parameterType != "bool") && (parameterType != "double")) {
                this->additionalIncludes.insert(parameterType);
            }

		}
	}


    QString tmp(theComponent->representation().c_str());
    if (tmp == QString("Image"))
        this->representation = IMAGE;
    else if (tmp  == QString("Mesh"))
        this->representation = MESH;
    else if (tmp == QString("None"))
        this->representation = NONE;
    else {
        QString msg = QString("Exception from Component generation: \n    Representation ") + this->representation + " not known. " + "\n";
        throw (msg);
    }
}

void ComponentGenerator::generateComponentFiles() throw (QString) {
	writeHFile();
	writeCFile();
}

void ComponentGenerator::writeHFile() throw (QString) {

	QString parentClassName = "";
	switch(representation) {
	case IMAGE:
		parentClassName = "ImageComponent";
		break;
	case MESH:
		parentClassName = "MeshComponent";
		break;
	case NONE:
		parentClassName = "Component";
		break;
	default:
		parentClassName = "";
		break;
	}
	
	QFile initFile(":/resources/Component.h.in");
	initFile.open(QIODevice::ReadOnly | QIODevice::Text);
	QTextStream in(&initFile);

    QFileInfo extFilePath;
    extFilePath.setFile(this->devDirectoryName, className + ".h");
    QFile extFile(extFilePath.absoluteFilePath());
    if (! extFile.open(QIODevice::WriteOnly | QIODevice::Text)) {
        QString msg = "Exception from extension generation: \n    Cannot write on file " + extFilePath.fileName() + "\n";
        throw (msg);
    }
    QTextStream out(&extFile);

	QString text;
	do {
		text = in.readLine();
		text.replace(QRegExp("@HEADDEF@"), className.toUpper());
		text.replace(QRegExp("@COMPONENTCLASSNAME@"), className);
		text.replace(QRegExp("@PARENTCLASSNAME@"), parentClassName);

		if (text.contains(QRegExp("@IF_NONE@"))) {
			if (representation != NONE) {
				do {
					text = in.readLine();
				} while ( (! text.contains(QRegExp("@ELSEIF_NONE@"))) &&
					      (! text.contains(QRegExp("@ENDIF_NONE@"))));
			}
		}
		else if (text.contains(QRegExp("@ELSEIF_NONE@"))) {
			if (representation == NONE) {
				do {
					text = in.readLine();
				} while (! text.contains(QRegExp("@ENDIF_NONE@")));
			}
		}
		else if (text.contains(QRegExp("@ENDIF_NONE@"))) {
			// Go to next line...
		}
		else {
			out << text << endl;
		}
	} while (! text.isNull());

	extFile.close();
	initFile.close();

}

void ComponentGenerator::writeCFile() throw (QString) {
	QString parentClassName = "";
	switch(representation) {
	case IMAGE:
		parentClassName = "ImageComponent";
		break;
	case MESH:
		parentClassName = "MeshComponent";
		break;
	case NONE:
		parentClassName = "Component";
		break;
	default:
		parentClassName = "";
		break;
	}
	
	QFile initFile(":/resources/Component.cpp.in");
	initFile.open(QIODevice::ReadOnly | QIODevice::Text);
	QTextStream in(&initFile);

    QFileInfo extFilePath;
    extFilePath.setFile(this->devDirectoryName, className + ".cpp");
    QFile extFile(extFilePath.absoluteFilePath());
    if (! extFile.open(QIODevice::WriteOnly | QIODevice::Text)) {
        QString msg = "Exception from extension generation: \n    Cannot write on file " + extFilePath.fileName() + "\n";
        throw (msg);
    }
    QTextStream out(&extFile);

	QString text;
	do {
		text = in.readLine();
		text.replace(QRegExp("@HEADDEF@"), className.toUpper());
		text.replace(QRegExp("@COMPONENTCLASSNAME@"), className);
		text.replace(QRegExp("@PARENTCLASSNAME@"), parentClassName);

		if (text.contains(QRegExp("@ADDITIONAL_INCLUDES@"))) {
			int nbIncludes = additionalIncludes.size();
			std::set<QString>::const_iterator it;
			for (it = additionalIncludes.begin(); it != additionalIncludes.end(); it++) {
				out << QString("#include <") << (*it) << ">" << endl;
			}
		}
		else if (text.contains(QRegExp("@IF_NONE@"))) {
			if (representation != NONE) {
				do {
					text = in.readLine();
				} while ( (! text.contains(QRegExp("@ELSEIF_NONE@"))) &&
					      (! text.contains(QRegExp("@ENDIF_NONE@"))));
			}
		}
		else if (text.contains(QRegExp("@ELSEIF_NONE@"))) {
			if (representation == NONE) {
				do {
					text = in.readLine();
				} while (! text.contains(QRegExp("@ENDIF_NONE@")));
			}
		}
		else if (text.contains(QRegExp("@ENDIF_NONE@"))) {
			// Go to next line...
		}
		else if (text.contains(QRegExp("@IF_MESH@"))) {
			if (representation != MESH) {
				do {
					text = in.readLine();
				} while ( (! text.contains(QRegExp("@ELSEIF_MESH@"))) &&
					      (! text.contains(QRegExp("@ENDIF_MESH@"))));
			}
		}
		else if (text.contains(QRegExp("@ELSEIF_MESH@"))) {
			if (representation == MESH) {
				do {
					text = in.readLine();
				} while (! text.contains(QRegExp("@ENDIF_MESH@")));
			}
		}
		else if (text.contains(QRegExp("@ENDIF_MESH@"))) {
			// Go to next line...
		}
		else if (text.contains(QRegExp("@IF_IMAGE@"))) {
			if (representation != IMAGE) {
				do {
					text = in.readLine();
				} while ( (! text.contains(QRegExp("@ELSEIF_IMAGE@"))) &&
					      (! text.contains(QRegExp("@ENDIF_IMAGE@"))));
			}
		}
		else if (text.contains(QRegExp("@ELSEIF_IMAGE@"))) {
			if (representation == IMAGE) {
				do {
					text = in.readLine();
				} while (! text.contains(QRegExp("@ENDIF_IMAGE@")));
			}
		}
		else if (text.contains(QRegExp("@ENDIF_IMAGE@"))) {
			// Go to next line...
		}
		else if (text.contains(QRegExp("@IF_ADDPROPERTIES@"))) {
			text = in.readLine();
			while (! text.contains(QRegExp("@ENDIF_ADDPROPERTIES@"))) {
				if ( ! parameters.isEmpty()) {
					out << text << endl;
				}
				text = in.readLine();
			}
		}
		else if (text.contains(QRegExp("@BEGIN_ADDPROPERTIES@"))) {
			text = in.readLine();
			int nbProps = parameters.size();
			int indexProp = 0;
			if (nbProps > 0) {
				QString * addOneProp = new QString[nbProps];
				for (indexProp = 0; indexProp < nbProps; indexProp++)
					addOneProp[indexProp] = "";
				while (! text.contains(QRegExp("@END_ADDPROPERTIES@")) && 
					   ! text.contains(QRegExp("@ELSE_ADDPROPERTIES@")) ) 
				{
					QMapIterator<QString, QPair<QString, QString> > it(parameters);
					indexProp = 0;
					while (it.hasNext()) {
						it.next();
						QString name = it.key();
						QString type = it.value().first;
						QString val  = it.value().second;
						QString toString = name;
						QString toType;
						if ((type == "int") || (type == "bool") || (type == "float") || (type == "double")) {
							toType = QString("to").append(QString(type.toUpper().at(0)));
							toType.append(type.right(type.length() - 1)).append("()");
						}
						else {
							toType = QString("value<").append(type).append(">()");
							if (type == "QVector3D") {
								toString.append(".x()");
							}
							else if (type != "QString") {
								toString.append(".toString()");
							}
						}

						QString textTmp = text;
						textTmp.replace(QRegExp("@PROPERTY_TYPE@"), type);
						textTmp.replace(QRegExp("@PROPERTY_NAME@"), name);
						textTmp.replace(QRegExp("@PROPERTY_NAME_STRING@"), toString);
						textTmp.replace(QRegExp("@PROPERTY_VALUE@"), val);
						textTmp.replace(QRegExp("@PROPERTY_TOTYPE@"), toType);
						addOneProp[indexProp].append(textTmp).append("\n");
						indexProp++;
					}
					text = in.readLine();
				}
				if (text.contains("@ELSE_ADDPROPERTIES@")) {
					while (! text.contains("@END_ADDPROPERTIES@")) {
						text = in.readLine();
					}
				}
				for (indexProp = 0; indexProp < nbProps; indexProp++) {
					out << addOneProp[indexProp];
				}
			}
			else {
				while ((! text.contains("@ELSE_ADDPROPERTIES@")) && (! text.contains("@END_ADDPROPERTIES@")))
					text = in.readLine();
				if (text.contains("@ELSE_ADDPROPERTIES@")) {
					text = in.readLine();
					while (! text.contains("@END_ADDPROPERTIES@")) {
						out << text << endl;
						text = in.readLine();
					}
				}
			}
		}
		
		else {
			out << text << endl;
		}
	} while (! text.isNull());

	extFile.close();
	initFile.close();
}


/*
    switch (representation) {
        case IMAGE:
            this->parentClass = QString("ImageComponent");
            break;
        case MESH:
            this->parentClass = QString("MeshComponent");
            this->additionalIncludes.insert(QString("vtkSmartPointer.h"));
            this->additionalIncludes.insert(QString("vtkPoints.h"));
            this->additionalIncludes.insert(QString("vtkPointData.h"));
            this->additionalIncludes.insert(QString("vtkCellArray.h"));
            this->additionalIncludes.insert(QString("vtkFloatArray.h"));
            this->additionalIncludes.insert(QString("vtkPolyData.h"));
            this->additionalIncludes.insert(QString("InteractiveViewer.h"));
            break;
        case NONE:
            this->parentClass = QString("Component");
			this->additionalIncludes.insert(QString("QFileInfo"));
            break;
        default:
            break;
    }
*/


//
//void ComponentGenerator::writeDeclareQObject() {
//    (*out) << "    Q_OBJECT" << endl;
//}
//
//
//void ComponentGenerator::writeDeclareConstructor() {
//    (*out) << "    /// Default Constructor " << endl;
//    (*out) << "    " << elementClassName << "(const QString & file) throw(AbortException);" << endl;
//    (*out) << endl;
//
//}
//
//
//void ComponentGenerator::writeDeclareSpecificMethods() {
//    // Images and Meshs already have a representation
//    if (representation == NONE) {
//        (*out) << "    /// do nothing to init the representation, as all representation are done in the sub-component" << endl;
//        (*out) << "    virtual void initRepresentation() {};" << endl;
//        (*out) << endl;
//    }
//    else if (representation == OTHER) {
//        (*out) << "    /** Inherited from Component. " << endl;
//        (*out) << "     * Instanciate the concrete representation (either InterfaceGeometry or InterfaceBitMap) if needed." << endl;
//        (*out) << "     * This method have to instanciate Slice (mySlice) or Geometry (myGeometry) that does all the work for this Component, i.e. the adaptee handler." << endl;
//        (*out) << "     * Generally this method should be called in the Component constructor." << endl;
//        (*out) << "     * " << endl;
//        (*out) << "     */" << endl;
//        (*out) << "    virtual void initRepresentation();" << endl;
//        (*out) << endl;
//    }
//    else if (representation == MESH) {
//        if (this->hasParameters) {
//            (*out) << "    /// camiTK's PROPERTY SERVICE. Returns the property object. " << endl;
//            (*out) << "    QObject * getPropertyObject();" << endl;
//        }
//    }
//}
//
//
//void ComponentGenerator::writeConstructor() {
//    (*out) << "// -------------------- default constructor  --------------------" << endl;
//    (*out) << elementClassName << "::" << elementClassName << "(const QString & file) throw(AbortException) " << endl;
//    if (parentClass == QString("Component")) 
//        (*out) << " : " << parentClass << "(file, QFileInfo(file).baseName())" << endl;
//    else if ((parentClass == QString("ImageComponent")) || (parentClass == QString("MeshComponent")))
//        (*out) << " : " << parentClass << "(file)" << endl;
//    (*out) << "{" << endl;
//    
//    switch (representation) {
//        case IMAGE:
//            (*out) << "    // Read an image from a file" << endl;
//            (*out) << "    if (!file.isNull()) { " << endl;
//            (*out) << "        // do watherver it takes... " << endl;
//            (*out) << "        " << endl;
//            (*out) << "        // Create a vtkImageData " << endl;
//            (*out) << "        vtkSmartPointer<vtkImageData> image = NULL;" << endl;
//            (*out) << endl;
//            (*out) << "        // Change the following lines to fill the image with what you have read " << endl;
//            (*out) << "        // See vtk documentation or tutorial to see how vtkImageData works" << endl;
//            (*out) << "        image = vtkSmartPointer<vtkImageData>::New();" << endl;
//            (*out) << "        image->SetDimensions(3,3,3);" << endl;
//            (*out) << "        image->SetNumberOfScalarComponents(1);" << endl;
//            (*out) << "        // Here, we arbitrarily fill the image data" << endl;
//            (*out) << "        int* dims = image->GetDimensions();" << endl;
//            (*out) << "        for (int z = 0; z < dims[2]; z++) {" << endl;
//            (*out) << "            for (int y=0; y<dims[1]; y++) { " << endl;
//            (*out) << "                for (int x=0; x<dims[0]; x++) {" << endl;
//            (*out) << "                    double val = ((x==1) && (y==1) && (z==1))? 1.0: 0.0; " << endl;
//            (*out) << "                    image->SetScalarComponentFromDouble(x,y,z,0,val);" << endl;
//            (*out) << "                }" << endl;
//            (*out) << "            } " << endl;
//            (*out) << "        } " << endl;
//            (*out) << endl;
//            if (this->hasParameters) {
//                writeParametersInitialization();
//            }
//            (*out) << endl;
//            (*out) << "        // Make this image the image of your component" << endl;
//            (*out) << "        // (this method is already implemented in ImageComponent)" << endl;
//            (*out) << "        setImageData(image, false);" << endl;
//            (*out) << endl;
//            (*out) << "    } " << endl;
//            (*out) << "    else { // there is no file name, so we cannot open an image..." << endl;
//            (*out) << "        throw (AbortException(\"No file found\"));" << endl;
//            (*out) << "    }" << endl;
//            (*out) << endl;
//            break;
//        case MESH:
//            (*out) << "    // Read an mesh from a file" << endl;
//            (*out) << "    if (!file.isNull()) { " << endl;
//            (*out) << "        // do watherver it takes... " << endl;
//            (*out) << "        " << endl;
//            (*out) << "        // Create a vtkPointSet " << endl;
//            (*out) << "        // In this example, we create a vtkPolyData, but any kind of vtkPointSet holds" << endl;
//            (*out) << "        // See vtk documentation for other types of vtkPointSet." << endl;
//            (*out) << "        vtkSmartPointer<vtkPolyData> data = NULL;" << endl;
//            (*out) << endl;
//            (*out) << "        // Change the following lines to fill the point set with what you have read " << endl;
//            (*out) << "        // See vtk documentation or tutorial to see how vtkPointSet works" << endl;
//            (*out) << "        // Here, we arbitrarilly create a cube (exemple from vtk)." << endl;
//            (*out) << "        int i;" << endl;
//            (*out) << "        static float x[8][3]={{0,0,0}, {1,0,0}, {1,1,0}, {0,1,0}, {0,0,1}, {1,0,1}, {1,1,1}, {0,1,1}};" << endl;
//            (*out) << "        static vtkIdType pts[6][4]={{0,1,2,3}, {4,5,6,7}, {0,1,5,4}, {1,2,6,5}, {2,3,7,6}, {3,0,4,7}};" << endl;
//            (*out) << "        // We create the building blocks of polydata including data attributes." << endl;
//            (*out) << "        data = vtkSmartPointer<vtkPolyData>::New(); " << endl;
//            (*out) << "        vtkSmartPointer<vtkPoints>     points  = vtkSmartPointer<vtkPoints>::New();" << endl;
//            (*out) << "        vtkSmartPointer<vtkCellArray>  polys   = vtkSmartPointer<vtkCellArray>::New();" << endl;
//            (*out) << "        vtkSmartPointer<vtkFloatArray> scalars = vtkSmartPointer<vtkFloatArray>::New();" << endl;
//            (*out) << "        // Load the point, cell, and data attributes." << endl;
//            (*out) << "        for (i=0; i<8; i++) points->InsertPoint(i,x[i]); " << endl;
//            (*out) << "        for (i=0; i<6; i++) polys->InsertNextCell(4,pts[i]); " << endl;
//            (*out) << "        for (i=0; i<8; i++) scalars->InsertTuple1(i,i); " << endl;
//            (*out) << "        // We now assign the pieces to the vtkPolyData. " << endl;
//            (*out) << "        data->SetPoints(points);" << endl;
//            (*out) << "        data->SetPolys(polys);" << endl;
//            (*out) << "        data->GetPointData()->SetScalars(scalars);" << endl;
//            (*out) << endl;
//            if (this->hasParameters) {
//                writeParametersInitialization();
//            }
//            (*out) << endl;
//            (*out) << "        // Make this point set the representation of your component" << endl;
//            (*out) << "        // (this method is already implemented in MeshComponent)" << endl;
//            (*out) << "        initRepresentation(data);" << endl;
//            (*out) << endl;
//            (*out) << "        // add it in the InteractiveViewer" << endl;
//            (*out) << "        setVisibility(InteractiveViewer::get3DViewer(), true);" << endl;   
//            (*out) << endl;
//            (*out) << "    } " << endl;
//            (*out) << "    else { // there is no file name, so we cannot open an image..." << endl;
//            (*out) << "        throw (AbortException(\"No file found\"));" << endl;
//            (*out) << "    }" << endl;
//            (*out) << endl;
//            break;
//        case NONE:
//            (*out) << " // Read the input file" << endl;
//            (*out) << endl;
//            if (this->hasParameters) {
//                writeParametersInitialization();
//            }
//            (*out) << " // Possibly create sub-components: " << endl;
//            (*out) << " // Component * subComp = new ....." << endl;
//            (*out) << " // add it as sub-component: " << endl;
//            (*out) << " // addChild(subComp);" << endl;
//            (*out) << endl;
//            break;
//        default:
//            break;
//    }
//
//    (*out) << "}" << endl;
//    (*out) << endl; 
//}
//
//
//void ComponentGenerator::writeDestructor() {
//    (*out) << "// Default destructor" << endl;
//    (*out) << elementClassName << "::~" << elementClassName << "() {" << endl;
//    (*out) << endl;
//    (*out) << "}" << endl;
//    (*out) << endl;
//}
//
//void ComponentGenerator::writeSpecificCMethods() {
//    if (representation == OTHER) {
//        (*out) << "void " << elementClassName << "::initRepresentation() {" << endl;
//        (*out) << endl;
//        (*out) << "}" << endl;
//        (*out) << endl;
//    }
//    else if (representation == MESH) {
//        if (this->hasParameters) {
//            (*out) << "    /// camiTK's PROPERTY SERVICE. Returns the property object. " << endl;
//            (*out) << "QObject * " << elementClassName << "::getPropertyObject() {" << endl;
//            (*out) << "    return this;" << endl;
//            (*out) << "}" << endl;
//        }
//    }
//}
//
//
//
