/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2012 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

// -- Core image component stuff classes
#include "ImageReconstructionAction.h"
#include "ImageComponent.h"
#include "MeshComponent.h"

// -- Core stuff classes
#include "Application.h"

// -- QT stuff
#include <QLineEdit>
#include <QFileDialog>
#include <QFileInfo>
#include <QMessageBox>

// -- stl stuff
#include <sstream>

// -- vtk filters stuff
#include <vtkImageResample.h>
#include <vtkContourFilter.h>
#include <vtkPolyDataNormals.h>
#include <vtkPolyDataConnectivityFilter.h>
#include <vtkPolyDataWriter.h>
#include <vtkPolyDataMapper.h>
#include <vtkMarchingCubes.h>
#include <vtkPointSet.h>
#include <vtkCallbackCommand.h>

#include <ImageComponent.h>
using namespace camitk;




// -------------------- constructor --------------------
ImageReconstructionAction::ImageReconstructionAction(ActionExtension* extension) : Action(extension) {
    this->setName("Reconstruction");
    this->setDescription("Action use to apply a reconstruction image component");
    this->setComponent("ImageComponent");
    this->setFamily("Reconstruction");
    this->addTag("Reconstruction");
}

// -------------------- destructor --------------------
ImageReconstructionAction::~ImageReconstructionAction() {
    if (actionWidget)
        delete actionWidget;
}

// -------------------- getWidget --------------------
QWidget * ImageReconstructionAction::getWidget() {
    if (actionWidget)
        delete actionWidget;

    actionWidget = new QWidget();
    myComponent = dynamic_cast<ImageComponent *>(getTargets().last());
    ui.setupUi(actionWidget);
    init();
    return actionWidget;
}

// ---------------------- init ----------------------------
void ImageReconstructionAction::init() {
    // images
    ui.lineImagesName->setText(myComponent->getName());

    connect(ui.PushButton5, SIGNAL(clicked()), this, SLOT(build3DModel()));
    connect(ui.keepComponentcheckBox, SIGNAL(toggled( bool )), this, SLOT(keepComponentcheckBox_toggled(bool)));
    connect(ui.subSamplecheckBox, SIGNAL(toggled( bool )), this, SLOT(subSamplecheckBox_toggled(bool)));

    // init thresholds
    double medianColor = (myComponent->getMaxColor() - myComponent->getMinColor()) / 2;
    std::ostringstream texte;
    texte << medianColor;
    ui.lineThresholdValue->setText(QString(texte.str().c_str()));

    // init subsample dimensions
    this->subSample = true;
    this->subSampledImageXDim = 64;
    std::ostringstream t_x_dim;
    t_x_dim << subSampledImageXDim;
    ui.lineXDimension->setText(QString(t_x_dim.str().c_str()));
    this->subSampledImageYDim = 64;
    std::ostringstream t_y_dim;
    t_y_dim << subSampledImageXDim;
    ui.lineYDimension->setText(QString(t_y_dim.str().c_str()));
    this->subSampledImageZDim = 64;
    std::ostringstream t_z_dim;
    t_z_dim << subSampledImageXDim;
    ui.lineZDimension->setText(QString(t_z_dim.str().c_str()));


    this->keepLargestComponent = true;
}


//**************************************************************************
//                            SLOT IMPLEMENTATION
//**************************************************************************
// ---------------------- build3DModel  ----------------------------
void ImageReconstructionAction::build3DModel() {
    // computes the model
    int threshold = ui.lineThresholdValue->text().toInt();
    this->subSampledImageXDim = ui.lineXDimension->text().toInt();
    this->subSampledImageYDim = ui.lineYDimension->text().toInt();
    this->subSampledImageZDim = ui.lineZDimension->text().toInt();

	vtkSmartPointer<vtkPointSet> resultPointSet = this->getMarchingCubesReconstruction(myComponent,
		threshold, keepLargestComponent, 
		subSample, subSampledImageXDim, subSampledImageYDim, subSampledImageZDim);

	MeshComponent * result = new MeshComponent(resultPointSet, myComponent->getName() + "_mesh");
	Application::refresh();

}

void ImageReconstructionAction::keepComponentcheckBox_toggled(bool toggled) {
    this->keepLargestComponent = toggled;
}

void ImageReconstructionAction::subSamplecheckBox_toggled(bool toggled) {
    this->subSample = toggled;
    ui.labelImageSize->setEnabled(toggled);
    ui.label_X_Size->setEnabled(toggled);
    ui.lineXDimension->setEnabled(toggled);
    ui.label_Y_Size->setEnabled(toggled);
    ui.lineYDimension->setEnabled(toggled);
    ui.label_Z_Size->setEnabled(toggled);
    ui.lineZDimension->setEnabled(toggled);

}

// -------------------- getMarchingCubesReconstruction --------------------
vtkSmartPointer<vtkPointSet> ImageReconstructionAction::getMarchingCubesReconstruction(
	ImageComponent * image, 
	int isoValue, bool keepLargestConnectedComponent,
    bool subsample, int subSampledDimX, int subSampledDimY, int subSampledDimZ)
    {
		Application::showStatusBarMessage
("Computing Marching Cube Reconstruction");
		Application::resetProgressBar();
		vtkSmartPointer<vtkCallbackCommand> progressCallback = vtkSmartPointer<vtkCallbackCommand>::New();
		progressCallback->SetCallback(&Application::vtkProgressFunction);

		vtkSmartPointer<vtkImageData> originalImageData = image->getImageData();
		vtkSmartPointer<vtkImageData> currentData = originalImageData;

        // For medical volumes, marching cubes generates a large number of triangles.
        // To be practical, we'll use a reduced resolution dataset.
        // For example, we take original 256^3 data and reduce it to 64^2 slices
        // by averaging neighboring pixels twice in the slice plane.
        // We call the resulting dataset quarter since it has 1/4 the resolution of the original data.
        vtkSmartPointer<vtkImageResample> resampleFilter = NULL;

        if (subsample) {
            resampleFilter = vtkSmartPointer<vtkImageResample>::New();
			int * previousDimensions = originalImageData->GetDimensions();


            double x_magnification, y_magnification, z_magnification;
            x_magnification = subSampledDimX / (double) previousDimensions[0];
            y_magnification = subSampledDimY / (double) previousDimensions[1];

            if (originalImageData->GetDataDimension() == 3)
                z_magnification = subSampledDimZ / (double) previousDimensions[2];

            resampleFilter->SetAxisMagnificationFactor(0, x_magnification);

            resampleFilter->SetAxisMagnificationFactor(1, y_magnification);

            if (originalImageData->GetDataDimension() == 3)
                resampleFilter->SetAxisMagnificationFactor(2, z_magnification);

            resampleFilter->SetInput(originalImageData);
			resampleFilter->AddObserver(vtkCommand::ProgressEvent, progressCallback);

            currentData = resampleFilter->GetOutput();

            currentData->Update();
        }

        ///TODO: Check that !!!
        // Add a black layer around the image to be sure to obtain a closed mesh
        int previousExtent[6];
        int newExtent[6];
        currentData->GetExtent(previousExtent);
        newExtent[0] = previousExtent[0] - 1;
        newExtent[1] = previousExtent[1] + 1;
        newExtent[2] = previousExtent[2] - 1;
        newExtent[3] = previousExtent[3] + 1;
        newExtent[4] = previousExtent[4] - 1;
        newExtent[5] = previousExtent[5] + 1;

        currentData->SetWholeExtent(newExtent);
        currentData->SetUpdateExtentToWholeExtent();

        // The filter we have chosen to use to compute 3D isosurface of the volume image
        // is vtkMarchingcubes. We could also use vtkContourFilter since it will automatically
        // create an instance of vtkMarchingCubes as it delegates to the fastest subclass
        // for a particular dataset type.
        vtkSmartPointer<vtkMarchingCubes> mcSurface = vtkSmartPointer<vtkMarchingCubes>::New();
        mcSurface->SetInput(currentData);
		mcSurface->AddObserver(vtkCommand::ProgressEvent, progressCallback);

        mcSurface->SetValue(0, isoValue);

        // The class vtkPolyDataNormals is used to generate
        // nice surface normals for the data. vtkMarchingCubes can also generate normals,
        // but sometimes better results are achieved when the normals are directly from the
        // surface (vtkPolyDataNormals) versus from the data (vtkMarchingCubes).
        vtkSmartPointer<vtkPolyDataNormals> normalsFilter = vtkSmartPointer<vtkPolyDataNormals>::New();

        //Specify the angle that defines a sharp edge. If the difference in angle
        // across neighboring polygons is greater than this value,
        // the shared edge is considered "sharp".
        normalsFilter->SetFeatureAngle(60.0);

        // Keep largest connected component
        vtkSmartPointer<vtkPolyDataConnectivityFilter> connectivityFilter = NULL;

        if (keepLargestConnectedComponent) {
            connectivityFilter = vtkPolyDataConnectivityFilter::New();
            connectivityFilter->SetInput(mcSurface->GetOutput());
            connectivityFilter->SetExtractionModeToLargestRegion();
            normalsFilter->SetInput(connectivityFilter->GetOutput());
        }
        else
            normalsFilter->SetInput(mcSurface->GetOutput());
		
		normalsFilter->AddObserver(vtkCommand::ProgressEvent, progressCallback);

		vtkSmartPointer<vtkPointSet> resultPointSet = normalsFilter->GetOutput();
		resultPointSet->Update();

        // One can make efficient data visualization using vtkTriangleFilter
        // and vtkStripper. But first, let us try without...

		// Cleaning...
		originalImageData = NULL;
        currentData = NULL;
		
		Application::showStatusBarMessage
("");
		Application::resetProgressBar();


        return resultPointSet;

	}

//**************************************************************************
