# Copyright (C) 2009 Canonical Ltd
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

from PyQt4 import QtCore, QtGui

from bzrlib.plugins.explorer.lib import (
    html_report_view,
    status_report,
    view_submit,
    )
from bzrlib.plugins.explorer.lib.i18n import gettext, N_


class WorkingTreeView(object):
    """The default view for branches & checkouts with working trees."""

    def __init__(self, model, action_callback):
        self.model = model
        self.action_callback = action_callback
        self.status_view = _StatusView(model, action_callback)

        # Decide which detail panels to display.
        # It may be a useful to define some preferences and to
        # leave it up to the user as to which detail panels get displayed.
        # In the short term though, the goal is simply to provide high
        # value ones which everyone ought to want. :-)
        self.detail_views = []
        branch = self.model.branch
        # Display the delta versus the submit branch iff one is
        # defined and it's local. Unfortunately though, merging a feature
        # branch into trunk sets the merged-from branch as the submit
        # branch which means we see a "silly" submit delta panel for trunk.
        # To work around that, we don't show the panel for bound branches
        # as trunk ought to be bound. We could look for the name
        # "trunk" instead but that sounds even more of a hack. :-(
        if branch.get_bound_location() is None:
            submit_url = branch.get_submit_branch() or branch.get_parent()
            if submit_url and submit_url.startswith("file:"):
                self.detail_views.append(
                    (view_submit.SubmitView(model, action_callback,),
                        gettext("Submit delta")),
                    )
        # TODO: add other panels to show what's on the shelf, etc.

    def ui(self):
        # Build the details panel
        count = len(self.detail_views)
        if count == 0:
            details = None
        elif count == 1:
            view, label = self.detail_views[0]
            details = view.ui()
        else:
            details = QtGui.QTabWidget()
            for view, label in self.detail_views:
                details.addTab(view.ui(), label)

        # Build the overall view
        status = self.status_view.ui()
        if details is None:
            return status
        else:
            # Status view above the details with 25% allocated to the latter
            splitter = QtGui.QSplitter()
            splitter.setOrientation(QtCore.Qt.Vertical)
            splitter.addWidget(status)
            splitter.addWidget(details)
            splitter.setStretchFactor(0, 3)
            splitter.setStretchFactor(1, 1)
            return splitter

    def refresh_view(self):
        self.status_view.refresh_view()
        for view, label in self.detail_views:
            view.refresh_view()


class _StatusView(html_report_view.HtmlReportView):
    """The status view showing what's about to be committed."""

    def report_text(self):
        model = self._model
        report = status_report.StatusReport(model.branch, model.tree,
            model.relpath)
        return report.full_report()
