# bug-triage -- bug triage and forward tool.
# Copyright (C) 2007  Gustavo R. Montesino
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
#
# Executes actions on Debian bugs

import os
import urllib
import webbrowser

import bt_exceptions

from btsutils import BugStatus
from btsutils.debbugs import debbugs

from bzutils.bugzilla import Bugzilla

from config import btstype
from config import packages
from config.config import Config

def bts_init():
    """Initializes the btsutls debbugs' object

    Returns a btsutils.debbugs.debbugs object intialized using the
    btsutils configuration"""

    config = Config()
    bts = debbugs()

    # SOAP stuff
    soap = {}
    soap["address"] = config.get_soap_address()
    soap["namespace"] = config.get_soap_namespace()
    soap["users"] = config.get_soap_users()
    bts.soap.connect(soap["address"], soap["namespace"], soap["users"])
    
    # HTML stuff
    bts.html.connect(config.get_html_address())
    bts.html.setUsers(config.get_html_users())

    return bts

def show(bug):
    """Show everything about a bug"""

    pid = os.fork()
    if pid == 0:
        try:
            os.execlp("xdg-open", "xdg-open", bug.getURL())
        except OSError:
            print "WARNING: couldn't find xdg-open, falling back to webbrowser module"
            webbrowser.open(bug.getURL())
            os._exit(os.EX_OK)

def followup(bug):
    """Send a followup message to a bug report"""

    address = "mailto: %s@bugs.debian.org" % bug.getBug()
    os.spawnlp(os.P_NOWAIT, "xdg-open", "xdg-open", address)

def upstream(bug):
    """Return the upstream bugs for a debbugs bug

    This function receives a btsutils bug instance. 
    
    If the bug was forwarded upstream, open the upstream bug into an
    webbrowser and returns False.
    
    if the bug wasn't forwarded upstream and has a valid entry into the
    packages configuration, returns a tuple with the upstream BTS type
    and a list of bugs.
    
    If the bug wasn't forwarded upstream and hasn't a valid entry into
    the packages configuration, returns None"""

    if bug.getForwarded() != '':
        pid = os.fork()
        if pid == 0:
            try:
                os.execlp("xdg-open", "xdg-open", bug.getForwarded())
            except OSError:
                print "WARNING: Couldn't find xdg-open, falling back to webbrowser module"
                webbrowser.open(bug.getForwarded())
                os._exit(os.EX_OK)
        else:
            return False
    else:
        pkg = packages.get(bug.getPackage())
        if pkg:
            if pkg.get_type() == btstype.BUGZILLA:
                data = pkg.get_data()

                if not data.has_key("server"):
                    raise InvalidBugzillaError("Bugzilla server URL not specified.")

                if not data.has_key("product"):
                    raise InvalidBugzillaError("Bugzilla product not specified.")

                server = Bugzilla(data["server"])

                chart = []
                chart.append((None, "product", "equals", data["product"]))

                if data.has_key("component"):
                    chart.append(("AND", "component", "equals", data["component"]))

                return (btstype.BUGZILLA, server.query(charts=[chart]))

            else:
                raise bt_exceptions.InvalidBTSError("Unknown bug tracking system: %s" %
                    package.get_type())

        else:
            return None

def upstream_new(bug):
    """Forwards a debbugs bug to a new upstream bug

    Currently, this function opens a webbrowser with some of the 
    fields pre-filled for bugzilla servers."""

    pkg = packages.get(bug.getPackage())

    if pkg.get_type() == btstype.BUGZILLA:

        bzdata = pkg.get_data()

        if not bzdata.has_key("server"):
            raise bt_exceptions.InvalidBugzillaError("Bugzilla server not specified.")

        if not bzdata.has_key("product"):
            raise bt_exceptions.InvalidBugzillaError("Bugzilla product not specified.")

        htmlquery = []

        # Product
        htmlquery.append(("product", bzdata["product"]))

        # Component
        if bzdata.has_key("component"):
            htmlquery.append(("component", bzdata["component"]))

        # Summary
        htmlquery.append(("short_desc", bug.getSummary()))

        # Description
        desc = "The following bug report was filled on Debian's BTS as bug #" + bug.getBug()
        desc = desc + ". Please see %s for more information." % bug.getURL();
        htmlquery.append(("comment", desc))

        # spawn the webbrowser
        url = "%s/enter_bug.cgi?%s" % (bzdata["server"].rstrip('/'), urllib.urlencode(htmlquery))
        os.spawnlp(os.P_NOWAIT, "xdg-open", "xdg-open", url)

    else:
        raise bt_Exceptions.InvalidBTSError("Unknown bug tracking system: %s" %
            package.get_type())



# vim: tabstop=4 expandtab shiftwidth=4
