/********************************************************************
 * Copyright (C) 2005, 2006 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BSCommander (Beesoft Commander).
 *
 * BsC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BsC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *******************************************************************/

/*------- include files:
-------------------------------------------------------------------*/
#include "MainWindow.h"
#include "Shared.h"
#include "Help.h"
#include "Settings.h"
#include "Workspace.h"
#include "About.h"
#include "Settings.h"
#include "Config.h"
#include "SystemInfo.h"
#include <qapplication.h>
#include <qframe.h>
#include <qaction.h>
#include <qpopupmenu.h>
#include <qmenubar.h>
#include <qtoolbar.h>
#include <qstatusbar.h>
#include <qpushbutton.h>
#include <qlayout.h>
#include <qtabbar.h>
#include <qwidgetstack.h>
#include <qtranslator.h>


/*------- local constants:
-------------------------------------------------------------------*/
const char* const MainWindow::APP_ICON            = "bsc.png";
const char* const MainWindow::RELOAD_MENU_ICON    = "reload.png";
const char* const MainWindow::FILTER_MENU_ICON    = "filter.png";
const char* const MainWindow::CMP_FILES_MENU_ICON = "cmpfiles.png";
const char* const MainWindow::DirSynchIcon        = "dirsynch.png";
const char* const MainWindow::DATETIME_MENU_ICON  = "datetime.png";
const char* const MainWindow::SELECT_MENU_ICON    = "select.png";
const char* const MainWindow::UNSELECT_MENU_ICON  = "unselect.png";
const char* const MainWindow::REVERS_MENU_ICON    = "revers.png";
const char* const MainWindow::FTP_START_MENU_ICON = "connect.png";
const char* const MainWindow::FTP_END_MENU_ICON   = "disconnect.png";
const char* const MainWindow::CONFIGURE_MENU_ICON = "configure.png";
const char* const MainWindow::EXIT_MENU_ICON      = "exit.png";
const char* const MainWindow::SystemInfoIcon      = "system.png";
const char* const MainWindow::SwitchMenuIcon      = "switch.png";
const char* const MainWindow::TabNewMenuIcon      = "tab_new.png";
const char* const MainWindow::TabRemoveMenuIcon   = "tab_remove.png";
const char* const MainWindow::FileJoinMenuIcon    = "joining.png";
//...................................................................
const char* const MainWindow::CONFIG_GROUP_NAME   = "MainWindow";
const char* const MainWindow::CONFIG_KEY_WIDTH    = "Width";
const char* const MainWindow::CONFIG_KEY_HEIGHT   = "Height";
const char* const MainWindow::EXIT_MENU_ACCEL     = "Ctrl+Q";
const char* const MainWindow::FILTER_MENU_ACCEL   = "Ctrl+F";
const char* const MainWindow::LANG_EN_MENU_TEXT   = "&English";
const char* const MainWindow::LANG_RU_MENU_TEXT   = "&Russian";
const char* const MainWindow::LANG_PL_MENU_TEXT   = "&Polski";
const char* const MainWindow::LANG_DE_MENU_TEXT   = "&Deutsch";
const char* const MainWindow::LANG_ES_MENU_TEXT   = "&Spanish";
const char* const MainWindow::LANG_CS_MENU_TEXT   = "&Czech";
//...................................................................
const char* const MainWindow::TabViewLabel        = QT_TR_NOOP( "&View" );
const char* const MainWindow::TabConsoleLabel     = QT_TR_NOOP( "&Console" );
const char* const MainWindow::MenuSystemText      = QT_TR_NOOP( "&System" );
const char* const MainWindow::MenuOperText        = QT_TR_NOOP( "&Operations" );
const char* const MainWindow::MenuLangText        = QT_TR_NOOP( "&Languages" );
const char* const MainWindow::MenuHelpText        = QT_TR_NOOP( "&Help" );
const char* const MainWindow::AboutQtMenuText     = QT_TR_NOOP( "About &Qt" );
const char* const MainWindow::AboutMenuText       = QT_TR_NOOP( "&About a Beesoft Commander" );
const char* const MainWindow::ReloadMenuText      = QT_TR_NOOP( "&Refresh files window" );
const char* const MainWindow::FilterMenuText      = QT_TR_NOOP( "&File filter" );
const char* const MainWindow::NcMenuText          = QT_TR_NOOP( "&NC selection mode" );
const char* const MainWindow::KdeMenuText         = QT_TR_NOOP( "&KDE selection mode" );
const char* const MainWindow::SystemInfoMenuText  = QT_TR_NOOP( "&System information" );
const char* const MainWindow::FileJoinMenuText    = QT_TR_NOOP( "Files &joining" );
const char* const MainWindow::FileJoinMenuAccel   = QT_TR_NOOP( "Ctrl+J" );
const char* const MainWindow::TreeMenuText        = QT_TR_NOOP( "&Tree of directories" );
const char* const MainWindow::SwitchMenuText      = QT_TR_NOOP( "&Switch panels" );
const char* const MainWindow::SwitchMenuAccel     = QT_TR_NOOP( "Ctrl+S");
const char* const MainWindow::TabNewMenuText      = QT_TR_NOOP( "&Create new tab" );
const char* const MainWindow::TabNewMenuAccel     = QT_TR_NOOP( "Ctrl+T" );
const char* const MainWindow::TabRemoveMenuText   = QT_TR_NOOP( "&Remove current tab" );
const char* const MainWindow::TabRemoveMenuAccel  = QT_TR_NOOP( "Ctrl+W" );
const char* const MainWindow::CmpFilesMenuText    = QT_TR_NOOP( "Compare &files" );
const char* const MainWindow::CmpDirMenuText      = QT_TR_NOOP( "Compare &directories" );
const char* const MainWindow::DirSynchMenuText    = QT_TR_NOOP( "Directories synchronisation" );
const char* const MainWindow::DateTimeMenuText    = QT_TR_NOOP( "&Last date/time modification" );
const char* const MainWindow::SelectMenuText      = QT_TR_NOOP( "&Select files via mask" );
const char* const MainWindow::UnselectMenuText    = QT_TR_NOOP( "&Unselect files via mask" );
const char* const MainWindow::ReversMenuText      = QT_TR_NOOP( "&Reverse selections" );
const char* const MainWindow::Md5CreateAccel      = QT_TR_NOOP( "Ctrl+5" );
const char* const MainWindow::Md5CreateMenuText   = QT_TR_NOOP( "Create MD5 control file" );
const char* const MainWindow::Md5CheckAccel       = QT_TR_NOOP( "Alt+5" );
const char* const MainWindow::Md5CheckMenuText    = QT_TR_NOOP( "Check MD5 file" );
const char* const MainWindow::FtpStartMenuText    = QT_TR_NOOP( "FTP connect" );
const char* const MainWindow::FtpEndMenuText      = QT_TR_NOOP( "FTP disconnect" );
const char* const MainWindow::ConfigureMenuText   = QT_TR_NOOP( "&Configure" );
const char* const MainWindow::ExitMenuText        = QT_TR_NOOP( "E&xit" );

//*******************************************************************
// CMainWindow
//*******************************************************************
MainWindow::MainWindow()
: QMainWindow         ()
, d_action_about_qt   ( new QAction( this ))
, d_action_about      ( new QAction( this ))
, d_action_reload     ( new QAction( this ))
, d_action_filter     ( new QAction( this ))
, d_action_nc         ( new QAction( this ))
, d_action_kde        ( new QAction( this ))
, d_action_exit       ( new QAction( this ))
, d_action_system_info( new QAction( this ))
, d_action_switch     ( new QAction( this ))
, d_action_tab_new    ( new QAction( this ))
, d_action_tab_remove ( new QAction( this ))
, d_action_joinfiles  ( new QAction( this ))
, d_action_tree       ( new QAction( this ))
, d_action_cmp_files  ( new QAction( this ))
, d_action_cmp_dirs   ( new QAction( this ))
, d_action_dir_synch  ( new QAction( this ))
, d_action_datetime   ( new QAction( this ))
, d_action_select     ( new QAction( this ))
, d_action_unselect   ( new QAction( this ))
, d_action_revers     ( new QAction( this ))
, d_action_md5_create ( new QAction( this ))
, d_action_md5_check  ( new QAction( this ))
, d_action_ftp_start  ( new QAction( this ))
, d_action_configure  ( new QAction( this ))
, d_action_ftp_end    ( new QAction( this ))
, d_action_lang_en    ( new QAction( this ))
, d_action_lang_ru    ( new QAction( this ))
, d_action_lang_pl    ( new QAction( this ))
, d_action_lang_de    ( new QAction( this ))
, d_action_lang_es    ( new QAction( this ))
, d_action_lang_cs    ( new QAction( this ))
, d_popup_system      ( new QPopupMenu( this ))
, d_popup_oper        ( new QPopupMenu( this ))
, d_popup_lang        ( new QPopupMenu( this ))
, d_popup_help        ( new QPopupMenu( this ))
, d_tool_bar          ( new QToolBar( this ))
, d_translator        ( new QTranslator( this ))
, d_workspace         ( 0 )
, d_lang_doit         ( TRUE )
, d_selection_doit    ( TRUE )
{
	for( int i = 0; i < BTN_NUMBER; ++i ) {
		d_function_buttons[ i ] = 0;
	}

	qApp->installTranslator( d_translator );
	read_lang_info();

	setIcon( QPixmap::fromMimeSource( APP_ICON ));
	setCaption( Shared::ProgramName + Shared::Spc + Shared::ProgramVersion );
	setAcceptDrops( FALSE );

	QFrame* const main_frame = new QFrame( this );
	setCentralWidget( main_frame );
		
	QVBoxLayout* const main_layout = new QVBoxLayout( main_frame );
	main_layout->setSpacing( Shared::LayoutMargin );
	main_layout->setMargin( Shared::LayoutMargin );
			
	QFrame* const work_area_frame = new QFrame( main_frame );
	main_layout->addWidget( work_area_frame );
	main_layout->setStretchFactor( work_area_frame, Shared::OverStretch );

	QVBoxLayout* const layout = new QVBoxLayout( work_area_frame );
	layout->addWidget( d_workspace = new Workspace( work_area_frame, this ));
			
	QFrame* const button_area_frame = new QFrame( main_frame );
	main_layout->addWidget( button_area_frame );
	create_button_area( button_area_frame );

	create_actions();
	init_menu( TRUE );
	init_tool_bar();
	
	connect( this       , SIGNAL( ftp_start()        ), d_workspace, SLOT( slot_ftp_start()    ));
	connect( this       , SIGNAL( ftp_end  ()        ), d_workspace, SLOT( slot_ftp_end()      ));
	connect( d_workspace, SIGNAL( end_of_ftp()       ), this       , SLOT( ftp_end_handle()      ));
	connect( this       , SIGNAL( lang_changed()     ), d_workspace, SLOT( lang_changed()      ));
	connect( this       , SIGNAL( selection_changed()), d_workspace, SLOT( selection_changed() )); 
	connect( d_workspace, SIGNAL( looks_refresh()    ), this       , SLOT( redraw_buttons()    ));
	
	update_lang();
}
// end of CMainWindow

//*******************************************************************
// ~MainWindow                                           DESTRUCTOR
//*******************************************************************
MainWindow::~MainWindow()
{
	Settings::save( CONFIG_GROUP_NAME, CONFIG_KEY_WIDTH, frameGeometry().width() );
	Settings::save( CONFIG_GROUP_NAME, CONFIG_KEY_HEIGHT, frameGeometry().height() );
}
// end of ~MainWindow

//###################################################################
//#                                                                 #
//#                        P R I V A T E                            #
//#                                                                 #
//###################################################################

//*******************************************************************
// polish                                                    PRIVATE
//*******************************************************************
void MainWindow::polish()
{
	static bool is_first_time = TRUE;
	
	if( is_first_time ) {
		int w, h;
	
		Settings::read( CONFIG_GROUP_NAME, CONFIG_KEY_WIDTH, w );
		Settings::read( CONFIG_GROUP_NAME, CONFIG_KEY_HEIGHT, h );
	
		if( w && h ) Shared::polish( this, QSize( w, h ));
		else         Shared::polish( this, 75, 75 );
		is_first_time = FALSE;
	}
}
// end of polish

//*******************************************************************
// create_actions                                            PRIVATE
//*******************************************************************
void MainWindow::create_actions ()
{
	const int lang = Config::instance()->lang();
	const int selection_mode = Config::instance()->selection_mode();
	
	//------- ABOUT QT ---------------------------------------
	connect( d_action_about_qt, SIGNAL( activated() ), qApp, SLOT( aboutQt() ));
	//------- ABOUT ------------------------------------------
	connect( d_action_about, SIGNAL( activated() ), this, SLOT( about() ));
	//------- RELOAD -----------------------------------------
	d_action_reload->setIconSet( QPixmap::fromMimeSource( RELOAD_MENU_ICON ));
	connect( d_action_reload, SIGNAL( activated() ), d_workspace, SIGNAL( reload() ));
	//------- FILTER -----------------------------------------
	d_action_filter->setAccel( QString(FILTER_MENU_ACCEL) );
	d_action_filter->setIconSet( QPixmap::fromMimeSource( FILTER_MENU_ICON ));
	d_action_filter->setToggleAction( TRUE );
	d_action_filter->setOn( TRUE );
	connect( d_action_filter, SIGNAL( toggled( bool ) ), d_workspace, SIGNAL( filter( bool ) ));
	//------- NC SELECTION MODE ------------------------------
	d_action_nc->setToggleAction( TRUE );
	d_action_nc->setOn( Config::NC == selection_mode );
	connect( d_action_nc, SIGNAL( toggled( bool ) ), this, SLOT( nc_selection( bool ) ));
	//------- KDE SELECTION MODE -----------------------------
	d_action_kde->setToggleAction( TRUE );
	d_action_kde->setOn( Config::KDE == selection_mode );
	connect( d_action_kde, SIGNAL( toggled( bool ) ), this, SLOT( kde_selection( bool ) ));
	//------- SYSTEM INFORMATION -----------------------------
	d_action_system_info->setIconSet( QPixmap::fromMimeSource( SystemInfoIcon ));
	connect( d_action_system_info, SIGNAL( activated() ), this, SLOT( system_info() ));
	//------- SWITCH PANELS ----------------------------------
	d_action_switch->setIconSet( QPixmap::fromMimeSource( SwitchMenuIcon ));
	d_action_switch->setAccel( QString( SwitchMenuAccel ));
	connect( d_action_switch, SIGNAL( activated() ), d_workspace, SLOT( slot_switch_panels() ));
	//------- CREATE NEW TAB ---------------------------------
	d_action_tab_new->setIconSet( QPixmap::fromMimeSource( TabNewMenuIcon ));
	d_action_tab_new->setAccel( QString( TabNewMenuAccel ));
	connect( d_action_tab_new, SIGNAL( activated() ), d_workspace, SLOT( slot_tab_new() ));
	//------- REMOVE CURRENT TAB -----------------------------
	d_action_tab_remove->setIconSet( QPixmap::fromMimeSource( TabRemoveMenuIcon ));
	d_action_tab_remove->setAccel( QString( TabRemoveMenuAccel ));
	connect( d_action_tab_remove, SIGNAL( activated() ), d_workspace, SLOT( slot_tab_remove() ));
	//------- JOIN FILES -------------------------------------
	d_action_joinfiles->setIconSet( QPixmap::fromMimeSource( FileJoinMenuIcon ));
	d_action_joinfiles->setAccel( QString( FileJoinMenuAccel ));
	connect( d_action_joinfiles, SIGNAL( activated() ), d_workspace, SLOT( slot_join_files() ));
	//------- TREE OF DIRECTORIES ----------------------------
	d_action_tree->setIconSet( QPixmap::fromMimeSource( Shared::TreeIcon ));
	connect( d_action_tree, SIGNAL( activated() ), d_workspace, SLOT( slot_tree() ));
	//------- COMPARE FILES ----------------------------------
	d_action_cmp_files->setIconSet( QPixmap::fromMimeSource( CMP_FILES_MENU_ICON ));
	connect( d_action_cmp_files, SIGNAL( activated() ), d_workspace, SLOT( cmp_files() ));
	//------- COMPARE DIRECTORIES ----------------------------
	d_action_cmp_dirs->setIconSet( QPixmap::fromMimeSource( Shared::CmpDirIcon ));
	connect( d_action_cmp_dirs, SIGNAL( activated() ), d_workspace, SLOT( cmp_dirs() ));
	//------- DIRECTORIES SYNCHRONISATION --------------------
	d_action_dir_synch->setIconSet( QPixmap::fromMimeSource( DirSynchIcon ));
	connect( d_action_dir_synch, SIGNAL( activated() ), d_workspace, SLOT( dir_synch() ));
	//------- COMPARE FILES ----------------------------------
	d_action_datetime->setIconSet( QPixmap::fromMimeSource( DATETIME_MENU_ICON ));
	connect( d_action_datetime, SIGNAL( activated() ), d_workspace, SLOT( date_time() ));
	//------- SELECT -----------------------------------------
	d_action_select->setIconSet( QPixmap::fromMimeSource( SELECT_MENU_ICON ));
	connect( d_action_select, SIGNAL( activated() ), d_workspace, SLOT( select() ));
	//------- UNSELECT ---------------------------------------
	d_action_unselect->setIconSet( QPixmap::fromMimeSource( UNSELECT_MENU_ICON ));
	connect( d_action_unselect, SIGNAL( activated() ), d_workspace, SLOT( unselect() ));
	//------- REVERS -----------------------------------------
	d_action_revers->setIconSet( QPixmap::fromMimeSource( REVERS_MENU_ICON ));
	connect( d_action_revers, SIGNAL( activated() ), d_workspace, SLOT( revers() ));
	//------- MD5 CREATE -----------------------------------------------
	d_action_md5_create->setAccel( QString( Md5CreateAccel ));
	connect( d_action_md5_create, SIGNAL( activated() ), d_workspace, SLOT( slot_md5_create() ));
	//------- MD5 CREATE -----------------------------------------------
	d_action_md5_check->setAccel( QString( Md5CheckAccel  ));
	connect( d_action_md5_check, SIGNAL( activated() ), d_workspace, SLOT( slot_md5_check() ));
	//------- FTP START --------------------------------------
	d_action_ftp_start->setIconSet( QPixmap::fromMimeSource( FTP_START_MENU_ICON ));
	connect( d_action_ftp_start, SIGNAL( activated() ), this, SLOT( slot_ftp_start() ));
	//------- FTP END ----------------------------------------
	d_action_ftp_end->setIconSet( QPixmap::fromMimeSource( FTP_END_MENU_ICON ));
	d_action_ftp_end->setEnabled( FALSE );
	connect( d_action_ftp_end, SIGNAL( activated() ), this, SLOT( ftp_end_handle() ));
	//------- CONFIGURE --------------------------------------
	d_action_configure->setIconSet( QPixmap::fromMimeSource( CONFIGURE_MENU_ICON ));
	connect( d_action_configure, SIGNAL( activated() ), d_workspace, SLOT( configure() ));
	//------- ENGLISH -----------------------------------------
	d_action_lang_en->setMenuText( LANG_EN_MENU_TEXT );
	d_action_lang_en->setToggleAction( TRUE );
	d_action_lang_en->setOn( Config::Lang_EN == lang );
	connect( d_action_lang_en, SIGNAL( toggled( bool ) ), this, SLOT( lang_en( bool ) ));
	//------- RUSSIAN -----------------------------------------
	d_action_lang_ru->setMenuText( LANG_RU_MENU_TEXT );
	d_action_lang_ru->setToggleAction( TRUE );
	d_action_lang_ru->setOn( Config::Lang_RU == lang );
	connect( d_action_lang_ru, SIGNAL( toggled( bool ) ), this, SLOT( lang_ru( bool ) ));
	//------- POLSKI -----------------------------------------
	d_action_lang_pl->setMenuText( LANG_PL_MENU_TEXT );
	d_action_lang_pl->setToggleAction( TRUE );
	d_action_lang_pl->setOn( Config::Lang_PL == lang );
	connect( d_action_lang_pl, SIGNAL( toggled( bool ) ), this, SLOT( lang_pl( bool ) ));
	//------- DEUTSCH ----------------------------------------
	d_action_lang_de->setMenuText( LANG_DE_MENU_TEXT );
	d_action_lang_de->setToggleAction( TRUE );
	d_action_lang_de->setOn( Config::Lang_DE == lang );
	connect( d_action_lang_de, SIGNAL( toggled( bool ) ), this, SLOT( lang_de( bool ) ));
	//------- SPANISH ----------------------------------------
	d_action_lang_es->setMenuText( LANG_ES_MENU_TEXT );
	d_action_lang_es->setToggleAction( TRUE );
	d_action_lang_es->setOn( Config::Lang_ES == lang );
	connect( d_action_lang_es, SIGNAL( toggled( bool ) ), this, SLOT( lang_es( bool ) ));
    //------- CZECH ----------------------------------------
	d_action_lang_cs->setMenuText( LANG_CS_MENU_TEXT );
	d_action_lang_cs->setToggleAction( TRUE );
	d_action_lang_cs->setOn( Config::Lang_CS == lang );
	connect( d_action_lang_cs, SIGNAL( toggled( bool ) ), this, SLOT( lang_cs( bool ) ));
	//------- KONIEC -----------------------------------------
	d_action_exit->setAccel( QString( EXIT_MENU_ACCEL ));
	d_action_exit->setIconSet( QPixmap::fromMimeSource( EXIT_MENU_ICON ));
	connect( d_action_exit, SIGNAL( activated() ), this,  SLOT( F10() ));
}
// end of create_actions

//*******************************************************************
// init_menu                                                 PRIVATE
//*******************************************************************
void MainWindow::init_menu( const bool in_first_time )
{
	if( FALSE == in_first_time ) menuBar()->clear();
	
	// ******* SYSTEM *******
	if( in_first_time ) init_popup_system_menu ();
	menuBar()->insertItem( tr(MenuSystemText), d_popup_system );
	
	// ******* OPERATIONS *******
	if( in_first_time ) init_popup_oper_menu();
	menuBar()->insertItem( tr(MenuOperText), d_popup_oper );	

	// ******* LANGUAGES *******
	if( in_first_time ) init_popup_lang_menu();
	menuBar()->insertItem( tr(MenuLangText), d_popup_lang );	

	// ******* HELP *******
	if( in_first_time ) init_popup_help_menu();
	menuBar()->insertItem( tr(MenuHelpText), d_popup_help );
}
// end of initMenu

//*******************************************************************
// init_popup_system_menu                                    PRIVATE
//*******************************************************************
void MainWindow::init_popup_system_menu ()
{
	d_action_reload->addTo( d_popup_system );
	d_action_filter->addTo( d_popup_system );
	d_popup_system->insertSeparator();
	d_action_nc->addTo( d_popup_system );
	d_action_kde->addTo( d_popup_system );
	d_popup_system->insertSeparator();
	d_action_configure->addTo( d_popup_system );
	d_popup_system->insertSeparator();
	d_action_exit->addTo( d_popup_system) ;
}
// end of init_system_submenu

//*******************************************************************
// init_popup_oper_menu                                      PRIVATE
//*******************************************************************
void MainWindow::init_popup_oper_menu ()
{
	d_action_tab_new->addTo( d_popup_oper );
	d_action_tab_remove->addTo( d_popup_oper );
	d_action_switch->addTo( d_popup_oper );
	d_popup_oper->insertSeparator();
	
	d_action_joinfiles->addTo( d_popup_oper );
	d_action_tree->addTo( d_popup_oper );
	d_action_cmp_files->addTo( d_popup_oper );
	d_action_cmp_dirs->addTo( d_popup_oper );
	d_action_dir_synch->addTo( d_popup_oper );
	d_action_datetime->addTo( d_popup_oper );
	d_popup_oper->insertSeparator();
	
	d_action_select->addTo( d_popup_oper );
	d_action_unselect->addTo( d_popup_oper );
	d_action_revers->addTo( d_popup_oper) ;
	d_popup_oper->insertSeparator();

	d_action_md5_create->addTo( d_popup_oper );
	d_action_md5_check->addTo( d_popup_oper) ;
	d_popup_oper->insertSeparator();

	d_action_ftp_start->addTo( d_popup_oper );
	d_action_ftp_end->addTo( d_popup_oper) ;
	d_popup_oper->insertSeparator();
	
	d_action_system_info->addTo( d_popup_oper );
}
// end of init_popup_oper_menu

//*******************************************************************
// init_popup_lang_menu                                      PRIVATE
//*******************************************************************
void MainWindow::init_popup_lang_menu()
{
	// in alphabetic order :)
	d_action_lang_cs->addTo( d_popup_lang );
	d_action_lang_de->addTo( d_popup_lang );
	d_action_lang_en->addTo( d_popup_lang );
	d_action_lang_pl->addTo( d_popup_lang );
	d_action_lang_ru->addTo( d_popup_lang );
	d_action_lang_es->addTo( d_popup_lang );
}
// end of init_popup_help_menu

//*******************************************************************
// init_popup_help_menu                                      PRIVATE
//*******************************************************************
void MainWindow::init_popup_help_menu()
{
	d_action_about_qt->addTo( d_popup_help );
	d_action_about->addTo( d_popup_help );
}
// end of init_popup_help_menu

//*******************************************************************
// init_tool_bar                                             PRIVATE
//*******************************************************************
void MainWindow::init_tool_bar()
{
	d_action_exit->addTo( d_tool_bar );
	d_tool_bar->addSeparator();
	
	d_action_filter->addTo( d_tool_bar );
	d_action_reload->addTo( d_tool_bar );
	d_tool_bar->addSeparator();

	d_action_tree->addTo( d_tool_bar );
	d_tool_bar->addSeparator();

//	d_action_datetime->addTo( d_tool_bar );
	d_action_cmp_files->addTo( d_tool_bar );
	d_action_cmp_dirs->addTo( d_tool_bar );
	d_tool_bar->addSeparator();

	d_action_select->addTo( d_tool_bar );
	d_action_unselect->addTo( d_tool_bar );
	d_action_revers->addTo( d_tool_bar );
	d_tool_bar->addSeparator();
	
	d_action_ftp_start->addTo( d_tool_bar );
	d_action_ftp_end->addTo( d_tool_bar );
//	d_tool_bar->addSeparator();
}
// end of init_tool_bar

//*******************************************************************
// create_button_area                                        PRIVATE
//*******************************************************************
void MainWindow::create_button_area( QFrame* const parent )
{
	QHBoxLayout* const layout = new QHBoxLayout( parent );
	if( layout ) {
		layout->setSpacing( 2 );
		
		for( int i = 0; i < BTN_NUMBER; ++i ) {
			//d_function_buttons[i] = new QPushButton( parent );
			d_function_buttons[i] = new QPushButton( parent );
			layout->addWidget( d_function_buttons[i] );
			d_function_buttons[i]->setFocusPolicy( NoFocus );
		}

		connect( d_function_buttons[BTN_HELP]  , SIGNAL( clicked() ), this       ,  SLOT( F1()     ));
		connect( d_function_buttons[BTN_ATTR]  , SIGNAL( clicked() ), d_workspace,  SLOT( slotF2() ));
		connect( d_function_buttons[BTN_VIEW]  , SIGNAL( clicked() ), d_workspace,  SLOT( slotF3() ));
		connect( d_function_buttons[BTN_EDIT]  , SIGNAL( clicked() ), d_workspace,  SLOT( slotF4() ));
		connect( d_function_buttons[BTN_COPY]  , SIGNAL( clicked() ), d_workspace,  SLOT( slotF5() ));
		connect( d_function_buttons[BTN_RENAME], SIGNAL( clicked() ), d_workspace,  SLOT( slotF6() ));
		connect( d_function_buttons[BTN_MKDIR] , SIGNAL( clicked() ), d_workspace,  SLOT( slotF7() ));
		connect( d_function_buttons[BTN_DELETE], SIGNAL( clicked() ), d_workspace,  SLOT( slotF8() ));
		connect( d_function_buttons[BTN_PACK]  , SIGNAL( clicked() ), d_workspace,  SLOT( slotF9() ));
		connect( d_function_buttons[BTN_QUIT]  , SIGNAL( clicked() ), this       ,  SLOT( F10()    ));
	}
}
// end of create_button_area

//*******************************************************************
// keyPressEvent                                             PRIVATE
//*******************************************************************
void MainWindow::keyPressEvent( QKeyEvent* e )
{
	switch( e->key() )
	{
		case Qt::Key_F1:
			e->accept();
			F1();
			break;
		case Qt::Key_F10:
			e->accept();
			F10();
			break;
	}
}
// end of keyPressEvent

//*******************************************************************
// closeEvent                                                PRIVATE
//*******************************************************************
void MainWindow::closeEvent (QCloseEvent* event)
{
	event->accept();
}
// end of closeEvent


//###################################################################
//#                                                                 #
//#                           S L O T S                             #
//#                                                                 #
//###################################################################


//*******************************************************************
// F1                                                   PRIVATE slot
//*******************************************************************
void MainWindow::F1()
{
	Help* dialog = new Help( this );
	if( dialog ) {
		dialog->exec();
		delete dialog;
		dialog = 0;
	}
}
// end of F1

//*******************************************************************
// F10                                                  PRIVATE slot
//*******************************************************************
void MainWindow::F10()
{
	close();
}
// end of F10

//*******************************************************************
// system_info                                          PRIVATE slot
//*******************************************************************
void MainWindow::system_info()
{
	SystemInfo* dlg = new SystemInfo( this );
	if( dlg ) {
		dlg->exec();
		delete dlg;
		dlg = 0;
	}
}
// end of system_info

//*******************************************************************
// about                                                PRIVATE slot
//*******************************************************************
void MainWindow::about()
{
	About* const dlg = new About( this );
	if( dlg ) {
		dlg->exec();
	}
}
// end of about

//*******************************************************************
// slot_ftp_start                                       PRIVATE slot
//*******************************************************************
void MainWindow::slot_ftp_start()
{
	d_action_ftp_start->setEnabled( FALSE );
	d_action_ftp_end->setEnabled( TRUE );
	emit ftp_start();
}
// end of slot_ftp_start

//*******************************************************************
// slot_ftp_end                                         PUBLIC slot
//*******************************************************************
void MainWindow::ftp_end_handle()
{
	d_action_ftp_start->setEnabled( TRUE );
	d_action_ftp_end->setEnabled( FALSE );
	emit ftp_end();
}
// end of slot_ftp_end

//*******************************************************************
// nc_selection                                         PRIVATE slot
//*******************************************************************
void MainWindow::nc_selection( const bool in_set )
{
	if( d_selection_doit ) {
		if( in_set ) {
			d_selection_doit = FALSE;
			d_action_kde->setOn( FALSE );
			d_selection_doit = TRUE;
			Settings::save( Config::AppConfigGroup, Config::SelectionModeKey, Config::NC );
			Config::instance()->refresh();
			emit selection_changed();
		}
		else {
			d_action_nc->setOn( TRUE );
		}
	}
}
// end of nc_selection

//*******************************************************************
// kde_selection                                        PRIVATE slot
//*******************************************************************
void MainWindow::kde_selection( const bool in_set )
{
	if( d_selection_doit ) {
		if( in_set ) {
			d_selection_doit = FALSE;
			d_action_nc->setOn( FALSE );
			d_selection_doit = TRUE;
			Settings::save( Config::AppConfigGroup, Config::SelectionModeKey, Config::KDE );
			Config::instance()->refresh();
			emit selection_changed();
		}
		else {
			d_action_kde->setOn( TRUE );
		}
	}
}
// end of kde_selection

//*******************************************************************
// lang_en                                              PRIVATE slot
//*******************************************************************
void MainWindow::lang_en( const bool in_set )
{
	if( d_lang_doit ) {
		if( in_set ) {
			d_lang_doit = FALSE;
			d_action_lang_ru->setOn( FALSE );
			d_action_lang_pl->setOn( FALSE );
			d_action_lang_de->setOn( FALSE );
			d_action_lang_es->setOn( FALSE );
			d_action_lang_cs->setOn( FALSE );
			d_lang_doit = TRUE;
			Settings::save( Config::AppConfigGroup, Config::LangKey, Config::Lang_EN );
			update_lang();
		}
		else {
			d_action_lang_en->setOn( TRUE );
		}
	}
}
// end of lang_en

//*******************************************************************
// lang_ru                                              PRIVATE slot
//*******************************************************************
void MainWindow::lang_ru( const bool in_set )
{
	if( d_lang_doit ) {
		if( in_set ) {
			d_lang_doit = FALSE;
			d_action_lang_en->setOn( FALSE );
			d_action_lang_pl->setOn( FALSE );
			d_action_lang_de->setOn( FALSE );
			d_action_lang_es->setOn( FALSE );
			d_action_lang_cs->setOn( FALSE );
			d_lang_doit = TRUE;
			Settings::save( Config::AppConfigGroup, Config::LangKey, Config::Lang_RU );
			update_lang();
		}
		else {
			d_action_lang_ru->setOn( TRUE );
		}
	}
}
// end of lang_ru

//*******************************************************************
// lang_pl                                              PRIVATE slot
//*******************************************************************
void MainWindow::lang_pl( const bool in_set )
{
	if( d_lang_doit ) {
		if( in_set ) {
			d_lang_doit = FALSE;
			d_action_lang_en->setOn( FALSE );
			d_action_lang_ru->setOn( FALSE );
			d_action_lang_de->setOn( FALSE );
			d_action_lang_es->setOn( FALSE );
			d_action_lang_cs->setOn( FALSE );
			d_lang_doit = TRUE;
			Settings::save( Config::AppConfigGroup, Config::LangKey, Config::Lang_PL );
			update_lang();
		}
		else {
			d_action_lang_pl->setOn( TRUE );
		}
	}
}
// end of lang_pl

//*******************************************************************
// lang_de                                              PRIVATE slot
//*******************************************************************
void MainWindow::lang_de( const bool in_set )
{
	if( d_lang_doit ) {
		if( in_set ) {
			d_lang_doit = FALSE;
			d_action_lang_en->setOn( FALSE );
			d_action_lang_ru->setOn( FALSE );
			d_action_lang_pl->setOn( FALSE );
			d_action_lang_es->setOn( FALSE );
			d_action_lang_cs->setOn( FALSE );
			d_lang_doit = TRUE;
			Settings::save( Config::AppConfigGroup, Config::LangKey, Config::Lang_DE );
			update_lang();
		}
		else {
			d_action_lang_de->setOn( TRUE );
		}
	}
}
// end of lang_pl

//*******************************************************************
// lang_es                                              PRIVATE slot
//*******************************************************************
void MainWindow::lang_es( const bool in_set )
{
	if( d_lang_doit ) {
		if( in_set ) {
			d_lang_doit = FALSE;
			d_action_lang_en->setOn( FALSE );
			d_action_lang_ru->setOn( FALSE );
			d_action_lang_de->setOn( FALSE );
			d_action_lang_pl->setOn( FALSE );
			d_action_lang_cs->setOn( FALSE );
			d_lang_doit = TRUE;
			Settings::save( Config::AppConfigGroup, Config::LangKey, Config::Lang_ES );
			update_lang();
		}
		else {
			d_action_lang_es->setOn( TRUE );
		}
	}
}
// end of lang_es

//*******************************************************************
// lang_cs                                              PRIVATE slot
//*******************************************************************
void MainWindow::lang_cs( const bool in_set )
{
	if( d_lang_doit ) {
		if( in_set ) {
			d_lang_doit = FALSE;
			d_action_lang_en->setOn( FALSE );
			d_action_lang_ru->setOn( FALSE );
			d_action_lang_de->setOn( FALSE );
			d_action_lang_pl->setOn( FALSE );
			d_action_lang_es->setOn( FALSE );
			d_lang_doit = TRUE;
			Settings::save( Config::AppConfigGroup, Config::LangKey, Config::Lang_CS );
			update_lang();
		}
		else {
			d_action_lang_cs->setOn( TRUE );
		}
	}
}
// end of lang_es

//*******************************************************************
// read_lang_info                                            PRIVATE
//*******************************************************************
void MainWindow::read_lang_info()
{
	Config::instance()->refresh();
	
	QString location = qApp->applicationDirPath();
	if( "/usr/bin" == location ) {
		location = "/usr/share/bsc/lang";
	}

	switch( Config::instance()->lang() ) {
		case Config::Lang_RU:
			d_translator->load( "bsc_ru.qm", location );	
			break;
		case Config::Lang_PL:
			d_translator->load( "bsc_pl.qm", location );	
			break;
		case Config::Lang_DE:
			d_translator->load( "bsc_de.qm", location );	
			break;
		case Config::Lang_ES:
			d_translator->load( "bsc_es.qm", location );	
			break;
		case Config::Lang_CS:
            d_translator->load( "bsc_cs.qm", location );
		    break;
		default:
			d_translator->clear();
	}
}
// end of read_lang_info

//*******************************************************************
// update_lang                                               PRIVATE
//*******************************************************************
void MainWindow::update_lang( const bool in_first_time )
{
	read_lang_info();
	retranslate_strings( in_first_time );
	emit lang_changed();
}
// end of update_lang

//*******************************************************************
// retranslate_strings                                       PRIVATE
//*******************************************************************
void MainWindow::retranslate_strings( const bool in_first_time )
{
	init_menu( in_first_time ) ;
	
	QString tmp;
	
	d_action_about_qt->setMenuText( tr(AboutQtMenuText) );
	d_action_about->setMenuText( tr(AboutMenuText) );
	d_action_reload->setMenuText( tmp = tr(ReloadMenuText) );
	d_action_reload->setToolTip( tmp.remove( Shared::Ampersand ));
	d_action_filter->setMenuText( tmp = tr(FilterMenuText) );
	d_action_filter->setToolTip( tmp.remove( Shared::Ampersand ) );
	d_action_nc->setMenuText( tmp = tr(NcMenuText) );
	d_action_nc->setToolTip( tmp.remove( Shared::Ampersand ) );
	d_action_kde->setMenuText( tmp = tr(KdeMenuText) );
	d_action_kde->setToolTip( tmp.remove( Shared::Ampersand ) );
	d_action_system_info->setMenuText( tmp = tr( SystemInfoMenuText ) );
	d_action_system_info->setToolTip( tmp.remove( Shared::Ampersand ) );
	d_action_switch->setMenuText( tmp = tr(SwitchMenuText) );
	d_action_switch->setToolTip( tmp.remove( Shared::Ampersand ) );
	d_action_tab_new->setMenuText( tmp = tr(TabNewMenuText) );
	d_action_tab_new->setToolTip( tmp.remove( Shared::Ampersand ) );
	d_action_tab_remove->setMenuText( tmp = tr(TabRemoveMenuText) );
	d_action_tab_remove->setToolTip( tmp.remove( Shared::Ampersand ) );
	d_action_joinfiles->setMenuText( tmp = tr(FileJoinMenuText) );
	d_action_joinfiles->setToolTip( tmp.remove( Shared::Ampersand ) );
	d_action_tree->setMenuText( tmp = tr(TreeMenuText) );
	d_action_tree->setToolTip( tmp.remove( Shared::Ampersand ) );
	d_action_cmp_files->setMenuText( tmp = tr(CmpFilesMenuText) );
	d_action_cmp_files->setToolTip( tmp.remove( Shared::Ampersand ) );
	d_action_cmp_dirs->setMenuText( tmp = tr(CmpDirMenuText) );
	d_action_cmp_dirs->setToolTip( tmp.remove( Shared::Ampersand ) );
	d_action_dir_synch->setMenuText( tmp = tr(DirSynchMenuText) );
	d_action_dir_synch->setToolTip( tmp.remove( Shared::Ampersand ) );
	d_action_datetime->setMenuText( tmp = tr(DateTimeMenuText) );
	d_action_datetime->setToolTip( tmp.remove( Shared::Ampersand ) );
	d_action_select->setMenuText( tmp = tr(SelectMenuText) );
	d_action_select->setToolTip( tmp.remove( Shared::Ampersand ) );
	d_action_unselect->setMenuText( tmp = tr(UnselectMenuText) );
	d_action_unselect->setToolTip( tmp.remove( Shared::Ampersand ) );
	d_action_revers->setMenuText( tmp = tr(ReversMenuText) );
	d_action_revers->setToolTip( tmp.remove( Shared::Ampersand ) );

	d_action_md5_create->setMenuText( tmp = tr(Md5CreateMenuText) );
	d_action_md5_create->setToolTip( tmp.remove( Shared::Ampersand ) );
	d_action_md5_check->setMenuText( tmp = tr(Md5CheckMenuText) );
	d_action_md5_check->setToolTip( tmp.remove( Shared::Ampersand ) );
	
	d_action_ftp_start->setMenuText( tmp = tr(FtpStartMenuText) );
	d_action_ftp_start->setToolTip( tmp.remove( Shared::Ampersand ) );
	d_action_ftp_end->setMenuText( tmp = tr(FtpEndMenuText) );
	d_action_ftp_end->setToolTip( tmp.remove( Shared::Ampersand ) );
	d_action_configure->setMenuText( tr(ConfigureMenuText) );
	d_action_exit->setMenuText( tmp = tr(ExitMenuText) );
	d_action_exit->setToolTip( tmp.remove( Shared::Ampersand ) );

	redraw_buttons();
}
// end of retranslate_strings

//*******************************************************************
// redraw_buttons                                       PRIVATE slot
//*******************************************************************
void MainWindow::redraw_buttons()
{
	QString label[BTN_NUMBER] = {
		tr( Shared::MenuHelp ),
		tr( Shared::MenuAccess ),
		tr( Shared::MenuView ),
		tr( Shared::MenuEdit ),
		tr( Shared::MenuCopy ),
		tr( Shared::MenuRename ),
		tr( Shared::MenuMkDir ),
		tr( Shared::MenuDelete ),
		tr( Shared::MenuPack ),
		tr( Shared::MenuQuit )
	};

	for( int i = 0; i < BTN_NUMBER; ++i ) {
		d_function_buttons[i]->setFont( Config::instance()->fn_font() );

		QString button_label;
		switch( Config::instance()->fn_disp_mode() ) {
			case Config::ONLY_FN:
				button_label = "F" + QString::number( i + 1 );
				break;
			case Config::ONLY_TEXT:
				button_label = label[i];
				break;
			case Config::FULL_DESC:
				button_label = "F" + QString::number( i + 1 ) + Shared::Spc + label[i];
				break;
			case Config::TWO_LINES:
				button_label = "F" + QString::number( i + 1 ) + "\n" + label[i];;
			break;
		}
		d_function_buttons[i]->setText( button_label );
	}
}
// end of redraw_buttons
