
/*
 *  Diverse Bristol audio routines.
 *  Copyright (c) by Nick Copeland <nickycopeland@hotmail.com> 1996,2009
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>
#include <sys/types.h>

#include "brightoninternals.h"
#include "brightonX11.h"

static float opacity = 1.0;
static int shiftflag = 0;
static int controlflag = 0;

extern void brightonKeyInput(brightonWindow*, int, int);
extern void printColorCacheStats(brightonWindow *);

extern int brightonMenu();

int
brightonKeyPress(brightonWindow *bwin, brightonEvent *event)
{
	brightonIResource *panel;

	if ((event->key == 'p') && controlflag)
	{
		brightonXpmWrite(bwin, "/tmp/brighton.xpm");
		printColorCacheStats(bwin);
	}

	/*
	 * All of the following up to the dispatch is for the top layer opacity, it
	 * could go into a separate file?
	 *
	 * T should just be a toggle between totally opaque and the current 
	 * setting. There are a few other keys that are natively interpreted for
	 * rather trivial reasons before being passed through to the GUI. These
	 * implicit mappings should be dropped.
	 */
	if ((event->key == 't') && (controlflag))
	{
		float hold;

		hold = bwin->opacity;
		bwin->opacity = opacity;
		opacity = hold;
		brightonFinalRender(bwin, 0, 0, bwin->width, bwin->height);
	} else if ((event->key == 65508) || (event->key == 65507)) {
		controlflag = 1;
	} else if ((event->key == 65505) || (event->key == 65506)) {
		shiftflag = 1;
	} else if ((event->key == 'o') && (controlflag)) {
		/*
		 * 'O' should be to make more opaque, 'o' more transparent. This is not
		 * as * trivial as it seems since the shift key is always a separate
		 * event.
		 */
		if (shiftflag)
		{
			if (bwin->opacity == 1.0f)
				bwin->opacity = 0.2f;
			else if ((bwin->opacity += 0.1f) > 1.0f)
				bwin->opacity = 1.0f;
			brightonFinalRender(bwin, 0, 0, bwin->width, bwin->height);
		} else {
			if (bwin->opacity <= 0.21f)
				bwin->opacity = 1.0f;
			else if ((bwin->opacity -= 0.2f) < 0.2f)
				bwin->opacity = 0.2f;
			brightonFinalRender(bwin, 0, 0, bwin->width, bwin->height);
		}
	}

	/*
	 * We now want to look to see if the request needs to be dispatched to any
	 * devices. This means 'if the mouse is within any device then pass the key
	 * event to that device.
	 */
	if ((bwin->flags & BRIGHTON_DEV_ACTIVE) && (bwin->activepanel != 0))
	{
		bwin->activepanel->configure(bwin, bwin->activepanel, event);
	} else if ((panel = brightonPanelLocator(bwin, event->x, event->y))
		> (brightonIResource *) NULL)
	{
		if (panel->configure)
			panel->configure(bwin, panel, event);
	}

	/*
	 * Finally, as this is a key press event I want to pass it through to the
	 * midi library. This is a rather ugly hack to have a more useful keyboard
	 * mapping. We should only really do this from the keyboard panel, but that
	 * if for later study.
	 */
	if (controlflag == 0)
		brightonKeyInput(bwin, event->key, 1);

	return(0);
}

int
brightonKeyRelease(brightonWindow *bwin, brightonEvent *event)
{
	brightonIResource *panel;

/*	printf("brightonKeyRelease(%i)\n", event->key); */

	if ((event->key == 65508) || (event->key == 65507))
		controlflag = 0;
	else if ((event->key == 65505) || (event->key == 65506))
		shiftflag = 0;

	if ((bwin->flags & BRIGHTON_DEV_ACTIVE) && (bwin->activepanel != 0))
		bwin->activepanel->configure(bwin, bwin->activepanel, event);
	else if ((panel = brightonPanelLocator(bwin, event->x, event->y))
		> (brightonIResource *) NULL)
	{
		if (panel->configure)
			panel->configure(bwin, panel, event);
	}

	if (controlflag == 0)
		brightonKeyInput(bwin, event->key, 0);
	return(0);
}

int
brightonButtonPress(brightonWindow *bwin, brightonEvent *event)
{
	if (event->key == BRIGHTON_BUTTON3) {
			brightonMenu(bwin, event->x, event->y, 100, 200);
		return(0);
	}

	/*
	 * We need to determine which device is under the selection, and force
	 * statefull delivery of events to that device for further motion.
	 */
	bwin->activepanel = 0;

	if ((bwin->activepanel = brightonPanelLocator(bwin, event->x, event->y))
		> (brightonIResource *) NULL)
	{
		bwin->flags |= BRIGHTON_DEV_ACTIVE;

		event->command = BRIGHTON_BUTTONPRESS;

		if (bwin->activepanel->configure)
			bwin->activepanel->configure(bwin, bwin->activepanel, event);
	} else
		bwin->flags &= ~BRIGHTON_DEV_ACTIVE;
	return(0);
}

int
brightonButtonRelease(brightonWindow *bwin, brightonEvent *event)
{
/*printf("brightonButtonRelease(%x, %x, %x)\n", bwin, bwin->activepanel, */
/*bwin->activepanel->configure); */

	event->command = BRIGHTON_BUTTONRELEASE;

	if (event->key == BRIGHTON_BUTTON3) {
			brightonMenu(bwin, event->x, event->y, 100, 200);
		return(0);
	}

	/*if (bwin->activepanel && bwin->activepanel->configure) */
	if ((bwin->flags & BRIGHTON_DEV_ACTIVE) && (bwin->activepanel != 0))
		bwin->activepanel->configure(bwin, bwin->activepanel, event);

	bwin->flags &= ~BRIGHTON_DEV_ACTIVE;
	bwin->activepanel = 0;
	return(0);
}

int
brightonMotionNotify(brightonWindow *bwin, brightonEvent *event)
{
	if ((bwin->flags & BRIGHTON_DEV_ACTIVE) && (bwin->activepanel != 0))
	{
		if (bwin->activepanel->configure)
			bwin->activepanel->configure(bwin, bwin->activepanel, event);
	}
	return(0);
}

int
brightonEnterNotify(brightonWindow *bwin, brightonEvent *event)
{
/*	printf("brightonEnterNotify()\n"); */
	BAutoRepeat(bwin->display, 0);
	return(0);
}

int
brightonLeaveNotify(brightonWindow *bwin, brightonEvent *event)
{
/*	printf("brightonLeaveNotify()\n"); */
	BAutoRepeat(bwin->display, 1);
	return(0);
}

int
brightonFocusIn(brightonWindow *bwin, brightonEvent *event)
{
/*	printf("brightonFocusIn()\n"); */
	return(0);
}

int
brightonFocusOut(brightonWindow *bwin, brightonEvent *event)
{
/*	printf("brightonFocusOut()\n"); */
	return(0);
}

int
brightonKeymapNotify(brightonWindow *bwin, brightonEvent *event)
{
	printf("brightonKeymapNotify()\n");
	return(0);
}

int
brightonExpose(brightonWindow *bwin, brightonEvent *event)
{
/*	printf("brightonExpose(%i %i %i %i)\n", */
/*		event->x, event->y, event->w, event->h); */

	BCopyArea(bwin->display, 
		event->x, event->y, event->w, event->h, event->x, event->y);
	return(0);
}

int
brightonGraphicsExpose(brightonWindow *bwin, brightonEvent *event)
{
/*	printf("brightonGraphicsExpose()\n"); */
	return(0);
}

int
brightonNoExpose(brightonWindow *bwin, brightonEvent *event)
{
/*	printf("brightonNoExpose()\n"); */
	return(0);
}

int
brightonVisibilityNotify(brightonWindow *bwin, brightonEvent *event)
{
/*	printf("brightonVisibilityNotify()\n"); */
	return(0);
}

int
brightonCreateNotify(brightonWindow *bwin, brightonEvent *event)
{
/*	printf("brightonCreateNotify()\n"); */
	return(0);
}

int
brightonDestroyNotify(brightonWindow *bwin, brightonEvent *event)
{
	/*
	 * Counter intuitive, we do nothing here. The process gets a SIGPIPE and
	 * the GUI intercepts this to request the engine to terminate the algo
	 * represented by this GUI. Engine will terminate when all algos have
	 * finished.
	 */
/*	printf("brightonDestroyNotify()\n"); */
	return(0);
}

int
brightonUnmapNotify(brightonWindow *bwin, brightonEvent *event)
{
/*	printf("brightonUnmapNotify()\n"); */
	return(0);
}

int
brightonMapNotify(brightonWindow *bwin, brightonEvent *event)
{
/*	printf("brightonMapNotify()\n"); */
	return(0);
}

int
brightonMapRequest(brightonWindow *bwin, brightonEvent *event)
{
/*	printf("brightonMapRequest()\n"); */
	return(0);
}

int
brightonReparentNotify(brightonWindow *bwin, brightonEvent *event)
{
/*	printf("brightonReparentNotify()\n"); */
	return(0);
}

int
brightonConfigureNotify(brightonWindow *bwin, brightonEvent *event)
{
/*	printf("brightonConfigureNotify(%i, %i, %i, %i)\n", */
/*		event->x, event->y, event->w, event->h); */

	if (bwin->flags & BRIGHTON_SET_SIZE)
	{
		bwin->flags &= ~BRIGHTON_SET_SIZE;
		return(0);
	}

	if ((bwin->width != event->w) || (bwin->height != event->h))
		brightonWorldChanged(bwin, event->w, event->h);

	if ((bwin->width != event->w) || (bwin->height != event->h))
		BResizeWindow(bwin->display, bwin, bwin->width, bwin->height);
	return(0);
}

int
brightonConfigureRequest(brightonWindow *bwin, brightonEvent *event)
{
	printf("brightonConfigureRequest()\n");
	return(0);
}

int
brightonGravityNotify(brightonWindow *bwin, brightonEvent *event)
{
	printf("brightonGravityNotify()\n");
	return(0);
}

int
brightonResizeRequest(brightonWindow *bwin, brightonEvent *event)
{
/*	printf("brightonResizeRequest()\n"); */

	if ((bwin->width != event->w) || (bwin->height != event->h))
	{
		/*
		 * Our size has changed. We need to re-render the background, and then 
		 * repaint it.
		 */
		brightonWorldChanged(bwin, event->w, event->h);

/*
		if ((bwin->width != event->w) || (bwin->height != event->h))
			BResizeWindow(bwin->display, bwin->win, bwin->width, bwin->height);
*/
	}
	return(0);
}

int
brightonCirculateNotify(brightonWindow *bwin, brightonEvent *event)
{
	printf("brightonCirculateNotify()\n");
	return(0);
}

int
brightonCirculateRequest(brightonWindow *bwin, brightonEvent *event)
{
	printf("brightonCirculateRequest()\n");
	return(0);
}

int
brightonPropertyNotify(brightonWindow *bwin, brightonEvent *event)
{
	printf("brightonPropertyNotify()\n");
	return(0);
}

int
brightonSelectionClear(brightonWindow *bwin, brightonEvent *event)
{
	printf("brightonSelectionClear()\n");
	return(0);
}

int
brightonSelectionRequest(brightonWindow *bwin, brightonEvent *event)
{
	printf("brightonSelectionRequest()\n");
	return(0);
}

int
brightonSelectionNotify(brightonWindow *bwin, brightonEvent *event)
{
	printf("brightonSelectionNotify()\n");
	return(0);
}

int
brightonColormapNotify(brightonWindow *bwin, brightonEvent *event)
{
	printf("brightonColormapNotify()\n");
	return(0);
}

int
brightonClientMessage(brightonWindow *bwin, brightonEvent *event)
{
	printf("brightonClientRequest()\n");
	return(0);
}

int
brightonMappingNotify(brightonWindow *bwin, brightonEvent *event)
{
	printf("brightonMappingNotify()\n");
	return(0);
}

int
brightonNullHandler(brightonWindow *bwin, brightonEvent *event)
{
	printf("brightonNullHandler()\n");
	return(0);
}

typedef int (*eventRoutine)(brightonWindow *, brightonEvent *);

/*
 * Check out the bEvent from the library, it can mask some events that are
 * considered "not important", but actually may be.
 */
eventRoutine defaultHandlers[BLASTEvent] = {
	brightonNullHandler,
	brightonNullHandler,
	brightonKeyPress,
	brightonKeyRelease,
	brightonButtonPress,
	brightonButtonRelease,
	brightonMotionNotify,
	brightonEnterNotify,
	brightonLeaveNotify,
	brightonFocusIn,
	brightonFocusOut,
	brightonKeymapNotify,
	brightonExpose,
	brightonGraphicsExpose,
	brightonNoExpose,
	brightonVisibilityNotify,
	brightonCreateNotify,
	brightonDestroyNotify,
	brightonUnmapNotify,
	brightonMapNotify,
	brightonMapRequest,
	brightonReparentNotify,
	brightonConfigureNotify,
	brightonConfigureRequest,
	brightonGravityNotify,
	brightonResizeRequest,
	brightonCirculateNotify,
	brightonCirculateRequest,
	brightonPropertyNotify,
	brightonSelectionClear,
	brightonSelectionRequest,
	brightonSelectionNotify,
	brightonColormapNotify,
	brightonClientMessage,
	brightonMappingNotify
};

void brightonInitDefHandlers(brightonWindow *bwin)
{
	int i;

	for (i = 0; i < BLASTEvent; i++)
		bwin->callbacks[i] = defaultHandlers[i];
}

void
brightonOldEventLoop(brightonDisplay **dlist)
{
	brightonDisplay *display;
	brightonEvent event;
	brightonWindow *bwin = (*dlist)->bwin;

	while (1) {
		/*
		 * BNextEvent will block due to it using XNextEvent. This makes the
		 * use of other functions rather difficult. Will rework this to use
		 * XCheckMaskEvent which will not block.
		 *
		 *	while (1) {
		 *		foreach (win) {
		 * 			BCheckMaskEvent(bwin, &event);
		 *		}
		 *	}
		 *
		 * We need this since the MIDI routines also need to be checked, so we
		 * will select on the ALSA SEQ port looking for controller changes so
		 * that they can modify the GUI display.
		 */
		BNextEvent(bwin->display, &event);

		if (event.command == BRIGHTON_NONE)
			continue;

		/*
		 * This may need to be changed to a semaphore, but I am hoping all GUI
		 * activity can occupy a single thread only.
		 */
		bwin->flags |= BRIGHTON_BUSY;

		/*
		 * Look for the right window.
		 */
		display = *dlist;

		while (display != 0)
		{
			if (event.wid == ((brightonWindow *) display->bwin)->win)
				break;

			if ((event.type == BRIGHTON_DESTROY)
				&& (((brightonWindow *) display->bwin)->parentwin == event.wid))
				break;

			display = display->next;
		}

		if ((display == 0 || (event.type < 0) || (event.type >= BLASTEvent)))
			continue;

		((brightonWindow *) display->bwin)->callbacks[event.type]
			(display->bwin, &event);

		bwin->flags &= ~BRIGHTON_BUSY;
	}
}

int
brightonEventLoop(brightonDisplay **dlist)
{
	brightonDisplay *display;
	brightonEvent event;
	brightonWindow *bwin = (*dlist)->bwin;

/*	while (1) { */
		/*
		 * BNextEvent will block due to it using XNextEvent. This makes the
		 * use of other functions rather difficult. Will rework this to use
		 * XCheckMaskEvent which will not block.
		 *
		 *	while (1) {
		 *		foreach (win) {
		 * 			BCheckMaskEvent(bwin, &event);
		 *		}
		 *	}
		 *
		 * We need this since the MIDI routines also need to be checked, so we
		 * will select on the ALSA SEQ port looking for controller changes so
		 * that they can modify the GUI display.
		 */
		while (BNextEvent(bwin->display, &event) > 0)
		{
			if (event.command == BRIGHTON_NONE)
				continue;

			/*
			 * This may need to be changed to a semaphore, but I am hoping all
			 * GUI activity can occupy a single thread only.
			 */
			bwin->flags |= BRIGHTON_BUSY;

			/*
			 * Look for the right window.
			 */
			display = *dlist;

			while (display != 0)
			{
				if (event.wid == ((brightonWindow *) display->bwin)->win)
					break;

				if ((event.type == BRIGHTON_DESTROY)
					&& (((brightonWindow *) display->bwin)->parentwin
						== event.wid))
					break;

				display = display->next;
			}

			if ((display == 0 || (event.type < 0)
				|| (event.type >= BLASTEvent)))
				continue;

			((brightonWindow *) display->bwin)->callbacks[event.type]
				(display->bwin, &event);

			bwin->flags &= ~BRIGHTON_BUSY;

			/*
			 * Configure events are typically big resizes which are long events.
			 * If we handle all of them (they come in buckets) at once it is
			 * possible the engine will detect GUI failure from active sense
			 */
			if (event.command == BRIGHTON_CONFIGURE)
				return(1);
		}

		/*
		 * This will now become a select on the ALSA SEQ socket looking for 
		 * MIDI events. Not certain how they will be linked into the GUI at
		 * the moment. For now this is just a sleep until the ALSA SEQ interface
		 * registration code has been finalised.
		 *
		 * What we will be looking for are events on a MIDI channel, we will
		 * look for that MIDI channel in our window lists. We want to respond
		 * to key events and reflect that in the GUI optionally, but not send
		 * the key events since the engine will handle all those. We also want 
		 * to look for controller values and have some configurable method to
		 * link those to our controls. Now this linkage will be done via the
		 * GUI, preferably, with the <Control><Middle Mouse><MIDI CC # motion>.
		 * Since the GUI handles this then we can dispatch events to another
		 * module that does the linkage. Need to be able to save and retrieve
		 * configurations - <Control><Middle> will go to this module, and all
		 * MIDI controller events as well, and it will make the linkage and
		 * dispatch the events.
		 * We should try and have a vkeydb type X event keymap.
		 */
/*		usleep(10000); */
/*	} */
	return(0);
}

