/*
*
*  A2DPD - Bluetooth A2DP daemon control application for Linux
*
*  Copyright (C) 2006-2007  Frédéric DALLEAU <frederic.dalleau@palmsource.com>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>

// D-Bus constants
#define A2DPD_SERVICE_NAME			"com.access.a2dpd"
#define A2DPD_SERVICE_PATH			"/com/access/a2dpd"
#define A2DPD_SERVER_INTERFACE_NAME		A2DPD_SERVICE_NAME ".server"

static struct {
	char* name;
	char* params;
	int syntax;
} methods [] = {
	{ "Startup",		"",				0 },
	{ "Exit",		"",				0 },
	{ "Connect",		"",				0 },
	{ "Disconnect",		"",				0 },
	{ "AVRCPDisconnect",	"",				0 },
	{ "StreamStart",	"",				0 },
	{ "StreamSuspend",	"",				0 },
	{ "Save",		"",				0 },
	{ "AutoConnect",	"<0|1|swap>",			1 },
	{ "SetDebug",		"<0|1>",			1 },
	{ "SetOutput",		"<filename>",			1 },
	{ "SetAddress",		"<none|auto|alsa|sco|bdaddr>",	1 },
	{ "SetVolume",		"0..16",			1 },
	{ "SetFlags",		"0..3",				1 },
	{ "SetReReadConfig",	"0|1",				1 },
	{ "GetAddress",		"",				2 },
	{ "GetState",		"",				2 },
};

int main(int argc, char *argv[])
{
	GError* gError = NULL;
	DBusGConnection *gconnection = NULL;
	DBusGProxy *gproxy = NULL;
	int bus = DBUS_BUS_SYSTEM;
	int success = 0;
	int error = 0;

	// Init
	g_type_init();

	while(!success && !error) {
		gconnection = dbus_g_bus_get(bus, &gError);

		if(gconnection != NULL) {
			gproxy = dbus_g_proxy_new_for_name(gconnection, A2DPD_SERVICE_NAME, A2DPD_SERVICE_PATH, A2DPD_SERVER_INTERFACE_NAME);

			if(gproxy!=NULL) {
				if(argc>1) {
					// Parse arguments
					char* method_name = "";
					char* addr = "";
					gchar* result = NULL;
					int syntax = 0;
					int i;
					for(i=0; i<sizeof(methods)/sizeof(methods[0]); i++) {
						// Case insensitive
						if(!strcasecmp(argv[1], methods[i].name)) {
							method_name=methods[i].name;
							syntax=methods[i].syntax;
							if(syntax==1) {
								addr = argv[2];
							}
						}
					}
		
					if(method_name[0]) {
						// Call method
						switch(syntax) {
						case 0:
							if(dbus_g_proxy_call(gproxy, method_name, &gError, G_TYPE_INVALID, G_TYPE_INVALID)) {
								success = 1;
								printf("Success: %s()\n", method_name);
							} else {
								//printf("%s() failed %s\n", method_name, gError->message);
							}
							break;
						case 1:
							if(dbus_g_proxy_call(gproxy, method_name, &gError, G_TYPE_STRING, addr, G_TYPE_INVALID, G_TYPE_INVALID)) {
								success = 1;
								printf("Success: %s(%s)\n", method_name, addr);
							} else {
								//printf("%s(%s) failed %s\n", method_name, addr, gError->message);
							}
							break;
						case 2:
							if(dbus_g_proxy_call(gproxy, method_name, &gError, G_TYPE_INVALID, G_TYPE_STRING, &result, G_TYPE_INVALID)) {
								success = 1;
								printf("Success: %s() == %s\n", method_name, result);
								g_free(result);
							} else {
								//printf("%s(%s) failed %s\n", method_name, addr, gError->message);
							}
							break;
						default:
							printf("bad syntax\n");
							error = 1;
						}
					} else {
						printf("Invalid method name %s\n", method_name);
						error = 1;
					}
				} else {
					// Display usage if at least method is not given
					int i;
					printf("Usage:\n");
					for(i=0; i<sizeof(methods)/sizeof(methods[0]); i++) {
						printf("\t%s\t%s\n", methods[i].name, methods[i].params);
					}
					error = 1;
				}
				g_object_unref(gproxy);
			} else {
				printf("ERROR: Couldn't create proxy to service\n");
				error = 1;
			}
		} else {
			printf("ERROR: Couldn't connect to %s bus\n", (bus==DBUS_BUS_SYSTEM)?"system":"session");
			error = 1;
		}

		if(gError!=NULL) {
			g_error_free(gError);
			gError = NULL;
		}

		// Try session bus if system bus failed
		if(!success && bus == DBUS_BUS_SYSTEM) {
			bus = DBUS_BUS_SESSION;
		// If session bus failed, leave program
		} else if(!success && bus == DBUS_BUS_SESSION) {
			break;
		}
	}

	return success?0:error?-1:-2;
}
