/* 
 * Copyright (c) 2010 Craig Heffner
 *
 * This software is provided under the MIT license. For the full text of this license, please see
 * the COPYING file included with this code, or visit http://www.opensource.org/licenses/MIT.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/wait.h>
#include "dd.h"
#include "nargv.h"
#include "common.h"

int dd_rule_count = 0;
struct dd_rule **dd_rules = NULL;

/* Get the index of an existing rule in dd_rules */
int dd_rule_index(char *match, char *extension)
{
	int i = 0, index = -1;

	for(i=0; i<dd_rule_count; i++)
	{
		if((strcmp(dd_rules[i]->match, match) == 0) && (strcmp(dd_rules[i]->ext, extension) == 0))
		{
			index = i;
			break;
		}
	}

	return index;
}

/* Add a DD rule to extract matching entries from the target file */
void add_dd_rule(char *match, char *extension, char *cmd)
{
	int index = -1;
	void *tmp = NULL;

	/* Check first to see if this rule already exists. */
	index = dd_rule_index(match, extension);

	/* If not, create room for a new rule */
	if(index == -1)
	{
		tmp = dd_rules;
		dd_rules = realloc(dd_rules, ((dd_rule_count+1) * sizeof(struct dd_rule *)));
		if(!dd_rules)
		{
			perror("realloc");
			if(tmp) free(tmp);
		}
		else
		{
			index = dd_rule_count;
			dd_rule_count++;
		}
	}
	else
	{
		/* If so, we will overwrite it */
		free_dd_rule_members(index);
	}
			
	if(index != -1)
	{
		dd_rules[index] = malloc(sizeof(struct dd_rule));
		if(dd_rules[index])
		{
			memset(dd_rules[index], 0, sizeof(struct dd_rule));
			
			dd_rules[index]->count = 0;
			dd_rules[index]->match = strdup(match);
			dd_rules[index]->ext = strdup(extension);
			if(cmd)
			{
				dd_rules[index]->cmd = strdup(cmd);
			}
			else 
			{
				dd_rules[index]->cmd = NULL;
			}
		}
	}
}

/* Frees all the members of the dd_rule structure at the given index in dd_rules */
void free_dd_rule_members(int i)
{
	if(dd_rules[i])
	{
		dd_rules[i]->count = 0;
		if(dd_rules[i]->match) free(dd_rules[i]->match);
		if(dd_rules[i]->ext) free(dd_rules[i]->ext);
		if(dd_rules[i]->cmd) free(dd_rules[i]->cmd);
	}
}

/* Frees all DD rule entries */
void free_dd_rules(void)
{
	int i = 0;

	if(dd_rules)
	{
		for(i=0; i<dd_rule_count; i++)
		{
			if(dd_rules[i])
			{
				free_dd_rule_members(i);
				free(dd_rules[i]);
			}
		}

		free(dd_rules);
	}

	dd_rule_count = 0;
}

/* Checks to see if a description matches any DD rules */
int matches_dd_rule(char *description)
{
	int i = 0, retval = -1;

	for(i=0; i<dd_rule_count; i++)
	{
		if((strcmp(dd_rules[i]->match, DD_WILDCARD) == 0) || 
		   (string_contains(description, dd_rules[i]->match)))
		{
			retval = i;
			dd_rules[i]->count++;
			break;
		}
	}

	return retval;
}

/* Save data to a file using the DD rule extension and offset value as the file name */
void dd(int index, uint32_t offset, char *data, size_t size, int verbose)
{
	FILE *fp = NULL;
	NARGV *nargv = NULL;
	int i = 0, fd = 0, status = 0;
	char file[FILENAME_MAX] = { 0 };

	if(index < dd_rule_count)
	{
		snprintf((char *) &file, FILENAME_MAX, "%X.%s", offset, dd_rules[index]->ext);

		fp = fopen((char *) &file, "wb");
		if(fp)
		{
			if(fwrite(data, 1, size, fp) != size)
			{
				perror("fwrite");
			}
	
			fclose(fp);

			/* If a command was executed, parse the command into an argv array and exec it */
			if(dd_rules[index]->cmd != NULL)
			{
				nargv = nargv_parse_line(dd_rules[index]->cmd);
				if(!nargv->error_code)
				{
					/* Replace all arguments of EXTRACT_PLACEHOLDER (%e) with the actual file name */
					for(i=0; i<nargv->argc; i++)
					{
						if(strcmp(nargv->argv[i], EXTRACT_PLACEHOLDER) == 0)
						{
							nargv->argv[i] = (char *) &file;
						}
					}
				
					if(!fork())
					{
						if(verbose <= 1)
						{
							fd = open(DEV_NULL, O_WRONLY);
							dup2(fd, 1);
							dup2(fd, 2);
							close(fd);
						}

						execvp(nargv->argv[0], nargv->argv);
						perror(nargv->argv[0]);
						exit(EXIT_FAILURE);
					}

					if(wait(&status) == -1)
					{
						perror("fork");
					}
				}

				nargv_free(nargv);
			}
		}
		else
		{
			perror("fopen");
		}
	}
}

/* Parses the --dd option and adds the appropriate dd rule */
void parse_dd_option(char *optarg)
{
	int i = 0;
	char *opt = NULL, *optptr = NULL, *result = NULL, *type = NULL, *extension = NULL, *cmd = NULL;

	opt = strdup(optarg);
	if(!opt)
	{
		perror("strdup");
	}
	else
	{
		optptr = opt;

		do
		{
			if((result = strtok(optptr, ":")))
			{
				switch(i)
				{
					case 0:
						type = result;
						break;
					case 1:
						extension = result;
						break;
					case 2:
						cmd = result;
						break;
					default:
						result = NULL;
						break;
				}

				i++;
				optptr = NULL;
			}
		} while(result != NULL);

		if(type && extension)
		{
			add_dd_rule(type, extension, cmd);
		}

		free(opt);
	}
}

/* Read an extract config file and add its rules to dd_rules */
void parse_extract_file(char *fname)
{
	FILE *fp = NULL;
	char *line = NULL;

	fp = fopen(fname, "r");
	if(fp)
	{
		while((line = freadline(fp)))
		{
			parse_dd_option(line);
			free(line);
		}

		fclose(fp);
	}
}

/* Read all the entries in all extract config files and add them to dd_rules */
void parse_extract_config(void)
{
	char *home_dir = NULL;
	char user_file[FILENAME_MAX] = { 0 };

	parse_extract_file(EXTRACT);

	home_dir = getenv("HOME");
	if(home_dir)
	{
		snprintf((char *) &user_file, FILENAME_MAX, "%s/%s/%s", home_dir, BINWALK_HOME_DIR, USER_EXTRACT);
		parse_extract_file((char *) &user_file);
	}		
}

