/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "books.h"
#include "directories.h"
#include "gwrappers.h"
#include "sqlite3.h"
#include "sqlite_reader.h"
#include "utilities.h"
#include <config.h>
#include "shell.h"
#include "languages.h"
#include "projectconfig.h"


typedef struct { 
  char *name;                 // English name.
  char *osis;                 // Osis name.
  char *paratext;             // Paratext ID.
  char *bibleworks;           // BibleWorks name.
  unsigned int id;            // Bibledit's internal id.
  BookType type;              // The type of the book.
  bool onechapter;            // The book has one chapter.
} book_record;


/*
This table gives the books Bibledit knows about.
The books are put in the standard order.

A note about this data.
The id should always be a number higher than 0, because 0 is taken for "not
found".
The id is connected to a book, and is used in the notes database too.
Therefore, ids are not supposed to be changed; new ones can be added though.
*/
typeof (book_record) books_table [] = {
 { "Front Matter",    "",       "FRT", "",    67, btFrontBackMatter, false },
 { "Genesis",         "Gen",    "GEN", "Gen",  1, btOldTestament,    false },
 { "Exodus",          "Exod",   "EXO", "Exo",  2, btOldTestament,    false },
 { "Leviticus",       "Lev",    "LEV", "Lev",  3, btOldTestament,    false },
 { "Numbers",         "Num",    "NUM", "Num",  4, btOldTestament,    false },
 { "Deuteronomy",     "Deut",   "DEU", "Deu",  5, btOldTestament,    false },
 { "Joshua",          "Josh",   "JOS", "Jos",  6, btOldTestament,    false },
 { "Judges",          "Judg",   "JDG", "Jdg",  7, btOldTestament,    false },
 { "Ruth",            "Ruth",   "RUT", "Rut",  8, btOldTestament,    false },
 { "1 Samuel",        "1Sam",   "1SA", "1Sa",  9, btOldTestament,    false },
 { "2 Samuel",        "2Sam",   "2SA", "2Sa", 10, btOldTestament,    false },
 { "1 Kings",         "1Kgs",   "1KI", "1Ki", 11, btOldTestament,    false },
 { "2 Kings",         "2Kgs",   "2KI", "2Ki", 12, btOldTestament,    false },
 { "1 Chronicles",    "1Chr",   "1CH", "1Ch", 13, btOldTestament,    false },
 { "2 Chronicles",    "2Chr",   "2CH", "2Ch", 14, btOldTestament,    false },
 { "Ezra",            "Ezra",   "EZR", "Ezr", 15, btOldTestament,    false },
 { "Nehemiah",        "Neh",    "NEH", "Neh", 16, btOldTestament,    false },
 { "Esther",          "Esth",   "EST", "Est", 17, btOldTestament,    false },
 { "Job",             "Job",    "JOB", "Job", 18, btOldTestament,    false },
 { "Psalms",          "Ps",     "PSA", "Psa", 19, btOldTestament,    false },
 { "Proverbs",        "Prov",   "PRO", "Pro", 20, btOldTestament,    false },
 { "Ecclesiastes",    "Eccl",   "ECC", "Ecc", 21, btOldTestament,    false },
 { "Song of Solomon", "Song",   "SNG", "Sol", 22, btOldTestament,    false },
 { "Isaiah",          "Isa",    "ISA", "Isa", 23, btOldTestament,    false },
 { "Jeremiah",        "Jer",    "JER", "Jer", 24, btOldTestament,    false },
 { "Lamentations",    "Lam",    "LAM", "Lam", 25, btOldTestament,    false },
 { "Ezekiel",         "Ezek",   "EZK", "Eze", 26, btOldTestament,    false },
 { "Daniel",          "Dan",    "DAN", "Dan", 27, btOldTestament,    false },
 { "Hosea",           "Hos",    "HOS", "Hos", 28, btOldTestament,    false },
 { "Joel",            "Joel",   "JOL", "Joe", 29, btOldTestament,    false },
 { "Amos",            "Amos",   "AMO", "Amo", 30, btOldTestament,    false },
 { "Obadiah",         "Obad",   "OBA", "Oba", 31, btOldTestament,    true  },
 { "Jonah",           "Jonah",  "JON", "Jon", 32, btOldTestament,    false },
 { "Micah",           "Mic",    "MIC", "Mic", 33, btOldTestament,    false },
 { "Nahum",           "Nah",    "NAM", "Nah", 34, btOldTestament,    false },
 { "Habakkuk",        "Hab",    "HAB", "Hab", 35, btOldTestament,    false },
 { "Zephaniah",       "Zeph",   "ZEP", "Zep", 36, btOldTestament,    false },
 { "Haggai",          "Hag",    "HAG", "Hag", 37, btOldTestament,    false },
 { "Zechariah",       "Zech",   "ZEC", "Zec", 38, btOldTestament,    false },
 { "Malachi",         "Mal",    "MAL", "Mal", 39, btOldTestament,    false },
 { "Matthew",         "Matt",   "MAT", "Mat", 40, btNewTestament,    false },
 { "Mark",            "Mark",   "MRK", "Mar", 41, btNewTestament,    false },
 { "Luke",            "Luke",   "LUK", "Luk", 42, btNewTestament,    false },
 { "John",            "John",   "JHN", "Joh", 43, btNewTestament,    false },
 { "Acts",            "Acts",   "ACT", "Act", 44, btNewTestament,    false },
 { "Romans",          "Rom",    "ROM", "Rom", 45, btNewTestament,    false },
 { "1 Corinthians",   "1Cor",   "1CO", "1Co", 46, btNewTestament,    false },
 { "2 Corinthians",   "2Cor",   "2CO", "2Co", 47, btNewTestament,    false },
 { "Galatians",       "Gal",    "GAL", "Gal", 48, btNewTestament,    false },
 { "Ephesians",       "Eph",    "EPH", "Eph", 49, btNewTestament,    false },
 { "Philippians",     "Phil",   "PHP", "Phi", 50, btNewTestament,    false },
 { "Colossians",      "Col",    "COL", "Col", 51, btNewTestament,    false },
 { "1 Thessalonians", "1Thess", "1TH", "1Th", 52, btNewTestament,    false },
 { "2 Thessalonians", "2Thess", "2TH", "2Th", 53, btNewTestament,    false },
 { "1 Timothy",       "1Tim",   "1TI", "1Ti", 54, btNewTestament,    false },
 { "2 Timothy",       "2Tim",   "2TI", "2Ti", 55, btNewTestament,    false },
 { "Titus",           "Titus",  "TIT", "Tit", 56, btNewTestament,    false },
 { "Philemon",        "Phlm",   "PHM", "Phm", 57, btNewTestament,    true  },
 { "Hebrews",         "Heb",    "HEB", "Heb", 58, btNewTestament,    false },
 { "James",           "Jas",    "JAS", "Jam", 59, btNewTestament,    false },
 { "1 Peter",         "1Pet",   "1PE", "1Pe", 60, btNewTestament,    false },
 { "2 Peter",         "2Pet",   "2PE", "2Pe", 61, btNewTestament,    false },
 { "1 John",          "1John",  "1JN", "1Jo", 62, btNewTestament,    false },
 { "2 John",          "2John",  "2JN", "2Jo", 63, btNewTestament,    false },
 { "3 John",          "3John",  "3JN", "3Jo", 64, btNewTestament,    true  },
 { "Jude",            "Jude",   "JUD", "Jud", 65, btNewTestament,    true  },
 { "Revelation",      "Rev",    "REV", "Rev", 66, btNewTestament,    false },
 { "Back Matter",     "",       "BAK", "",    68, btFrontBackMatter, false },
 { "Other Material",  "",       "OTH", "",    69, btOtherMaterial,   false }
};


void books_order (const ustring& project, vector<unsigned int>& books)
// Books read from the database usually are out of order. 
// Reorder the books in agreement with the user's settings.
// When Bibledit got new defined books, there will be books not yet in the user's
// settings. These will be reordered depending on their relative position with
// the existing books.
{
  // Configuration
  ProjectConfiguration projectconfig (project);
  vector <unsigned int> project_books = projectconfig.book_order ();
  
  // Sort part or all of the books following the project.
  vector <unsigned int> books_sorted;
  {
    set <unsigned int> bookset (books.begin(), books.end());
    for (unsigned int i = 0; i < project_books.size(); i++) {
      if (bookset.find (project_books[i]) != bookset.end()) {
        books_sorted.push_back (project_books[i]);
      }     
    }
  }

  // Give a sequence to each book already sorted.
  vector <unsigned int> books_sorted_sequences;
  for (unsigned int i = 0; i < books_sorted.size(); i++) {
    unsigned int sequence;
    for (unsigned int i2 = 0; i2 < sizeof (books_table) / sizeof (*books_table); i2++) {
      if (books_sorted[i] == books_table[i2].id) {
        sequence = i2;
        break;
      }
    }
    books_sorted_sequences.push_back (sequence);
  }

  // Store any books still left out.
  vector <unsigned int> books_left;
  {
    set <unsigned int> bookset (books_sorted.begin(), books_sorted.end());
    for (unsigned int i = 0; i < books.size(); i++) {
      if (bookset.find (books[i]) == bookset.end()) {
        books_left.push_back (books[i]);
      }     
    }
  }

  // Give a sequence to each book left out.
  vector <unsigned int> books_left_sequences;
  for (unsigned int i = 0; i < books_left.size(); i++) {
    unsigned int sequence;
    for (unsigned int i2 = 0; i2 < sizeof (books_table) / sizeof (*books_table); i2++) {
      if (books_left[i] == books_table[i2].id) {
        sequence = i2;
        break;
      }
    }
    books_left_sequences.push_back (sequence);
  }

  // If we had no books, just add the ones left out, after sorting them.
  if (books_sorted.empty()) {
    books.clear();
    quick_sort (books_left_sequences, books_left, 0, books_left_sequences.size());
    for (unsigned int i = 0; i < books_left.size(); i++) {
      books.push_back (books_left[i]);
    }
    return;
  }

  // Depending on the sequence of each book left out, insert it at the right 
  // location in the already sorted books.
  // The algorithm is that we check to which book each of them comes nearest,
  // and the insert it before of after that location, depending on whether it 
  // follows or precedes it.
  for (unsigned int i = 0; i < books_left.size(); i++) {
    unsigned int smallest_absolute_distance = 1000;
    size_t insert_location = 1000;
    unsigned int sequence_left = books_left_sequences[i];
    for (unsigned int i2 = 0; i2 < books_sorted.size(); i2++) {
      unsigned int sequence_sorted = books_sorted_sequences[i2];
      unsigned int absolute_distance = ABS (sequence_sorted - sequence_left);
      if (absolute_distance < smallest_absolute_distance) {
        smallest_absolute_distance = absolute_distance;
        insert_location = i2;
      }
    }
    if (sequence_left > books_sorted_sequences[insert_location])
      insert_location++;
    if (insert_location >= books_sorted.size()) {
      books_sorted.push_back (books_left[i]);
      books_sorted_sequences.push_back (books_left_sequences[i]);
    } else {
      vector<unsigned int>::iterator book_iter = books_sorted.begin();
      vector<unsigned int>::iterator sequence_iter = books_sorted_sequences.begin();
      for (size_t i2 = 0; i2 < insert_location; i2++) {
        book_iter++;
        sequence_iter++;
      }
      books_sorted.insert (book_iter, books_left[i]);
      books_sorted_sequences.insert (sequence_iter, books_left_sequences[i]);
    }
  }

  // Store the books already sorted, with the possible ones left out, in the
  // books variable.
  books.clear();
  for (unsigned int i = 0; i < books_sorted.size(); i++) {
    books.push_back (books_sorted[i]);
  }
}


void books_standard_order (vector<unsigned int>& books)
// This orders the books into the standard order.
{
  set <unsigned int> books_set (books.begin(), books.end());
  books.clear();
  for (unsigned int i = 0; i < sizeof (books_table) / sizeof (*books_table); i++) {
    if (books_set.find (books_table[i].id) != books_set.end())
      books.push_back (books_table[i].id);
  }
}


unsigned int books_name_to_id (const ustring& language, const ustring& book)
/*
This returns the id of "book" in "language". 

The id is Bibledit's internal id for a given book. 
This id uniquely identifies the book. 

We could have identified books by the Paratext ID, or by the OSIS name, but as 
there aren't Paratext IDs or OSIS names for every book that Bibledit supports, 
we need to have our own id.
*/
{
  unsigned int id = 0;

  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    rc = sqlite3_open(language_get_filename (language).c_str (), &db);
    if (rc) throw runtime_error (sqlite3_errmsg(db));
    sqlite3_busy_timeout (db, 1000);
    SqliteReader sqlitereader (0);
    char * sql;
    sql = g_strdup_printf ("select id from books where bookcf = '%s';", book.casefold().c_str());
    rc = sqlite3_exec(db, sql, sqlitereader.callback, &sqlitereader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
    if (sqlitereader.ustring0.size() > 0) {
      id = convert_to_int (sqlitereader.ustring0[0]);
    } else {
      sql = g_strdup_printf ("select id from altbooks where bookcf = '%s';", book.casefold().c_str());
      rc = sqlite3_exec(db, sql, sqlitereader.callback, &sqlitereader, &error);
      g_free (sql);
      if (rc != SQLITE_OK) {
        throw runtime_error (error);
      }
      if (sqlitereader.ustring0.size() > 0) {
        id = convert_to_int (sqlitereader.ustring0[0]);      
      }
    }
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  sqlite3_close (db);

  return id;
}


unsigned int books_abbreviation_to_id (const ustring& language, const ustring& abbreviation)
// This returns the id of "abbreviation" in "language". 
{
  unsigned int id = 0;

  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    rc = sqlite3_open(language_get_filename (language).c_str (), &db);
    if (rc) throw runtime_error (sqlite3_errmsg(db));
    sqlite3_busy_timeout (db, 1000);
    SqliteReader sqlitereader (0);
    char * sql;
    sql = g_strdup_printf ("select id from books where abbrevcf = '%s';", abbreviation.casefold().c_str());
    rc = sqlite3_exec(db, sql, sqlitereader.callback, &sqlitereader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
    if (sqlitereader.ustring0.size() > 0)
      id = convert_to_int (sqlitereader.ustring0[0]);
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  sqlite3_close (db);

  return id;
}


unsigned int books_abbreviation_to_id_loose (const ustring& language, const ustring& abbreviation)
// This returns the id of "abbreviation" in "language". It uses a looser
// searching algorithm.
{
  unsigned int id = 0;

  ustring myabbrev (abbreviation);
  replace_text (myabbrev, ".", "");
  
  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    rc = sqlite3_open(language_get_filename (language).c_str (), &db);
    if (rc) throw runtime_error (sqlite3_errmsg(db));
    sqlite3_busy_timeout (db, 1000);
    SqliteReader sqlitereader (0);
    char * sql;
    sql = g_strdup_printf ("select id from books where abbrevcf glob ('%s*');", myabbrev.casefold().c_str());
    rc = sqlite3_exec(db, sql, sqlitereader.callback, &sqlitereader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
    if (sqlitereader.ustring0.size() > 0)
      id = convert_to_int (sqlitereader.ustring0[0]);
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  sqlite3_close (db);

  return id;
}


ustring books_id_to_name  (const ustring& language, unsigned int id)
// Returns the language's human readable bookname from the Bibledit id.
{
  ustring name;

  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    rc = sqlite3_open(language_get_filename (language).c_str (), &db);
    if (rc) throw runtime_error (sqlite3_errmsg(db));
    sqlite3_busy_timeout (db, 1000);
    SqliteReader sqlitereader (0);
    char * sql;
    sql = g_strdup_printf ("select book from books where id = %d;", id);
    rc = sqlite3_exec(db, sql, sqlitereader.callback, &sqlitereader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
    if (sqlitereader.ustring0.size() > 0)
      name = sqlitereader.ustring0[0];
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  sqlite3_close (db);

  return name;
}


ustring books_id_to_abbreviation (const ustring& language, unsigned int id)
// Returns the language's abbreviation of the Bibledit id.
{
  ustring name;

  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    rc = sqlite3_open(language_get_filename (language).c_str (), &db);
    if (rc) throw runtime_error (sqlite3_errmsg(db));
    sqlite3_busy_timeout (db, 1000);
    SqliteReader sqlitereader (0);
    char * sql;
    sql = g_strdup_printf ("select abbrev from books where id = %d;", id);
    rc = sqlite3_exec(db, sql, sqlitereader.callback, &sqlitereader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
    if (sqlitereader.ustring0.size() > 0)
      name = sqlitereader.ustring0[0];
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  sqlite3_close (db);

  return name;
}


ustring books_id_to_paratext (unsigned int id)
{
  for (unsigned int i = 0; i < sizeof (books_table) / sizeof (*books_table); i++)
    if (id == books_table[i].id)
      return books_table[i].paratext;
  return "";
}


unsigned int books_paratext_to_id (const ustring& paratext)
{
  ustring s1 (paratext.casefold ());
  for (unsigned int i = 0; i < sizeof (books_table) / sizeof (*books_table); i++) {
    ustring s2 (books_table[i].paratext);
    s2 = s2.casefold();
    if (s1 == s2) {
      return books_table[i].id;
    }
  }
  return 0;
}


ustring books_id_to_bibleworks (unsigned int id)
{
  for (unsigned int i = 0; i < sizeof (books_table) / sizeof (*books_table); i++)
    if (id == books_table[i].id)
      return books_table[i].bibleworks;
  return "";
}


unsigned int books_bibleworks_to_id (const ustring& bibleworks)
{
  ustring s1 (bibleworks.casefold ());
  for (unsigned int i = 0; i < sizeof (books_table) / sizeof (*books_table); i++) {
    ustring s2 (books_table[i].bibleworks);
    s2 = s2.casefold();
    if (s1 == s2) {
      return books_table[i].id;
    }
  }
  return 0;
}


ustring books_id_to_osis (unsigned int id)
{
  for (unsigned int i = 0; i < sizeof (books_table) / sizeof (*books_table); i++)
    if (id == books_table[i].id)
      return books_table[i].osis;
  return "";
}


unsigned int books_osis_to_id (const ustring& osis)
{
  ustring s1 (osis.casefold ());
  for (unsigned int i = 0; i < sizeof (books_table) / sizeof (*books_table); i++) {
    ustring s2 (books_table[i].osis);
    s2 = s2.casefold();
    if (s1 == s2) {
      return books_table[i].id;
    }
  }
  return 0;
}


ustring books_id_to_english (unsigned int id)
{
  for (unsigned int i = 0; i < sizeof (books_table) / sizeof (*books_table); i++)
    if (id == books_table[i].id)
      return books_table[i].name;
  return "";
}


unsigned int books_english_to_id (const ustring& english)
{
  ustring s1 (english.casefold ());
  for (unsigned int i = 0; i < sizeof (books_table) / sizeof (*books_table); i++) {
    ustring s2 (books_table[i].name);
    s2 = s2.casefold();
    if (s1 == s2) {
      return books_table[i].id;
    }
  }
  return 0;
}


BookType books_id_to_type (unsigned int id)
{
  if (id < sizeof (books_table) / sizeof (*books_table))
    return books_table[id].type;
  return btUnknown;
}


vector <unsigned int> books_type_to_ids (BookType type)
// Gives the book ids of a certain type. 
// If the unknown type is given, it means "doesn't matter", so it gives all ids.
{
  vector <unsigned int> ids;
  for (unsigned int i = 0; i < sizeof (books_table) / sizeof (*books_table); i++) {
    if ((type == books_table[i].type) || (type == btUnknown)) {
      ids.push_back (books_table[i].id);
    }
  }
  return ids;
}


bool books_id_to_one_chapter (unsigned int id)
// Gives whether this is a book with one chapter.
{
  for (unsigned int i = 0; i < sizeof (books_table) / sizeof (*books_table); i++)
    if (id == books_table[i].id)
      return books_table[i].onechapter;
  return false;
}


/*
There are still many functions around that employ books_id_to_english or
books_english_to_id. Check them all, and see if they can be modified.
*/
