/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef WFMOREACTOR_H_
#define WFMOREACTOR_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <ace/config-lite.h>

#ifdef ACE_WIN32

#define HAVE_WFMO_REACTOR

#include "Reactor.h"
#include "NonCopyable.h"
#include "types.h"
#include <winsock2.h>
#include <sigc++/sigc++.h>
#include <memory>
#include <stddef.h>

class AbstractSynchFactory;
class ACE_Lock;
namespace ReactorHelpers {
	struct Handler;
	class HandlerRepository;
	class TimerQueue;
};


class WFMOReactor : public Reactor
{
	DECLARE_NON_COPYABLE(WFMOReactor)
public:
	WFMOReactor(
		AbstractSynchFactory const& synch_factory);
	
	virtual ~WFMOReactor();
	
	virtual ReactorHandlerId findHandler(ACE_HANDLE handle) const;
	
	virtual ReactorHandlerId registerHandler(
	    ACE_HANDLE handle, EventHandlerPtr const& handler, IOEvents events);
	
	virtual void unregisterHandler(ReactorHandlerId const& id);
	
	virtual void unregisterAllHandlers();
	
	virtual void enableEvents(ReactorHandlerId const& id, IOEvents events);
	
	virtual void disableEvents(ReactorHandlerId const& id, IOEvents events);
	
	virtual void setEvents(ReactorHandlerId const& id, IOEvents events);
	
	virtual IOEvents getEvents(ReactorHandlerId const& id) const;
	
	virtual ReactorTimerId registerTimer(
		EventHandlerPtr const& handler, TimeDelta const* timeout = 0);
	
	virtual void rescheduleTimer(
		ReactorTimerId const& id, TimeDelta const* timeout = 0);
	
	virtual void unregisterTimer(ReactorTimerId const& id);
	
	virtual void unregisterAllTimers();
	
	virtual TimeDelta getRemainingTime(ReactorTimerId const& timer_id) const;
	
	virtual Status handleEvents();
	
	virtual Status runEventLoop();
	
	virtual void wakeup();
	
	virtual void stop();
	
	virtual void restart();
	
	virtual sigc::signal<void>& beforeSleepSignal();
private:
	typedef void (EventHandlerBase::*HandlerFuncPtr)(ACE_HANDLE);
	
	struct IODispatchIter
	{
		unsigned pos;
		unsigned end;
		long revents;
		unsigned char phase;
		
		static unsigned const INITIAL_POS = static_cast<unsigned>(-1);

		static unsigned char const WRITE_PHASE = 0;
		static unsigned char const EXCEPT_PHASE = 1;
		static unsigned char const READ_PHASE = 2;
		static unsigned char const INITIAL_PHASE = 2;
		static unsigned char const MIN_PHASE = 0;
		static unsigned char const MAX_PHASE = 2;

		IODispatchIter();
		
		void reset(unsigned pos, unsigned end);
		
		bool next(WFMOReactor& reactor);
	};
	friend struct WFMOReactor::IODispatchIter;
	
	unsigned updateWaitSet();
	
	DWORD getDemuxTimeout() const;
	
	bool continueIODispatching();
	
	void dispatchIO(ReactorHelpers::Handler const& handler);
	
	void dispatchIOEvent(
		ACE_HANDLE handle, EventHandlerBase* eh, HandlerFuncPtr func);
	
	long queryEvent(unsigned pos);
	
	void destroyEvent(ReactorHelpers::Handler const& handler);
	
	std::auto_ptr<ReactorHelpers::HandlerRepository> m_ptrHandlers;
	std::auto_ptr<ReactorHelpers::TimerQueue> m_ptrTimers;
	std::auto_ptr<ACE_Lock> m_ptrMutex; // available in WaitForMultipleObjects() and in event handlers
	std::auto_ptr<ACE_Lock> m_ptrDemuxMutex; // available outside of WaitForMultipleObjects()
	IODispatchIter m_ioIter;
	bool m_isInsideDemux;
	bool m_isStopped;
	int32_t m_isStoppedAtomic;
	sigc::signal<void> m_beforeSleepSignal;
	WSAEVENT m_events[MAXIMUM_WAIT_OBJECTS];
	ReactorHelpers::Handler const* m_handlers[MAXIMUM_WAIT_OBJECTS];
};

#endif // ACE_WIN32

#endif
