/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "FileOps.h"
#include "FileVersion.h"
#include "AutoClosingSAP.h"
#include "OperationLog.h"
#include "Log.h"
#include <ace/config-lite.h>
#include <ace/FILE_Addr.h>
#include <ace/FILE_IO.h>
#include <ace/FILE_Connector.h>
#include <ace/SOCK_Acceptor.h>
#include <ace/OS_NS_sys_stat.h> // for stat(), fstat(), mkdir()
#include <ace/OS_NS_stdio.h> // for rename()
#include <ace/OS_NS_unistd.h> // for unlink()
#include <wx/string.h>
#include <errno.h>

namespace wxGUI
{

namespace FileOps
{

bool readFile(wxFileName const& file_path, std::string& target,
	FileVersion* version, bool log_errors)
{
	wxString const file_path_str(file_path.GetFullPath());
	AutoClosingSAP<ACE_FILE_IO> file;
	ACE_FILE_Addr addr(file_path_str.c_str());
	ACE_FILE_Connector connector;
	
	if (connector.connect(file, addr, 0, ACE_Addr::sap_any, 0, O_RDONLY) == -1) {
		if (log_errors) {
			Log* log = OperationLog::instance();
			log->appendRecord(
				_T("Could not open ")+file_path_str,
				log->getErrorStyle()
			);
		}
		return false;
	}
	
	std::string content;
	
	char buf[4096];
	ssize_t bytes_read = 0;
	while ((bytes_read = file.recv(buf, sizeof(buf))) > 0) {
		content.append(buf, bytes_read);
	}
	if (bytes_read != 0) {
		if (log_errors) {
			Log* log = OperationLog::instance();
			log->appendRecord(
				_T("Error reading ")+file_path.GetFullPath(),
				log->getErrorStyle()
			);
		}
		return false;
	}
	
	target.swap(content);
	
	if (version) {
		ACE_stat st;
		if (ACE_OS::fstat(file.get_handle(), &st) == -1) {
			*version = FileVersion::undefined();
		} else {
			*version = FileVersion(st.st_mtime);
		}
	}
	
	return true;
}

bool writeFile(wxFileName const& file_path, std::string const& data,
	FileVersion* version, bool log_errors)
{
	wxString const file_path_str(file_path.GetFullPath());
	AutoClosingSAP<ACE_FILE_IO> file;
	ACE_FILE_Addr addr(file_path_str.c_str());
	ACE_FILE_Connector connector;
	
	int const flags = O_WRONLY|O_CREAT|O_TRUNC;
	if (connector.connect(file, addr, 0, ACE_Addr::sap_any, 0, flags) == -1) {
		if (log_errors) {
			Log* log = OperationLog::instance();
			log->appendRecord(
				_T("Could not open ")+file_path_str,
				log->getErrorStyle()
			);
		}
		return false;
	}
	
	if (file.send_n(data.c_str(), data.size()) == -1) {
		if (log_errors) {
			Log* log = OperationLog::instance();
			log->appendRecord(
				_T("Error writing ")+file_path.GetFullPath(),
				log->getErrorStyle()
			);
		}
		return false;
	}
	
	if (version) {
		ACE_stat st;
		if (ACE_OS::fstat(file.get_handle(), &st) == -1) {
			*version = FileVersion::undefined();
		} else {
			*version = FileVersion(st.st_mtime);
		}
	}
	
	return true;
}

bool renameFile(wxFileName const& from_path,
	wxFileName const& to_path, bool log_errors)
{
	wxString const from_path_str(from_path.GetFullPath());
	wxString const to_path_str(to_path.GetFullPath());
	if (ACE_OS::rename(from_path_str.c_str(), to_path_str.c_str()) == -1) {
		if (log_errors) {
			Log* log = OperationLog::instance();
			log->appendRecord(
				_T("Error renaming ")+from_path_str+
				_T(" to ")+to_path_str,
				log->getErrorStyle()
			);
		}
		return false;
	}
	return true;
}

bool deleteFile(wxFileName const& file_path, bool log_errors)
{
	wxString const file_path_str(file_path.GetFullPath());
	if (ACE_OS::unlink(file_path_str.c_str()) == -1 && errno != ENOENT) {
		if (log_errors) {
			Log* log = OperationLog::instance();
			log->appendRecord(
				_T("Error deleting ")+file_path_str,
				log->getErrorStyle()
			);
		}
		return false;
	}
	return true;
}

FileVersion readFileVersion(wxFileName const& file_path)
{
	ACE_stat st;
	if (ACE_OS::stat(file_path.GetFullPath().c_str(), &st) == -1) {
		return FileVersion::undefined();
	} else {
		return FileVersion(st.st_mtime);
	}
}

bool createDir(wxFileName const& dir_path, bool log_errors)
{
	bool ok = true;
	wxString const dir_path_str(dir_path.GetFullPath());
	
	if (ACE_OS::mkdir(dir_path_str.c_str()) == -1) {
		if (errno != EEXIST) {
			ok = false;
		} else {
			ACE_stat st;
			if (ACE_OS::stat(dir_path_str.c_str(), &st) == -1) {
				ok = false;
			} else if (!(st.st_mode & S_IFDIR)) {
				ok = false;
			}
		}
	}
	
	if (!ok && log_errors) {
		Log* log = OperationLog::instance();
		log->appendRecord(
			_T("Could not create directory ")+dir_path_str,
			log->getErrorStyle()
		);
	}
	
	return ok;
}

} // namespace FileOps

} // namespace wxGUI
