/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "TrayIcon.h"
#include "Application.h"
#include "CompiledImages.h"
#include "Log.h"
#include "OperationLog.h"
#include "ArraySize.h"
#include "WorkerThreadPoolExSingleton.h"
#include <gtkmm/window.h>
#include <glibmm/main.h>

using namespace std;

namespace GtkGUI
{

TrayIcon::TrayIcon()
{
	mouseClickSignal().connect(
		sigc::mem_fun(*this, &TrayIcon::onMouseClick)
	);
	closeSignal().connect(
		sigc::mem_fun(*this, &TrayIcon::onClose)
	);
	
	loadIcons();
	
	OperationLog::instance()->getUnseenWarningsSignal().connect(
		sigc::mem_fun(*this, &TrayIcon::onUnseenWarnings)
	);
	
	WorkerThreadPoolExSingleton::instance()->acceptingConnectionsSignal().connect(
		sigc::mem_fun(*this, &TrayIcon::onAcceptingConnections)
	);
	
	onStateChange();
}

TrayIcon::~TrayIcon()
{
}

void
TrayIcon::displayNetworkActivity()
{
	m_timerConnection.disconnect();
	m_timerConnection = Glib::signal_timeout().connect(
		sigc::mem_fun(*this, &TrayIcon::onTimer), 2000
	);
	onStateChange();
}

void
TrayIcon::loadIcons()
{
	using namespace CompiledImages;
	
	int const delay1 = 180;
	
	initFrame(m_normalFrames[0], tray_0_png, delay1);
	initFrame(m_normalFrames[1], tray_1_png, delay1);
	initFrame(m_normalFrames[2], tray_2_png, delay1);
	initFrame(m_normalFrames[3], tray_3_png, delay1);
	initFrame(m_normalFrames[4], tray_4_png, delay1);
	initFrame(m_normalFrames[5], tray_5_png, delay1);
	initFrame(m_normalFrames[6], tray_6_png, delay1);
	
	initFrame(m_warningFrames[0], tray_warn_0_png, delay1);
	initFrame(m_warningFrames[1], tray_warn_1_png, delay1);
	initFrame(m_warningFrames[2], tray_warn_2_png, delay1);
	initFrame(m_warningFrames[3], tray_warn_3_png, delay1);
	initFrame(m_warningFrames[4], tray_warn_4_png, delay1);
	initFrame(m_warningFrames[5], tray_warn_5_png, delay1);
	initFrame(m_warningFrames[6], tray_warn_6_png, delay1);
	
	m_ptrInoperativeIcon = tray_inoperative_png.getPixbuf();
}

void
TrayIcon::initFrame(Frame& frame, CompiledImage const& image, int delay)
{
	frame.icon = image.getPixbuf();
	frame.delay_ms = delay;
}

void
TrayIcon::setNormalIcon()
{
	setIcon(m_normalFrames[0].icon, "BFilter");
}

void
TrayIcon::setInoperativeIcon()
{
	setIcon(m_ptrInoperativeIcon, "BFilter [inoperative]", 0);
}

void
TrayIcon::setWarningIcon()
{
	setIcon(m_warningFrames[0].icon, "BFilter [warnings in log]", 0);
}

void
TrayIcon::setNormalAnimation()
{
	setAnimation(
		m_normalFrames, m_normalFrames + ARRAY_SIZE(m_normalFrames),
		"BFilter"
	);
}

void
TrayIcon::setWarningAnimation()
{
	setAnimation(
		m_warningFrames, m_warningFrames + ARRAY_SIZE(m_warningFrames),
		"BFilter [warnings in log]"
	);
}

void
TrayIcon::onClose()
{
	Application::instance()->requestAppExit();
}

void
TrayIcon::onMouseClick(GdkEventButton* evt)
{
	Application::instance()->showTrayMenu(evt->button, evt->time);
}

bool
TrayIcon::onTimer()
{
	m_timerConnection.disconnect();
	onStateChange();
	return false;
}

void
TrayIcon::onAcceptingConnections(bool accepting)
{
	onStateChange();
}

void
TrayIcon::onUnseenWarnings(bool warnings)
{
	onStateChange();
}

void
TrayIcon::onStateChange()
{
	if (!WorkerThreadPoolExSingleton::instance()->isAcceptingConnections()) {
		setInoperativeIcon();
		return;
	}
	
	Log* log = OperationLog::instance();
	if (m_timerConnection.connected()) {
		if (log->hasUnseenWarnings()) {
			setWarningAnimation();
		} else {
			setNormalAnimation();
		}
	} else {
		if (log->hasUnseenWarnings()) {
			setWarningIcon();
		} else {
			setNormalIcon();
		}
	}
}

} // namespace GtkGUI
