/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "SkinnedButton.h"

namespace GtkGUI
{

SkinnedButton::SkinnedButton(Glib::RefPtr<Gdk::Pixbuf> const& skin)
:	m_image(skin),
	m_ptrNormalSkin(skin),
	m_ptrHoverSkin(skin),
	m_ptrDisabledSkin(skin),
	m_isEnabled(true),
	m_isMouseOver(false)
{
	Gdk::EventMask const events = Gdk::BUTTON_PRESS_MASK
		|Gdk::BUTTON_RELEASE_MASK
		|Gdk::ENTER_NOTIFY_MASK
		|Gdk::LEAVE_NOTIFY_MASK;
	
	add_events(events);
	add(m_image);
	m_image.show();
	
	signal_map().connect(
		sigc::mem_fun(*this, &SkinnedButton::tryUpdateCursor)
	);
	signal_button_release_event().connect(
		sigc::mem_fun(*this, &SkinnedButton::onButtonRelease)
	);
	signal_enter_notify_event().connect(
		sigc::mem_fun(*this, &SkinnedButton::onMouseEnter)
	);
	signal_leave_notify_event().connect(
		sigc::mem_fun(*this, &SkinnedButton::onMouseLeave)
	);
}

SkinnedButton::~SkinnedButton()
{
}

void
SkinnedButton::setNormalSkin(Glib::RefPtr<Gdk::Pixbuf> const& skin)
{
	m_ptrNormalSkin = skin;
	updateSkin();
}

void
SkinnedButton::setHoverSkin(Glib::RefPtr<Gdk::Pixbuf> const& skin)
{
	m_ptrHoverSkin = skin;
	updateSkin();
}

void
SkinnedButton::setDisabledSkin(Glib::RefPtr<Gdk::Pixbuf> const& skin)
{
	m_ptrDisabledSkin = skin;
	updateSkin();
}

void
SkinnedButton::setHoverCursor(Gdk::Cursor const& cursor)
{
	m_hoverCursor = cursor;
	tryUpdateCursor();
}

void
SkinnedButton::setDisabledCursor(Gdk::Cursor const& cursor)
{
	m_disabledCursor = cursor;
	tryUpdateCursor();
}

void
SkinnedButton::setEnabled(bool enabled)
{
	if (enabled != m_isEnabled) {
		m_isEnabled = enabled;
		updateSkin();
		tryUpdateCursor();
	}
}

void
SkinnedButton::updateSkin()
{
	if (!m_isEnabled) {
		m_image.set(m_ptrDisabledSkin);
	} else if (m_isMouseOver) {
		m_image.set(m_ptrHoverSkin);
	} else {
		m_image.set(m_ptrNormalSkin);
	}
}

void
SkinnedButton::tryUpdateCursor()
{
	Gdk::Cursor const& cursor = (m_isEnabled ? m_hoverCursor : m_disabledCursor);
	Glib::RefPtr<Gdk::Window> window(m_image.get_window());
	if (window) {
		window->set_cursor(cursor);
	}
}

bool
SkinnedButton::onButtonRelease(GdkEventButton* evt)
{
	if (!m_isEnabled) {
		return false; // stop event propagation
	}
	if (evt->button == 1) {
		m_signalClicked.emit((Gdk::ModifierType)evt->state);
	}
	return false; // propagate event further
}

bool
SkinnedButton::onMouseEnter(GdkEventCrossing*)
{
	m_isMouseOver = true;
	updateSkin();
	return false; // propagate event further
}

bool
SkinnedButton::onMouseLeave(GdkEventCrossing*)
{
	m_isMouseOver = false;
	updateSkin();
	return false; // propagate event further
}

} // namespace GtkGUI
