/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "GifImageInspector.h"
#include "GifContentIdentifier.h"
#include "ScopedDecInc.h"
#include "StringUtils.h"
#include "types.h"
#include <cstring>

using namespace std;

GifImageInspector::GifImageInspector()
:	m_state(ST_SIGNATURE),
	m_delayState(DST_DELAY_NOT_ENCOUNTERED),
	m_status(IN_PROGRESS),
	m_dataSize(0),
	m_curPos(m_data.begin()),
	m_curOffset(0),
	m_width(0),
	m_height(0),
	m_isAnimated(false)
{
}

GifImageInspector::~GifImageInspector()
{
}

void
GifImageInspector::reset()
{
	m_state = ST_SIGNATURE;
	m_delayState = DST_DELAY_NOT_ENCOUNTERED;
	m_status = IN_PROGRESS;
	m_data.clear();
	m_dataSize = 0;
	m_curPos = m_data.begin();
	m_curOffset = 0;
	m_width = 0;
	m_height = 0;
	m_isAnimated = false;	
}

void
GifImageInspector::consumeDataChunk(SplittableBuffer& data, bool eof)
{
	{
		ScopedDecInc<SplittableBuffer::ByteIterator> di(m_curPos);
		// if m_curPos was at end(), it would remain there
		m_dataSize += data.size();
		m_data.appendDestructive(data);
	}
	if (m_status == IN_PROGRESS) {
		processNewData();
		if (eof && m_status == IN_PROGRESS) {
			m_status = FAILED;
		}
	}
}

void
GifImageInspector::processNewData()
{
	switch (m_state) {
		case ST_SIGNATURE: {
			if (!checkSignature()) {
				return;
			}
			advance(3);
			m_state = ST_HEADER;
			goto case_ST_HEADER;
		}
		case ST_HEADER:
		case_ST_HEADER: {
			if (m_dataSize < 13) {
				return;
			}
			SplittableBuffer::ByteIterator pos(m_curPos);
			pos += 3;
			m_width = unsigned(uint8_t(*pos));
			++pos;
			m_width += unsigned(uint8_t(*pos)) << 8;
			++pos;
			m_height = unsigned(uint8_t(*pos));
			++pos;
			m_height += unsigned(uint8_t(*pos)) << 8;
			++pos;
			uint8_t flag = *pos;
			advance(10);
			if (flag & 0x80) {
				m_curBlockSize = 3*(1<<((flag&7)+1));
				m_state = ST_GLOBAL_COLOR_TABLE;
				goto case_ST_GLOBAL_COLOR_TABLE;
			} else {
				m_state = ST_BLOCK_BEGIN;
				goto case_ST_BLOCK_BEGIN;
			}
		}
		case ST_GLOBAL_COLOR_TABLE:
		case_ST_GLOBAL_COLOR_TABLE: {
			if (m_dataSize < m_curOffset + m_curBlockSize) {
				return;
			}
			advance(m_curBlockSize);
			m_state = ST_BLOCK_BEGIN;
			goto case_ST_BLOCK_BEGIN;
		}
		case ST_BLOCK_BEGIN:
		case_ST_BLOCK_BEGIN: {
			if (m_dataSize <= m_curOffset) {
				return;
			}
			uint8_t block_id = *m_curPos;
			switch (block_id) {
				case 0x2c: {
					if (m_delayState == DST_WAITING_FOR_GRAPHIC_BLOCK) {
						m_delayState = DST_WAITING_FOR_ANOTHER_GRAPHIC_BLOCK;
					} else if (m_delayState == DST_WAITING_FOR_ANOTHER_GRAPHIC_BLOCK) {
						m_isAnimated = true;
						m_status = COMPLETE;
						return;
					}
					m_state = ST_IMAGE_DESCRIPTOR;
					goto case_ST_IMAGE_DESCRIPTOR;
				}
				case 0x21: {
					m_state = ST_EXTENSION_BLOCK_BEGIN;
					goto case_ST_EXTENSION_BLOCK_BEGIN;
				}
				case 0x3b: {
					m_status = COMPLETE;
					return;
				}
				default: {
					m_status = FAILED;
					return;
				}
			}
		}
		case ST_IMAGE_DESCRIPTOR:
		case_ST_IMAGE_DESCRIPTOR: {
			if (m_dataSize < m_curOffset + 10) {
				return;
			}
			advance(9);
			uint8_t flag = *m_curPos;
			advance();
			if (flag & 0x80) {
				m_curBlockSize = 3*(1<<((flag&7)+1));
				m_state = ST_LOCAL_COLOR_TABLE;
				goto case_ST_LOCAL_COLOR_TABLE;
			} else {
				m_state = ST_IMAGE_DATA;
				goto case_ST_IMAGE_DATA;
			}
		}
		case ST_LOCAL_COLOR_TABLE:
		case_ST_LOCAL_COLOR_TABLE: {
			if (m_dataSize < m_curOffset + m_curBlockSize) {
				return;
			}
			advance(m_curBlockSize);
			m_state = ST_IMAGE_DATA;
			goto case_ST_IMAGE_DATA;
		}
		case ST_IMAGE_DATA:
		case_ST_IMAGE_DATA: {
			if (m_dataSize <= m_curOffset) {
				return;
			}
			advance(); // skipping the LZW minimum code size
			m_state = ST_IMAGE_DATA_SUBBLOCK_OR_TERMINATOR;
			goto case_ST_IMAGE_DATA_SUBBLOCK_OR_TERMINATOR;
		}
		case ST_IMAGE_DATA_SUBBLOCK_OR_TERMINATOR:
		case_ST_IMAGE_DATA_SUBBLOCK_OR_TERMINATOR: {
			if (m_dataSize <= m_curOffset) {
				return;
			}
			m_curBlockSize = uint8_t(*m_curPos);
			advance();
			if (m_curBlockSize == 0) {
				m_state = ST_BLOCK_BEGIN;
				goto case_ST_BLOCK_BEGIN;
			} else {
				m_state = ST_IMAGE_DATA_SUBBLOCK;
				goto case_ST_IMAGE_DATA_SUBBLOCK;
			}
		}
		case ST_IMAGE_DATA_SUBBLOCK:
		case_ST_IMAGE_DATA_SUBBLOCK: {
			if (m_dataSize < m_curOffset + m_curBlockSize) {
				return;
			}
			advance(m_curBlockSize);
			m_state = ST_IMAGE_DATA_SUBBLOCK_OR_TERMINATOR;
			goto case_ST_IMAGE_DATA_SUBBLOCK_OR_TERMINATOR;
		}
		case ST_EXTENSION_BLOCK_BEGIN:
		case_ST_EXTENSION_BLOCK_BEGIN: {
			if (m_dataSize < m_curOffset + 2) {
				return;
			}
			advance();
			uint8_t ext_byte = *m_curPos;
			advance();
			if (ext_byte == 0xf9) {
				m_state = ST_GRAPHIC_CONTROL_EXTENSION_SUBBLOCK_OR_TERMINATOR;
				goto case_ST_GRAPHIC_CONTROL_EXTENSION_SUBBLOCK_OR_TERMINATOR;
			} else if (ext_byte == 0xff) {
				m_state = ST_APPLICATION_EXTENSION_SUBBLOCK_OR_TERMINATOR;
				goto case_ST_APPLICATION_EXTENSION_SUBBLOCK_OR_TERMINATOR;
			} else {
				if (ext_byte == 0xff) { // netscape extension
					m_isAnimated = true;
					m_status = COMPLETE;
					return;
				} else if (ext_byte == 0x01) { // plain text extension
					if (m_delayState == DST_WAITING_FOR_GRAPHIC_BLOCK) {
						m_delayState = DST_WAITING_FOR_ANOTHER_GRAPHIC_BLOCK;
					} else if (m_delayState == DST_WAITING_FOR_ANOTHER_GRAPHIC_BLOCK) {
						m_isAnimated = true;
						m_status = COMPLETE;
						return;
					}
				}
				m_state = ST_OTHER_EXTENSION_SUBBLOCK_OR_TERMINATOR;
				goto case_ST_OTHER_EXTENSION_SUBBLOCK_OR_TERMINATOR;
			}
		}
		case ST_GRAPHIC_CONTROL_EXTENSION_SUBBLOCK_OR_TERMINATOR:
		case_ST_GRAPHIC_CONTROL_EXTENSION_SUBBLOCK_OR_TERMINATOR: {
			if (m_dataSize <= m_curOffset) {
				return;
			}
			m_curBlockSize = uint8_t(*m_curPos);
			advance();
			if (m_curBlockSize == 0) {
				m_state = ST_BLOCK_BEGIN;
				goto case_ST_BLOCK_BEGIN;
			} else {
				m_state = ST_GRAPHIC_CONTROL_EXTENSION_SUBBLOCK;
				goto case_ST_GRAPHIC_CONTROL_EXTENSION_SUBBLOCK;
			}
		}
		case ST_GRAPHIC_CONTROL_EXTENSION_SUBBLOCK:
		case_ST_GRAPHIC_CONTROL_EXTENSION_SUBBLOCK: {
			if (m_curBlockSize != 4) {
				m_status = FAILED;
				return;
			}
			if (m_dataSize < m_curOffset + m_curBlockSize) {
				return;
			}
			if (*(m_curPos+1) != 0x00 || *(m_curPos+2) != 0x00) {
				if (m_delayState == DST_DELAY_NOT_ENCOUNTERED) {
					m_delayState = DST_WAITING_FOR_GRAPHIC_BLOCK;
				}
			}
			advance(m_curBlockSize);
			m_state = ST_GRAPHIC_CONTROL_EXTENSION_SUBBLOCK_OR_TERMINATOR;
			goto case_ST_GRAPHIC_CONTROL_EXTENSION_SUBBLOCK_OR_TERMINATOR;
		}
		case ST_APPLICATION_EXTENSION_SUBBLOCK_OR_TERMINATOR:
		case_ST_APPLICATION_EXTENSION_SUBBLOCK_OR_TERMINATOR: {
			if (m_dataSize <= m_curOffset) {
				return;
			}
			m_curBlockSize = uint8_t(*m_curPos);
			advance();
			if (m_curBlockSize == 0) {
				m_state = ST_BLOCK_BEGIN;
				goto case_ST_BLOCK_BEGIN;
			} else {
				m_state = ST_APPLICATION_EXTENSION_SUBBLOCK;
				goto case_ST_APPLICATION_EXTENSION_SUBBLOCK;
			}
		}
		case ST_APPLICATION_EXTENSION_SUBBLOCK:
		case_ST_APPLICATION_EXTENSION_SUBBLOCK: {
			if (m_dataSize < m_curOffset + m_curBlockSize) {
				return;
			}
			static char const netscape_sig[] = "NETSCAPE2.0";
			static char const* netscape_sig_end = netscape_sig + sizeof(netscape_sig) - 1;
			bool is_netscape_extension = StringUtils::equal(
				m_curPos, m_curPos + m_curBlockSize,
				netscape_sig, netscape_sig_end
			);
			advance(m_curBlockSize);
			if (is_netscape_extension) {
				m_state = ST_NETSCAPE_EXTENSION_SUBBLOCK2_OR_TERMINATOR;
				goto case_ST_NETSCAPE_EXTENSION_SUBBLOCK2_OR_TERMINATOR;
			} else {
				m_state = ST_OTHER_EXTENSION_SUBBLOCK_OR_TERMINATOR;
				goto case_ST_OTHER_EXTENSION_SUBBLOCK_OR_TERMINATOR;
			}
		}
		case ST_NETSCAPE_EXTENSION_SUBBLOCK2_OR_TERMINATOR:
		case_ST_NETSCAPE_EXTENSION_SUBBLOCK2_OR_TERMINATOR: {
			if (m_dataSize <= m_curOffset) {
				return;
			}
			m_curBlockSize = uint8_t(*m_curPos);
			advance();
			if (m_curBlockSize == 0) {
				m_state = ST_BLOCK_BEGIN;
				goto case_ST_BLOCK_BEGIN;
			} else {
				m_state = ST_NETSCAPE_EXTENSION_SUBBLOCK2;
				goto case_ST_NETSCAPE_EXTENSION_SUBBLOCK2;
			}
		}
		case ST_NETSCAPE_EXTENSION_SUBBLOCK2:
		case_ST_NETSCAPE_EXTENSION_SUBBLOCK2: {
			if (m_dataSize < m_curOffset + m_curBlockSize) {
				return;
			}
			if (m_curBlockSize == 3 && *m_curPos == 0x01) {
				uint16_t repeats = uint8_t(*(m_curPos+1));
				repeats += unsigned(uint8_t(*(m_curPos+2))) << 8;
				if (repeats != 1) {
					m_isAnimated = true;
					m_status = COMPLETE;
					return;
				}
			}
			advance(m_curBlockSize);
			m_state = ST_OTHER_EXTENSION_SUBBLOCK_OR_TERMINATOR;
			goto case_ST_OTHER_EXTENSION_SUBBLOCK_OR_TERMINATOR;
		}
		case ST_OTHER_EXTENSION_SUBBLOCK_OR_TERMINATOR:
		case_ST_OTHER_EXTENSION_SUBBLOCK_OR_TERMINATOR: {
			if (m_dataSize <= m_curOffset) {
				return;
			}
			m_curBlockSize = uint8_t(*m_curPos);
			advance();
			if (m_curBlockSize == 0) {
				m_state = ST_BLOCK_BEGIN;
				goto case_ST_BLOCK_BEGIN;
			} else {
				m_state = ST_OTHER_EXTENSION_SUBBLOCK;
				goto case_ST_OTHER_EXTENSION_SUBBLOCK;
			}
		}
		case ST_OTHER_EXTENSION_SUBBLOCK:
		case_ST_OTHER_EXTENSION_SUBBLOCK: {
			if (m_dataSize < m_curOffset + m_curBlockSize) {
				return;
			}
			advance(m_curBlockSize);
			m_state = ST_OTHER_EXTENSION_SUBBLOCK_OR_TERMINATOR;
			goto case_ST_OTHER_EXTENSION_SUBBLOCK_OR_TERMINATOR;
		}
	}
}

bool
GifImageInspector::checkSignature()
{
	typedef GifContentIdentifier CI;
	SplittableBuffer::ByteIterator it(m_curPos);
	char const* sigp = CI::SIGNATURE;
	char const* const sige = CI::SIGNATURE + sizeof(CI::SIGNATURE);
	for (;; ++it, ++sigp) {
		if (sigp == sige) {
			return true;
		} else if (it.isAtRightBorder()) {
			return false;
		} else if (*sigp != char(*it)) {
			m_status = FAILED;
			return false;
		}
	}
}
