/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "AsyncReaderWriter.h"
#include "ScopedIncDec.h"
#include <ace/config-lite.h>
#include <ace/IPC_SAP.h>
#include <ace/OS_NS_sys_socket.h>
#include <cassert>

using namespace std;

AsyncReaderWriter::AsyncReaderWriter()
:	m_refCount(0),
	m_pReactor(0),
	m_handle(ACE_INVALID_HANDLE),
	m_pReadAddr(0),
	m_readTodo(0),
	m_pWriteAddr(0),
	m_writeTodo(0),
	m_tryReadingNestLevel(0),
	m_tryWritingNestLevel(0),
	m_isReadScheduled(0),
	m_isWriteScheduled(0)
{
}

AsyncReaderWriter::~AsyncReaderWriter()
{
	unregisterHandler();
	assert(m_refCount == 0);
}

void
AsyncReaderWriter::activate(
	Listener& listener, Reactor& reactor, ACE_HANDLE handle)
{
	assert(handle != ACE_INVALID_HANDLE);
	
	deactivate();
	
	/*
	We have to register the handler here, and not in
	startReading() or startWriting(), because by that
	time the reactor may be already destroyed. If we do
	it here, the reactor will call unref() on us in the
	process of its destruction.
	*/
	try {
		m_handlerId = reactor.registerHandler(
			handle, Reactor::EventHandlerPtr(this),
			Reactor::NO_EVENTS
		);
	} catch (Reactor::Exception const& e) {
		listener.onGenericError();
		return;
	}
	
	m_observerLink.setObserver(&listener);
	m_pReactor = &reactor;
	m_handle = handle;
	
	setNonBlockingMode(handle);
}

void
AsyncReaderWriter::deactivate()
{
	m_observerLink.setObserver(0);
	if (m_handlerId) {
		assert(m_pReactor);
		stopReading();
		stopWriting();
		unregisterHandler();
	}
	assert(m_refCount == 0);
}

void
AsyncReaderWriter::startReading(void* addr, size_t len)
{
	if (!m_pReactor) {
		return;
	}
	if (len == 0) {
		stopReading();
		return;
	}
	
	
	assert(m_handle != ACE_INVALID_HANDLE);
	assert(m_handlerId);
	
	m_pReadAddr = (char*)addr;
	m_readTodo = len;
	
	m_pReactor->enableEvents(m_handlerId, Reactor::READ);
	
	if (m_tryReadingNestLevel) {
		m_isReadScheduled = 1;
	} else {
		do {
			m_isReadScheduled = 0;
			tryReading();
		} while (m_isReadScheduled);
	}
}

void
AsyncReaderWriter::startWriting(void const* addr, size_t len)
{
	if (!m_pReactor) {
		return;
	}
	if (len == 0) {
		stopWriting();
		return;
	}
	
	assert(m_handle != ACE_INVALID_HANDLE);
	assert(m_handlerId);
	
	m_pWriteAddr = (char*)addr;
	m_writeTodo = len;
	
	m_pReactor->enableEvents(m_handlerId, Reactor::WRITE);
	
	if (m_tryWritingNestLevel) {
		m_isWriteScheduled = 1;
	} else {
		do {
			m_isWriteScheduled = 0;
			tryWriting();
		} while (m_isWriteScheduled);
	}
}

void
AsyncReaderWriter::stopReading()
{
	if (!m_handlerId) {
		return;
	}
	
	assert(m_handlerId);
	
	m_pReactor->disableEvents(m_handlerId, Reactor::READ);
	m_pReadAddr = 0;
	m_readTodo = 0;
	m_isReadScheduled = 0;
}

void
AsyncReaderWriter::stopWriting()
{
	if (!m_handlerId) {
		return;
	}
	
	assert(m_handlerId);
	
	m_pReactor->disableEvents(m_handlerId, Reactor::WRITE);
	m_pWriteAddr = 0;
	m_writeTodo = 0;
	m_isWriteScheduled = 0;
}

void
AsyncReaderWriter::ref()
{
	++m_refCount;
}

void
AsyncReaderWriter::unref()
{
	if (--m_refCount == 0) {
		m_pReactor = 0;
		m_handlerId = ReactorHandlerId();
	}
}

void
AsyncReaderWriter::handleRead(ACE_HANDLE)
{
	tryReading();
}

void
AsyncReaderWriter::handleWrite(ACE_HANDLE)
{
	tryWriting();
}

void
AsyncReaderWriter::unregisterHandler()
{
	if (m_handlerId) {
		assert(m_pReactor);
		m_pReactor->unregisterHandler(m_handlerId);
		m_handlerId = ReactorHandlerId();
	}
}

void
AsyncReaderWriter::tryReading()
{
	if (m_readTodo == 0) {
		return;
	}
	
	assert(m_pReactor);
	assert(m_handlerId);
	
	ScopedIncDec<unsigned char> inc_dec(m_tryReadingNestLevel);
	
	ssize_t r = ACE_OS::recv(m_handle, m_pReadAddr, m_readTodo, 0);
	if (r <= 0) {
		if (errno != EWOULDBLOCK) {
			stopReading();
			Listener* listener = m_observerLink.getObserver();
			if (listener) {
				listener->onReadError();
			}
		}
		return;
	}
	
	m_pReadAddr += r;
	m_readTodo -= r;
	if (m_readTodo == 0) {
		m_pReactor->disableEvents(m_handlerId, Reactor::READ);
		Listener* listener = m_observerLink.getObserver();
		if (listener) {
			listener->onReadDone();
		}
	}
}

void
AsyncReaderWriter::tryWriting()
{
	if (m_writeTodo == 0) {
		return;
	}
	
	assert(m_pReactor);
	assert(m_handlerId);
		
	ScopedIncDec<unsigned char> inc_dec(m_tryWritingNestLevel);
	
	ssize_t r = ACE_OS::send(m_handle, m_pWriteAddr, m_writeTodo, 0);
	if (r < 0) {
		if (errno != EWOULDBLOCK) {
			stopWriting();
			Listener* listener = m_observerLink.getObserver();
			if (listener) {
				listener->onWriteError();
			}
		}
		return;
	}
	
	m_pWriteAddr += r;
	m_writeTodo -= r;
	if (m_writeTodo == 0) {
		m_pReactor->disableEvents(m_handlerId, Reactor::WRITE);
		Listener* listener = m_observerLink.getObserver();
		if (listener) {
			listener->onWriteDone();
		}
	}
}

void
AsyncReaderWriter::setNonBlockingMode(ACE_HANDLE handle)
{
	class SAP : public ACE_IPC_SAP {};
	SAP sap;
	sap.set_handle(handle);
	sap.enable(ACE_NONBLOCK);
}
