'''This module implements a templating generator based on Genshi'''
__revision__ = '$Revision: 4971 $'

from genshi.template import TemplateLoader, \
                            TextTemplate, MarkupTemplate, TemplateError
try:
    from genshi.template import NewTextTemplate
    have_ntt = True
except:
    have_ntt = False
import logging
import Bcfg2.Server.Plugin
import genshi.core, genshi.input

logger = logging.getLogger('Bcfg2.Plugins.TGenshi')

def removecomment(stream):
    """A genshi filter that removes comments from the stream."""
    for kind, data, pos in stream:
        if kind is genshi.core.COMMENT:
            continue
        yield kind, data, pos

class TemplateFile:
    '''Template file creates Genshi template structures for the loaded file'''
    def __init__(self, name, properties, specific, encoding):
        self.name = name
        self.properties = properties
        self.specific = specific
        self.encoding = encoding
        if self.specific.all:
            matchname = self.name
        elif self.specific.group:
            matchname = self.name[:self.name.find('.G')]
        else:
            matchname = self.name[:self.name.find('.H')]
        if matchname.endswith('.txt'):
            self.template_cls = TextTemplate
        elif matchname.endswith('.newtxt'):
            if not have_ntt:
                logger.error("Genshi NewTextTemplates not supported by this version of Genshi")
            else:
                self.template_cls = NewTextTemplate
        else:
            self.template_cls = MarkupTemplate
        
    def handle_event(self, event):
        '''Handle all fs events for this template'''
        if event.code2str() == 'deleted':
            return
        try:
            loader = TemplateLoader()
            self.template = loader.load(self.name, cls=self.template_cls,
                                        encoding=self.encoding)
        except TemplateError, terror:
            logger.error('Genshi template error: %s' % terror)
        except genshi.input.ParseError, perror:
            logger.error('Genshi parse error: %s' % perror)
            
    def bind_entry(self, entry, metadata):
        '''Build literal file information'''
        fname = entry.get('realname', entry.get('name'))
        try:
            stream = self.template.generate( \
                name=fname, metadata=metadata, path=self.name,
                properties=self.properties).filter(removecomment)
            if have_ntt:
                ttypes = [TextTemplate, NewTextTemplate]
            else:
                ttypes = [TextTemplate]
            if True in [isinstance(self.template, t) for t in ttypes]:
                textdata = stream.render('text')
                if type(textdata) == unicode:
                    entry.text = textdata
                else:
                    if self.encoding != 'ascii':
                        logger.debug("Override encoding of %s TGenshi template to %s" % (self.name, self.encoding))
                    entry.text = unicode(textdata, self.encoding)
            else:
                xmldata = stream.render('xml')
                if type(xmldata) == unicode:
                    entry.text = xmldata
                else:
                    if self.encoding != 'ascii':
                        logger.debug("Override encoding of %s TGenshi template to %s" % (self.name, self.encoding))
                    entry.text = unicode(xmldata, self.encoding)
        except TemplateError, terror:
            logger.error('Genshi template error: %s' % terror)
            raise Bcfg2.Server.Plugin.PluginExecutionError
        except AttributeError, err:
            logger.error('Genshi template loading error: %s' % err)
            raise Bcfg2.Server.Plugin.PluginExecutionError

class TGenshi(Bcfg2.Server.Plugin.GroupSpool):
    '''
    The TGenshi generator implements a templating
    mechanism for configuration files
    '''
    __name__ = 'TGenshi'
    __version__ = '$Id: TGenshi.py 4971 2008-11-12 23:02:13Z desai $'
    __author__ = 'jeff@ocjtech.us'
    use_props = True
    filename_pattern = 'template\.(txt|newtxt|xml)'
    es_child_cls = TemplateFile
