# Balazar
# Copyright (C) 2005 Jean-Baptiste LAMY
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import soya
import balazar, balazar.sound, balazar.controller as controller, balazar.base as base
from balazar.game_interface import *
from balazar.item           import *


class Inventory(widget.Label, soya.World):
  def __init__(self, character, items = None):
    soya.World.__init__(self)
    widget.Label.__init__(self)
    
    self.text  = _(u"Inventory")
    self.color = (0.0, 0.0, 1.0, 1.0)
    
    self.selection       = None
    
    self.character = character
    self.items     = items or character.items
    self.camera    = soya.Camera(self)
    self.camera.set_xyz(0.0, 0.4, 2.5)
    self.camera.partial = 1
    
    self.light = soya.Light(self)
    self.light.set_xyz(3.0, 2.0, 1.0)
    self.light.diffuse = (0.8, 0.8, 0.8, 1.0)
    
    self.atmosphere = soya.NoBackgroundAtmosphere()
    self.atmosphere.ambient = (0.2, 0.0, 0.2, 1.0)
    
    self.focus = soya.World(self)
    self.focus.atmosphere = soya.NoBackgroundAtmosphere()
    self.focus.atmosphere.ambient = (2.0, 2.0, 2.0, 1.0)
    
    self.clones    = []
    self.old_items = []
    
  def check(self): return 1
  
  def destroy(self):
    balazar.sound.play("menu1.wav")
    self.master.destroy()
  
  def set_focus(self, focus):
    if not focus: self.destroy()
    
  def resize(self, parent_left, parent_top, parent_width, parent_height):
    self.left   = parent_left
    self.top    = parent_top
    self.width  = min(80 * len(self.items) + 150, soya.get_screen_width() - 28)
    self.height = 250
    
    self.camera.set_viewport(soya.get_screen_width() - self.width, 0, self.width, 250)
    
  def render(self):
    self.camera.render()
    widget.Label.render(self)
    
  def widget_begin_round(self):
    import copy
    if self.old_items != self.items:
      # self.items have changed !!!
      self.select_item(None)
      
      d = dict([(clone.item, clone) for clone in self.clones])
      
      for clone in self.clones[:]:
        if not clone.item.owner is self.character:
          clone.parent.remove(clone)
          
      self.clones = []
      x = -0.55 * (len(self.items) - 1)
      for item in self.items:
        clone = d.get(item)
        if not clone:
          clone = copy.deepcopy(item)
          self.add(clone)
          clone.item = item
          clone.set_inventory_pos()
        else:
          clone.zap()
        clone.x = x
        x += 1.1
        
        self.clones.append(clone)
        
      self.old_items = self.items[:]
      
    for item in self.clones: item.begin_round()
      
  def widget_advance_time(self, proportion):
    if self.selection and not isinstance(self.selection, Photo): self.selection.rotate_lateral(4.0)
    
    for item in self.clones: item.advance_time(proportion)
    
  def select_item(self, item):
    if self.selection:
      self.focus.remove(self.selection)
      self.add(self.selection)
      
    self.selection = item
    if self.selection:
      self.remove(self.selection)
      self.focus.add(self.selection)
      self.text  = u"%s : %s" % (_(u"Inventory"), base.display_name(self.selection))
      
      if   -self.camera.x + self.selection.x >  4.0: self.camera.x = self.selection.x - 4.0
      elif -self.camera.x + self.selection.x < -4.0: self.camera.x = self.selection.x + 4.0
    else:
      self.text  = _(u"Inventory")
      
  def validate(self):
    import balazar.character
    
    balazar.sound.play("menu1.wav")
    item = self.selection.item
    if isinstance(item, EquipableItem):


      #print item, item.owner, self.character, item.owner is self.character
      
      if item.equiped: self.character.doer.do_action(balazar.character.ItemAction(balazar.character.ACTION_UNEQUIP_ITEM, item))
      else:            self.character.doer.do_action(balazar.character.ItemAction(balazar.character.ACTION_EQUIP_ITEM  , item))
      
    elif isinstance(item, Map):
      balazar.game_interface.Bubble(soya.root_widget, self.character, balazar.game_interface.MapViewer(item)).set_focus(1)
      balazar.sound.play("menu2.wav")
      
    elif isinstance(item, UseableItem):
      self.character.doer.do_action(balazar.character.ItemAction(balazar.character.ACTION_USE_ITEM, item))
      
    else:
      
      # XXX Useable Item
      
      pass
    
  def key_down(self, key):
    if   key == controller.CONTROL_LEFT:
      if self.clones:
        if not self.selection: self.select_item(self.clones[len(self.clones) // 2])
        else:
          i = self.clones.index(self.selection)
          if i > 0: self.select_item(self.clones[i - 1])
        return 1
    
    elif key == controller.CONTROL_RIGHT:
      if self.clones:
        if not self.selection: self.select_item(self.clones[len(self.clones) // 2])
        else:
          i = self.clones.index(self.selection)
          if i < len(self.clones) - 1: self.select_item(self.clones[i + 1])
        return 1
    
    elif key == controller.CONTROL_MENU:
      if self.selection: self.validate()
      self.destroy()
      return 1
    
    elif (key == controller.CONTROL_ITEM) or (key == controller.CONTROL_JUMP):
      if self.selection:
        balazar.sound.play("menu1.wav")
        self.character.doer.do_action(balazar.character.ItemAction(balazar.character.ACTION_DROP_ITEM, self.selection.item))
      return 1
    
    elif key == controller.CONTROL_QUIT:
      self.destroy()
      return 1
 
  

class PhotoViewer(widget.Label):
  def __init__(self, photo):
    widget.Label .__init__(self)
    
    from balazar.item import photo_model
    
    self.photo = photo
    
    self.material = soya.Material(soya.image_from_pil(photo.image.resize((1024, 512))))
    
    self.text  = photo.full_display_name()
    self.color = (0.0, 0.0, 1.0, 1.0)
    
  def check(self): return 1
  
  def destroy(self):
    balazar.sound.play("menu1.wav")
    self.master.destroy()
  
  def set_focus(self, focus):
    if not focus: self.destroy()
    
  def resize(self, parent_left, parent_top, parent_width, parent_height):
    self.left   = parent_left
    self.top    = parent_top
    self.width  = soya.get_screen_width () * 0.8
    self.height = soya.get_screen_height() * 0.8 + 50
    
    self._changed = -2
    
  def build_display_list(self):
    widget.Label.build_display_list(self)
    
    self.material.activate()
    soyaopengl.glDisable(soyaopengl.GL_CULL_FACE)
    soyaopengl.glBegin(soyaopengl.GL_QUADS)
    soyaopengl.glTexCoord2f(0.0, 0.0); soyaopengl.glVertex2i(self.left             , self.top + 50)
    soyaopengl.glTexCoord2f(1.0, 0.0); soyaopengl.glVertex2i(self.left + self.width, self.top + 50) 
    soyaopengl.glTexCoord2f(1.0, 1.0); soyaopengl.glVertex2i(self.left + self.width, self.top + self.height) 
    soyaopengl.glTexCoord2f(0.0, 1.0); soyaopengl.glVertex2i(self.left             , self.top + self.height)
    soyaopengl.glEnd()
    soya.DEFAULT_MATERIAL.activate()
    
  def key_down(self, key):
    self.destroy()
    return 1
   
    
