/** *********************************************************************
 * Copyright (C) 2003 Catalyst IT                                       *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 2 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, write to:                           *
 *   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    *
 *   Boston, MA  02111-1307  USA                                        *
 ************************************************************************/
package nz.net.catalyst.lucene.server;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import nz.net.catalyst.Log;
import nz.net.catalyst.StringPair;

/**
 * Read, store and write a Transmission.  A transmission consists of a
 * Command word and an ordered list of headers.  The headers consist
 * of a keyword and value.  The header list may contain duplicate
 * keywords.  Only one keyword is treated specially by this class:
 * Content-Length.  This is used in case a body is requested.
 */

public class ApplicationMessage implements IPackage, Constants
{
  private final Transmission input;

  private static final Map VALID = new HashMap();
  static
  {
    Map map = new HashMap();
    map.put(YES.toUpperCase(), YES);
    map.put(NO.toUpperCase(), NO);
    VALID.put(FIELD_INDEXED.toUpperCase(), map);
    VALID.put(FIELD_STORED.toUpperCase(), map);
    map = new HashMap();
    map.put(ID.toUpperCase(), ID);
    map.put(DATE.toUpperCase(), DATE);
    map.put(TEXT.toUpperCase(), TEXT);
    VALID.put(FIELD_TYPE.toUpperCase(), map);
  }

  ApplicationMessage(Transmission transmission)
  {
    input = transmission;
  }

  private Transmission error(String message)
  {
    Transmission response = new Transmission(ECommand.CONTROL_RESPONSE);
    response.setSerial(input.getSerial());
    response.add(ERROR, message);
    return response;
  }


  Transmission execute()
  {
    List headers = input.getHeadersView();

    String appName = input.get(APPLICATION, NO_APP);

    if (appName == null)
    {
      return error('"' + APPLICATION + "\" header required for \"" +
                   APPLICATION_SUB_COMMAND  + "\" sub-command");
    }

    appName = appName.trim();
    if (appName.length() == 0)
    {
      return error('"' + APPLICATION +
                   "\" header must not have blank value in \"" +
                   APPLICATION_SUB_COMMAND  + "\" sub-command");
    }

    Application app = Application.get(appName);

    if (app == null)
      app = new Application(appName);

    Field field = null;

    Log.debug("Processing Application " + appName);

    for (Iterator it = headers.iterator(); it.hasNext(); )
    {
      StringPair header = (StringPair)it.next();
      String key = header.getKeyString();
      String value = header.getValueString();

      if (key.equalsIgnoreCase(APPLICATION) ||
          key.equalsIgnoreCase(SUB_COMMAND) ||
          key.equalsIgnoreCase(CONTENT_LENGTH) ||
          key.equalsIgnoreCase(BODY) ||
          key.equalsIgnoreCase(ID))
        continue;

      if (key.equalsIgnoreCase(FIELD_NAME))
      {
        if (value.length() == 0)
        {
          field = null;
          app.remove(Application.FIELD_DEFS);
        }
        else
        {
          if (field != null)
          {
            app.appendProperty(Application.FIELD_DEFS, field.toString(), ", ");
            Log.debug(Application.FIELD_DEFS + ": " +
                               app.getProperty(Application.FIELD_DEFS));
          }


          field = new Field(value);
        }
      }
      else if (key.equalsIgnoreCase(FIELD_TYPE) ||
               key.equalsIgnoreCase(FIELD_INDEXED) ||
               key.equalsIgnoreCase(FIELD_STORED))
      {
        Map map = (Map)VALID.get(key.toUpperCase());
        String goodValue;

        if (field != null && map != null &&
            (goodValue = (String)map.get(value.toUpperCase())) != null)
        {
          if (key.equalsIgnoreCase(FIELD_TYPE))
            field.type = goodValue;
          else if (key.equalsIgnoreCase(FIELD_INDEXED))
            field.indexed = goodValue;
          else if (key.equalsIgnoreCase(FIELD_STORED))
            field.stored = goodValue;
        }
      }
      else if (value.length() == 0)
        app.remove(key);
      else
      {
        app.appendProperty(key, value);
        Log.debug(key + ": " + app.getProperty(key.toLowerCase()));
      }
    }
    if (field != null)
    {
      app.appendProperty(Application.FIELD_DEFS, field.toString(), ", ");
      Log.debug(Application.FIELD_DEFS + ": " +
                         app.getProperty(Application.FIELD_DEFS));
    }

    app.store();
    Transmission response = new Transmission(ECommand.CONTROL_RESPONSE);
    response.setSerial(input.getSerial());
    return response;
  }

  private static class Field
  {
    final String name;
    String type = "Text";
    String indexed = "Yes";
    String stored = "No";

    Field(String name)
    {
      this.name = name;
    }

    public String toString()
    {
      return name + ' ' + type + ' ' + indexed + ' ' + stored;
    }
  }
}

