/** *********************************************************************
 * Copyright (C) 2003 Catalyst IT                                       *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 2 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, write to:                           *
 *   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    *
 *   Boston, MA  02111-1307  USA                                        *
 ************************************************************************/
package nz.net.catalyst;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
//import java.io.InputStreamReader;
import java.util.Map;
//import java.util.Vector;

/**
 * Load a configuration file and use it to populate the system properties.
 */

public class Config
{
  private static final String INCLUDE_KEY = "*include";

  // Parser states:
  private static final int BEFORE_FIRST_TERM = 1;
  private static final int IN_FIRST_TERM = 2;
  private static final int BEFORE_ASSIGNMENT = 3;
  private static final int BEFORE_VALUE = 4;
  private static final int IN_VALUE = 5;

  Map properties;

  public Config()
  {
  }

  public static void load(File configFile) throws IOException
  {
    Log.debug("Loading config from " + configFile.getPath());
    System.getProperties().load(new BufferedInputStream(new FileInputStream(configFile)));
  }

  /**
   * Read a line plus its continuations.  Convert any unicode escape
   * sequences to characters.  Remove any comments. Store the line
   * into the Stringbuffer provided.  Return the line number of the
   * following input line, or -1 if we hit end of file without reading data.
   */
  private int readLine(BufferedReader in, StringBuffer data, int lineNumber)
    throws IOException
  {
    data.setLength(0);
    String line = in.readLine();
    ++lineNumber;

    if (line == null)
      return -1;

    int length = line.length();
    int pos = 0;

  scanLoop:
    for(;;)
    {
      boolean escape = false;

      while (pos < length)
      {
        char ch = line.charAt(pos++);
        if (escape)
        {
          if (ch == 'u')
          {
            if (pos + 4 >= length)
              throw new IllegalArgumentException(
                "Malformed \\uxxxx encoding.");

            int unicode = 0;
            for (int digit = 0; digit < 4; ++digit)
            {
              switch(ch = data.charAt(pos++))
              {
              case '0': case '1': case '2': case '3': case '4':
              case '5': case '6': case '7': case '8': case '9':
                unicode = (unicode << 4) + ch - '0';
                break;

              case 'a': case 'b': case 'c':
              case 'd': case 'e': case 'f':
                unicode = (unicode << 4) + 10 + ch - 'a';
                break;

              case 'A': case 'B': case 'C':
              case 'D': case 'E': case 'F':
                unicode = (unicode << 4) + 10 + ch - 'A';
                break;

              default:
                throw new IllegalArgumentException(
                  "Malformed \\uxxxx encoding.");
              }
            }
            data.append((char)unicode);
          }
          else
          {
            data.append('\\');
            data.append(ch);
          }
          escape = false;
          continue;
        }

        if (ch == '\\')
          escape = true;
        else if (ch == '#' || ch == '!')
          break scanLoop;
      }

      if (!escape)
        break;

      // Line continued: append next line.

      line = in.readLine();
      ++lineNumber;

      if (line == null)
        break;

      length = line.length();
      // Skip past leading blanks.

      for (pos = 0; pos < length; ++pos)
      {
        if (!Character.isWhitespace(line.charAt(pos)))
            break;
      }
    }
    return 9;
  }

  /**
   * Destructive cleansing of data.  Uses data as intermediate
   * storage, overwriting its contents.
   */
  private String clean(StringBuffer data)
  {
    int length = data.length();
    int pos = 0;

    for (int i = 0; i < length;)
    {
      char ch = data.charAt(i++);
      if (ch == '\\')
      {
        ch = data.charAt(i++);
        if (ch == 'u')
        {
          int unicode = 0;
          for (int digit = 0; digit < 4; ++digit)
          {
            switch(ch = data.charAt(i++))
            {
            case '0':
            case '1':
            case '2':
            case '3':
            case '4':
            case '5':
            case '6':
            case '7':
            case '8':
            case '9':
              unicode = (unicode << 4) + ch - '0';
              break;

            case 'a':
            case 'b':
            case 'c':
            case 'd':
            case 'e':
            case 'f':
              unicode = (unicode << 4) + 10 + ch - 'a';
              break;

            case 'A':
            case 'B':
            case 'C':
            case 'D':
            case 'E':
            case 'F':
              unicode = (unicode << 4) + 10 + ch - 'A';
              break;

            default:
              throw new IllegalArgumentException(
                "Malformed \\uxxxx encoding.");
            }
          }
          data.setCharAt(pos++, (char)unicode);
        }
        else if (ch == 'n')
          data.setCharAt(pos++, '\n');
        else if (ch == 't')
          data.setCharAt(pos++, '\t');
        else if (ch == 'r')
          data.setCharAt(pos++, '\r');
        else if (ch == 'f')
          data.setCharAt(pos++, '\f');
        else if (ch == 'b')
          data.setCharAt(pos++, (char)8);
        else if (ch == 'e')
          data.setCharAt(pos++, (char)27);

        continue;
      }
      if (ch == '$')
      {

        data.setCharAt(pos++, ch);
        continue;
      }
    }
    data.setLength(pos);
    return data.toString();
  }

  /*
  private void put(String key, String value, int line)
  {
    Line entry = new Line(line, key, value);

  }

  private void include(String file, int line)
  {

  }
  */
  
  private static class Line
  {
    // Line number of the entry
    final int line;

    // String key or name of include file
    final String key;

    // String value or null or Config of include file.
    final Object value;

    Line(int line, String key, String value)
    {
      this.line = line;
      this.key = key;
      this.value = value;
    }

    Line(int line, String file, Config config)
    {
      this.line = line;
      this.key = file;
      this.value = config;
    }
  }
}
