#ifndef READHANDLERS_H
#define READHANDLERS_H

/**
 *
 * Interfaces that provide uniform access to data in Audio Video streams,
 * local and networked files  (Avi, Asf, maybe more later)
 *
 */

#include "avifile.h"
#include "StreamInfoPriv.h"
#include "string.h"

AVM_BEGIN_NAMESPACE;

// StreamPacket implemented in ReadStream.cpp
class StreamPacket
{
protected:
    uint8_t* memory;
    size_t size;
    size_t read;
    int flags;
    framepos_t position;
    int64_t timestamp;  // timestamp in microseconds
    uint_t refcount;

public:
    static const size_t MAX_PACKET_SIZE = 10000000;
    static const int64_t NO_TIMESTAMP = -1LL;

    StreamPacket(size_t bsize = 0, void* mem = 0);
    virtual ~StreamPacket();
    /** return number of read bytes from packet */
    size_t Read(void* to, size_t read_size);
    uint8_t* GetData() const { return memory + read; }
    size_t GetSize() const { return size - read; }
    size_t GetRead() const { return read; }
    int GetFlags() const { return flags; }
    void SetFlags(int f)  { flags = f; }
    framepos_t GetPos() const { return position; }
    void SetPos(framepos_t p)  { position = p; }
    int64_t GetTimestamp() const { return timestamp; }
    void SetTimestamp(int64_t t)  { timestamp = t; }
    double GetTime() const { return (double)timestamp / 1000000.; }
    void SetTime(double t)  { timestamp = (int64_t)(t * 1000000.); }
    void SwapData(StreamPacket& p)  { uint8_t* m = memory; memory = p.memory; p.memory = m; }
    void AddRef() { refcount++; }
    void Release() { refcount--; if (!refcount) delete this; }
};

class IMediaReadStream : public IStream
{
public:

    virtual ~IMediaReadStream() {}

    /* 1 cache full, 0 cache empty */
    virtual double CacheSize() const				= 0;

    /* Clear the internal stream cache */
    virtual void ClearCache()					= 0;

    /**
     * Return structure describing main stream header
     *
     * NOTE: format is in the stream based one!
     * Use GetStreamInfo to obtain standard set of informations
     * about stream
     * this call will be eliminated later!!
     */
    virtual size_t GetHeader(void *header, size_t size) const = 0;

    /** Retrieval of stream-specific format data */
    // XXXXXXXXXXXXX
    virtual size_t GetFormat(void *format = 0, size_t size = 0) const	= 0;

    /** Stream frame time - might be just average for whole stream */
    // XXXXXXXXXXXXX
    virtual double GetFrameTime() const				= 0;

    /** Length of the stream in samples - equals with last frame position */
    virtual framepos_t GetLength() const			= 0;

    /** Length of the stream in seconds */
    // XXXXXXXXXXXXX
    virtual double GetLengthTime() const			= 0;

    /**
     * position of previous, next and nearest key frame
     * related to frame pos ( ERR means current frame )
     */
    virtual framepos_t GetNearestKeyFrame(framepos_t pos = ERR) const = 0;
    virtual framepos_t GetNextKeyFrame(framepos_t pos = ERR) const = 0;
    virtual framepos_t GetPrevKeyFrame(framepos_t pos = ERR) const = 0;

    /** Timestamp of sample no. lSample ( ERR = current sample ) */
    virtual double GetTime(framepos_t lSample = ERR) const = 0;

    /** Returns 0 for variable frame size stream (usually video), >0 for audio */
    virtual size_t GetSampleSize() const			= 0;

    /**
     * Various useful stream information
     *
     * this is preffered way to obtain information about stream!
     */
    virtual StreamInfo* GetStreamInfo() const			= 0;

    /** Type of stream - usually fcc 'vids' or 'auds' */
    virtual StreamType GetType() const				= 0;

    /**
     * Return true if frame no. pos ( ERR means current frame ) is keyframe
     */
    virtual bool IsKeyFrame(framepos_t pos = ERR) const		= 0;

    /** Obtain whole packet - new prefered way for reading streams */
    virtual StreamPacket* ReadPacket()				= 0;

    /** Seek to frame position pos in stream */
    virtual int Seek(framepos_t pos)				= 0;

    /** Seek to time */
    virtual int SeekTime(double time)				= 0;

    virtual int SkipFrame()					= 0;
    virtual int SkipTo(double time)				= 0;

    // this is currently cruel hack
    virtual int FixAvgBytes(uint_t bps) { return -1; }
};

class IMediaReadHandler : public IStream
{
public:
    virtual ~IMediaReadHandler() {}

    /* vids/auds, 0-... */
    virtual IMediaReadStream* GetStream(uint_t stream_id, StreamType type) = 0;
    virtual size_t GetStreamCount(StreamType type)		= 0;
    virtual size_t GetHeader(void* header, size_t size)		= 0;

    /* All calls for network-oriented readers should be interruptible  */
    virtual void Interrupt() 					= 0;

    /* in case of asynchronous opening */
    virtual bool IsOpened() 					= 0;
    virtual bool IsValid()					= 0;
    virtual bool IsRedirector() 				= 0;
    virtual bool GetURLs(avm::vector<avm::string> &urls) 	= 0;
};

/*  Read handler for AVI file  */
IMediaReadHandler* CreateAviReadHandler(const char *pszFile, int flags);

/*  Read handler for ASF/WMV/ASX file  */
IMediaReadHandler* CreateAsfReadHandler(const char *pszFile);

/*  Read handler for everything :) file  */
IMediaReadHandler* CreateFFReadHandler(const char *pszFile);

AVM_END_NAMESPACE;

#endif // READHANDLERS_H
