#!/usr/bin/env python
# Copyright (C) Andrew Mitchell 2006,2007

#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.

#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
############################################################################

import os, sys
import debconf
import config
import string
import time
import ConfigParser

from Version import VERSION,INSTALLED

def dyn_import(name):
    try:
        mod = __import__(name)
        components = string.split(name, '.')
        for comp in components[1:]:
            mod = getattr(mod, comp)
        return mod
    except (AttributeError,ValueError), mesg:
        raise ImportError, mesg

import warnings

def deprecated(func):
    """This is a decorator which can be used to mark functions
    as deprecated. It will result in a warning being emitted
    when the function is used."""
    # From http://wiki.python.org/moin/PythonDecoratorLibrary
    def newFunc(*args, **kwargs):
        warnings.warn("Call to deprecated function %s." % func.__name__,
                      category=DeprecationWarning)
        return func(*args, **kwargs)
    newFunc.__name__ = func.__name__
    newFunc.__doc__ = func.__doc__
    newFunc.__dict__.update(func.__dict__)
    return newFunc


class IAuthConfig(object):
    packages = []
    requires = []
    name = ""
    persistent = []
    detected = False

    # variable:'new value' pairs for delayed commits
    changed = {}

    def __init__(self, authconfig):
        self.cfg = authconfig
        # setup keys
        # catch 22 - pulling the config requires the keys to be setup
        # Instead, use the default values in the config to set the variable
        # However, the load call will overwrite them iff they exist
        self.persistent += self.config().keys()
        for item in self.persistent:
            if not item in self.__dict__:
                #print "Setting %s = %s" % (item, self.config()[item][2])
                self.__setattr__(item, self.config()[item][2] )

        # load from storage
        self.load()

    def config(self):
        """Return a list of configuration options in a standardised dictionary format

        {'id':[type,description,default,options],...}
        eg
        {'server':['string','Kerberos Server','auth-server.localdomain','']}
        each id will be prefixed by the method name, ie server becomes kerberos_server
        options are dependant on the config type
        types are modelled on debconf (for good reason):
         - boolean
         - password
         - string
         - select
         - multiselect
         - note
         - text
         - error
         - title
        
        """
        return {}

    def preserve(self, filename):
        """Backup the system configuration file"""
        current_time = int(time.time())
        fh_r = open(filename,'r')
        fh_w = open('%s.%s' % (filename,current_time),"w")
        for line in fh_r.read():
            fh_w.write(line)
        fh_r.close()
        fh_w.close()

        

    def restore(self, filename):
        """Restore the system configuration file (will need some version information)"""

    def enable(self):
        """Enable this authentication method"""

    def disable(self):
        """Disable this authentication method"""
        
    def preinst(self, mode):
        """Execute the maintainer script stanzas"""

    def postinst(self, mode):
        """Execute the maintainer script stanzas"""

    def prerm(self, mode):
        """Execute the maintainer script stanzas"""

    def postrm(self, mode):
        """Execute the maintainer script stanzas"""

    def store(self, id, data):
        """Store the configuration data that was given. This can just be in-memory storage until commit() is called"""
        #print "Storing [%s]%s:%s" % (self.name, id,data)
        # section is method name
        try:
            if not self.cfg.cf.has_section(self.name):
                self.cfg.cf.add_section(self.name) 
            self.cfg.cf.set(self.name, id, data)
        except ConfigParser.NoSectionError:
            self.cfg.cf.add_section(self.name)
            self.cfg.cf.set(self.name, id, data)

    def save(self):
        """Call store() for each persistent setting in the class"""
        for prop in self.persistent:
            #print "Saving %s" % prop
            self.store(prop, self.__getattribute__(prop))

    def load(self):
        """Call get() for each persistent setting in the class & set them in the object"""
        #print "Persistent settings are %s" % self.persistent
        for prop in self.persistent:
            try:
                result = self.get(prop)
                if result:
                    self.__dict__[prop] = result
                    obj = self.__getattribute__(prop)
                    #print "Loaded %s = %s" % (prop, obj)
            except ConfigParser.NoOptionError:
                pass
            except ConfigParser.NoSectionError:
                pass
            
        
    def get(self, id):
        return self.cfg.cf.get(self.name, id)


    def commit(self):
        """Commit stored changes"""
        print "Committing changes"

    def detect(self, params):
        """Return True/False based on whether the method thinks it can connect, based on the data given"""

class Debconf:
    """From http://aspn.activestate.com/ASPN/Cookbook/Python/Recipe/52558"""

    class __impl:
        db = None

        def __init__(self):
            if self.db is None:
                #print "Initiating debconf"
                self.db = debconf.DebconfCommunicator('authtool')

        def get(self, item):
            return self.db.get(item)

        def set(self, item, value):
            return self.db.set(item, value)

        def unlock(self):
            """Stops debconf instance - must create when finished"""
            self.db.shutdown()

        def reinit(self):
            # Can't call __init__, need a new object this time
            self.db = debconf.DebconfCommunicator('authtool')
        
    __instance = None
    
    def __init__(self):
        """ Create debconf instance """
        # Check whether we already have an instance
        if Debconf.__instance is None:
            # Create and remember instance
            Debconf.__instance = Debconf.__impl()

        # Store instance reference as the only member in the handle
        self.__dict__['_Debconf__instance'] = Debconf.__instance

    def __getattr__(self, attr):
        """ Delegate access to implementation """
        return getattr(self.__instance, attr)

    def __setattr__(self, attr, value):
        """ Delegate access to implementation """
        return setattr(self.__instance, attr, value)


        

    
class Authtools:

    modules={}
    method_cfgs={}

    def __init__(self):

        self.cfg = config.AuthConfig()

        # load all methods in the auth_methods dir
        if INSTALLED:
            path = '/usr/share/authtool/auth_methods'
            sys.path.append('/usr/share/authtool')
            #path = '/usr/share/pycentral/authtool/site-packages/auth_methods'
        else:
            path = 'auth_methods'
        files = os.listdir(path)
        #print sys.path
        #print files
        dirs = os.listdir(path)
        for method in dirs:
            if (method[-3:] != '.py') and (method[-4:] != '.pyc'):
                #print "Importing %s" % method
                self.modules[method] = dyn_import("auth_methods.%s.%s" % (method,method))
        #print self.modules
        
        #return
#         for method in files:
#             if method[-3:] == '.py':
#                 self.modules[method[:-3]] = dyn_import("auth_methods.%s" % method[:-3])
#                 #self.modules.append( dyn_import("auth_methods.%s" % method[:-3]) )

        #print "#####"
        for method in config.methods:
            obj = getattr(self.modules[string.lower(method)],"%sConfig" % method)
            self.method_cfgs[method] = obj(self.cfg)
        #print self.method_cfgs

    def get_method_obj(self, method):
        """Return the MethodConfig"""
        try:
            return self.method_cfgs[method]
        except:
            print "Error fetching %s from %s", (method, self.method_cfgs)
        

    def get_methods(self):
        """Get a list of the authentication methods supported"""
        return config.methods

    def method_display(self, method):
        return self.method_cfgs[method].display

    def get_method_config(self, method):
        # 3AM code here - 3:20, removed the recursion
        # get the config of this method & all of its subsequent methods
        cfg_all = {}
        m = self.get_method_obj(method)
        if m.requires != [] and m.requires != None:
            for item in m.requires:
                obj =  self.get_method_obj(item)
                #print obj.config()
                if obj.config() is not None:
                    cfg_all[item] = obj.config()


        cfg = m.config()
        if cfg is not None:
            cfg_all[method] = cfg
        return cfg_all

    def set_method_config(self, method, config):
        """Set the method's config & use each config's storage mechanism (debconf, etc) to save them"""
        pass

    def get_current(self):
        """Return the current authentication method"""
        return self.cfg.current

    def set_current(self, method):
        """Set the current authentication method
        Needs to be followed by a reconfigure()"""
        print "Setting ", method
        self.cfg.current = method

    # Should really use commit/rollback transaction style for *all* operations
    def commit_current(self):
        self.cfg.save()

    def disable(self, method, verbose=False):
        """Trigger the disabling code for the method, but first its dependencies"""
        #print "Now in disable(%s)" % method
        m = self.get_method_obj(method)
        if m.requires != [] and m.requires != None:
            for item in m.requires:
                self.disable(item)
        self.method_cfgs[method].disable()


    def enable(self, method, verbose=False):
        """Trigger the enabling code for the method, but first its dependencies"""
        if verbose:
            print "Now in enable(%s)" % method
        m = self.get_method_obj(method)
        if m.requires != [] and m.requires != None:
            #if verbose:
            #    print "Requires ", m.requires
            for item in m.requires:
                ## add to seen list to avoid loops
                self.enable(item)
        #if verbose:
        #    print "Enabling %s" % method
        #    print self.method_cfgs[method]
        self.method_cfgs[method].enable()

    def required_packages(self, method):
        """For the selected authentication method, check what other methods are required,
           and what Debian packages need to be installed for it to work
           Returns a list of packages to install, frontends can install the packages themselves
           with python-apt
           """

        #print "\nLooking up for %s" % method
        try:
            requires = self.method_cfgs[method].requires
        except AttributeError:
            requires = None
        #if requires:
        #    print "Requires other method: %s" % requires

        #print "Checking packages for %s" % method
        try:
            packages = self.method_cfgs[method].packages
        except AttributeError:
            packages = None
        #print "Depends on packages: %s" % packages
        return packages
            
    def reload_config(self, conffile):
        # Reconfigure each auth method according to the config file
        # Best option is to just read in a new config object
        self.cfg = config.AuthConfig(conffile)

        for method in config.methods:
            obj = self.get_method_obj(method)
            # Reset the config reference in the auth method
            obj.cfg = self.cfg
            obj.load()
        # Current method will be changed
        # Use 'authtool' to transform config file to a working configuration
        


    def reconfigure(self):
        """Reconfigure the system authentication according to the current
        authtools configuration"""
        pass


if __name__ == '__main__':

    at = Authtools()

    
