/*  Audacious-DUMB Plug-in to enable audacious to use DUMB for IT/XM/S3M/MOD files
 *  Copyright (C) 2006-2007 Christian Birchinger
 *  Audacious port based on previous plugins for XMMS and BMP
 *  Copyright (C) 2002-2003  Ben Davis
 *  Incorporates code from the wav plug-in,
 *  Copyright (C) 1998-2000  Peter Alm, Mikael Alm, Olle Hallnas, Thomas Nilsson and 4Front Technologies
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "audacious-dumb.h"

#include <audacious/output.h>

#include <gtk/gtk.h>
#include "interface.h"
#include "support.h"

DuhFile *duh_file = NULL;
static GThread *decode_thread;
static gboolean disable_amiga_mods = FALSE;
static const gchar duh_custom_title_format[] = "${file-name}${?title: - ${title}}";
static gint output_frequency;
static const gint output_frequencies[]={4000, 8000, 11025, 22050, 44100, 48000, 64000, 96000};
static GtkWidget *prefs;

gchar *fmts[] = { "mod", "s3m", "it", "xm", "duh", NULL };

InputPlugin duh_ip = {
    .description = "DUMB Plugin " VERSION,
    .init = duh_init,
    .about = duh_about,
    .configure = duh_configure,
    .play_file = play_file,
    .stop = stop,
    .pause = duh_pause,
    .seek = seek,
    .get_song_info = get_song_info,
    .file_info_box = file_info_box,
    .get_song_tuple = get_tuple_info,
    .is_our_file_from_vfs = is_our_file_from_vfs,
    .vfs_extensions = fmts,
    .mseek = mseek,
    .probe_for_tuple = duh_probe_for_tuple
};

InputPlugin *duh_iplist[] = { &duh_ip, NULL };

SIMPLE_INPUT_PLUGIN(duh_ip, duh_iplist)

static void duh_init(void)
{
	ConfigDb *db;
	db = aud_cfg_db_open();

	aud_cfg_db_get_bool(db, "DUMB", "disable_amiga_mods", &disable_amiga_mods);
	if (!disable_amiga_mods) /* force db creation since defined false equals none-existant */
		aud_cfg_db_set_bool(db, "DUMB", "disable_amiga_mods", FALSE);

	aud_cfg_db_get_int(db, "DUMB", "output_frequency", &output_frequency);	
	if (output_frequency < 4000 || output_frequency > 96000) {
		output_frequency = 44100;
		aud_cfg_db_set_int(db, "DUMB", "output_frequency", output_frequency);
	}
		
	aud_cfg_db_close(db);

	/* dumb_register_stdfiles(); */
	/*  Seems only needed when opening files with dumb directly * 
	 *  and not when VFS puts them to a memory location first   */
	
	dumb_it_max_to_mix = 256;
}

static int is_our_file_from_vfs(gchar *filename, VFSFile *file)
{
	gchar magic[4];
	const gchar *ext;

	aud_vfs_fread(magic, 1, 4, file);
	if (!memcmp(magic, XM_MAGIC, 4))
                return TRUE;
	if (!memcmp(magic, IT_MAGIC, 4))
		return TRUE;

	aud_vfs_fseek(file, 44, SEEK_SET);
	aud_vfs_fread(magic, 1, 4, file);
	if (!memcmp(magic, S3M_MAGIC, 4))
		return TRUE;

	aud_vfs_fseek(file, 1080, SEEK_SET);
	aud_vfs_fread(magic, 1, 4, file);   
	if (!memcmp(magic, MOD_MAGIC_FASTTRACKER6, 4))
		return TRUE;
	if (!memcmp(magic, MOD_MAGIC_FASTTRACKER8, 4))
		return TRUE;

	if (!disable_amiga_mods) {
		if (!memcmp(magic, MOD_MAGIC_PROTRACKER4, 4))
			return TRUE;
		if (!memcmp(magic, MOD_MAGIC_PROTRACKER4X, 4))
			return TRUE;
		if (!memcmp(magic, MOD_MAGIC_NOISETRACKER, 4))
			return TRUE;
		if (!memcmp(magic, MOD_MAGIC_STARTRACKER4, 4))
			return TRUE;
		if (!memcmp(magic, MOD_MAGIC_STARTRACKER8, 4))
			return TRUE;
		if (!memcmp(magic, MOD_MAGIC_STARTRACKER4X, 4))
			return TRUE;
		if (!memcmp(magic, MOD_MAGIC_STARTRACKER8X, 4))
			return TRUE;
		if (!memcmp(magic, MOD_MAGIC_FASTTRACKER4, 4))
			return TRUE;
	}

	ext = strrchr(filename, '.');
	if (ext) {
		if (!strcasecmp(ext, ".duh")) return TRUE;
		if (!strcasecmp(ext, ".it")) return TRUE;
		if (!strcasecmp(ext, ".xm")) return TRUE;
		if (!strcasecmp(ext, ".s3m")) return TRUE;
		if (!disable_amiga_mods) {
			if (!strcasecmp(ext, ".mod")) return TRUE;
		}
	}

	return FALSE;
}

static gint duh_universal_load_vfs(DUH **duh, const char *filename, VFSFile *extfd, const short testmask)
{
	DUMBFILE *filedumb;

	gint ext_index = -1;
	VFSFile *fd = NULL;
	off_t filesize, readsize;
	gchar *filemap;
	
	*duh = NULL;

	if (extfd)
		fd = extfd;
	else {
		fd = aud_vfs_fopen(filename, "rb");
		g_return_val_if_fail(fd != NULL, -1);
	}

	filesize = aud_vfs_fsize(fd);
	g_return_val_if_fail(filesize > 0, -1);
	
	filemap = malloc(filesize);
	g_return_val_if_fail(filemap != NULL, -1);

	readsize = aud_vfs_fread(filemap, 1, filesize, fd);

	if (!extfd)
		aud_vfs_fclose(fd);

	if (readsize == 0) {
		g_warning("audacious-dumb: Couldn't read from %s", filename);
		free(filemap); filemap = NULL;
		return -1;
	}

	/*
	   The close and re-opening after reach read attempt is needed.

	   Quote from the documentation:

	   WARNING: The behaviour of these functions are undefined if you pass a
	   DUMBFILE from which data have already been read; they are likely
	   not to work. This oversight will be fixed in future releases.
	*/

	int count, total_count = sizeof(uniread) / sizeof(uniread_struct_t);

	for (count = 0; count < total_count; count++) {
		if (uniread[count].testmask & testmask) {
			filedumb = dumbfile_open_memory(filemap, readsize);
			*duh = uniread[count].read(filedumb);
			dumbfile_close(filedumb);
			if (*duh) {
				ext_index = count;
				break;
			}
		}
	}

	free(filemap); filemap = NULL;

	return ext_index;
}

static Tuple *get_tuple_info_from_duh(DUH *duh, char *filename)
{
	const gchar *mod_title_raw;
	gchar *mod_title;

	g_return_val_if_fail(filename != NULL, NULL);
	g_return_val_if_fail(duh != NULL, NULL);

	Tuple *tuple = aud_tuple_new_from_filename(filename);

	aud_tuple_associate_string(tuple, FIELD_FORMATTER, NULL, duh_custom_title_format);
	aud_tuple_associate_string(tuple, FIELD_QUALITY, NULL, "sequenced");
	aud_tuple_associate_int(tuple, FIELD_LENGTH, NULL, (int)(duh_get_length(duh) * 1000LL >> 16) );
	
	mod_title_raw = duh_get_tag(duh, "TITLE");
	if (mod_title_raw) {
		/* DOS (CP850) all my XM IT and S3M use this (MODs should be 7bit ASCII) */ 
		mod_title = g_convert(mod_title_raw, -1, "UTF-8", "CP850", NULL, NULL, NULL);
		/* Raw copy but GTK2 doesn't like none-UTF8 stuff*/ 
		/* mod_title = g_strdup(mod_title_raw); */

		g_strstrip(mod_title);

		if (mod_title[0])
			aud_tuple_associate_string(tuple, FIELD_TITLE, NULL, mod_title);

		g_free(mod_title); mod_title = NULL;
	}
	mod_title_raw = NULL;

	return tuple;
}

static Tuple *__get_tuple_info(char *filename, VFSFile *fd)
{
	Tuple *input = NULL;
	DUH *duh = NULL;
	int ext_index;

	g_return_val_if_fail(filename != NULL, NULL);

	ext_index = duh_universal_load_vfs(&duh, filename, fd, UNIREAD_ALL);
	if (ext_index == -1)
		return NULL; /* Return empty Tuple if module was not loadable */
	
	input = get_tuple_info_from_duh(duh, filename);
	aud_tuple_associate_string(input, FIELD_CODEC, NULL, uniread[ext_index].description);	

	return input;
}

static Tuple *get_tuple_info(char *filename)
{
	return __get_tuple_info(filename, NULL);
}

static Tuple *duh_probe_for_tuple(char *filename, VFSFile *fd)
{
	if (!is_our_file_from_vfs(filename, fd))
		return NULL;
		
	aud_vfs_rewind(fd);
		
	return __get_tuple_info(filename, fd);
}

static void get_song_info(char *filename, char **title, int *length)
{
	Tuple *tuple = get_tuple_info(filename);
	
        (*length) = -1;
        (*title) = NULL;
	
	*length = aud_tuple_get_int(tuple, FIELD_LENGTH, NULL);
	*title = aud_tuple_formatter_process_string(tuple, duh_custom_title_format);

	tuple_free(tuple);
}

static gchar *get_title(DUH *duh, char *filename)
{
        Tuple *tuple = NULL;
	gchar *title;
	
	tuple = get_tuple_info_from_duh(duh, filename);
	title = aud_tuple_formatter_process_string(tuple, duh_custom_title_format);

	aud_tuple_free(tuple);

	return title;
}

static void install_callbacks(DUH_SIGRENDERER *sr)
{
	DUMB_IT_SIGRENDERER *itsr = duh_get_it_sigrenderer(sr);
	dumb_it_set_loop_callback(itsr, &dumb_it_callback_terminate, NULL);
	dumb_it_set_xm_speed_zero_callback(itsr, &dumb_it_callback_terminate, NULL);
}

static void *play_loop(void *arg)
{
	char data[2048 * 2];
	int blk_size, rate;
	unsigned int bytes;
	int actual_read;
	
	InputPlayback *playback = arg;

	blk_size = 512 * (duh_file->bits_per_sample / 8) * duh_file->channels;
	rate = duh_file->samples_per_sec * duh_file->channels * (duh_file->bits_per_sample / 8);
	while (playback->playing)
	{
		if (duh_file->seek_to != -1)
		{
			duh_file->position = (unsigned long)((gint64)duh_file->seek_to * (gint64)rate / 1000L);
			duh_end_sigrenderer(duh_file->file);
			duh_file->file = duh_start_sigrenderer(duh_file->duh, 0, duh_file->channels, (unsigned long)((gint64)duh_file->seek_to << 16) / 1000L);
			install_callbacks(duh_file->file);
			playback->output->flush(duh_file->seek_to);
			duh_file->seek_to = -1;
		}

		if (!playback->eof)
		{
			bytes = blk_size;
			if (duh_file->length - duh_file->position < bytes)
				bytes = duh_file->length - duh_file->position;
			if (bytes > 0)
			{
				actual_read = duh_render(duh_file->file, 16, 0, 1.0f, 65536.0f / duh_file->samples_per_sec, bytes / ((duh_file->bits_per_sample / 8) * duh_file->channels), data);
				actual_read *= (duh_file->bits_per_sample / 8) * duh_file->channels;

				if (actual_read == 0)
					playback->eof = TRUE;
				else
				{
					if (playback->playing && duh_file->seek_to == -1)
						playback->pass_audio(playback, (duh_file->bits_per_sample == 16) ? FMT_S16_LE : FMT_U8,
						duh_file->channels, bytes, data, &playback->playing);

					duh_file->position += actual_read;
				}
			}
			else
				playback->eof = TRUE;
		}
		else
		{
			playback->output->buffer_free ();
			playback->output->buffer_free ();
			while (playback->output->buffer_playing())
				g_usleep(10000);
			playback->playing = 0;
		} 
	}

	duh_end_sigrenderer(duh_file->file);
	unload_duh(duh_file->duh);
	return NULL;
}

static void play_file(InputPlayback *playback)
{
	char *name;
	int rate;
	char *filename = playback->filename;

	playback->error = FALSE;

	duh_file = g_malloc(sizeof (DuhFile));
	memset(duh_file, 0, sizeof (DuhFile));

	duh_universal_load_vfs(&duh_file->duh, filename, NULL, UNIREAD_ALL);
	if (!duh_file->duh) {
		g_warning("audacious-dumb: Unable to play %s", filename);
		g_free(duh_file);
		duh_file = NULL;
		return;
	}

	if ((duh_file->file = duh_start_sigrenderer(duh_file->duh, 0, 2, 0)))
	{
		install_callbacks(duh_file->file);
		duh_file->channels = 2;
		duh_file->samples_per_sec = output_frequency;
		duh_file->bits_per_sample = 16;
		duh_file->length = (long long)duh_get_length(duh_file->duh) * (output_frequency * 2 * 2) >> 16;
		duh_file->position = 0;
		playback->playing = 1;

		if (playback->output->open_audio((duh_file->bits_per_sample == 16) ? FMT_S16_LE : FMT_U8, duh_file->samples_per_sec, duh_file->channels) == 0)
		{
			playback->error = TRUE;
			duh_end_sigrenderer(duh_file->file);
			unload_duh(duh_file->duh);
			g_free(duh_file);
			duh_file = NULL;
			return;
		}
		name = get_title(duh_file->duh, filename);
		rate = duh_file->samples_per_sec * duh_file->channels * (duh_file->bits_per_sample / 8);
		playback->set_params(playback, name, 1000LL * duh_file->length / rate, 8 * rate, duh_file->samples_per_sec, duh_file->channels);
		g_free(name);
		duh_file->seek_to = -1;
		decode_thread = g_thread_self();
		playback->set_pb_ready(playback);
		play_loop(playback);
	}
}

static void stop(InputPlayback *playback)
{
	if (duh_file && playback->playing)
	{
		playback->playing = 0;
		g_thread_join(decode_thread);
		playback->output->close_audio();
		g_free(duh_file);
		duh_file = NULL;
	}
}

static void duh_pause(InputPlayback *playback, short p)
{
	playback->output->pause(p);
}

static void mseek(InputPlayback *playback, gulong millisecond)
{
	duh_file->seek_to = millisecond;

	playback->eof = FALSE;

	while (duh_file->seek_to != -1)
		g_usleep(10000);
}

static void seek(InputPlayback *playback, int time)
{
	gulong millisecond = time * 1000;
	mseek(playback, millisecond);
}

static void file_info_box(char *filename)
{
	GtkWidget *songinfo;
	GtkStyle *style = NULL;
	GtkTextBuffer *TextBuffer;
	PangoFontDescription *font_desc = NULL;

	DUMB_IT_SIGDATA *sd;
	DUH *duh;

	if ( duh_universal_load_vfs(&duh, filename, NULL, UNIREAD_MOD) == -1 )
		return;
	
	sd = duh_get_it_sigdata(duh);

	songinfo = create_songinfo ();
	if (!songinfo) return;

	font_desc = pango_font_description_from_string ("monospace 9");

	if (font_desc) {
		style = gtk_style_copy(gtk_widget_get_style(songinfo));
		style->font_desc = font_desc;
	}
	
	{
		int n = dumb_it_sd_get_n_instruments(sd);
		if (n) {
			GtkCList *l = GTK_CLIST(lookup_widget(songinfo, "songinfo_instruments_clist"));
			int i;
			if (style) gtk_widget_set_style(GTK_WIDGET(l), style);
			gtk_clist_column_titles_passive(l);
			gtk_clist_set_column_justification(l, 0, GTK_JUSTIFY_RIGHT);
			for (i = 0; i < n; i++) {
				char str[3];
				const gchar *text[] = {
					str,
					(gchar*) dumb_it_sd_get_instrument_name(sd, i),
					(gchar*) dumb_it_sd_get_instrument_filename(sd, i)
				};
				sprintf(str, "%d", i+1);
				gtk_clist_insert(l, i, (gchar **)text);
			}
		} else
			gtk_notebook_remove_page(GTK_NOTEBOOK(lookup_widget(songinfo, "songinfo_notebook")), 2);
	}

	{
		GtkCList *l = GTK_CLIST(lookup_widget(songinfo, "songinfo_samples_clist"));
		int i, n = dumb_it_sd_get_n_samples(sd);
		if (style) gtk_widget_set_style(GTK_WIDGET(l), style);
		gtk_clist_column_titles_passive(l);
		gtk_clist_set_column_justification(l, 0, GTK_JUSTIFY_RIGHT);
		for (i = 0; i < n; i++) {
			char str[3];
			const gchar *text[] = {
				str,
				(gchar*) dumb_it_sd_get_sample_name(sd, i),
				(gchar*) dumb_it_sd_get_sample_filename(sd, i)
			};
			sprintf(str, "%d", i+1);
			gtk_clist_insert(l, i, (gchar **)text);
		}
	}

	{
		GtkTextView *l = GTK_TEXT_VIEW(lookup_widget(songinfo, "songinfo_message_text"));
		const char *message = (char*) dumb_it_sd_get_song_message(sd);
		if (message) {
			char *m = strdup(message);
			int i;
			for (i = 0; m[i]; i++)
				if (m[i] == '\r') m[i] = '\n';

			TextBuffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (lookup_widget(songinfo, "songinfo_message_text")));
			if (style) gtk_widget_set_style(GTK_WIDGET(l), style);
			gtk_text_buffer_set_text(TextBuffer, (gchar*) m, -1);

			free(m);
		} else
			gtk_notebook_remove_page(GTK_NOTEBOOK(lookup_widget(songinfo, "songinfo_notebook")), 0);
	}

	/* I hope that's ok here. Previously there was no unload_duh at all (leak?) */
	if (duh)
		unload_duh(duh);

	gtk_widget_show (songinfo);
}

static void duh_configure(void) {
	if (prefs) return;
	prefs = create_prefs();
	if (!prefs) return;

	gint output_frequency_idx = 4; /* 44100 default/fallback */
	int i;
	gchar freqstring[6];

	for (i = 0; i < (int) (sizeof(output_frequencies) / sizeof(output_frequencies[0])); i++) {
		snprintf(freqstring, sizeof(freqstring), "%d", output_frequencies[i]);
		gtk_combo_box_append_text (GTK_COMBO_BOX(lookup_widget(prefs, "prefs_freq_combobox")), freqstring);
		if (output_frequencies[i] == output_frequency)
			output_frequency_idx = i;
	}
	
 	gtk_combo_box_set_active(GTK_COMBO_BOX(lookup_widget(prefs, "prefs_freq_combobox")), output_frequency_idx);

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(lookup_widget(prefs, "prefs_checkbox_disableamiga")), disable_amiga_mods);

	gtk_signal_connect(GTK_OBJECT(prefs), "destroy",         
			   GTK_SIGNAL_FUNC(gtk_widget_destroyed), &prefs);

	gtk_signal_connect_object(GTK_OBJECT(lookup_widget(prefs, "prefs_button_cancel")), "clicked",
				  GTK_SIGNAL_FUNC(gtk_widget_destroy), 
				  GTK_OBJECT(prefs));

	gtk_signal_connect_object(GTK_OBJECT(lookup_widget(prefs, "prefs_button_ok")), "clicked",
				  GTK_SIGNAL_FUNC(duh_configure_ok_cb), 
				  NULL);

	gtk_widget_show(prefs);
}

static void duh_configure_ok_cb(GtkButton *w, gpointer data) {
	ConfigDb *db;
	gint output_frequency_idx;

	db = aud_cfg_db_open();

	disable_amiga_mods = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(prefs, "prefs_checkbox_disableamiga")));
	aud_cfg_db_set_bool(db, "DUMB", "disable_amiga_mods", disable_amiga_mods);

	output_frequency_idx = gtk_combo_box_get_active(GTK_COMBO_BOX(lookup_widget(prefs, "prefs_freq_combobox")));
	output_frequency = output_frequencies[output_frequency_idx];
	aud_cfg_db_set_int(db, "DUMB", "output_frequency", output_frequency);	

	aud_cfg_db_close(db);
	gtk_widget_destroy(prefs);
}

static void duh_about(void) {
	static GtkWidget *about;

	if (about) return;

	about = audacious_info_dialog("About DUMB Plugin",
				  "DUMB Input Plugin " VERSION "\n"
				  "by Christian Birchinger <joker@netswarm.net>\n"
				  "\n"
				  "Based on the original XMMS plugin by Ben Davis\n"
				  "and the BMP port by Michael Doering\n"
				  "\n"
				  "Built with DUMB version " DUMB_VERSION_STR,
				  "Ok", FALSE, NULL, NULL);

	g_signal_connect(G_OBJECT(about), "destroy",
			 G_CALLBACK(gtk_widget_destroyed), &about);
}
