# -*- mode: python; coding: utf-8 -*-

from __future__ import with_statement

import os, sys, string, stat

import glob
import subprocess as subp
import logging
import time
import select
import signal
import copy
import exceptions, traceback
import socket
import StringIO
from functools import partial

from atheist.const import *
from atheist.utils import *
import atheist.log
from atheist.gvar import Log

from pyarco.Type import Record, SortedDict, merge_uniq
from pyarco.Pattern import Singleton
from pyarco.Thread import ThreadFunc
from pyarco.UI import ellipsis, high, cout, ProgressBar
from pyarco.Conio import *


def check_type(val, cls):
    if not isinstance(val, cls):
        raise TypeError(("A %s is required, not %s." % \
              (cls.__name__, val.__class__.__name__)))

    return val


def pretty(value):
    RESULT_STR = {\
        FAIL:      [LIGHT_RED,        STR_STATUS[FAIL]],
        OK:        [GREEN,            STR_STATUS[OK]],
        NOEXEC:    [GREY,             STR_STATUS[NOEXEC]],
        ERROR:     [BOLD , LIGHT_RED, STR_STATUS[ERROR]],
        UNKNOWN:   [GREY,             STR_STATUS[UNKNOWN]],
        TODO:      [PURPLE,           STR_STATUS[TODO]],
        mild_FAIL: [RED,              STR_STATUS[mild_FAIL]]}

    return "[%s]" % cout(*RESULT_STR[value]+[NORM])


def file_template():
    # FIXME: Construir a partir de 'task_attrs'
    return '''# -*- mode:python; coding:utf-8 -*-

# Test(cmd,
#      cwd         = '.',
#      delay       = 0,
#      desc        = 'test template',
#      detach      = False,
#      env         = {},
#      expected    = 0,
#      must_fail   = False,
#      path        = '.',
#      template    = [],
#      timeout     = 5,
#      save_stderr = False,
#      save_stdout = False,
#      signal      = signal.SIGKILL)
'''


def exec_file(fname, _global=None, _local=None, template={}):

    def error():
        print_exc(traceback.format_exc(), Log)
        Log.error("Errors in the testcase definition '%s'" % fname)
        sys.exit(1)

    try:
        with file(fname) as fd:
            script = string.Template(fd.read()).safe_substitute(template)

        code = compile(script, fname, 'exec')
        exec(code) in _global, _local  # EXEC USER FILE

    except IOError, e:
        if e.errno == 2:
            Log.warning("No such file or directory: '%s'" % fname)
            return

        error()

    except Exception, e:
        error()

class Public:
    "Base class for classes available for the client tasks"

class Plugin:
    "Base class for plugins"

    @classmethod
    def add_options(cls, parser):
        pass

    @classmethod
    def config(cls, mng):
        pass


class Reporter:
    def __init__(self, mng, dst, handler):
        self.cfg = mng.cfg
        self.dst = dst
        self.width = 80
        self.logger = logging.getLogger(dst)
        self.logger.propagate = 0
        self.logger.setLevel(logging.INFO)
        self.logger.addHandler(handler)

    def do_render(self, mng):
        Log.debug("%s: sending report to '%s'" % \
                       (self.__class__.__name__, self.dst))
        report = self.render(mng)
        if report:
            self.logger.info(report)

    def render(self, mng):
        self.report = []
        if self.cfg.report_detail == 0: return ''
        for taskset in mng.itercases():
            self.build_taskcase(taskset)

        return str.join('\n', self.report)

    def pretty(self, value):
        return "[%s]" % STR_STATUS[value]

    def build_taskcase(self, taskcase):

#        TESTCASE_FORMAT = '{result} TaskCase: {path}'
#        self.report.append(
#            TESTCASE_FORMAT.format(
#                result = pretty(taskcase.result),
#                path   = compath(taskcase.fname)
#                ))

#        print taskcase.fname
        self.report.append("%s TaskCase: %s" %  (
                self.pretty(taskcase.result),
                compath(taskcase.fname)))

        if taskcase.result in [OK,NOEXEC] and self.cfg.report_detail < 2:
            return

        tree_draw(taskcase.tasks, cb=self.build_task)


    def build_task(self, task, conn, tree, level=0):

        #FIXME: retcode y cmd son específicos de Subprocess y deberían especializarse
        if not hasattr(task, 'retcode'):
            retcode = 'na'
        elif task.retcode is None:
            retcode = '-'
        else:
            retcode = int(task.retcode)

        cmd = task.cmd if hasattr(task, 'cmd') else ''

        if task.desc:
            cmd_desc = '%s | %s' % (ellipsis(cmd, self.width*0.25-(3*level), True),
                                    task.desc)
        else:
            cmd_desc = ellipsis(cmd, self.width*0.75)


#        TASK_SUMMARY = u"{result} {tree} {type}-{index:<3}{mode}({retcode:>2}:{expected:2})  {desc}"

#        self.report.append(
#            TASK_SUMMARY.format(
#                result   = self.pretty(task.result),
#                tree     = tree,
#                retcode  = retcode,
#                expected = task.expected,
#                mode     = task.get_mode(),
#                type     = task.acro,
#                index    = task.indx,
#                desc     = cmd_desc
#                ))

        self.report.append("%s %s %s-%-3s%s(%2s:%2s)  %s" % (
                self.pretty(task.result),
                tree,
                task.acro,
                task.indx,
                task.get_mode(),
                retcode,
                task.expected,
                cmd_desc))

        if (self.cfg.report_detail > 2 or task.result != OK) and \
                isinstance(task, CompositeTask):
            tree_draw(task.children, cb=self.build_task, conn=conn, args=[level+1])

        if not task.conditions_fail and (self.cfg.report_detail < 4):
            return

        tree_draw(task.pre, cb=self.build_condition, conn=conn, args=['pre', task])
        tree_draw(task.post, cb=self.build_condition, conn=conn, args=['post', task])


    def build_condition(self, condition, conn, tree, kind, task):

#        CONDITION_SUMMARY = u"{value} {tree} {kind:<4} {name} {info}"
#        self.report.append(
#            CONDITION_SUMMARY.format(
#                value  = self.pretty(condition.value),
#                tree   = tree,
#                kind   = 'pre: ' if condition in task.pre else 'post:',
#                name   = condition.name,
#                info   = condition.basic_info()
#                ))

        self.report.append("%s %s %s: %s %s" % (
                self.pretty(condition.value),
                tree,
                kind,
                condition.name,
                condition.basic_info()))



class FailReporter(Reporter):
    '''A reporter to use only on fails'''

    def do_render(self, mng):
        if not mng.ALL():
            Reporter.do_render(self, mng)

    def render(self, mng):
        retval = '''
## Atheist report ##\n
SOME TASKS FAILED!\n
user: %s
host: %s
date: %s
argv: %s
pwd: %s
\n''' % (os.getlogin(),
       socket.gethostname(),
       time.asctime(),
       str.join(' ', sys.argv),
       os.getcwd())

        retval += Reporter.render(self, mng)

        return retval


class ConsoleReporter(Reporter):
    def __init__(self, mng):
        Reporter.__init__(self, mng, 'console', logging.StreamHandler())
        self.width = mng.cfg.screen_width - 21

    def pretty(self, value):
        return pretty(value)


class TypedList(list):
    def __init__(self, cls, *values):
        self._cls = cls
        for i in values:
            self += i

    def append(self, val):
        Log.error("Use operator += instead")

    def __iadd__(self, val):
        if isinstance(val, list) or isinstance(val, tuple):
            for i in val: self += i
            return self

        check_type(val, self._cls)
        list.append(self, val)
        return self

    def copy(self):
        retval = TypedList(self._cls)
        retval.extend(self)
        return retval

    def prepend(self, val):
        self.insert(0, val)

    @classmethod
    def isA(cls, val, klass):
        "checks 'val' is a TypedList(cls)"
        assert isinstance(val, list), val
        for i in val:
            assert isinstance(i, klass)


class ConditionList(TypedList):
    def __init__(self, values=[]):
        TypedList.__init__(self, Condition)
        for i in values:
            self += i

    def __iadd__(self, val):
        if val in self:
            Log.debug("Trying to add a duplicate condition, ignored")
            return self
#        print "adding", id(val), val
        check_type(val, self._cls)
        return TypedList.__iadd__(self, val.clone())

    def __add__(self, other):
        retval = ConditionList()
        for i in self: retval += i
        for i in other: retval += i
        return retval

    def clone(self):
        return ConditionList(self)

    def remove_dups(self):
        return ConditionList(remove_dups(self))

    def __str__(self):
        retval = '<ConditionList \n['
        for x in self:
            retval += '%s:%s\n ' % (id(x), x)
        retval += ']>'
        return retval

class Condition(object):
    def __init__(self):
        self._value = NOEXEC
        self.task = None

    @property
    def value(self):
        return self._value

    def before(self, task, condlist, pos):
        """ Executed just BEFORE task execution """

        if self.task is not None and task is not self.task:
            Log.error("Conditions can not be shared: %s in %s, use clone()" % (self, self.task))
            sys.exit(1)

        self._value = NOEXEC
        self.task = task


    def after(self):
        """ Exectuted just AFTER task execution """

    def run(self):
        raise NotImplemented

    def evaluate(self):
        self._value = self.run()
        assert self._value is not None
        return self._value

    def clone(self):
        retval = copy.copy(self)
        retval._value = NOEXEC
        retval.task = None
        return retval

    def __eq__(self, other):
        return isinstance(other, self.__class__)

    def __repr__(self):
        return str(self)

    def __str__(self):
        return ("<%s %s %s>" % (pretty(self._value), self.name, self.basic_info())).encode('utf-8')

    @property
    def name(self):
        return self.__class__.__name__

    def basic_info(self):
        return ''

    def more_info(self):
        return ''


class Callback(Condition, Public):
    def __init__(self, func, args=()):
        self.func = func
        assert isinstance(args, tuple)
        self.args = args
        Condition.__init__(self)

    def run(self):
        try:
            retval = self.func(*self.args)
            if retval is None:
                Log.warning("%s() returns None" % self.func.__name__)
            return bool(retval)

        except Exception, e:
            Log.error("%s: %s" % (self.__class__.__name__, e))
            print_exc(traceback.format_exc(limit=1), Log)

        return False

    def __eq__(self, other):
        return Condition.__eq__(self, other) and \
            self.func == other.func and \
            self.args == other.args

    def basic_info(self):
        return "%s: '%s'" % (self.func.__name__, ellipsis(self.args))



class EnvVarDefined(Condition, Public):
    def __init__(self, varname, varvalue=None):
        Condition.__init__(self)
        self.varname = varname
        self.varvalue = varvalue

    def run(self):
        retval = self.varname in os.environ.keys()
        if retval and self.varvalue:
            retval = (os.environ[self.varname] == self.varvalue)
        return retval

    def __eq__(self, other):
        return Condition.__eq__(self, other) and \
            self.varname == other.varname and \
            self.varvalue == other.varvalue

    def basic_info(self):
        return "'%s'" % self.varname


class FileExists(Condition, Public):
    def __init__(self, fname):
        self.fname = fname
        Condition.__init__(self)

    def __eq__(self, other):
        return Condition.__eq__(self, other) and self.fname == other.fname

    def run(self):
        assert isinstance(self.fname, str), "Not a valid filename"
        return os.path.exists(self.fname)

    def basic_info(self):
        return "'%s'" % self.fname


class DirExists(FileExists, Public):
    def run(self):
        return FileExists.run(self) and os.path.isdir(self.fname)


class FileContains(Condition, Public):
    def __init__(self, data, fname=None,  strip='', whole=False, times=1):
        assert fname != '', "fname must be None or no-empty string"

        self.fname = fname
        self.times = check_type(times, int)
        self.whole = check_type(whole, bool)
        self.strip = check_type(strip, str)

        if whole:
            assert isinstance(data, str), "In whole-mode, data must be a string"

        self.contains = self.check_contains(data)
        Condition.__init__(self)


    def before(self, task, condlist, pos):
        task.log.debug('before %s' % self)

        Condition.before(self, task, condlist, pos)
        if self.fname == None:
            task.enable_outs(enable_stdout=True)
            self.fname = task.stdout
            return

        exists = FileExists(self.fname)

        if exists in condlist:
            return
#            raise Exception, "%s %s" % (len(condlist), condlist)

        condlist[pos:pos] = [exists]
        exists.before(task, condlist, pos)
#        print condlist
        return


        orig = condlist[:]
        del condlist[:]
#        condlist.extend(orig[:pos] + [exists] + orig[pos:])

        for c in orig[:pos] + [exists] + orig[pos:]:
            condlist += c


    def check_contains(self, data):
        if isinstance(data, str):
            return [data]
        elif isinstance(data, list):
            assert all([isinstance(x, str) for x in data]),\
                "FileContains args must be strings"
            return data

        raise exceptions.TypeError(data)


    def run(self):
        if self.fname is None:
            Log.critical("Condition is not initialized!")
            return ERROR

        if not os.path.exists(self.fname):
            Log.error("'%s' does not exists" % compath(self.fname))
            return ERROR

        try:
            fd = open(self.fname)
            fcontent = fd.read()
            fd.close()
        except IOError, e:
            Log.error(e)
            return ERROR

        if self.strip: fcontent = fcontent.strip(self.strip)

        if self.whole:
            return fcontent == self.contains[0]

        #FIXME: check contains order
        return all([fcontent.count(x) >= self.times for x in self.contains])


    def __eq__(self, other):
        return Condition.__eq__(self, other) and \
            self.fname == other.fname and \
            self.contains == other.contains and \
            self.times == other.times and \
            self.whole == other.whole and \
            self.strip == other.strip

    def basic_info(self):
        return "'%s' (%s) content:'%s'" % (self.fname, self.times,
                                         ellipsis(self.contains))


class FileEquals(Condition, Public):
    def __init__(self, fname1, fname2=None):
        self.fname1 = fname1
        self.fname2 = fname2
        Condition.__init__(self)

    def before(self, task, condlist, pos):
        Condition.before(self, task, condlist, pos)
        if self.fname2 == None:
            task.enable_outs(enable_stdout=True)
            self.fname2 = task.stdout

        # auto-prepend a FileExists condition
# FIXME
#        for fname in [self.fname1, self.fname2]:
#            condlist.auto += FileExists(fname)

    def __eq__(self, other):
        return Condition.__eq__(self, other) and \
            self.fname1 == other.fname1 and \
            self.fname2 == other.fname2

    def run(self):
        return not os.system('diff %s %s > /dev/null' % (self.fname1, self.fname2))

    def basic_info(self):
        return "'%s' == '%s'" % (self.fname1, self.fname2)


class FileIsNewerThan(Condition, Public):
    def __init__(self, fname1, fname2):
        self.fname1 = fname1
        self.fname2 = fname2
        Condition.__init__(self)

    def run(self):
        time1 = os.stat(self.fname1)[stat.ST_MTIME]
        time2 = os.stat(self.fname2)[stat.ST_MTIME]
        return int(time1 > time2);

    def __eq__(self, other):
        return Condition.__eq__(self, other) and \
            self.fname1 == other.fname1 and \
            self.fname2 == other.fname2

    def basic_info(self):
        return "'%s' is newer than '%s'" % (self.fname1, self.fname2)


class ProcessRunning(Condition, Public):
    def __init__(self, pid):
        assert isinstance(pid, int)
        self.pid = pid
        Condition.__init__(self)

    def run(self):
        return os.system("ps ef %s > /dev/null" % self.pid) == 0

    def __eq__(self, other):
        return Condition.__eq__(self, other) and \
            self.pid == other.pid

    def basic_info(self):
        return "'%s'" % self.pid


class AtheistVersion(Condition, Public):
    def __init__(self, version):
        self.version = self.str_to_float(version)
        Condition.__init__(self)

    def run(self):
        Log.debug("%s: inst:%s req:%s" %
                   (self.__class__.__name__, VERSION, self.version))
        return self.str_to_float(VERSION) >= self.version

    @staticmethod
    def str_to_float(string):
        retval = str.join('.', string.split('.')[:2])
        try:
            retval = float(retval)
        except ValueError, e:
            Log.error(e)
            sys.exit(1)

        return retval


#FIXME: Clonación de Condition decoradas
class ConditionDecorator(Condition, Public):
    def __init__(self, *conds):
        Condition.__init__(self)
        self.children = ConditionList()
        for c in conds:
            check_type(c, Condition)
            self.children += c

    def clone(self):
        retval = Condition.clone(self)
        retval.children = self.children.clone()
        return retval

    def before(self, task, condlist, pos):
        for c in self.children:
            c.before(task, condlist, pos)
        Condition.before(self, task, condlist, pos)

    def run(self):
        raise NotImplemented

    def __eq__(self, other):
        return Condition.__eq__(self, other) and set(self.children) == set(other.children)

    @property
    def name(self):
        return "%s (%s" % (self.__class__.__name__,
                          str.join(' ', [x.name for x in self.children]))

    def basic_info(self):
        return str.join(' ', [x.basic_info() for x in self.children])

    def more_info(self):
        return str.join(' ', [x.more_info() for x in self.children])


class Not(ConditionDecorator, Public):
    def __init__(self, condition):
        ConditionDecorator.__init__(self, condition)

    def run(self):
        val = self.children[0].run()
        if isinstance(val, bool):
            return not(val)

        return ERROR



# # FIXME:test this
# class Or(ConditionDecorator, Public):
#    def run(self):
#        return any([c.run() for c in self.conds])


class Poll(ConditionDecorator, Public):
    def __init__(self, condition, interval=1, timeout=5):
        ConditionDecorator.__init__(self, condition)
        self.interval = interval
        self.timeout = timeout

    def run(self):
        enlapsed = 0
        while 1:
#            if self.task.mng.abort: break
            Log.debug("Polling condition: %s" % self.children[0])
            if self.children[0].run(): return True
            if self.timeout is not None and enlapsed > self.timeout: break
            time.sleep(self.interval)
            enlapsed += self.interval

        return False


class RemoteTestFactory(Public):
    def __init__(self, account):
        self.account = account
        try:
            self.user, self.host = self.account.split('@')
        except ValueError:
            self.host = self.account
            self.user = os.getlogin()

    def __call__(self, cmd, **values):
        values.update({'shell':True})
        return Test('ssh %s "%s"' % (self.account, cmd), **values)



class BufFile:

    def __init__(self, fd):
        self.data = ''
        self.fileno = fd.fileno()

    def readline(self):
        new = os.read(self.fileno, 2048)
        if not new:
            yield self.data
            raise StopIteration

        self.data += new
        if not '\n' in self.data:
            yield ''
            raise StopIteration

        for line in self.data.split()[:-1]:
            yield line



class FileWrapper:

    def __init__(self, console_out, file_handler, tag, only_on_fail=False):

        def file_write_tag(text, fd=sys.__stdout__):
            # FIXME: imprime como lineas textos que no acaban en \n.
            # Necesitamos un readline()
            for line in text.split('\n'):
                if not line: continue
                fd.write("%s| %s\n" % (tag, line))
                fd.flush()

        def file_write(text, fd):
            fd.write(text)
            fd.flush()

        #--

        self.out_funcs = []
        self.last_task_out = None

        if console_out:
            self.out_funcs.append(file_write_tag)

        elif only_on_fail:
            self.last_task_out = StringIO.StringIO()
            self.out_funcs.append(partial(file_write_tag, fd=self.last_task_out))

        if file_handler:
            self.out_funcs.append(partial(file_write, fd=file_handler))

    def write(self, data):
        for func in self.out_funcs: func(data)

    def flush(self):
        pass

    def print_fail_out(self):
        if not self.last_task_out:
            return # it was not asked

        print self.last_task_out.getvalue(),
        sys.__stdout__.flush()


class Task(Record):

    acro = 'Task'
    allows = ['tid', 'cwd', 'delay', 'desc', 'dirty', 'expected', 'loglevel', 'must_fail', 'template',
              'save_stdout', 'save_stderr', 'stderr', 'stdout', 'todo']
    forbid = []

    @classmethod
    def set_mng(cls, mng):
        cls.mng = mng

    @classmethod
    def validate_kargs(cls, _dict):
        for k,v in _dict.items():
            assert task_attrs.has_key(k), "'%s' is not a valid keyword" % k
            assert k in cls.allows, \
            "'%s' is not an allowed keyword for '%s'" % (k, cls.__name__)

            check_type(v, task_attrs[k].type_)

#            if v == task_attrs[k].default:
#                Log.info("Default value for '%s' is already '%s'. Remove it." % (k, v))

        return _dict


    def __init__(self, **kargs):
        #current_ids = [x.id for x in TC] + [x.id for x in ts]  # All
        current_ids = [x.tid for x in self.mng.ts]    # Only this testcase

        assert 'tid' not in kargs.keys() or \
            kargs['tid'] not in current_ids, "Duplicate task id: '%s'" % kargs['tid']

        try:
            self.ts = kargs['parent'].children
        except KeyError:
            self.ts = self.mng.ts

        self.ts.append(self)
        self.kargs = kargs

        defaults = {}
        for k,v in task_attrs.items(): defaults[k] = v.default
        self.__dict__.update(defaults)

        if kargs.has_key('template'):
            check_type(kargs['template'], list)
            for i in kargs['template']:
                self.__dict__.update(self.validate_kargs(i))

        self.__dict__.update(self.validate_kargs(kargs))

        self._indx = self.mng.next()
        self.fname = ''
        self.gen = TypedList(str)
        self.pre =  ConditionList()
        self.post = ConditionList()

        self._mode = None               # normal, setup, teardown
        self.result = NOEXEC
        self.conditions_fail = False

        self.thread = None
        self.time_start = None
        self.time_end = None

#        self.already_setup = False
        self.stderr_fd = None
        self.stdout_fd = None
        self.last_task_out = None

        self.save_stdout |= self.stdout != ''
        self.save_stderr |= self.stderr != ''

        self.enable_outs(self.save_stdout, self.save_stderr)

        self.wrap_outs = False

        #- logger
        self.log = logging.getLogger(self.name)
        if self.log.handlers:
            return # the logger already exists

        term = logging.StreamHandler()
        term.setFormatter(log.XXLoggingFormatter(\
                self.mng.cfg.timetag + '[%(levelinitial)s] %(name)s: %(message)s',
                log.DATEFORMAT))

        if self.loglevel is not None:
            loglevel = self.loglevel
        else:
            loglevel = self.mng.cfg.loglevel

        term.setLevel(loglevel)
        self.log.setLevel(loglevel)
        self.log.addHandler(term)
        self.log.propagate = 0


    def get_mode(self):
        return self._mode

    def set_mode(self, mode):
        self._mode = mode


    def enable_outs(self, enable_stdout=False, enable_stderr=False):

        def enable_out(task, name):
            assert name in ['stdout', 'stderr'], "Internal Error"

            out = getattr(task, name)
            if not out:
                out = '/tmp/atheist/%s_%s.%s' % (task.name, os.getpid(), name[-3:])
                setattr(task, name, out)

            setattr(task, "save_%s" % name, True)
            task.gen += out

        if enable_stdout: enable_out(self, 'stdout')
        if enable_stderr: enable_out(self, 'stderr')


    # FIXME: Debería dejar la Task en el estado inicial, necesario para --until-failure
    def setup(self):
        self.log.debug('setup')

        self.result = UNKNOWN

        for i,c in enumerate(self.pre[:]):
            c.before(self, self.pre, i)

        for i,c in enumerate(self.post[:]):
#            print i,c
            c.before(self, self.post, i)

        self.pre = self.pre.remove_dups()
        self.post = self.post.remove_dups()

        self.gen = remove_dups(self.gen)

        for i in self.gen:
            self.pre.prepend(Not(FileExists(i)))
            self.post.prepend(FileExists(i))

        if self.pre or self.post:
            self.result = OK


    @property
    def indx(self):
        return self._indx

    @property
    def description(self):
        return  "%s:%s" % (self.name, self.desc)

    @property
    def name(self):
        return self.initial + str(self.indx)

    @property
    def initial(self):
        retval = self.__class__.__name__[0]
        return retval


    def describe(self, pre_keys=[]):
        self.setup()

        dic = self.__dict__.copy()
        dic['fname'] = compath(dic['fname'])

        keys = pre_keys[:] + ['fname']
        if self.result is not NOEXEC: keys.append('result')

        # add standard attrs with no-default value
        for k,v in task_attrs.items():
            if getattr(self,k) != v.default:
                keys.append(k)

        attrs = []
        for i in keys:
            value = dic[i]
            if isinstance(value, str):
                value = "'%s'" % value.replace('\n', '\n' + ' '*16)

            attrs.append("%s: %s" % (i.rjust(14), value))

        # multivalued attributes
        for a in ['pre','post','gen']:
            for i in getattr(self, a):
                attrs.append("%s %s" % ('%s:' % a.rjust(14), i))

        return '%3s:%s\n%s\n' % (self.indx, self.__class__.__name__,
                                 str.join('\n', attrs))

    def __repr__(self):
        return str(self)

    def __str__(self):
        return "%s: <%s(%s)>" % \
            (high(self.name), self.__class__.__name__, self.str_param())

    def str_param(self):
        return ''

    def clone(self, *args):
        kargs = self.kargs
        kargs['desc'] = '(copy of %s) %s' % (self.name, kargs['desc'])

        retval = self.__class__(*args, **kargs)

        for c in self.pre:
            retval.pre += c
        for c in self.post:
            retval.post += c
        retval.gen = self.gen.copy()
        return retval


    def do_exec_task(self):
        # key-val sanity
        if self.dirty and not self.gen:
            self.log.warning("'dirty' without 'gen' has no sense.")

        self.outwrap = FileWrapper(self.mng.cfg.stdout,
                                   self.stdout_fd,
                                   tag='%s:out' % self.name,
                                   only_on_fail=self.mng.cfg.stdout_on_fail)

        self.errwrap = FileWrapper(self.mng.cfg.stderr,
                                   self.stderr_fd,
                                   tag='%s:err' % self.name)


        try:
            if self.wrap_outs:
                sys.stdout = self.outwrap
                sys.stderr = self.errwrap

            self.exec_task()

        finally:
            sys.stdout = sys.__stdout__
            sys.stderr = sys.__stderr__


        if self.result != OK:
            self.outwrap.print_fail_out()

        if self.result == FAIL and self.todo:
            self.result = TODO

        if self.result == FAIL and not self.check:
            self.result = mild_FAIL

        self.outwrap = None
        self.errwrap = None

    def is_running(self):
        raise NotImplementedError, self

    def exec_task(self):
        raise NotImplementedError, self

    def terminate(self):
        raise NotImplementedError, self

    def close(self):
        '''Close file descriptors: stdin, stdout, etc.'''


class Subprocess(Task):

    allows = Task.allows + ['detach', 'env', 'path', 'timeout', 'shell', 'signal']

    def __init__(self, cmd, **kargs):
        self.cmd = '' if cmd is None else cmd
        assert isinstance(self.cmd, (str, types.NoneType))

        Task.__init__(self, **kargs)

        # add the user variables to dft environ
        environ = os.environ.copy()
        for k,v in self.env.items():
            environ[k] = string.Template(v).safe_substitute(environ)

        self.env = environ
        self.ps = None
        self.retcode = None

    def exec_task(self):
        if not self.cmd: return

        if self.path:
            os.environ['PATH'] += ':'+self.path

        if self.shell:
            cmd = ['/bin/bash', '-c', "%s" % self.cmd]
        else:
            if set('><|').intersection(set(self.cmd)):
                self.log.warning("The command '%s' has special characters." % self.cmd)
                self.log.warning("Perhaps it require a shell...")

            cmd = self.cmd.split()

        try:
            self.ps = subp.Popen(
                cmd,
                close_fds  = True,
                stdin      = subp.PIPE,
                stdout     = subp.PIPE,
                stderr     = subp.PIPE,
                shell      = False,
                bufsize    = 0,
                cwd        = self.cwd,
                env        = self.env,
                preexec_fn = os.setsid)
        except OSError, e:
            self.log.error("%s: '%s'" % (e.strerror, self.cmd))
            self.result = ERROR
            return

        self.log.debug("starts (pid: %s)" % self.ps.pid)

        read_fds = [self.ps.stdout, self.ps.stderr]
        read_ready = select.select(read_fds,[],[], 0.05)[0]

        while self.ps.poll() is None or read_fds:
            for fd in read_ready:
# http://mail.python.org/pipermail/python-bugs-list/2001-March/004491.html
                time.sleep(0.01)
                data = os.read(fd.fileno(), 2048)
                if not data:
                    read_fds.remove(fd)
                elif fd is self.ps.stdout:
                    self.outwrap.write(data)
                elif fd is self.ps.stderr:
                    self.errwrap.write(data)

            try:
                read_ready = select.select(read_fds,[],[], 0.2)[0]
            except select.error:
                self.mng.abort()
                self.result = ERROR
                return

            if self.timeout == 0 or time.time()-self.tini < self.timeout:
                continue

            self.log.debug("timeout expired (%.1f>%s), sending signal %s to %s" % \
                               (time.time() - self.tini, self.timeout,
                                self.signal, self.ps.pid))
            self.kill()


        self.outwrap.write(self.ps.stdout.read())
        self.errwrap.write(self.ps.stderr.read())
        self.retcode = self.ps.returncode

        if self.shell and self.retcode == 127:
            self.log.error("No such file or directory: '%s'" % self.cmd)
            self.result = ERROR
            return

        self.result = (self.retcode == self.expected)

        if self.must_fail:
            self.result = self.retcode != 0

        self.log.debug("%s finish with %s" % (pretty(self.result), self.retcode))

    def is_running(self):
        return self.ps is not None and self.ps.poll() is None

    def str_param(self):
        return "'%s'" % self.cmd.split('\n')[0]

    def describe(self):
        return Task.describe(self, ['cmd'])

    def clone(self):
        return Task.clone(self, self.cmd)

    def terminate(self):
        self.kill(self.signal)

    def kill(self, n=None, assure=True):
        if n is None:
            n = self.signal

        self.log.debug("sending signal %s" % n)

        try:
            os.killpg(self.ps.pid, n)
        except OSError, e:
            if e.errno == 3: return # OSError(3, 'No such process')
            Log.debug("%s (%s)" % (e, self.ps.pid))
            return
        except AttributeError:
            self.log.warning("did not even starts")
            return

        if not assure:
            return

        tini = time.time()
        while 1:
            if self.ps.poll() is not None: break
            time.sleep(0.5)

            if (time.time() - tini) > 5:
                self.log.warning('is still alive after 5 seconds!')
                self.kill(signal.SIGKILL, assure=False)

            if (time.time() - tini) > 3:
                self.kill(n, assure=False)



    def close(self):
        if not self.ps: return
        self.ps.stdin.close()
        self.ps.stdout.close()
        self.ps.stderr.close()



class Test(Subprocess, Public):
    acro = 'Test'


class TestFunc(Task, Public):
    acro = 'Func'
    allows = ['delay', 'desc', 'expected', 'must_fail', 'template', 'tid']

    def __init__(self, func, args=(), **kargs):
        assert callable(func)
        self.func = func
        self.args = args

        if not kargs.has_key('desc'):
            kargs['desc'] = "%s(%s)" % (func.__name__, str(self.args))

        Task.__init__(self, **kargs)


    def exec_task(self):
        sys.stdout = self.outwrap
        sys.stderr = self.errwrap

        try:
            self.retcode = self.func(*self.args)

            if not isinstance(self.retcode, int):
                Log.error("Function '%s' should return an integer" % self.func.__name__)

            if self.retcode is None:
                self.result = UNKNOWN
            else:
                self.result = (self.retcode == self.expected)

        except Exception, e:
            self.log.error(e)
            self.result = ERROR

        finally:
            sys.stdout = sys.__stdout__
            sys.stderr = sys.__stderr__


        if self.must_fail and self.result == FAIL:
            self.result = OK


class Command(Subprocess, Public):
    acro = 'Cmnd'

    def __init__(self, cmd, **kargs):
        Subprocess.__init__(self, cmd, **kargs)
        self.check = False


class CompositeTask(Task, Public):
    acro = 'Comp'
    allows = ['detach']

    """ The CompositeTask let you specify a predicate to evalue a set of tasks:

    CompositeTask(all, t1, t2, t3) is OK if all the tasks are OK.

    #--

    def at_least_two(tasks):
        return [bool(x) for x in tasks].count(True) >= 2

    CompositeTask(at_least_two, t1, t2, t3) is OK when 2 or more tasks are OK.
    """

    def __init__(self, oper, *tasks, **kargs):
        self.oper = oper
        self.children = TypedList(Task, tasks)
        for t in self.children:
            assert t.detach == False, 'CompositeTask children must not detachable'

        Task.__init__(self, **kargs)

        for t in tasks:
            t.parent = self
            self.ts.remove(t)

        self.desc = "%s %s" % (oper.__name__, [x.name for x in tasks])

    def set_mode(self, mode):
        self._mode = mode
        for t in self.children:
            t.set_mode(mode)

    def exec_task(self):
        self.gen =  merge_uniq(*[t.gen for t in self.children])

        for t in self.children:
            Log.info(t)
            run_task(t)

        self.result = self.oper([t.result is OK for t in self.children if t.check])
        self.log.debug('%s finish' % pretty(self.result))

    def terminate(self):
        self.log.debug('Terminating child tasks')
        for t in self.children:
            t.terminate()

    def is_running(self):
        return any(x.is_running() for x in self.children)

    def str_param(self):
        return str([x.name for x in self.children])


task_attrs = SortedDict({
    'tid':         Record(type_=str,  default=None),
    'check':       Record(type_=bool, default=True),
    'cwd':         Record(type_=str,  default=None),
    'delay':       Record(type_=int,  default=0),
    'desc':        Record(type_=str,  default=''),
    'detach':      Record(type_=bool, default=False),
    'dirty':       Record(type_=bool, default=False),
    'env':         Record(type_=dict, default=os.environ.copy()),
    'expected':    Record(type_=int,  default=0),
    'loglevel':    Record(type_=int,  default=None),
    'must_fail':   Record(type_=bool, default=False),
    'parent':      Record(type_=CompositeTask, default=None),
    'path':        Record(type_=str,  default=''),
    'template':    Record(type_=list, default=[]),
    'timeout':     Record(type_=int,  default=5),
    'save_stderr': Record(type_=bool, default=False),
    'save_stdout': Record(type_=bool, default=False),
    'shell':       Record(type_=bool, default=False),
    'signal':      Record(type_=int,  default=signal.SIGKILL),
    'stderr':      Record(type_=str,  default=''),
    'stdout':      Record(type_=str,  default=''),
    'todo':        Record(type_=bool, default=False),
    })



class Daemon(Command):
    acro = 'Daem'
    forbid = ['detach', 'timeout']

    def __init__(self, cmd, **kargs):
        Command.__init__(self, cmd, **kargs)
        self.timeout = 0
        self.detach = True
        self.check = False

class Template(dict, Public):
    def __init__(self, **kargs):
        self.update(kargs)


def run_task(task, keep_going=False, recurse=False, end_callback=lambda:None):
    if task.detach and recurse==False:
        task.log.debug("detaching")
        task.thread = ThreadFunc(run_task, (task,),
                                 {'keep_going':keep_going, 'recurse':True,
                                  'end_callback':end_callback})
        time.sleep(0.0)
        return None

    task.setup()

    # pre-conditions
    for c in task.pre:
        value = c.evaluate()
        task.log.info("Pre:  %s" % c)
        if value != OK:
            task.conditions_fail = True
            task.result = FAIL
            if not keep_going:
                return False

    if not task.conditions_fail:
        if task.stdout:
            task.stdout_fd = open(task.stdout, 'w')

        if task.stderr:
            task.stderr_fd = open(task.stderr, 'w')

        time.sleep(task.delay)

        task.tini = time.time()
        try:
            task.do_exec_task()
        except select.error:
            task.mng.abort()

        if task.stdout_fd:
            task.stdout_fd.close()

        if task.stderr_fd:
            task.stderr_fd.close()

        task.close()

    #FIXME: no debería ejecutar las post-condiciones si el comando falla

    # post-conditions
    for c in task.post:
        c.after()
        value = c.evaluate()
        task.log.info("Post: %s" % c)
        if value != OK:
            task.conditions_fail = True
            task.result = False
            if not keep_going:
                return False

    end_callback()

    return task.result


class TaskCase(object):
    def __init__(self, fname, mng, template={}):
        fname = os.path.abspath(fname)
        self.fname = fname
        self.mng = mng
        self.ts = mng.ts

        aux_fname = os.path.splitext(compath(fname))[0]

        self.template = template
        self.testdir =  os.path.abspath(os.path.dirname(fname))
        self.template.update({
                'basedir':  compath(),
                'fullbasedir':  os.path.abspath(compath()),
                'testdir':  compath(self.testdir),
                'fulltestdir':  os.path.abspath(compath(self.testdir)),
                'dirname':  " ERROR: ['$dirname' is deprecated: use '$testdir' instead] ",
                'fname':    aux_fname,
                'testname': os.path.basename(aux_fname),
                'atheist':  './athcmd.py'})

        del self.ts[:]

        if not self.mng.cfg.skip_hooks:
            setup = os.path.join(self.testdir, SETUP)
            if os.path.exists(setup):
                self.process_file(setup, task_mode.SETUP)

        self.process_file(fname, task_mode.MAIN)

        if not self.mng.cfg.skip_hooks:
            teardown = os.path.join(self.testdir, TEARDOWN)
            if os.path.exists(teardown):
                self.process_file(teardown, task_mode.TEARDOWN)

#        for t in self.ts:
#            for i in t.gen:
#                t.pre.prepend(Not(FileExists(i)))
#                t.post.prepend(FileExists(i))

#            if isinstance(t, CompositeTask):
#                for j in t.tasks:
#                    self.ts.remove(j)

        self.tasks = self.ts[:]
        self.result = NOEXEC

    def __str__(self):
        return "<TaskCase %s:%s>" % (self.fname, self.result)

    def __repr__(self):
        return str(self)

    def process_file(self, fname, mode):
        Log.debug("%s loading" % compath(fname))

        before = self.ts[:]
        env = self.mng.exec_env.copy()
        env['ts'] = self.ts
        env['__file__'] = os.path.abspath(fname)

        exec_file(fname, env, None, self.template)

        for t in set(self.ts) - set(before):
            t.fname = fname
            t.set_mode(mode)


    def run(self, ob=lambda:None):

        def make_cleaning():
            generated = []

            for t in self.tasks:
                if t.result == NOEXEC: continue
                if self.mng.cfg.clean and t.dirty:
                    t.log.warning("dirty-task, not removing gen: %s" % t.gen)
                else:
                    generated.extend(t.gen)

            if not generated:
                return

            if self.mng.cfg.clean:
                remove_generated(generated)
            else:
                self.save_dirty_filelist(generated)


        Log.info("Test case %s" % (compath(self.fname)+' ').ljust(80, '-'))

        if not self.tasks:
            return

        for t in self.tasks:
            if self.mng.aborted: break

            Log.info(t)
            result = run_task(t,
                              keep_going=self.mng.cfg.keep_going,
                              end_callback=ob)

            if result == False and t.check == True \
                    and not self.mng.cfg.keep_going:
                t.log.info("FAIL, skipping remaining tasks ('keep-going mode' disabled)")
                break


        while 1:
            unfinished = [x for x in self.tasks if x.thread \
                              and x.thread.isAlive() \
                              and x.timeout != 0]

            if not unfinished: break

            for t in unfinished:
                t.log.debug("waiting to finish detached task")

                if self.mng.aborted:
                    t.terminate()

                t.thread.join(1)


        daemons = [x for x in self.tasks if x.thread \
                       and x.thread.isAlive() \
                       and x.timeout == 0]
        if daemons:
            Log.debug("-- Killing remaining daemons...")
            for t in daemons:
                t.terminate()
                t.thread.join(1)

        self.result = all([x.result==OK for x in self.tasks if x.check])

        make_cleaning()


    def save_dirty_filelist(self, filelist):
        assert filelist

        gen_fname = os.path.basename(self.fname).replace('.test', '').replace('.', '').strip('_')
        gen_fname = os.path.join(ATHEIST_TMP, '%s_%s.gen' % (gen_fname, os.getpid()))
        filelist = remove_dups(filelist + [gen_fname])

        gen_fd = open(gen_fname, 'w')
        Log.debug("Storing generated files in %s:\n%s" % (gen_fname, filelist))

#        gen_fd.writelines([x+'\n' for x in filelist])
        for fname in filelist:
            Log.warning("dirty-mode: not removing '%s'" % fname)
            gen_fd.write(fname + '\n')

        gen_fd.flush()
        gen_fd.close()



#    def status(self):
#        return all([t.result for t in self.tasks if t.check])


class task_mode:
    MAIN,SETUP,TEARDOWN = ['-', 's', 't']


def remove_generated(filelist=None):

    def get_old_generated():
        retval = []
        for fname in glob.glob('/tmp/atheist/*.gen'):
            Log.debug("reading generated files from '%s'" % fname)

            try:
                with file(fname) as fd:
                    retval.extend([x.strip('\n') for x in fd.readlines()])
            except IOError, e:
                Log.error(e)
                continue

        return retval


    if filelist is None: # borrar todo lo generado por atheist
        filelist = get_old_generated()

    basedir = compath()

    for f in filelist:
        if not (f.startswith('/tmp') or f.startswith(basedir)):
            Log.warning('Removing files out of /tmp or %s is forbbiden: %s' % (basedir, f))
            continue

        if os.path.isdir(f):
            Log.warning("- removing directory '%s'" % compath(f))
            os.system('rm -r %s' % f)

        elif os.path.isfile(f):
            Log.debug("- removing file '%s'" % compath(f))
            try:
                os.remove(f)
            except OSError, e:
                Log.debug(e)
        else:
            Log.warning("%s is not a file nor directory!" % compath(f))


class Runner(object):
    def __init__(self, mng):
        self.mng = mng
        self.pb = ProgressBar(len(mng),
                              width=int(mng.cfg.screen_width),
                              label='Run ',
                              disable= mng.cfg.disable_bar or \
                                  mng.cfg.verbosity or \
                                  mng.cfg.quiet or \
                                  mng.cfg.stdout or \
                                  mng.cfg.stderr)

        self.pb.listen_logger(Log, mng.cfg.loglevel)

    def run(self):
        self.pb.reset()
        self.mng.run(self.pb.inc)
        self.pb.clean()

    def summary(self):
        for r in self.mng.reporters:
            r.do_render(self.mng)

        if self.mng.ntests:
            print(self.mng.str_stats())
